# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac

class Sff8024Values:
   identifiers = { 0x00: "unknown or unspecified",
                   0x01: "GBIC",
                   0x02: "module/connector soldered to motherboard "
                         "(using SFF-8472)",
                   0x03: "SFP/SFP+/SFP28 and later",
                   0x04: "300 pin XBI",
                   0x05: "XENPAK",
                   0x06: "XFP",
                   0x07: "XFF",
                   0x08: "XFP-E",
                   0x09: "XPAK",
                   0x0a: "X2",
                   0x0b: "DWDM-SFP/SFP+ (not using SFF-8472)",
                   0x0c: "QSFP (INF-8438)",
                   0x0d: "QSFP+ or later "
                         "(SFF-8436, SFF-8635, SFF-8665, SFF-8685 et al)",
                   0x0e: "CXP or later",
                   0x0f: "shielded mini multilane HD 4X",
                   0x10: "shielded mini multilane HD 8X",
                   0x11: "QSFP28 or later (SFF-8665 et al)",
                   0x12: "CXP2 (aka CXP28) or later",
                   0x13: "CDFP (Style 1/Style2)",
                   0x14: "shielded mini multilane HD 4X fanout cable",
                   0x15: "shielded mini multilane HD 8X fanout cable",
                   0x16: "CDFP (Style 3)",
                   0x17: "microQSFP",
                   0x18: "QSFP-DD Double Density 8X Pluggable Transceiver "
                         "(INF-8628)",
                   0x19: "OSFP 8X Pluggable Transceiver",
                   0x1A: "SFP-DD Double Density 2X Pluggable Transceiver",
                   0x1B: "DSFP Dual Small Form Factor Pluggable Transceiver",
                   0x1C: "x4 MiniLink/OcuLink",
                   0x1D: "x8 MiniLink",
                   0x1E: "QSFP+ or later with CMIS" }

   encodings = { 0x00: "Unspecified",
                 0x01: "8B/10B",
                 0x02: "4B/5B",
                 0x03: "NRZ",
                 0x07: "256B/257B (transcoded FEC-enabled data)",
                 0x08: "PAM4" }

   connectors = { 0x00: "unknown or unspecified",
                  0x01: "SC (Subscriber Connector)",
                  0x02: "fibre channel style 1 copper connector",
                  0x03: "fibre channel style 2 copper connector",
                  0x04: "BNC/TNC (Bayonet/Threaded Neill-Concelman)",
                  0x05: "fibre channel coax headers",
                  0x06: "fiber jack",
                  0x07: "LC (Lucent Connector)",
                  0x08: "MT-RJ (Mechanical Transfer - Registered Jack)",
                  0x09: "MU (Multiple Optical)",
                  0x0a: "SG",
                  0x0b: "Optical Pigtail",
                  0x0c: "MPO 1x12 (Multifiber Parallel Optic)",
                  0x0d: "MPO 2x16",
                  0x20: "HSSDC II (High Speed Serial Data Connector)",
                  0x21: "copper pigtail",
                  0x22: "RJ45 (Registered Jack)",
                  0x23: "no separable connector",
                  0x24: "MXC 2x16",
                  0x25: "CS optical connector",
                  0x26: "SN (previously Mini CS) optical connector",
                  0x27: "MPO 2x12",
                  0x28: "MPO 1x16" }

   complianceExtended = { 0x00: "unspecified",
                          0x01: "100G AOC (Active Optical Cable) or 25GAUI C2M AOC",
                          0x02: "100GBASE-SR4 or 25GBASE-SR",
                          0x03: "100GBASE-LR4 or 25GBASE-LR",
                          0x04: "100GBASE-ER4 or 25GBASE-ER",
                          0x05: "100GBASE-SR10",
                          0x06: "100G CWDM4",
                          0x07: "100G PSM4 parallel SMF",
                          0x08: "100G ACC (Active Copper Cable) or 25GAUI C2M ACC",
                          0x09: "obsolete",
                          0x0B: "100GBASE-CR4 or 25GBASE-CR CA-L",
                          0x0C: "25GBASE-CR CA-S",
                          0x0D: "25GBASE-CR CA-N",
                          0x10: "40GBASE-ER4",
                          0x11: "4 x 10GBASE-SR",
                          0x12: "40G PSM4 parallel SMF",
                          0x13: "G959.1 profile P1I1-2D1",
                          0x14: "G959.1 profile P1S1-2D2",
                          0x15: "G959.1 profile P1L1-2D2",
                          0x16: "10GBASE-T with SFI electrical interface",
                          0x17: "100G CLR4",
                          0x18: "100G AOC or 25GAUI C2M AOC",
                          0x19: "100G ACC or 25GAUI C2M ACC",
                          0x1A: "100GE-DWDM2",
                          0x1B: "100G 1550nm WDM (4 wavelengths)",
                          0x1C: "10GBASE-T short reach (30 meters)",
                          0x1D: "5GBASE-T",
                          0x1E: "2.5GBASE-T",
                          0x1F: "40G SWDM4",
                          0x20: "100G SWDM4",
                          0x21: "100G PAM4 BiDi",
                          0x22: "4WDM-10 MSA",
                          0x23: "4WDM-20 MSA",
                          0x24: "4WDM-40 MSA",
                          0x25: "100GBASE-DR",
                          0x26: "100G-FR or 100GBASE-FR1, CAUI-4 (no FEC)",
                          0x27: "100G-LR or 100GBASE-LR1, CAUI-4 (no FEC)",
                          0x2A: "100G-FR or 100GBASE-FR1, CAUI-4 (RS-FEC)",
                          0x2B: "100G-LR or 100GBASE-LR1, CAUI-4 (RS-FEC)",
                          0x2D: "100G-ER1-30 MSA, CAUI-4 (no FEC)",
                          0x2E: "100G-ER1-40 MSA, CAUI-4 (no FEC)",
                          0x34: "100G-ER1-30 MSA (RS-FEC)",
                          0x35: "100G-ER1-40 MSA (RS-FEC)",
                          0x30: "Active Copper Cable with 50GAUI, 100GAUI-2 or "
                                "200GAUI-4 C2M",
                          0x31: "Active Optical Cable with 50GAUI, 100GAUI-2 or "
                                "200GAUI-4 C2M",
                          0x32: "Active Copper Cable with 50GAUI, 100GAUI-2 or "
                                "200GAUI-4 C2M",
                          0x33: "Active Optical Cable with 50GAUI, 100GAUI-2 or "
                                "200GAUI-4 C2M",
                          0x40: "50GBASE-CR, 100GBASE-CR2, or 200GBASE-CR4",
                          0x41: "50GBASE-SR, 100GBASE-SR2, or 200GBASE-SR4",
                          0x42: "50GBASE-FR or 200GBASE-DR4",
                          0x43: "200GBASE-FR4",
                          0x44: "200G 1550 nm PSM4",
                          0x45: "50GBASE-LR",
                          0x46: "200GBASE-LR4",
                          0x50: "64GFC EA",
                          0x51: "64GFC SW",
                          0x52: "64GFC LW",
                          0x53: "128GFC EA",
                          0x54: "128GFC SW",
                          0x55: "128GFC LW" }

class Sff8472Values:
   identifiers = Sff8024Values.identifiers

   extIdentifiers = { 0x00: "not specified",
                      0x01: "GBIC is compliant with MOD_DEF 1",
                      0x02: "GBIC is compliant with MOD_DEF 2",
                      0x03: "GBIC is compliant with MOD_DEF 3",
                      0x04: "GBIC/SFP function is defined by two-wire "
                            "interface ID only",
                      0x05: "GBIC is compliant with MOD_DEF 5",
                      0x06: "GBIC is compliant with MOD_DEF 6",
                      0x07: "GBIC is compliant with MOD_DEF 7" }

   encodings = Sff8024Values.encodings.copy()
   encodings.update( { 0x04: "Manchester",
                       0x05: "SONET scrambled",
                       0x06: "64B/66B" } )

   connectors = Sff8024Values.connectors

   compliance10G = [ "10GBASE-SR", "10GBASE-LR", "10GBASE-LRM", "10GBASE-ER" ]

   complianceInfini = [ "1X Copper Passive", "1X Copper Active", "1X LX", "1X SX" ]

   complianceEth = [ "1000BASE-SX", "1000BASE-LX", "1000BASE-CX", "1000BASE-T",
                     "100BASE-LX/LX10", "100BASE-FX", "BASE-BX10", "BASE-PX" ]

   fibreLength = [ "medium distance (M)", "long distance (L)",
                   "intermediate distance (I)", "short distance (S)",
                   "very long distance (V)" ]

   fibreTech = [ "Longwave laser (LL)", "Shortwave laser with OFC (SL)",
                 "Shortwave laser w/o OFC (SN)", "Electrical intra-enclosure (EL)",
                 "Electrical inter-enclosure (EL)", "Longwave laser (LC)",
                 "Shortwave laser, linear Rx (SA)" ]

   sfpTech = [ "Unallocated", "Unallocated", "Passive Cable", "Active Cable" ]

   fibreMedia = [ "Single Mode (SM)", "Unallocated", "Multimode, 50um (M5, M5E)",
                  "Multimode, 62.5um (M6)", "Video Coax (TV)", "Miniture Coax (MI)",
                  "Twisted Pair (TP)", "Twin Axial Pair (TW)" ]

   fibreSpeed = [ "100 MBytes/sec", "Extended", "200 MBytes/sec",
                  "3200 MBytes/sec", "400 MBytes/sec", "1600 MBytes/sec",
                  "800 MBytes/sec", "1200 MBytes/sec" ]

   complianceExtended = Sff8024Values.complianceExtended

   revisions = { 0x00: "not specified",
                 0x01: "SFF-8472 rev 9.3",
                 0x02: "SFF-8472 rev 9.5",
                 0x03: "SFF-8472 rev 10.2",
                 0x04: "SFF-8472 rev 10.4",
                 0x05: "SFF-8472 rev 11.0",
                 0x06: "SFF-8472 rev 11.3",
                 0x07: "SFF-8472 rev 11.4",
                 0x08: "SFF-8472 rev 12.3" }

class Sff8636Values:
   identifiers = Sff8024Values.identifiers

   encodings = Sff8024Values.encodings.copy()
   encodings.update( { 0x04: "SONET scrambled",
                       0x05: "64B/66B",
                       0x06: "Manchester" } )

   connectors = Sff8024Values.connectors

   complianceEth = [ "40G active cable", "40GBASE-LR4", "40GBASE-SR4", "40GBASE-CR4",
                     "10GBASE-SR", "10GBASE-LR", "10GBASE-LRM", "Extended" ]

   complianceGig = [ "1000BASE-SX", "1000BASE-LX", "1000BASE-CX", "1000BASE-T",
                     "Reserved", "Reserved", "Reserved", "Reserved" ]

   fibreLength = Sff8472Values.fibreLength

   fibreTech = ( [ "Reserved", "Reserved", "Reserved", "Reserved" ] +
                 Sff8472Values.fibreTech[ : -1 ] +
                 [ "Reserved" ] )

   fibreMedia = Sff8472Values.fibreMedia[ : ]
   fibreMedia[ 1 ] = "Multi-mode 50um (OM3)"

   fibreSpeed = Sff8472Values.fibreSpeed

   complianceExtended = Sff8024Values.complianceExtended

   transmitters = [ "850nm VCSEL",
                    "1310nm VCSEL",
                    "1550nm VCSEL",
                    "1310nm FP",
                    "1310nm DFB",
                    "1550nm DFB",
                    "1310nm EML",
                    "1550nm EML",
                    "Other / Undefined",
                    "1490 nm DFB",
                    "Copper cable unequalized",
                    "Copper cable passive equalized",
                    "Copper cable, near and far end limiting active equalizers",
                    "Copper cable, far end limiting active equalizers",
                    "Copper cable, near end limiting active equalizers",
                    "Copper cable, linear active equalizers" ]

   rateSelectVer = [ 'reserved', 'version 1', 'version 2', 'reserved' ]

   revisions = { 0x00: "not specified",
                 0x01: "SFF-8436 rev 4.8 or earlier",
                 0x02: "SFF-8436 rev 4.8 or earlier, bytes 186-189 from SFF-8636",
                 0x03: "SFF-8636 rev 1.3 or earlier",
                 0x04: "SFF-8636 rev 1.4",
                 0x05: "SFF-8636 rev 1.5",
                 0x06: "SFF-8636 rev 2.0",
                 0x07: "SFF-8636 rev 2.5, 2.6, and 2.7",
                 0x08: "SFF-8636 rev 2.8, 2.9 and 2.10" }

   emphasisType = { 0x00: "peak-to-peak amplitude stays constant, \
or no information stays constant, or no info, or not implemented",
                    0x01: "steady state amplitude stays constant",
                    0x02: "average of peak-to-peak and steady state \
amplitudes stay constant" }

class Cmis8x16xValues:
   identifiers = Sff8024Values.identifiers

   twiSpeed = { 0x0: "400 kHz",
                0x1: "1 MHz" }

   moduleEncodings = { 0x00: "Undefined",
                       0x01: "Optical Interfaces: MMF",
                       0x02: "Optical Interfaces: SMF",
                       0x03: "Passive Cu",
                       0x04: "Active Cables",
                       0x05: "BASE-T" }
   moduleEncodings.update( ( i, "Custom" ) for i in range( 0x40, 0x90 ) )

   apselLaneAssignments = list( map( str, range( 1, 9 ) ) )

   connectors = Sff8024Values.connectors

   mediaTech = { 0x00: "850nm VCSEL",
                 0x01: "1310nm VCSEL",
                 0x02: "1550nm VCSEL",
                 0x03: "1310nm FP",
                 0x04: "1310nm DFB",
                 0x05: "1550nm DFB",
                 0x06: "1310nm EML",
                 0x07: "1550nm EML",
                 0x08: "Others",
                 0x09: "1490 nm DFB",
                 0x0A: "Copper cable unequalized",
                 0x0B: "Copper cable passive equalized",
                 0x0C: "Copper cable, near and far end limiting active equalizers",
                 0x0D: "Copper cable, far end limiting active equalizers",
                 0x0E: "Copper cable, near end limiting active equalizers",
                 0x0F: "Copper cable, linear active equalizers" }

   implementedBanks = [ "0", "0 and 1", "reserved", "reserved" ]

   stateDuration = [ "Maximum state duration is less than 1 ms",
                     "1 ms <= maximum state duration < 5 ms",
                     "5 ms <= maximum state duration < 10 ms",
                     "10 ms <= maximum state duration < 50 ms",
                     "50 ms <= maximum state duration < 100 ms",
                     "100 ms <= maximum state duration < 500 ms",
                     "500 ms <= maximum state duration < 1 s",
                     "1 s <= maximum state duration < 5 s",
                     "5 s <= maximum state duration < 10 s",
                     "10 s <= maximum state duration < 1 min",
                     "1 min <= maximum state duration < 5 min",
                     "5 min <= maximum state duration < 10 min",
                     "10 min <= maximum state duration < 50 min",
                     "Maximum state duration >= 50 min",
                     "Reserved",
                     "Reserved" ]

   # TODO: field name: "Tx clock recovery groups"
   txClockRecovery = [ "lanes 1-8",
                       "lanes 1-4, 5-8",
                       "lanes 1-2, 3-4, 5-6, 7-8",
                       "lanes 1, 2, 3, 4, 5, 6, 7, 8" ]

class CoherentCmisValues:
   # These parameters are listed in EnhancedDomParamType. We are leaving out
   # the parameters that are not defined by VDM
   vdmParamTypeDecode = {
         # CMIS 4 - table 8-99
         b'laserAge': 'Laser age',
         b'tecCurrent': 'TEC current',
         b'laserFreqErr': 'Laser frequency error',
         b'laserTemp': 'Laser temperature',
         b'snr': 'eSNR media input',
         b'snrHost': 'eSNR host input',
         b'pam4Transitions': 'PAM4 level transition parameter media input',
         b'pam4TransitionsHost': 'PAM4 level transition parameter host input',
         b'preFecBERMin': 'Pre-FEC BER minimum media input',
         b'preFecBERMinHost': 'Pre-FEC BER minimum host input',
         b'preFecBERMax': 'Pre-FEC BER maximum media input',
         b'preFecBERMaxHost': 'Pre-FEC BER maximum host input',
         b'preFecBERAvg': 'Pre-FEC BER average media input',
         b'preFecBERAvgHost': 'Pre-FEC BER average host input',
         b'preFecBERCurr': 'Pre-FEC BER current value media input',
         b'preFecBERCurrHost': 'Pre-FEC BER current value host input',
         b'errFramesMin': 'Errored frames minimum media input',
         b'errFramesMinHost': 'Errored frames minimum host input',
         b'errFramesMax': 'Errored frames maximum media input',
         b'errFramesMaxHost': 'Errored frames maximum host input',
         b'errFramesAvg': 'Errored frames average media input',
         b'errFramesAvgHost': 'Errored frames average host input',
         b'errFramesCur': 'Errored frames current value media input',
         b'errFramesCurHost': 'Errored frames current value host input',
         # C-CMIS - table 7
         b'txModBiasXIMonitor': 'Modulator bias X/I',
         b'txModBiasXQMonitor': 'Modulator bias X/Q',
         b'txModBiasYIMonitor': 'Modulator bias Y/I',
         b'txModBiasYQMonitor': 'Modulator bias Y/Q',
         b'txModBiasXPhaseMonitor': 'Modulator bias X_phase',
         b'txModBiasYPhaseMonitor': 'Modulator bias Y_phase',
         b'cdHighShortLink': 'CD - high granularity, short link',
         b'cdLowLongLink': 'CD - low granularity, long link',
         b'dgd': 'DGD',
         b'sopmd': 'SOPMD',
         b'pdl': 'PDL',
         b'osnr': 'OSNR',
         b'esnr': 'eSNR',
         b'cfo': 'CFO',
         b'evm': 'EVM_modem',
         b'txPower': 'Tx power',
         b'rxTotalPower': 'Rx total power',
         b'rxSignalPower': 'Rx signal power',
         b'sopRoc': 'SOP ROC',
         b'mer': 'MER',
         b'clkRecoveryLoop': 'Clock recovery loop',
         b'sopmdLg': 'SOPMD low granularity',
         b'snrMargin': 'SNR margin',
         b'qFactor': 'Q-factor',
         b'qMargin': 'Q-margin'
         }

def _pgA0h( *byteList ):
   return [ ( 0xA0, byteList ) ]

def _pg00h( *byteList ):
   return [ ( 0x00, byteList ) ]

def _pg01h( *byteList ):
   return [ ( 0x01, byteList ) ]

def _pg03h( *byteList ):
   return [ ( 0x03, byteList ) ]

def _pg04h( *byteList ):
   return [ ( 0x04, byteList ) ]

def _pg13h( *byteList ):
   return [ ( 0x13, byteList ) ]

def _pg20h( *byteList ):
   return [ ( 0x20, byteList ) ]

def _pg21h( *byteList ):
   return [ ( 0x21, byteList ) ]

def _pg22h( *byteList ):
   return [ ( 0x22, byteList ) ]

def _pg23h( *byteList ):
   return [ ( 0x23, byteList ) ]

def _pg2Fh( *byteList ):
   return [ ( 0x2F, byteList ) ]

def _pg40h( *byteList ):
   return [ ( 0x40, byteList ) ]

# Get a decode function to map a bitfield to a list
def _mapBitField( fieldVals ):
   return lambda x: ",".join( fieldVals[ i ] for i in range( 0, x.bit_length() )
                              if x & ( 1 << i ) ) or "None"
# Get a decode function to grab value from a dictionary and fall back to a string

def _dictGet( dictionary, string ):
   return lambda x: dictionary.get( x, string )
# Get a decode function to multiply a field by a value

def _mult( coeff ):
   return lambda x: coeff * int( x )
# Get a decode function to divide a field by a value

def _div( divisor ):
   return lambda x: int( x ) // divisor
# A decode function to determine if average power or OMA

def _avgOrOma( x ):
   return "average power" if x else "OMA"

def farDec( x ):
   return x or "Unknown"

def nbool( x ):
   return not x

def decodeQsfp( pgData, model ):
   decode = Sff8636Values

   # Identifier
   micVal = pgData.moduleInfoContents
   iden = model.addField( "Identifier", _pg00h( 0, 128 ) )
   iden.addField( "Lower", _pg00h( 0 ), raw=micVal.identifier0,
                   decode=_dictGet( decode.identifiers, "Reserved" ) )
   iden.addField( "Upper", _pg00h( 128 ), raw=pgData.identifier,
                   decode=_dictGet( decode.identifiers, "Reserved" ) )

   # Status fields
   stat = model.addField( "Status fields", _pg00h( 1, 2 ) )
   revDec = _dictGet( decode.revisions, "Unknown revision (0x%X)" % micVal.revision )
   rev = stat.addField( "Revision compliance", _pg00h( 1 ), raw=micVal.revision,
                        decode=revDec )
   rev.note = "parsed via SFF-8636 rev 2.10"
   stat.addField( "Flat mem", _pg00h( 2 ), raw=micVal.flatMem, decode=bool )

   # Far end
   farVal = pgData.cableConfiguration.farEnd
   far = model.addField( "Far end implementation", _pg00h( 113 ), raw=farVal.reg )
   far.addField( "Num ends", _pg00h( 113 ), raw=farVal.numEnds, decode=farDec )
   far.addField( "Num channels per end", _pg00h( 113 ), raw=farVal.numChannelsPerEnd,
                 decode=farDec )

   # Near end
   nearVal = pgData.cableConfiguration.nearEnd
   near = model.addField( "Near end implementation", _pg00h( 113 ), raw=nearVal.reg )
   near.addField( "Channel 1", _pg00h( 113 ), raw=nearVal.channel1, decode=nbool )
   near.addField( "Channel 2", _pg00h( 113 ), raw=nearVal.channel2, decode=nbool )
   near.addField( "Channel 3", _pg00h( 113 ), raw=nearVal.channel3, decode=nbool )
   near.addField( "Channel 4", _pg00h( 113 ), raw=nearVal.channel4, decode=nbool )

   # Extended identifier
   extIdVal = Tac.newInstance( "Xcvr::Sff8436ExtendedIdentifier",
                               pgData.identifierExtended )
   extId = model.addField( "Extended identifier", _pg00h( 129 ) )
   extId.addField( "Power Class", _pg00h( 129 ), raw=extIdVal.powerClass,
                   decode=lambda x: x + 1 )
   extId.addField( "Power Class 8 implemented", _pg00h( 129 ),
                   raw=extIdVal.powerClass8Implemented, decode=bool )
   extId.addField( "CLEI in Page 02h", _pg00h( 129 ), raw=extIdVal.cleiCode,
                   decode=bool )
   extId.addField( "CDR present in TX", _pg00h( 129 ), raw=extIdVal.cdrPresentInTx,
                   decode=bool )
   extId.addField( "CDR present in RX", _pg00h( 129 ), raw=extIdVal.cdrPresentInRx,
                   decode=bool )
   extId.addField( "High Power Class", _pg00h( 129 ), raw=extIdVal.highPowerClass,
                   decode=lambda x: x + 4 if x else "unused" )

   # Max Power Dissipation
   maxPwrDis = model.addField( "Max Power Dissipation", _pg00h( 107 ),
                               raw=pgData.powerClass8Data,
                               decode=_div( 10.0 ), units="W" )
   if not extIdVal.powerClass8Implemented:
      maxPwrDis.note = "ignore, see Power Class 8 implemented"

   # Connector
   model.addField( "Connector", _pg00h( 130 ), raw=pgData.connector,
                   decode=_dictGet( decode.connectors, "Reserved" ) )

   # Transceiver compliance
   compVal = Tac.newInstance( "Xcvr::Sff8436TransceiverCompliance",
                              pgData.compliance )
   # Bit shift left by 72 since need 8 * 9 bits of space
   compRaw = ( pgData.secondaryComplianceExtended << 72 )
   compRaw += ( pgData.compliance << 8 ) + pgData.complianceExtended
   # Compliance has 9 subsections and is placed after register 00h:113 in the eeprom
   # Since secondary compliance is located at register 00h:116.
   comp = model.addField( "Compliance",
                          _pg00h( 116, list( range( 131, 139 ) ), 192 ),
                          raw=compRaw, _numBytes=10 )
   comp.addField( "Secondary Compliance", _pg00h( 116 ),
                  raw=pgData.secondaryComplianceExtended,
                  decode=_dictGet( decode.complianceExtended, "Reserved" ) )
   comp.addField( "Ethernet", _pg00h( 131 ), raw=compVal.ethernet,
                  decode=_mapBitField( decode.complianceEth ),
                  possibleValues=decode.complianceEth )
   comp.addField( "Gigabit", _pg00h( 134 ), raw=compVal.gigabit,
                  decode=_mapBitField( decode.complianceGig ),
                  possibleValues=decode.complianceGig )
   fibre = comp.addField( "Fibre Channel", _pg00h( list( range( 135, 139 ) ) ) )
   fibre.addField( "Link length", _pg00h( 135 ), raw=compVal.fibreLength,
                   decode=_mapBitField( decode.fibreLength ),
                   possibleValues=decode.fibreLength )
   fibre.addField( "Transmitter technology", _pg00h( 135, 136 ),
                   raw=compVal.fibreTech, decode=_mapBitField( decode.fibreTech ) )
   fibre.addField( "Transmission media", _pg00h( 137 ), raw=compVal.fibreMedia,
                   decode=_mapBitField( decode.fibreMedia ),
                   possibleValues=decode.fibreMedia )
   fibre.addField( "Speed", _pg00h( 138 ), raw=compVal.fibreSpeed,
                   decode=_mapBitField( decode.fibreSpeed ),
                   possibleValues=decode.fibreSpeed )
   comp.addField( "Extended", _pg00h( 192 ), raw=pgData.complianceExtended,
                  decode=_dictGet( decode.complianceExtended, "Reserved" ) )

   # Encoding
   model.addField( "Encoding", _pg00h( 139 ), raw=pgData.encoding,
                     decode=_dictGet( decode.encodings, "Reserved" ) )

   # Bitrate
   bit = model.addField( "Bitrate", _pg00h( 140, 222 ) )
   nomBit = bit.addField( "Nominal", _pg00h( 140 ), raw=pgData.bitrateNominal,
                          decode=_mult( 100 ), units="Mbps" )
   extBit = bit.addField( "Extended", _pg00h( 222 ), raw=pgData.bitrateExtended,
                          decode=_mult( 250 ), units="Mbps" )
   if pgData.higherBitrate:
      nomBit.note = "ignore, see extended bitrate"
   else:
      extBit.note = "ignore, see nominal bitrate"

   # Length
   length = model.addField( "Length", _pg00h( list( range( 142, 147 ) ) ) )
   length.addField( "SMF", _pg00h( 142 ), raw=pgData.lengthSmf,
                    decode=int, units="km" )
   length.addField( "OM3", _pg00h( 143 ), raw=pgData.lengthOm3,
                    decode=_mult( 2 ), units="m" )
   length.addField( "OM2", _pg00h( 144 ), raw=pgData.lengthOm2,
                    decode=int, units="m" )
   om1Len = length.addField( "OM1", _pg00h( 145 ), raw=pgData.lengthOm1,
                             decode=int, units="m" )
   om4Len = length.addField( "OM4", _pg00h( 146 ), raw=pgData.lengthCableOrOm4,
                             decode=_mult( 2 ), units="m" )
   cabLen = length.addField( "Cable assembly", _pg00h( 146 ),
                             raw=pgData.lengthCableOrOm4, decode=int, units="m" )
   devTechVal = Tac.newInstance( "Xcvr::Sff8436DeviceTech", pgData.deviceTech )
   if devTechVal.transmitterType in [ 0b1010, 0b1011, 0b1101, 0b1111 ]:
      om1Len.note = "ignore, see copper cable attenuation"
   if pgData.connector != 0x23 and devTechVal.transmitterType == 0:
      cabLen.note = "ignore, type is separable 850nm VCSEL"
   else:
      om4Len.note = "ignore, type is not separable 850nm VCSEL"

   # Copper attenuation
   copAttnVal = Tac.newInstance( "Xcvr::Sff8436CableAttn", pgData.cableAttn )
   copAttn = model.addField( "Copper cable attenuation",
                             _pg00h( 145, list( range( 186, 190 ) ) ) )
   attn25p78 = copAttn.addField( "25.78 GHz", _pg00h( 145 ), raw=pgData.lengthOm1,
                                 decode=int, units="dB" )
   copAttn.addField( "2.5 GHz", _pg00h( 186 ), raw=copAttnVal.attn2p5Ghz,
                     decode=int, units="dB" )
   copAttn.addField( "5.0 GHz", _pg00h( 187 ), raw=copAttnVal.attn5p0Ghz,
                     decode=int, units="dB" )
   copAttn.addField( "7.0 GHz", _pg00h( 188 ), raw=copAttnVal.attn7p0Ghz,
                     decode=int, units="dB" )
   copAttn.addField( "12.9 GHz", _pg00h( 189 ), raw=copAttnVal.attn12p9Ghz,
                     decode=int, units="dB" )
   if devTechVal.transmitterType < 0b1010:
      copAttn.note = "ignore, optical variant detected"
   elif devTechVal.transmitterType not in [ 0b1010, 0b1011, 0b1101, 0b1111 ]:
      attn25p78.note = "ignore, see OM1 length"

   # Device tech
   devTech = model.addField( "Device Tech", _pg00h( 147 ) )
   devTech.addField( "Transmitter Type", _pg00h( 147 ),
                     raw=devTechVal.transmitterType,
                     decode=lambda x: decode.transmitters[ x ] )
   devTech.addField( "Active wavelength control", _pg00h( 147 ),
                     raw=devTechVal.activeWavelengthControl, decode=bool )
   devTech.addField( "Cooled Transmitter", _pg00h( 147 ),
                     raw=devTechVal.cooledTransmitter, decode=bool )
   devTech.addField( "Detector Type", _pg00h( 147 ),
                     raw=devTechVal.detectorType,
                     decode=lambda x: "APD" if x else "Pin" )
   devTech.addField( "Tunable Transmitter", _pg00h( 147 ),
                     raw=devTechVal.cooledTransmitter, decode=bool )

   # Wavelength
   wave = model.addField( "Wavelength", _pg00h( 186, 187 ),
                          raw=pgData.laserWavelength, decode=_div( 1000.0 ),
                          units="nm" )
   wave.addField( "Tolerance", _pg00h( 188, 189 ), raw=pgData.wavelengthTolerance,
                  decode=_div( 200.0 ), units="nm", prefix="+/-" )
   if devTechVal.transmitterType >= 0b1010:
      wave.note = "ignore, non-optical variant detected"

   # Case Temp
   model.addField( "Max case temp", _pg00h( 190 ), raw=pgData.maxCaseTemp,
                     decode=int, units="C" )

   # Options
   opsVal = Tac.newInstance( "Xcvr::Sff8436Options", pgData.options )
   ops = model.addField( "Options", _pg00h( 193, 194, 195 ), raw=pgData.options,
                         _numBytes=3 )
   ops.addField( "Tx input adaptive equalizer freeze capable",
                 _pg00h( 193 ), decode=bool,
                 raw=opsVal.txInputAdaptiveEqualizerFreezeCapable )
   ops.addField( "Tx input equalization auto adaptive capable",
                 _pg00h( 193 ), decode=bool,
                 raw=opsVal.txInputEqualizationAutoAdaptiveCapable )
   ops.addField( "Tx input equalization fixed programmable settings implemented",
                 _pg00h( 193 ), decode=bool,
                 raw=opsVal.txInputEqualizationFixedProgrammableSettingsImplemented )
   ops.addField( "Rx output emphasis fixed programmable settings implemented",
                 _pg00h( 193 ), decode=bool,
                 raw=opsVal.rxOutputEmphasisFixedProgrammableSettingsImplemented )
   ops.addField( "Rx output amplitude fixed programmable settings implemented",
                 _pg00h( 193 ), decode=bool,
                 raw=opsVal.rxOutputAmplitudeFixedProgrammableSettingsImplemented )
   ops.addField( "Tx CDR on/off control implemented", _pg00h( 194 ),
                 raw=opsVal.txCdrOnOffControlImplemented, decode=bool )
   ops.addField( "Rx CDR on/off control implemented", _pg00h( 194 ),
                 raw=opsVal.rxCdrOnOffControlImplemented, decode=bool )
   ops.addField( "Tx CDR LOL flag implemented", _pg00h( 194 ),
                 raw=opsVal.txCdrLolFlagImplemented, decode=bool )
   ops.addField( "Rx CDR LOL flag implemented", _pg00h( 194 ),
                 raw=opsVal.rxCdrLolFlagImplemented, decode=bool )
   ops.addField( "Rx squelch disable implemented", _pg00h( 194 ),
                 raw=opsVal.rxSquelchDisableImplemented, decode=bool )
   ops.addField( "Rx output disable implemented", _pg00h( 194 ),
                 raw=opsVal.rxOutputDisableImplemented, decode=bool )
   ops.addField( "Tx squelch disable implemented", _pg00h( 194 ),
                 raw=opsVal.txSquelchDisableImplemented, decode=bool )
   ops.addField( "Tx squelch implemented", _pg00h( 194 ),
                 raw=opsVal.txSquelchImplemented, decode=bool )
   ops.addField( "Memory page 02h provided", _pg00h( 195 ),
                 raw=opsVal.memoryPage02hProvided, decode=bool )
   ops.addField( "Memory page 01h provided", _pg00h( 195 ),
                 raw=opsVal.memoryPage01hProvided, decode=bool )
   ops.addField( "RATE_SELECT implemented", _pg00h( 195 ),
                 raw=opsVal.rateSelectImplemented, decode=bool )
   ops.addField( "TX_DISABLE implemented", _pg00h( 195 ),
                 raw=opsVal.txDisableImplemented, decode=bool )
   ops.addField( "TX_FAULT implemented", _pg00h( 195 ),
                 raw=opsVal.txFaultImplemented, decode=bool )
   ops.addField( "Tx squelch implemented to reduce", _pg00h( 195 ),
                 raw=opsVal.txSquelchImplementedToReduce, decode=_avgOrOma )
   ops.addField( "Tx LOS implemented", _pg00h( 195 ),
                 raw=opsVal.txLosImplemented, decode=bool )
   ops.addField( "Pages 20-21h implemented", _pg00h( 195 ),
                 raw=opsVal.pages20h21hImplemented, decode=bool )

   # Date
   dateRaw = pgData.dateCode.encode().hex()
   date = model.addField( "Date code", _pg00h( list( range( 212, 220 ) ) ),
                          raw=int( dateRaw, base=16 ) if dateRaw else 0,
                          decode=lambda x: '"%s"' % pgData.dateCode )
   date.note = "YYMMDDLL"

   # DOM
   domVal = Tac.newInstance( "Xcvr::Sff8436DomType", pgData.diagsMonitoringType )
   dom = model.addField( "DOM type", _pg00h( 220 ) )
   dom.addField( "Temperature monitoring implemented", _pg00h( 220 ),
                 raw=domVal.temperatureMonitoringImplemented, decode=bool )
   dom.addField( "Supply voltage monitoring implemented", _pg00h( 220 ),
                 raw=domVal.supplyVoltageMonitoringImplemented, decode=bool )
   dom.addField( "Rx power measurements type", _pg00h( 220 ),
                 raw=domVal.rxPowerMeasurementsType, decode=_avgOrOma )
   dom.addField( "Tx power measurement supported", _pg00h( 220 ),
                 raw=domVal.txPowerMeasurementSupported, decode=bool )

   # Extended Options
   ehOpsVal = pgData.enhancedOptions
   ehOps = model.addField( "Enhanced Options", _pg00h( 221 ) )
   ehOps.addField( "Initialization complete flag implemented", _pg00h( 221 ),
                   raw=ehOpsVal.initCompleteImplemented, decode=bool )
   ehOps.addField( "Rate select declaration", _pg00h( 221 ),
                   raw=ehOpsVal.rateSelectDeclaration, decode=bool )
   ehOps.addField( "Application select table declaration", _pg00h( 221 ),
                   raw=ehOpsVal.applicationSelectTableDeclaration, decode=bool )
   ehOps.addField( "TC readiness flag implemented", _pg00h( 221 ),
                   raw=ehOpsVal.tcReadinessFlagImplemented, decode=bool )

   # Rate Select
   rateSel = model.addField( "Extended rate select", _pg00h( 141, 194, 195, 221 ) )
   rateSel.addField( "Rate select compliance", _pg00h( 141 ),
                     raw=pgData.extendedRateSelectCompliance & 0x03,
                     decode=lambda x: decode.rateSelectVer[ x ] )
   rateSel.addField( "Tx CDR on/off control implemented", _pg00h( 194 ),
                     raw=opsVal.txCdrOnOffControlImplemented, decode=bool )
   rateSel.addField( "Rx CDR on/off control implemented", _pg00h( 194 ),
                     raw=opsVal.rxCdrOnOffControlImplemented, decode=bool )
   rateSel.addField( "RATE_SELECT implemented", _pg00h( 195 ),
                     raw=opsVal.rateSelectImplemented, decode=bool )
   rateSel.addField( "Rate select declaration", _pg00h( 221 ),
                     raw=ehOpsVal.rateSelectDeclaration, decode=bool )

   # Checksums
   check = model.addField( "Checksums", _pg00h( list( range( 128, 224 ) ) ) )
   check.addField( "Base", _pg00h( 191 ),
                   raw=pgData.baseChecksum[ "page00Upper" ] )
   check.addField( "Base computed", _pg00h( list( range( 128, 191 ) ) ),
                   raw=pgData.baseChecksumCalculated[ "page00Upper" ] )
   check.addField( "Extended", _pg00h( 223 ),
                   raw=pgData.extendedChecksum )
   check.addField( "Extended computed", _pg00h( list( range( 192, 223 ) ) ),
                   raw=pgData.extendedChecksumCalculated )

   # Equalizer, Emphasis, Amplitude and Timing
   if micVal.flatMem:
      # Only want to include page 03 if the module has it
      return
   maxTxInEqualizer = 10
   maxRxOutEmphasis = 7
   txRxEmph = model.addField( "Tx Eq & Rx Emphasis Magnitude ID", _pg03h( 224 ) )
   txRxEmph.addField( "Max Tx input equalization", _pg03h( 224 ),
                      raw=pgData.maxTxInputEqualization,
                      decode=lambda x: str( x ) + " dB"
                      if x <= maxTxInEqualizer else "Reserved" )
   txRxEmph.addField( "Max Rx output emphasis supported", _pg03h( 224 ),
                      raw=pgData.maxRxOutputEmphasis,
                      decode=lambda x: str( x ) + " dB"
                      if x <= maxRxOutEmphasis else "Reserved" )
   rxOutAmpVal = Tac.newInstance( "Xcvr::Sff8436AmplitudeSupport",
                                  pgData.amplitudeSupport )
   rxOutAmp = model.addField( "Rx output amplitude support indicators",
                              _pg03h( 225 ) )
   rxOutAmp.addField( "Rx output amplitude 0 support", _pg03h( 225 ),
                     raw=rxOutAmpVal.rxOutputAmpSupport0,
                     decode=bool )
   rxOutAmp.addField( "Rx output amplitude 1 support", _pg03h( 225 ),
                     raw=rxOutAmpVal.rxOutputAmpSupport1,
                     decode=bool )
   rxOutAmp.addField( "Rx output amplitude 2 support", _pg03h( 225 ),
                     raw=rxOutAmpVal.rxOutputAmpSupport2,
                     decode=bool )
   rxOutAmp.addField( "Rx output amplitude 3 support", _pg03h( 225 ),
                     raw=rxOutAmpVal.rxOutputAmpSupport3,
                     decode=bool )
   rxOutAmp.addField( "Rx output emphasis type", _pg03h( 225 ),
                     raw=pgData.rxOutputEmphasisType,
                     decode=_dictGet( decode.emphasisType, "Reserved" ) )
   adverSupport = model.addField( "TxDis & RxLOSL fast mode, Tx force squelch,"
                                  " & controllable FEC support", _pg03h( 227 ) )
   adverSupport.addField( "TxDis fast mode support", _pg03h( 227 ),
                         raw=pgData.txDisFastMode, decode=bool )
   adverSupport.addField( "RxLOSL fast mode support", _pg03h( 227 ),
                         raw=pgData.rxLoslFastMode, decode=bool )
   adverSupport.addField( "Tx force squelch implemented", _pg03h( 227 ),
                         raw=pgData.txForceSquelch, decode=bool )
   adverSupport.addField( "Controllable media-side FEC support", _pg03h( 227 ),
                         raw=pgData.controlMediaSideFEC, decode=bool )
   adverSupport.addField( "Controllable host-side FEC support", _pg03h( 227 ),
                         raw=pgData.controlHostSideFEC, decode=bool )
   model.addField( "Maximum TC stabilization time", _pg03h( 228 ),
                  raw=pgData.maxTcStabilizeTime, decode=int, units="s" )
   model.addField( "Maximum CTLE settling time", _pg03h( 229 ),
                  raw=pgData.maxCtleSettleTime, decode=_mult( 100 ), units="ms" )

def decodeSfp( pgData, model ):
   decode = Sff8472Values

   # Identifier
   model.addField( "Identifier", _pgA0h( 0 ), raw=pgData.identifier,
                   decode=_dictGet( decode.identifiers, "Reserved" ) )

   # Extended identifier
   model.addField( "Extended identifier", _pgA0h( 1 ), raw=pgData.extIdentifier,
                   decode=_dictGet( decode.extIdentifiers, "Unallocated" ) )

   # Connector
   model.addField( "Connector", _pgA0h( 2 ), raw=pgData.connector,
                   decode=_dictGet( decode.connectors, "Reserved" ) )

   # Transceiver compliance
   compVal = Tac.newInstance( "Xcvr::Sff8472TransceiverCompliance",
                              pgData.compliance )
   compRaw = ( pgData.compliance << 8 ) + pgData.complianceExtended
   comp = model.addField( "Compliance", _pgA0h( list( range( 3, 11 ) ), 36 ),
                          raw=compRaw, _numBytes=9 )
   comp.addField( "10G", _pgA0h( 3 ), raw=compVal.tenG,
                  decode=_mapBitField( decode.compliance10G ),
                  possibleValues=decode.compliance10G )
   comp.addField( "Infiniband", _pgA0h( 3 ), raw=compVal.infini,
                  decode=_mapBitField( decode.complianceInfini ),
                  possibleValues=decode.complianceInfini )
   comp.addField( "Ethernet", _pgA0h( 6 ), raw=compVal.ethernet,
                  decode=_mapBitField( decode.complianceEth ),
                  possibleValues=decode.complianceEth )
   fibre = comp.addField( "Fibre Channel", _pgA0h( list( range( 7, 11 ) ) ) )
   fibre.addField( "Link length", _pgA0h( 7 ), raw=compVal.fibreLength,
                   decode=_mapBitField( decode.fibreLength ),
                   possibleValues=decode.fibreLength )
   fibre.addField( "Transmitter technology", _pgA0h( 7, 8 ), raw=compVal.fibreTech,
                   decode=_mapBitField( decode.fibreTech ),
                   possibleValues=decode.fibreTech )
   fibre.addField( "Transmission media", _pgA0h( 9 ), raw=compVal.fibreMedia,
                   decode=_mapBitField( decode.fibreMedia ),
                   possibleValues=decode.fibreMedia )
   fibre.addField( "Speed", _pgA0h( 10 ), raw=compVal.fibreSpeed,
                   decode=_mapBitField( decode.fibreSpeed ),
                   possibleValues=decode.fibreSpeed )
   comp.addField( "SFP+ technology", _pgA0h( 8 ), raw=compVal.sfpTech,
                  decode=_mapBitField( decode.sfpTech ),
                  possibleValues=decode.sfpTech )
   comp.addField( "Extended", _pgA0h( 36 ), raw=pgData.complianceExtended,
                  decode=_dictGet( decode.complianceExtended, "Reserved" ) )

   # Encoding
   model.addField( "Encoding", _pgA0h( 11 ), raw=pgData.encoding,
                   decode=_dictGet( decode.encodings, "Reserved" ) )

   # Bitrate
   bit = model.addField( "Bitrate", _pgA0h( 12, 66, 67 ) )
   nomBit = bit.addField( "Nominal", _pgA0h( 12 ), raw=pgData.bitrateNominal,
                          decode=_mult( 100 ), units="MBd" )
   maxBit = bit.addField( "Max", _pgA0h( 66 ), raw=pgData.bitrateMax,
                          decode=_mult( pgData.bitrateNominal ),
                          units="MBd", prefix="+" )
   minBit = bit.addField( "Min", _pgA0h( 67 ), raw=pgData.bitrateMin,
                          decode=_mult( pgData.bitrateNominal ),
                          units="MBd", prefix="-" )
   extBit = bit.addField( "Extended", _pgA0h( 66 ), raw=pgData.bitrateMax,
                          decode=_mult( 250 ), units="MBd" )
   # Err is defined as +/-1% over nominal when byte 12 (bitrate nominal) is 0xff
   # Nominal bitrate is then defined by byte 66 (bitrate max) * 250, so we want to
   # scale the percentage in byte 67 (bitrate min) by that amount / 100
   errBit = bit.addField( "Error", _pgA0h( 67 ), raw=pgData.bitrateMin,
                          decode=_mult( pgData.bitrateMax * 2.5 ),
                          units="MBd", prefix="+/-" )
   if pgData.bitrateNominal == 0xff:
      nomBit.note = "ignore, see extended bitrate"
      minBit.note = maxBit.note = "ignore, see bitrate error"
   else:
      extBit.note = "ignore, see nominal bitrate"
      errBit.note = "ignore, see min/max bitrate"

   # Note: skip byte 13 (rate identifier) as we don't use it

   # Length
   length = model.addField( "Length", _pgA0h( list( range( 14, 20 ) ) ) )
   smfLen = length.addField( "SMF", _pgA0h( 14, 15 ) )
   smfLen.addField( "Kilometers", _pgA0h( 14 ), raw=pgData.length9umKm,
                    decode=int, units="km" )
   smfMLen = smfLen.addField( "Meters", _pgA0h( 15 ), raw=pgData.length9um100m,
                              decode=_mult( 100 ), units="m" )
   length.addField( "OM2", _pgA0h( 16 ), raw=pgData.lengthOm2,
                    decode=_mult( 10 ), units="m" )
   length.addField( "OM1", _pgA0h( 17 ), raw=pgData.lengthOm1,
                    decode=_mult( 10 ), units="m" )
   om4Len = length.addField( "OM4", _pgA0h( 18 ), raw=pgData.lengthCopper,
                             decode=_mult( 10 ), units="m" )
   cabLen = length.addField( "Cable assembly", _pgA0h( 18 ), raw=pgData.lengthCopper,
                             decode=int, units="m" )
   length.addField( "OM3", _pgA0h( 19 ), raw=pgData.lengthOm3,
                    decode=_mult( 10 ), units="m" )
   if not compVal.sfpTech:
      cabLen.note = "ignore, optical variant detected"
   else:
      om4Len.note = "ignore, non-optical variant detected"
   if pgData.length9um100m == 0xff:
      smfMLen.note = "ignore, see Kilometers"

   # Active cable spec
   actCblVal = Tac.newInstance( "Xcvr::Sff8472ActiveAttenuation",
                                pgData.activeAttenuation )
   actCbl = model.addField( "Active cable spec", _pgA0h( 60, 61 ) )
   actCbl.addField( "Compliant to FC-PI-4 limiting", _pgA0h( 60 ),
                    raw=actCblVal.compliantToFcPi4Limiting, decode=bool )
   actCbl.addField( "Compliant to SFF-8431 limiting", _pgA0h( 60 ),
                    raw=actCblVal.compliantToSff8431Limiting, decode=bool )
   actCbl.addField( "Compliant to FC-PI-4 appendix H", _pgA0h( 60 ),
                    raw=actCblVal.compliantToFcPi4AppendixH, decode=bool )
   actCbl.addField( "Compliant to SFF-8431 appendix E", _pgA0h( 60 ),
                    raw=actCblVal.compliantToSff8431AppendixE, decode=bool )

   # Passive cable spec
   pasCblVal = Tac.newInstance( "Xcvr::Sff8472PassiveAttenuation",
                                pgData.passiveAttenuation )
   pasCbl = model.addField( "Passive cable spec", _pgA0h( 60, 61 ) )
   pasCbl.addField( "Compliant to FC-PI-4 appendix H", _pgA0h( 60 ),
                    raw=pasCblVal.compliantToFcPi4AppendixH, decode=bool )
   pasCbl.addField( "Compliant to SFF-8431 appendix E", _pgA0h( 60 ),
                    raw=pasCblVal.compliantToSff8431AppendixE, decode=bool )

   # Wavelength
   wave = model.addField( "Wavelength", _pgA0h( 60, 61 ),
                          raw=pgData.laserWavelengthInteger, decode=int, units="nm" )
   if compVal.sfpTechActive:
      wave.note = pasCbl.note = "ignore, active variant detected"
   elif compVal.sfpTechPassive:
      wave.note = actCbl.note = "ignore, passive variant detected"
   else:
      actCbl.note = pasCbl.note = "ignore, optical variant detected"

   # Options
   opsVal = Tac.newInstance( "Xcvr::Sff8472Options", pgData.options )
   ops = model.addField( "Options", _pgA0h( 64, 65 ), raw=pgData.options,
                         _numBytes=2 )
   ops.addField( "High power level 3 declaration", _pgA0h( 64 ),
                 raw=opsVal.highPowerLevel, decode=bool )
   ops.addField( "Paging implemented", _pgA0h( 64 ),
                 raw=opsVal.pagingImplemented, decode=bool )
   ops.addField( "CDR indicator", _pgA0h( 64 ),
                 raw=opsVal.cdrIndicator, decode=bool )
   ops.addField( "Cooled transmitter", _pgA0h( 64 ),
                 raw=opsVal.cooledTransmitter, decode=bool )
   pwrLvl = ops.addField( "Power level declaration", _pgA0h( 64 ),
                 raw=opsVal.powerLevel, decode=lambda x: int( x ) + 1 )
   if opsVal.highPowerLevel:
      pwrLvl.note = "ignore, see High power level 3 declaration"
   ops.addField( "Linear receiver output implemented", _pgA0h( 64 ),
                 raw=opsVal.lnRxOutputImplemented, decode=bool )
   ops.addField( "Receiver decision threshold implemented", _pgA0h( 65 ),
                 raw=opsVal.rxDecisionThresholdImplemented, decode=bool )
   ops.addField( "Tunable transmitter", _pgA0h( 65 ),
                 raw=opsVal.tunableTransmitter, decode=bool )
   ops.addField( "RATE_SELECT implemented", _pgA0h( 65 ),
                 raw=opsVal.rateSelectImplemented, decode=bool )
   ops.addField( "TX_DISABLE implemented", _pgA0h( 65 ),
                 raw=opsVal.txDisableImplemented, decode=bool )
   ops.addField( "TX_FAULT implemented", _pgA0h( 65 ),
                 raw=opsVal.txFaultImplemented, decode=bool )
   ops.addField( "Signal detect implemented", _pgA0h( 65 ),
                 raw=opsVal.signalDetectImplemented, decode=bool )
   ops.addField( "RX_LOS implemented", _pgA0h( 65 ),
                 raw=opsVal.rxLosImplemented, decode=bool )

   # Date
   dateRaw = pgData.dateCode.encode().hex()
   date = model.addField( "Date code", _pgA0h( list( range( 84, 92 ) ) ),
                          raw=int( dateRaw, base=16 ) if dateRaw else 0,
                          decode=lambda x: '"%s"' % pgData.dateCode )
   date.note = "YYMMDDLL"

   # DOM
   domVal = Tac.newInstance( "Xcvr::Sff8472DomType", pgData.diagsMonitoringType )
   dom = model.addField( "DOM type", _pgA0h( 92 ) )
   dom.addField( "Legacy DOM", _pgA0h( 92 ),
                 raw=domVal.legacyBit, decode=bool )
   dom.addField( "Implemented", _pgA0h( 92 ),
                 raw=domVal.implemented, decode=bool )
   dom.addField( "Internally calibrated", _pgA0h( 92 ),
                 raw=domVal.internallyCalibrated, decode=bool )
   dom.addField( "Externally calibrated", _pgA0h( 92 ),
                 raw=domVal.externallyCalibrated, decode=bool )
   dom.addField( "Received power measurement type", _pgA0h( 92 ),
                 raw=domVal.receivedPowerMeasurementType, decode=_avgOrOma )
   dom.addField( "Address change required", _pgA0h( 92 ),
                 raw=domVal.addressChangeRequired, decode=bool )

   # Extended Options
   ehOpsVal = Tac.newInstance( "Xcvr::Sff8472EnhancedOptions",
                               pgData.enhancedOptions )
   ehOps = model.addField( "Enhanced Options", _pgA0h( 93 ) )
   ehOps.addField( "Optional alarm/warning flags implemented", _pgA0h( 93 ),
                   raw=ehOpsVal.alarmFlagsImplemented, decode=bool )
   ehOps.addField( "Soft TX_DISABLE control and monitoring implemented",
                   _pgA0h( 93 ), raw=ehOpsVal.swTxDisableImplemented, decode=bool )
   ehOps.addField( "Soft TX_FAULT monitoring implemented", _pgA0h( 93 ),
                   raw=ehOpsVal.swTxFaultImplemented, decode=bool )
   ehOps.addField( "Soft RX_LOS monitoring implemented", _pgA0h( 93 ),
                   raw=ehOpsVal.swRxLosImplemented, decode=bool )
   ehOps.addField( "Soft RATE_SELECT control and monitoring implemented",
                   _pgA0h( 93 ), raw=ehOpsVal.swRateSelectImplemented, decode=bool )
   ehOps.addField( "Soft application select control implemented",
                   _pgA0h( 93 ), raw=ehOpsVal.applicationSelectImplemented,
                   decode=bool )
   ehOps.addField( "Soft rate select control implemented (per SFF8431)",
                   _pgA0h( 93 ), raw=ehOpsVal.swRateSelectSff8431Implemented,
                   decode=bool )

   # Revision
   revDec = _dictGet( decode.revisions, "Unknown revision (0x%X)" % pgData.revision )
   rev = model.addField( "Revision compliance", _pgA0h( 94 ), raw=pgData.revision,
                         decode=revDec )
   rev.note = "parsed via Sff-8472 rev 12.3"

   # Checksums
   check = model.addField( "Checksums", _pgA0h( list( range( 0, 96 ) ) ) )
   check.addField( "Base", _pgA0h( 63 ), raw=pgData.storedCcBase )
   check.addField( "Base computed", _pgA0h( list( range( 0, 63 ) ) ),
                   raw=pgData.computedCcBase )
   check.addField( "Extended", _pgA0h( 95 ), raw=pgData.storedCcExt )
   check.addField( "Extended computed", _pgA0h( list( range( 64, 95 ) ) ),
                   raw=pgData.computedCcExt )

# Decode the information of enhanced DOM configuration registers.
def decodeQsfpEnhancedDom( xcvrStatus, model ):
   enhDomParamTypeDecode = {
         b'laserAge': 'Laser age',
         b'tecCurrent': 'TEC current',
         b'laserFreqErr': 'Laser frequency error',
         b'laserTemp': 'Laser temperature',
         b'snr': 'eSNR media input',
         b'pam4Transitions': 'PAM4 level transition parameter media input',
         b'preFecBERMin': 'Pre-FEC BER minimum media input',
         b'preFecBERMax': 'Pre-FEC BER maximum media input',
         b'preFecBERCurr': 'Pre-FEC BER current value media input',
         b'preFecBER': 'Pre-FEC BER average media input',
         b'preFecBERAcc': 'Pre-FEC BER average media input during the'
                         ' last monitoring interval',
         b'uncorrectedBERMin': 'Uncorrected BER minimum media input',
         b'uncorrectedBERMax': 'Uncorrected BER maximum media input',
         b'uncorrectedBER': 'Uncorrected BER average media input',
         b'uncorrectedBERCurr': 'Uncorrected BER current value media input',
         b'uncorrectedBERAcc': 'Uncorrected BER average media input during the'
                              ' last monitoring interval',
         b'paramUnknown': 'Reserved/Vendor specific',
         }
   enhDomConfig = model.addField( "Enhanced dom parameter configuration",
                                  _pg20h( 200, 247 ) )
   totalParams = xcvrStatus.enhancedDomRegisterData.totalParameters
   for paramId in range( totalParams ):
      enhDomParamVal = Tac.Value( "Xcvr::Sff8436ParameterConfig" )
      enhDomParamType = Tac.Type( "Xcvr::EnhancedDomParamType" )
      enhDomParamVal = xcvrStatus.enhancedDomParamConfigRegisterData[ paramId + 1 ]
      if enhDomParamVal.paramType != enhDomParamType.paramUnsupported:
         enhDomParam = enhDomConfig.addField( "Parameter %d" % ( paramId + 1 ),
                                              _pg20h( 200 + 2 * paramId,
                                                      200 + 2 * paramId + 1 ) )
         enhDomParam.addField( "Threshold ID", _pg20h( 200 + 2 * paramId ),
                               raw=enhDomParamVal.thresholdId, decode=int )
         enhDomParam.addField( "Channel specific", _pg20h( 200 + 2 * paramId ),
                               raw=enhDomParamVal.channelSpecific, decode=bool )
         enhDomParam.addField( "Channels", _pg20h( 200 + 2 * paramId ),
                               raw=enhDomParamVal.channels, decode=int )
         rawParamType = enhDomParamVal.paramType.encode().hex()
         def getEnhDomParamLongName( rawParam ):
            # Follow-up will mut will remove this conversion
            paramType = bytes.fromhex( f'{rawParam:x}' )
            return enhDomParamTypeDecode.get( paramType, paramType )
         enhDomParam.addField( "Parameter type", _pg20h( 200 + 2 * paramId + 1 ),
                               raw=int( rawParamType, base=16 ),
                               decode=getEnhDomParamLongName )

def scaleDec( x ):
   return 10 ** ( ( ( x >> 6 ) & 0x3 ) - 1 ) * ( x & 0x3F )

def decodeCmis( pgData, model ):
   decode = Cmis8x16xValues

   # Identifier
   iden = model.addField( "Identifier", _pg00h( 0, 128 ) )
   iden.addField( "Lower", _pg00h( 0 ), raw=pgData.identifier,
                  decode=_dictGet( decode.identifiers, "Reserved" ) )
   iden.addField( "Upper", _pg00h( 128 ), raw=pgData.identifierExtended,
                  decode=_dictGet( decode.identifiers, "Reserved" ) )

   # Revision Compliance
   model.addField( "Revision compliance", _pg00h( 1 ), raw=pgData.fullVersionId,
                   decode=lambda x: "version %X.%X" % ( ( x >> 4 ) & 0xF, x & 0xF ) )

   # Coherent CMIS (C-CMIS) revision compliance
   if pgData.cmisCoherentRevision is not None:
      model.addField( "C-CMIS Revision compliance", _pg40h( 128 ),
                      raw=pgData.cmisCoherentRevision,
                      decode=( lambda x: "C-CMIS version %X.%X" %
                               ( ( x >> 4 ) & 0xF, x & 0xF ) ) )
   # Status fields
   stat = model.addField( "Status fields", _pg00h( 2 ) )
   stat.addField( "Flat mem", _pg00h( 2 ), raw=pgData.flatMem, decode=bool )
   stat.addField( "TWI max speed", _pg00h( 2 ), raw=pgData.maxSmbusSpeedSupported,
                  decode=_dictGet( decode.twiSpeed, "Reserved" ) )

   sfpDdIdentifier = Tac.enumValue( "Xcvr::Sff8024::IdentifierCode", "idenSfpDd" )
   if ( pgData.fullVersionId >= 0x40 or
        ( pgData.fullVersionId >= 0x20 and pgData.identifier == sfpDdIdentifier ) ):
      # Active and Inactive Firmware rev is only defined for
      # CMIS4+ modules and SFP-DD2+
      model.addField( "Active firmware revision",
                      _pg00h( list( range( 39, 41 ) ) ),
                      raw=pgData.moduleActiveFirmwareRev,
                      decode=lambda x: "{0}.{1} (0x{0:02X} 0x{1:02X})".format(
                         ( x >> 8 ) & 0xFF, x & 0xFF ) )

      # Inactive Firmware rev is only defined on upperPage01h
      if not pgData.flatMem:
         model.addField( "Inactive firmware revision",
                         _pg01h( list( range( 128, 130 ) ) ),
                         raw=pgData.moduleInactiveFirmwareRev,
                         decode=lambda x: "{0}.{1} (0x{0:02X} 0x{1:02X})".format(
                            ( x >> 8 ) & 0xFF, x & 0xFF ) )
   elif pgData.fullVersionId >= 0x30:
      # Only Active Firmware rev is defined for CMIS3 modules, and
      # it is in a different location
      model.addField( "Active firmware revision",
                      _pg01h( list( range( 128, 130 ) ) ),
                      raw=pgData.moduleActiveFirmwareRev,
                      decode=lambda x: "{0}.{1} (0x{0:02X} 0x{1:02X})".format(
                         ( x >> 8 ) & 0xFF, x & 0xFF ) )

   # Application Select
   apsel = model.addField( "Application select",
                           _pg00h( list( range( 85, 118 ) ) ) )
   apsel.addField( "Module type encoding", _pg00h( 85 ), raw=pgData.moduleType,
                   decode=_dictGet( decode.moduleEncodings, "Reserved" ) )
   if not pgData.flatMem:
      # Only want to include page 01 if the module has it
      apsel.addBytes( 1, list( range( 176, 191 ) ), list( range( 223, 251 ) ) )

   def moduleMediaDesc( index, appVal ):
      return appVal.moduleMediaEntry( pgData.moduleType, pgData.vendorType ).appName

   def appBuilder( index, appVal, page, baseByte ):
      if not appVal.primaryReg:
         # Don't display empty applications
         return
      startByte = baseByte + index * 4
      app = apsel.addField( "Application %d" % ( index + 1 ),
                            page( list( range( startByte, startByte + 4 ) ) ) )
      hostDec = appVal.hostElectricalEntry().appName
      app.addField( "Host electrical interface", page( startByte ),
                    raw=appVal.hostElectricalInterface,
                    decode=lambda x: hostDec if "Unknown" not in hostDec else None )
      modDec = moduleMediaDesc( index, appVal )
      app.addField( "Module media interface", page( startByte + 1 ),
                    raw=appVal.moduleMediaInterface,
                    decode=lambda x: modDec if "Unknown" not in modDec else None )
      app.addField( "Host lane count", page( startByte + 2 ),
                    raw=appVal.hostLaneCount, decode=int )
      app.addField( "Media lane count", page( startByte + 2 ),
                    raw=appVal.mediaLaneCount, decode=int )
      app.addField( "Host lane assignment options",
                    page( startByte + 3 ),
                    raw=appVal.hostLaneAssignment,
                    decode=_mapBitField( decode.apselLaneAssignments ) )
      if not pgData.flatMem:
         # Only want to include page 01 if the module has it
         app.addBytes( 1, 176 + index )
         app.addField( "Media lane assignment options", _pg01h( 176 + index ),
                       raw=appVal.mediaLaneAssignment,
                       decode=_mapBitField( decode.apselLaneAssignments ) )
   for i, appVal in pgData.application.items():
      if i <= 8:
         appBuilder( i - 1, appVal, _pg00h, 86 )
      elif not pgData.flatMem:
         appBuilder( i - 1, appVal, _pg01h, 223 - 8 * 4 )

   # Date
   dateRaw = pgData.dateCode.encode().hex()
   date = model.addField( "Date code", _pg00h( list( range( 182, 190 ) ) ),
                          raw=int( dateRaw, base=16 ) if dateRaw else 0,
                          decode=lambda x: '"%s"' % pgData.dateCode )
   date.note = "YYMMDDLL"

   # Module Power Characteristics
   mdlPwrType = "Xcvr::CmisModulePowerCharacteristics"
   mdlPwrVal = Tac.newInstance( mdlPwrType, pgData.modulePower )
   mdlPwr = model.addField( "Module power", _pg00h( 200, 201 ) )
   pwrClsVal = Tac.enumValue( mdlPwrType + "::CmisPowerClass", mdlPwrVal.powerClass )
   pwrCls = mdlPwr.addField( "Power class", _pg00h( 200 ), raw=pwrClsVal,
                             decode=lambda x: x + 1 )
   mdlPwr.addField( "Max power", _pg00h( 201 ), raw=mdlPwrVal.maxPower,
                    decode=_mult( 0.25 ), units="W" )
   if pwrClsVal == 0x7:
      pwrCls.note = "see Max power"

   # Length
   length = model.addField( "Length", _pg00h( 202 ) )
   length.addField( "Cable assembly", _pg00h( 202 ), raw=pgData.cableLength,
                    decode=scaleDec, units="m" )

   # Connector
   model.addField( "Connector", _pg00h( 203 ), raw=pgData.connector,
                   decode=_dictGet( decode.connectors, "Reserved" ) )

   # Copper attenuation
   # TODO: need a decode for this field

   # Near end
   # TODO: need a decode for this field

   # Far end
   # TODO: need a decode for this field
   # TODO: need to figure out how best to show the breakout
   #   hardcode?
   #   look at a few algorithmic options

   # Media interface technology
   model.addField( "Media interface technology", _pg00h( 212 ),
                   raw=pgData.mediaInterfaceTech,
                   decode=_dictGet( decode.mediaTech, "Reserved" ) )

   # Checksums
   check = model.addField( "Checksums", _pg00h( 128 ) )
   if 0x00 in pgData.checksum:
      check.addBytes( 0, list( range( 128, 223 ) ) )
      check.addField( "Page 00", _pg00h( 222 ), raw=pgData.checksum[ 0x00 ] )
      check.addField( "Page 00 computed", _pg00h( list( range( 128, 222 ) ) ),
                      raw=pgData.checksumCalculated[ 0x00 ] )

   if pgData.flatMem:
      # The module only has page 00h, so stop after decoding it
      return

   if 0x01 in pgData.checksum:
      check.addBytes( 1, list( range( 128, 256 ) ) )
      check.addField( "Page 01", _pg01h( 255 ), raw=pgData.checksum[ 0x01 ] )
      check.addField( "Page 01 computed", _pg01h( list( range( 128, 255 ) ) ),
                      raw=pgData.checksumCalculated[ 0x01 ] )

   # Module Firmware and Hardware revisions
   # TODO: need a decode for these fields

   # Optical Length
   length.addBytes( 1, list( range( 132, 137 ) ) )
   length.addField( "SMF", _pg01h( 132 ), raw=pgData.lengthSmf,
                    decode=scaleDec, units="km" )
   length.addField( "OM5", _pg01h( 133 ), raw=pgData.lengthOm5,
                    decode=_mult( 2 ), units="m" )
   length.addField( "OM4", _pg01h( 134 ), raw=pgData.lengthOm4,
                    decode=_mult( 2 ), units="m" )
   length.addField( "OM3", _pg01h( 135 ), raw=pgData.lengthOm3,
                    decode=_mult( 2 ), units="m" )
   length.addField( "OM2", _pg01h( 136 ), raw=pgData.lengthOm2,
                    decode=int, units="m" )

   # Wavelength
   # TODO: need a decode for these fields

   # Implemented Memory Pages
   pgBkVal = Tac.newInstance( "Xcvr::CmisImplementedPagesAndBanks",
                              pgData.implPagesAndBanks )
   pgBk = model.addField( "Implemented pages and banks", _pg01h( 142 ) )
   pgBk.addField( "VDM pages implemented", _pg01h( 142 ),
                  raw=pgBkVal.vdmImpl, decode=bool )
   pgBk.addField( "Diagnostic pages implemented", _pg01h( 142 ),
                  raw=pgBkVal.diagImpl, decode=bool )
   pgBk.addField( "Page 03h implemented", _pg01h( 142 ), raw=pgBkVal.pg03hImpl,
                  decode=bool )
   pgBk.addField( "Banks implemented", _pg01h( 142 ), raw=pgBkVal.banks,
                  decode=lambda x: decode.implementedBanks[ x ] )

   # Decode function for duration data
   def stateDurationFn( x ):
      return decode.stateDuration[ x ]

   # Data path timing
   dPath = model.addField( "Data path timings", _pg01h( 144 ) )
   dPathInitVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                 pgData.dataPathInitDuration.encoding )
   dPath.addField( "Init", _pg01h( 144 ), raw=dPathInitVal,
                   decode=stateDurationFn )
   dPathDeinitVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                   pgData.dataPathDeinitDuration.encoding )
   dPath.addField( "Deinit", _pg01h( 144 ), raw=dPathDeinitVal,
                   decode=stateDurationFn )

   # Module Characteristics Advertising
   # TODO: need a decode for some of these fields
   mdlChar = model.addField( "Module characteristics advertising",
                             _pg01h( list( range( 145, 155 ) ) ) )
   auxMonType = Tac.Type( "Xcvr::CmisAuxMonitorType" )

   def auxTypeToBool( auxType ):
      return auxType in [ auxMonType.auxTecCurrent, auxMonType.auxVcc2 ]
   mdlChar.addField( "Aux 1 monitor type", _pg01h( 145 ),
                     raw=auxTypeToBool( pgData.aux1MonitorType ),
                     decode=lambda x: "TEC current" if x else "reserved" )
   mdlChar.addField( "Aux 2 monitor type", _pg01h( 145 ),
                     raw=auxTypeToBool( pgData.aux2MonitorType ),
                     decode=lambda x: "TEC current" if x else "Laser Temperature" )
   mdlChar.addField( "Aux 3 monitor type", _pg01h( 145 ),
                     raw=auxTypeToBool( pgData.aux3MonitorType ),
                     decode=lambda x: "Vcc2" if x else "Laser Temperature" )
   mdlChar.addField( "Rx output amplitude codes supported", _pg01h( 153 ),
                     raw=pgData.implRxOutputAmplitudeCode,
                     decode=_mapBitField( list( map( str, list( range( 4 ) ) ) ) ) )
   mdlChar.addField( "Max tx input eq", _pg01h( 153 ),
                     raw=pgData.maxTxInputEqualization,
                     decode=int, units="dB" )
   mdlChar.addField( "Max rx output eq post-cursor", _pg01h( 154 ),
                     raw=pgData.maxRxOutputPostEmphasis,
                     decode=int, units="dB" )
   mdlChar.addField( "Max rx output eq pre-cursor", _pg01h( 154 ),
                     raw=pgData.maxRxOutputPreEmphasis,
                     decode=lambda x: 0.5 * x, units="dB" )

   # Implemented Controls Advertisement
   ctlType = "Xcvr::CmisImplControlsAdvertisement"
   ctlVal = Tac.newInstance( ctlType, pgData.implControlsAdvertisement )
   ctl = model.addField( "Implemented controls advertisement", _pg01h( 155, 156 ) )
   ctl.addField( "Wavelength control implemented", _pg01h( 155 ),
                 raw=ctlVal.wavelengthCtrlImpl, decode=bool )
   ctl.addField( "Tunable transmitter", _pg01h( 155 ),
                 raw=ctlVal.tunableTxImpl, decode=bool )
   txSqlVal = Tac.enumValue( ctlType + "::CmisTxSquelch", ctlVal.txSquelchImpl )
   txSql = ctl.addField( "Tx squelch", _pg01h( 155 ) )
   txSql.addField( "OMA", _pg01h( 155 ), raw=txSqlVal & 0b01, decode=bool )
   txSql.addField( "Average power", _pg01h( 155 ), raw=txSqlVal & 0b10, decode=bool )
   ctl.addField( "Tx force squelch implemented", _pg01h( 155 ),
                 raw=ctlVal.txForceSquelchImpl, decode=bool )
   ctl.addField( "Tx squelch disable implemented", _pg01h( 155 ),
                 raw=ctlVal.txSquelchDisableImpl, decode=bool )
   ctl.addField( "Tx disable implemented", _pg01h( 155 ),
                 raw=ctlVal.txDisableImpl, decode=bool )
   ctl.addField( "Tx polarity flip implemented", _pg01h( 155 ),
                 raw=ctlVal.txPolFlipImpl, decode=bool )
   ctl.addField( "Rx squelch disable implemented", _pg01h( 156 ),
                 raw=ctlVal.rxSquelchDisableImpl, decode=bool )
   ctl.addField( "Rx disable implemented", _pg01h( 156 ),
                 raw=ctlVal.rxDisableImpl, decode=bool )
   ctl.addField( "Rx polarity flip implemented", _pg01h( 156 ),
                 raw=ctlVal.rxPolFlipImpl, decode=bool )

   # Implemented Flags Advertisement
   flgVal = Tac.newInstance( "Xcvr::CmisImplFlagsAdvertisement",
                             pgData.implFlagsAdvertisement )
   flg = model.addField( "Implemented flags advertisement", _pg01h( 157, 158 ) )
   flg.addField( "Tx adaptive input eq fault flag implemented", _pg01h( 157 ),
                 raw=flgVal.txAdaptiveInputEqFaultImpl, decode=bool )
   flg.addField( "Tx CDR LOL flag implemented", _pg01h( 157 ),
                 raw=flgVal.txCdrLolImpl, decode=bool )
   flg.addField( "Tx LOS flag implemented", _pg01h( 157 ),
                 raw=flgVal.txLosImpl, decode=bool )
   flg.addField( "Tx fault flag implemented", _pg01h( 157 ),
                 raw=flgVal.txFaultImpl, decode=bool )
   # TODO: CDR LOL?
   flg.addField( "Rx LOL flag implemented", _pg01h( 158 ),
                 raw=flgVal.rxCdrLolImpl, decode=bool )
   flg.addField( "Rx LOS flag implemented", _pg01h( 158 ),
                 raw=flgVal.rxLosImpl, decode=bool )

   # Implemented Monitors Advertisement
   monType = "Xcvr::CmisImplMonitorsAdvertisement"
   monVal = Tac.newInstance( monType, pgData.implMonitorsAdvertisement )
   mon = model.addField( "Implemented monitors advertisement", _pg01h( 159, 160 ) )
   mon.addField( "Custom monitor implemented", _pg01h( 159 ),
                 raw=monVal.customMonitorImpl, decode=bool )
   mon.addField( "Aux 3 monitor implemented", _pg01h( 159 ),
                 raw=monVal.aux3MonitorImpl, decode=bool )
   mon.addField( "Aux 2 monitor implemented", _pg01h( 159 ),
                 raw=monVal.aux2MonitorImpl, decode=bool )
   mon.addField( "Aux 1 monitor implemented", _pg01h( 159 ),
                 raw=monVal.aux1MonitorImpl, decode=bool )
   mon.addField( "Internal 3.3V monitor implemented", _pg01h( 159 ),
                 raw=monVal.voltageMonitorImpl, decode=bool )
   mon.addField( "Temperature monitor implemented", _pg01h( 159 ),
                 raw=monVal.tempMonitorImpl, decode=bool )
   txBiasVal = Tac.enumValue( monType + "::CmisTxBiasMultiplier",
                              monVal.txBiasMultiplier )
   mon.addField( "Tx bias current multiplier", _pg01h( 160 ),
                 raw=txBiasVal, decode=lambda x: 1 << x )
   mon.addField( "Rx optical input power monitor implemented", _pg01h( 160 ),
                 raw=monVal.rxPowerMonitorImpl, decode=bool )
   mon.addField( "Tx optical output power monitor implemented", _pg01h( 160 ),
                 raw=monVal.txPowerMonitorImpl, decode=bool )
   mon.addField( "Tx bias monitor implemented", _pg01h( 160 ),
                 raw=monVal.txBiasMonitorImpl, decode=bool )

   # Implemented Signal Integrity Controls
   sigVal = Tac.newInstance( "Xcvr::CmisImplSignalIntegrityControl",
                             pgData.implSignalIntegrityControl )
   sig = model.addField( "Implemented signal integrity controls",
                         _pg01h( 161, 162 ) )
   sig.addField( "Tx input eq store/recall buffer count", _pg01h( 161 ),
                 raw=sigVal.txInputEqStoreRecallBufCount, decode=int )
   sig.addField( "Tx input eq freeze implemented", _pg01h( 161 ),
                 raw=sigVal.txInputEqualizationFreezeImpl, decode=bool )
   sig.addField( "Adaptive tx input eq implemented", _pg01h( 161 ),
                 raw=sigVal.txInputAdaptiveEqualizationImpl, decode=bool )
   sig.addField( "Tx input eq fixed manual control implemented", _pg01h( 161 ),
                 raw=sigVal.txInputFixedEqualizationImpl, decode=bool )
   sig.addField( "Tx CDR bypass control implemented", _pg01h( 161 ),
                 raw=sigVal.txCdrBypassCtrlImpl, decode=bool )
   sig.addField( "Tx CDR implemented", _pg01h( 161 ),
                 raw=sigVal.txCdrImpl, decode=bool )
   sig.addField( "Staged set 1 implemented", _pg01h( 162 ),
                 raw=sigVal.stagedSetOneImpl, decode=bool )
   rxOut = sig.addField( "Rx output eq control implemented", _pg01h( 162 ) )
   rxOut.addField( "Post-cursor", _pg01h( 162 ),
                   raw=sigVal.rxOutputPostEmphasisImpl, decode=bool )
   rxOut.addField( "Pre-cursor", _pg01h( 162 ),
                   raw=sigVal.rxOutputPreEmphasisImpl, decode=bool )
   sig.addField( "Rx output amplitude control implemented", _pg01h( 162 ),
                 raw=sigVal.rxOutputAmplitudeImpl, decode=bool )
   sig.addField( "Rx CDR bypass control implemented", _pg01h( 162 ),
                 raw=sigVal.rxCdrBypassCtrlImpl, decode=bool )
   sig.addField( "Rx CDR implemented", _pg01h( 162 ),
                 raw=sigVal.rxCdrImpl, decode=bool )

   # The CMIS 3.0 specification does not define register 167 and 168 of upper
   # page 01h.  Only add these fields for 4.0 compliant and beyond.
   if pgData.fullVersionId >= 0x40:
      # Module power timings
      modPwrTime = model.addField( "Module power timings", _pg01h( 167 ) )
      modulePwrDownVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                        pgData.modulePwrDownDuration.encoding )
      modPwrTime.addField( "Down", _pg01h( 167 ), raw=modulePwrDownVal,
                           decode=stateDurationFn )
      modulePwrUpVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                      pgData.modulePwrUpDuration.encoding )
      modPwrTime.addField( "Up", _pg01h( 167 ), raw=modulePwrUpVal,
                           decode=stateDurationFn )

      # Data path Tx timings
      txTime = model.addField( "Data path Tx timings", _pg01h( 168 ) )
      txTurnOffVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                    pgData.dataPathTxTurnOffDuration.encoding )
      txTime.addField( "Off", _pg01h( 168 ), raw=txTurnOffVal,
                       decode=stateDurationFn )
      txTurnOnVal = Tac.enumValue( "Xcvr::CmisStateDurationEncoding",
                                   pgData.dataPathTxTurnOnDuration.encoding )
      txTime.addField( "On", _pg01h( 168 ), raw=txTurnOnVal,
                       decode=stateDurationFn )

   decodeCoherent = CoherentCmisValues()
   # 01:155, bit 6 indicated that tunable transmitter is implemented, then
   # page 04h is required to be supported
   # Laser Capabilities Advertising
   if ctlVal.tunableTxImpl:
      # freqTuningSupport : CmisLaserCapabilitiesFreqTuningSupport
      freqTuningSuppVal = pgData.freqTuningSupport
      freqTuningSupp = model.addField( "Frequency tuning support",
                                       _pg04h( 128, 129 ) )
      freqTuningSupp.addField( "Tunable wavelength", _pg04h( 128 ),
                               raw=freqTuningSuppVal.tunableWavelength,
                               decode=bool )
      gridCapVal = freqTuningSuppVal.grid
      gripCap = freqTuningSupp.addField( "Grid spacing capabilities",
                                         _pg04h( 128 ) )
      gripCap.addField( "150 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing150000M, decode=bool )
      gripCap.addField( "100 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing100000M, decode=bool )
      gripCap.addField( "75 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing75000M, decode=bool )
      gripCap.addField( "50 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing50000M, decode=bool )
      gripCap.addField( "33 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing33000M, decode=bool )
      gripCap.addField( "25 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing25000M, decode=bool )
      gripCap.addField( "12.5 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing12500M, decode=bool )
      gripCap.addField( "6.25 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing6250M, decode=bool )
      gripCap.addField( "3.125 GHz grid supported", _pg04h( 128 ),
                        raw=gridCapVal.gridSpacing3125M, decode=bool )
      freqTuningSupp.addField( "Fine tuning support", _pg04h( 129 ),
                               raw=freqTuningSuppVal.fineTuningSupport,
                               decode=bool )
      # supportedChannelBoundary : ChannelBoundary[GridSpacing]
      supportedChannelBoundariesVal = pgData.supportedChannelBoundary
      supportedChannelBoundaries = model.addField( "Supported channel boundaries",
                                                   _pg04h( list( range( 130,
                                                                        166 ) ) ) )
      lowChannelByte = { 'gridSpacing3125M': 130, 'gridSpacing6250M': 134,
                         'gridSpacing12500M': 138, 'gridSpacing25000M': 142,
                         'gridSpacing50000M': 146, 'gridSpacing100000M': 150,
                         'gridSpacing33000M': 154, 'gridSpacing75000M': 158,
                         'gridSpacing150000M': 162 }

      for gridSpacing in supportedChannelBoundariesVal:
         channelBoundaryVal = supportedChannelBoundariesVal[ gridSpacing ]
         frequencyRange = channelBoundaryVal.frequencyRange()
         grid = channelBoundaryVal.grid
         if not getattr( gridCapVal, grid, False ):
            # Do not display the minimum and maximum channel/frequency for modules
            # that do not support this grid.
            continue
         lowChannel = channelBoundaryVal.lowChannel
         lowestFreq = frequencyRange.lowestFrequency
         highChannel = channelBoundaryVal.highChannel
         highestFreq = frequencyRange.highestFrequency
         gridSize = int( grid[ len( 'gridSpacing' ) : -len( 'M' ) ] ) / 1000
         gridLabel = ( '%f' % gridSize ).rstrip( '0' ).rstrip( '.' )
         channelBoundary = supportedChannelBoundaries.addField(
                     "%s GHz grid" % gridLabel,
                     _pg04h( list( range( lowChannelByte[ grid ],
                                          lowChannelByte[ grid ] + 4 ) ) ) )
         channelBoundary.addField( "Lowest channel",
                                   _pg04h( lowChannelByte[ grid ],
                                           lowChannelByte[ grid ] + 1 ),
                                   raw=lowChannel, decode=int )
         channelBoundary.addField( "Lowest frequency",
                                   _pg04h( lowChannelByte[ grid ],
                                           lowChannelByte[ grid ] + 1 ),
                                   raw=lowestFreq, decode=int, units="MHz" )
         channelBoundary.addField( "Highest channel",
                                   _pg04h( lowChannelByte[ grid ] + 2,
                                           lowChannelByte[ grid ] + 3 ),
                                   raw=highChannel, decode=int )
         channelBoundary.addField( "Highest frequency",
                                   _pg04h( lowChannelByte[ grid ] + 2,
                                           lowChannelByte[ grid ] + 3 ),
                                   raw=highestFreq, decode=int, units="MHz" )
      # freqFineTuningCapabilities : CmisFreqFineTuningCapabilities
      if freqTuningSuppVal.fineTuningSupport:
         freqFineTuningCapsVal = pgData.freqFineTuningCapabilities
         freqFineTuningCaps = model.addField( "Fine-tuning capabilities",
                                              _pg04h( list( range( 190, 196 ) ) ) )
         freqFineTuningCaps.addField( "Fine-tuning resolution", _pg04h( 190, 191 ),
                                      raw=freqFineTuningCapsVal.resolution,
                                      decode=int, units="MHz" )
         freqFineTuningCaps.addField( "Fine-tuning low offset", _pg04h( 192, 193 ),
                                      raw=freqFineTuningCapsVal.lowOffset,
                                      decode=int, units="MHz" )
         freqFineTuningCaps.addField( "Fine-tuning high offset", _pg04h( 194, 195 ),
                                      raw=freqFineTuningCapsVal.highOffset,
                                      decode=int, units="MHz" )
      # prgOutputPowerAdvertisement : CmisPrgOutputPowerAdvertisement
      prgOutputVal = pgData.prgOutputPowerAdvertisement
      prgOutput = model.addField( "Programmable output power advertisement",
                                  _pg04h( list( range( 196, 202 ) ) ) )
      prgOutput.addField( "Lane programmable output power supported", _pg04h( 196 ),
                          raw=prgOutputVal.supported, decode=bool )
      if prgOutputVal.supported:
         prgOutput.addField( "Min programmable output power", _pg04h( 198, 199 ),
                             raw=prgOutputVal.minPower,
                             decode=lambda x: x / 100, units='dB' )
         prgOutput.addField( "Max programmable output power", _pg04h( 200, 201 ),
                             raw=prgOutputVal.maxPower,
                             decode=lambda x: x / 100, units='dB' )

   if pgData.fullVersionId >= 0x40:
      addCmisLoopbackCapsToModel( pgData, model )

   # 01:142 bit 6 inidcates that VDM pages are implemented at pages 20h-2Fh
   if pgBkVal.vdmImpl:
      # 20h-23h VDM config pages
      vdmConfigVal = pgData.vdmConfig
      # There are 64 VDM parameters defined per page ( 2 bytes each )
      paramsPerPage = 64
      # We can have up to 4 config pages
      vdmConfigPages = [ None, None, None, None ]
      _pgs = [ _pg20h, _pg21h, _pg22h, _pg23h ]
      vdmConfig = model.addField( "VDM configuration",
                                  _pg20h( list( range( 128, 256 ) ) ) +
                                  _pg21h( list( range( 128, 256 ) ) ) +
                                  _pg22h( list( range( 128, 256 ) ) ) +
                                  _pg23h( list( range( 128, 256 ) ) ) )

      # Decode function to be used to find the full parameter description as given
      # by CMIS.
      def getVdmParamLongName( rawParam ):
         # Follow-up will mut will remove this conversion
         paramType = bytes.fromhex( f'{rawParam:x}' )
         return decodeCoherent.vdmParamTypeDecode.get( paramType,
                                                      "Unknown parameter" )

      for vdmParamVal in vdmConfigVal.values():
         pageNum, paramId = divmod( vdmParamVal.id, paramsPerPage )
         if pageNum > 3:
            # Cmis 4 supports up to 4 pages. In case we will get a version with
            # more than 4 pages, we won't show them until we'll add support
            continue
         _pg = _pgs[ pageNum ]
         if not vdmConfigPages[ pageNum ]:
            vdmConfigPages[ pageNum ] = vdmConfig.addField(
                                           "VDM group %d" % ( pageNum + 1 ),
                                           _pg( list( range( 128, 256 ) ) ) )
         vdmPage = vdmConfigPages[ pageNum ]
         vdmParam = vdmPage.addField( "Parameter %d" % ( vdmParamVal.id + 1 ),
                                      _pg( 128 + 2 * paramId,
                                           128 + 2 * paramId + 1 ) )
         vdmParam.addField( "Threshold ID", _pg( 128 + 2 * paramId ),
                            raw=vdmParamVal.thresholdsId, decode=int )
         vdmParam.addField( "Lane", _pg( 128 + 2 * paramId ),
                            raw=vdmParamVal.lane, decode=int )
         rawParamType = vdmParamVal.type.encode().hex()
         vdmParam.addField( "Parameter type", _pg( 128 + 2 * paramId + 1 ),
                            raw=int( rawParamType, base=16 ),
                            decode=getVdmParamLongName )
      # 2Fh - VDM advertisement and control
      model.addField( "Number of VDM groups supported", _pg2Fh( 128 ),
                      raw=pgData.maxVdmPageGroup, decode=int )

loopBackRegBitDefs = [
   ( 6, "Simultaneous host and media side loopbacks" ),
   ( 5, "Per lane media side loopbacks" ),
   ( 4, "Per lane host side loopbacks" ),
   ( 3, "Host side input loopback" ),
   ( 2, "Host side output loopback" ),
   ( 1, "Media side input loopback" ),
   ( 0, "Media side output loopback" ),
]

def addCmisLoopbackCapsToModel( pgData, model ):
   if not pgData.testPatternContents:
      return
   # Convert decimal to 8-bit binary string, left padded by 0s.  Then reverse the
   # string so we can index bit 0 of the register using index 0 into our string.
   lbCapsReg = '{:08b}'.format(
       pgData.testPatternContents.prbsLoopbackCaps )[ : : -1 ]
   loopbackCaps = model.addField( "Loopback capabilities", _pg13h( 128 ) )

   def decoder( v ):
      return bool( int( v ) )
   for idx, name in loopBackRegBitDefs:
      loopbackCaps.addField( name,
                            _pg13h( 128 ),
                            raw=lbCapsReg[ idx ],
                            decode=decoder )
