# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Tac
import EthIntfLib
import LazyMount
import XcvrErrorCliLib as xecl
from XcvrLib import ( getXcvrSlotName, getXcvrStatus, isCmisTransceiver,
                      isPrimaryIntf )
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCli import getAllIntfsWrapper
import CliGlobal

XEM = CliDynamicPlugin( 'XcvrErrorModel' )

# --------------------------------------------------------------------------------
#
# "show transceiver error [ interface <INTFS> ]"
#
# --------------------------------------------------------------------------------

gv = CliGlobal.CliGlobal( xcvrStatusDir=None, xcvrErrorSliceDir=None )

def getCmisErrorStatus( intfName, xcvrStatus ):
   """
   Retrieves the cmisErrorStatus object from gv.xcvrErrorSliceDir
   for a particular interface.

   Parameters
   __________
   intfName: str
      The name of the interface
   xcvrStatus:
      The xcvrStatus object for the given interface
   """
   isModular = Cell.cellType() == "supervisor"
   sliceName = 'FixedSystem'
   # Supervisors having xcvr slots will be present in FixedSystem slice
   if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
   xcvrErrorDir = gv.xcvrErrorSliceDir[ sliceName ]

   # XcvrError currently only supports CMIS transceivers.
   # It is possible that xcvrStatus exists before cmisErrorDir.
   if isCmisTransceiver( xcvrStatus ) and xcvrErrorDir.cmisErrorDir is not None:
      slot = getXcvrSlotName( intfName )
      return xcvrErrorDir.cmisErrorDir.cmisErrorStatus.get( slot )
   return None

def _populateXcvrErrorInfo( xcvrErrorInfo, errorKey, model, lastReason=False ):
   """
   Populates the xcvr error CLI model for a given error with data from
   cmisErrorStatus if the error is supported by the modules' CMIS version.

   Parameters
   __________
   xcvrErrorInfo : Xcvr::XcvrErrorInfo
      The error tracked inside cmisErrorStatus
   errorKey : str
      One of the entries in the list errorKeys in XcvrErrorCliLib that identifies
      the error data to populate
   model : TransceiverErrorValues instance
      The mapping between an error and its values in the CLI model
   lastReason : bool
      Indicates if the error has a reason field to populate
   """
   # Check that the error is supported in this module's CMIS version.
   if xcvrErrorInfo.supported:
      modelKey = xecl.lowerCamelCase( errorKey )
      model.transceiverErrors[ modelKey ] = XEM.TransceiverErrorValueFormat()
      errorValue = model.transceiverErrors[ modelKey ]
      errorValue.count = xcvrErrorInfo.state.count
      errorValue.lastError = xcvrErrorInfo.state.last
      # Most errors tracked do not have a reason associated with them.
      if lastReason:
         errorValue.lastReason = xcvrErrorInfo.reason

def _populateXcvrErrors( portName, intfName ):
   """
   Populates the CLI model for 'show transceiver error' for all the errors
   tracked inside cmisErrorStatus.

   Parameters
   __________
   portName : str
      The port name for the module
   intfName : str
      The interface name for the module
   """
   model = None
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( portName ) )
   if xcvrStatus is None:
      return model

   cmisErrorStatus = getCmisErrorStatus( intfName, xcvrStatus )

   # XcvrError currently only supports CMIS transceivers.
   # Make sure cmisErrorStatus and its attrs have been instantiated.
   if cmisErrorStatus is None or cmisErrorStatus.cmisConfigState is None:
      return model

   model = XEM.TransceiverErrorValues()
   for errorKey in xecl.errorKeys:
      if errorKey == xecl.moduleFaultKey:
         _populateXcvrErrorInfo( cmisErrorStatus.moduleFault, errorKey, model,
                                 lastReason=True )
      elif errorKey == xecl.modulePwrDwnKey:
         _populateXcvrErrorInfo( cmisErrorStatus.modulePowerDownTimeout, errorKey,
                                 model )
      elif errorKey == xecl.modulePwrUpKey:
         _populateXcvrErrorInfo( cmisErrorStatus.modulePowerUpTimeout, errorKey,
                                 model )
      elif errorKey == xecl.cmisConfigKey:
         _populateXcvrErrorInfo( cmisErrorStatus.cmisConfigState, errorKey, model,
                                 lastReason=True )
      elif errorKey == xecl.mixedApplicationsKey:
         _populateXcvrErrorInfo( cmisErrorStatus.mixedApplicationState, errorKey,
                                 model, lastReason=True )
      elif errorKey == xecl.smbusKey:
         _populateXcvrErrorInfo( cmisErrorStatus.smbusError, errorKey, model,
                                 lastReason=True )
      elif errorKey == xecl.faultStateKey:
         _populateXcvrErrorInfo( cmisErrorStatus.faultyState, errorKey, model,
                                 lastReason=True )
      elif errorKey == xecl.moduleFirmwareKey:
         _populateXcvrErrorInfo( cmisErrorStatus.moduleFirmwareError, errorKey,
                                 model )
      elif errorKey == xecl.dpFirmwareKey:
         _populateXcvrErrorInfo( cmisErrorStatus.dpFirmwareError, errorKey, model )
      elif errorKey == xecl.dpDeinitKey:
         _populateXcvrErrorInfo( cmisErrorStatus.dpDeinitTimeout, errorKey, model )
      elif errorKey == xecl.vdmFreezeKey:
         _populateXcvrErrorInfo( cmisErrorStatus.vdmFreezeTimeout, errorKey, model )
      elif errorKey == xecl.vdmUnfreezeKey:
         _populateXcvrErrorInfo( cmisErrorStatus.vdmUnfreezeTimeout, errorKey,
                                 model )
   return model

def _showXcvrErrorInterfaces( mode, intf, mod ):
   model = XEM.TransceiverErrorInterfaces()

   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   for intfName in intfNames:
      # If this is not a primary interface, print nothing
      if not isPrimaryIntf( intfName ):
         continue
      portName = getXcvrSlotName( intfName )
      if portName not in model.ports:
         xcvrErrorModel = _populateXcvrErrors( portName, intfName )
         if xcvrErrorModel is None:
            continue
         model.ports[ portName ] = xcvrErrorModel

   return model

def transceiverErrorCmdHandler( mode, args ):
   intf = args.get( 'INTFS' )
   mod = args.get( 'MOD' )
   return _showXcvrErrorInterfaces( mode, intf, mod )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrErrorSliceDir = LazyMount.mount( em, "hardware/xcvr/status/error/slice",
                                        "Tac::Dir", "ri" )
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
