# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliGlobal
from CliPlugin import IntfCli
from CliPlugin.PhyTxEqLaneConfigCli import isPrimaryIntf
import ConfigMount
from TypeFuture import TacLazyType

gv = CliGlobal.CliGlobal( polarityOverrideDir=None )

EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
FabricIntfId = TacLazyType( 'Arnet::FabricIntfId' )
InterfaceSlotDescriptor = \
  TacLazyType( 'Hardware::L1Topology::InterfaceSlotDescriptor' )
IntfSlotGroupId = TacLazyType( 'L1::IntfSlotTypes::IntfSlotGroupId' )
Polarity = TacLazyType( 'Hardware::L1Topology::Polarity' )
PolarityOverrideWriter = \
  TacLazyType( 'Hardware::L1Topology::PolarityOverrideWriter' )

def intfSlotDescFromIntfName( intfName ):
   isFixedSystem = Cell.cellType() == 'fixed'
   if EthIntfId.isEthIntfId( intfName ):
      slotPrefix = 'Ethernet'
      slotGroupId = IntfSlotGroupId.unknownIntfSlotGroupId if isFixedSystem \
                    else EthIntfId.module( intfName )
      intfSlotId = EthIntfId.port( intfName )
   elif FabricIntfId.isFabricIntfId( intfName ):
      slotPrefix = 'Fabric'
      slotGroupId = IntfSlotGroupId.unknownIntfSlotGroupId if isFixedSystem \
                    else FabricIntfId.cardSlot( intfName )
      intfSlotId = FabricIntfId.intfSlot( intfName )
   else:
      # Currently, this function should only be used for front panel interface types
      # of ethernet or fabric
      return None

   return InterfaceSlotDescriptor( slotPrefix, slotGroupId, intfSlotId )

def handler( mode, args ):
   intfName = mode.intf.name
   if not isPrimaryIntf( intfName ):
      mode.addWarning( f'Intf: { intfName } is not a primary interface' )
      return

   intfSlotDesc = intfSlotDescFromIntfName( intfName )
   if not intfSlotDesc:
      mode.addWarning(
         'Command is only supported on Fabric and Ethernet interfaces' )
      return

   lanes = args[ 'LANES' ].values()
   tx = 'transmitter' in args
   if 'standard' in args:
      polarity = Polarity.polarityDefault
   else:
      polarity = Polarity.polaritySwap

   polarityOverrideWriter = PolarityOverrideWriter( gv.polarityOverrideDir )
   for lane in lanes:
      polarityOverrideWriter.addLanePolarityOverride( intfSlotDesc,
                                                      lane,
                                                      tx,
                                                      polarity )

      # TODO BUG1000059 - When we move to port level config, the primaryIntfId
      # attribute will be removed and this function should be removed as well.
      polarityOverrideWriter.setPrimaryIntfIdForSlotPolarityOverride( intfSlotDesc,
                                                                      intfName )

def noOrDefaultHandler( mode, args ):
   intfName = mode.intf.name
   if not isPrimaryIntf( intfName ):
      mode.addWarning( f'Intf: { intfName } is not a primary interface' )
      return

   intfSlotDesc = intfSlotDescFromIntfName( intfName )
   if not intfSlotDesc:
      mode.addWarning(
         'Command is only supported on Fabric and Ethernet interfaces' )
      return

   lanes = args[ 'LANES' ].values()
   tx = 'transmitter' in args

   polarityOverrideWriter = PolarityOverrideWriter( gv.polarityOverrideDir )
   for lane in lanes:
      polarityOverrideWriter.delLanePolarityOverride( intfSlotDesc,
                                                      lane,
                                                      tx )

class XcvrPolarityCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      intfSlotDesc = intfSlotDescFromIntfName( intfName )
      del gv.polarityOverrideDir.slotPolarityOverride[ intfSlotDesc ]

def Plugin( entityManager ):
   gv.polarityOverrideDir = ConfigMount.mount(
      entityManager,
      'intfSlot/config/polarity/override',
      'Hardware::L1Topology::PolarityOverrideDir',
      'wi' )
   IntfCli.Intf.registerDependentClass( XcvrPolarityCleaner )
