#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrConfigCli import ( intfCliReject,
                                      getXcvrConfigCliDir,
                                      getXcvrConfigCliForConfigCommand,
                                      xcvrConfigCliDelete )
from CliPlugin.XcvrPrbsCli import detailKw, getTestPatternStatusDir
import CliGlobal
import TypeFuture
from XcvrPrbsLib import ( prbsCliHostKw, prbsCliMediaKw,
                          prbsCliCheckerKw, fecEnableKw,
                          aristaUnsupportedPatterns )
from XcvrPrbsLib import ( ClockSourceEnum, SideEnum )
from XcvrPrbsLib import prbsCliSideToTacSide
from XcvrLib import getXcvrSlotName, isCmisType

gv = CliGlobal.CliGlobal( xcvrStatusDir=None )
PresenceEnum = TypeFuture.TacLazyType( "Xcvr::XcvrPresence" )
XPM = CliDynamicPlugin( 'XcvrPrbsModel' )

# ------------------------------------------------------------------------------
#
# [ no | default ] transceiver diag test pattern <PATTERN>
# {line | system} {transmitter | receiver} [fec]
#
# --------------------------------------------------------------------------------

def transceiverDiagTestPatternCmdHandler( mode, args ):
   intfName = mode.intf.name

   hostOrMedia = args[ 'HOST_MEDIA' ]
   genOrCheck = args[ 'GENERATOR_CHECKER' ]
   pattern = args[ 'PATTERN_TYPE' ]

   side = prbsCliSideToTacSide( hostOrMedia, genOrCheck )
   # We can collect the user options in a new
   # sideConfig object for xcvrConfigCli
   newSideConfig = Tac.Value( "Xcvr::TestPatternSideConfig" )
   newSideConfig.pattern = pattern
   newSideConfig.enabled = True
   if genOrCheck == prbsCliCheckerKw:
      # Default clock for receivers is 'recovered'
      newSideConfig.clockSource = ClockSourceEnum.recovered
   newSideConfig.fec = fecEnableKw in args

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName,
                                                     xcvrConfigCliDir,
                                                     True )
   prbsConfig = xcvrConfigCli.testPatternConfig

   if not prbsConfig:
      xcvrConfigCli.testPatternConfig = ( intfName, )
      prbsConfig = xcvrConfigCli.testPatternConfig
      # Otherwise we just add the side config then change generationId
      # to trigger the reactor
   prbsConfig.sideConfigIs( newSideConfig, side )
   prbsConfig.generationId += 1

def transceiverDiagTestPatternCmdNoOrDefaultHandler( mode, args ):
   # no/default command is unguarded, need to make sure that if we replace
   # a module, we can still delete the test pattern configuration
   intfName = mode.intf.name
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName,
                                                     xcvrConfigCliDir,
                                                     False )
   if not xcvrConfigCli:
      return

   # If we don't have a testPatternConfig object, we have nothing
   # to delete, except the xcvrConfigCli in some instances
   if xcvrConfigCli.testPatternConfig:
      hostOrMedia = args.get( 'HOST_MEDIA' )
      genOrCheck = args.get( 'GENERATOR_CHECKER' )
      if hostOrMedia and genOrCheck:
         side = prbsCliSideToTacSide( hostOrMedia, genOrCheck )
         inactiveSideConfig = Tac.Value( "Xcvr::TestPatternSideConfig" )
         xcvrConfigCli.testPatternConfig.sideConfigIs( inactiveSideConfig, side )
         xcvrConfigCli.testPatternConfig.generationId += 1
      else:
         xcvrConfigCli.testPatternConfig = None

   if xcvrConfigCliDelete( xcvrConfigCli ):
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern
#
# ------------------------------------------------------------------------------

def _populateSideConfigModel( options ):
   """
   This model must always be populated, since a non-null options object will
   always have all the necessary attribute. Thus all options parsed here are assumed
   to be active

   Parameters
   ----------
   options: Xcvr::TestPatternSideConfig
      This object represents the config/options for a side. Can be configured
      ( has been entered by the user ) or operational ( is actively being used by
      the module )
   """
   model = XPM.InterfacesTransceiverTestPatternSideConfig()
   model.pattern = options.pattern
   model.fec = options.fec
   return model

def _populateSideConfigsModel( configuredOptions, status, intfName, side ):
   """
   In order to display a side config, there are 3 conditions:
   1. The base object ( TestPatternConfig or TestPatternStatus ) must exist
   2. The side config object ( TestPatternConfig.get( some side ) or
      TestPatternSideStatus.operationalConfig ) must exist
   3. The config must be active ( indicated by the 'enabled' attribute )

   Parameters
   ----------
   configuredOptions: Xcvr::TestPatternConfig
      Holds the configured options ( most recently entered by the user )
      for all sides
   status: Xcvr::TestPatternSideStatus
      Holds the operational options ( the options currently running in the
      module ) for all sides.
   intfName: Arnet::IntfId
      The id for the interface to whom these options objects belong
   side: Xcvr::TestPatternSide
      The side to whom the options parsed in the function belong
   """
   # If the options for that side are empty, then we'll just print none
   model = XPM.InterfacesTransceiverTestPatternSideConfigs()

   if configuredOptions and configuredOptions.sideConfig( side ):
      sideConfig = configuredOptions.sideConfig( side )
      if sideConfig.enabled:
         model.configuredOptions = _populateSideConfigModel( sideConfig )
   if status and status.sideStatus( side ):
      operationalOptions = status.sideStatus( side ).operationalConfig
      if operationalOptions and operationalOptions.enabled:
         model.operationalOptions = _populateSideConfigModel( operationalOptions )
   return model

def _populateConfigModel( intfName, side ):
   """
   This function parses the configuration for a given interface, either on the line
   or media side

   Parameters
   ----------
   intfName: Arnet::IntfId
      The interface whose options are parsed
   side:
      The physical side ( line/media or host/system ) of the options that are parsed
   """
   model = XPM.InterfacesTransceiverTestPatternConfiguration()

   prbsDir = getTestPatternStatusDir( intfName )
   # If there's no prbsDir, then prbs is not supported by any transceiver in the dut,
   # so we don't have to render anything.
   if not prbsDir:
      return model
   prbsStatus = prbsDir.testPatternStatus.get( intfName )

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     False )

   # If the config object is not present, we still want to pass a mock object
   # into the populating function so that we'll render a 'none'
   prbsConfiguredOptions = None
   if xcvrConfigCli and xcvrConfigCli.testPatternConfig:
      prbsConfiguredOptions = xcvrConfigCli.testPatternConfig

   if side == prbsCliHostKw:
      model.transmitter = _populateSideConfigsModel( prbsConfiguredOptions,
                                                     prbsStatus, intfName,
                                                     SideEnum.hostGenerator )
      model.receiver = _populateSideConfigsModel( prbsConfiguredOptions,
                                                  prbsStatus, intfName,
                                                  SideEnum.hostChecker )
   elif side == prbsCliMediaKw:
      model.transmitter = _populateSideConfigsModel( prbsConfiguredOptions,
                                                     prbsStatus, intfName,
                                                     SideEnum.mediaGenerator )
      model.receiver = _populateSideConfigsModel( prbsConfiguredOptions,
                                                  prbsStatus, intfName,
                                                  SideEnum.mediaChecker )
   return model

def showInterfacesTransceiverTestPatternConfiguration( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   baseModel = XPM.InterfacesTransceiverTestPatternConfigurationCollection()
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return baseModel

   for intf in intfs:
      intfName = intf.name
      # Don't want secondary interfaces showing up in the output
      if intfCliReject( intf ):
         continue
      slotName = getXcvrSlotName( intfName )
      status = gv.xcvrStatusDir.xcvrStatus.get( slotName )
      if( not status or status.presence != PresenceEnum.xcvrPresent or
          not isCmisType( status.xcvrType ) ):
         continue
      systemModel = _populateConfigModel( intfName, prbsCliHostKw )
      lineModel = _populateConfigModel( intfName, prbsCliMediaKw )
      baseModel.systemInterfaces[ intfName ] = systemModel
      baseModel.lineInterfaces[ intfName ] = lineModel
   return baseModel

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern capabilities
#
# ------------------------------------------------------------------------------

def _populateSideCapsModel( sideCaps ):
   model = XPM.InterfacesTransceiverTestPatternSideCapabilities()
   if not sideCaps:
      return model
   sidePatterns = sideCaps.patternCapabilities
   model.supportedPatterns = [ pattern for pattern in sidePatterns
                               if ( sidePatterns[ pattern ] and
                                    pattern not in aristaUnsupportedPatterns ) ]
   model.fecSupported = ( sideCaps.fecSupported and sideCaps.preFecSupported and
                          sideCaps.postFecSupported )
   return model

def _populateCapabilitiesModel( intfName ):
   model = XPM.InterfacesTransceiverTestPatternCapabilities()

   prbsDir = getTestPatternStatusDir( intfName )
   if not prbsDir:
      return model

   prbsCaps = prbsDir.capabilities.get( intfName )

   if not prbsCaps:
      return model

   model.systemTransmitter = _populateSideCapsModel( prbsCaps.hostGeneratorCaps )
   model.systemReceiver = _populateSideCapsModel( prbsCaps.hostCheckerCaps )
   model.lineTransmitter = _populateSideCapsModel( prbsCaps.mediaGeneratorCaps )
   model.lineReceiver = _populateSideCapsModel( prbsCaps.mediaCheckerCaps )
   return model

def showInterfacesTransceiverTestPatternCapabilities( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   baseModel = XPM.InterfacesTransceiverTestPatternCapabilitiesCollection()
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return baseModel

   for intf in intfs:
      intfName = intf.name
      # Don't want secondary interfaces showing up in the output
      if intfCliReject( intf ):
         continue
      slotName = getXcvrSlotName( intfName )
      status = gv.xcvrStatusDir.xcvrStatus.get( slotName )
      if( not status or status.presence != PresenceEnum.xcvrPresent or
          not isCmisType( status.xcvrType ) ):
         continue
      model = _populateCapabilitiesModel( intfName )
      baseModel.interfaces[ intfName ] = model
   return baseModel

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern counters
#
# ------------------------------------------------------------------------------

def _populateLaneCountersModel( prbsLaneStatus, detailed ):
   """
   This function parses the test pattern counters and link status
   for a given lane belonging to an interface.

   Parameters
   ----------
   prbsLaneStatus: Xcvr::XcvrTestPatternLaneStatus
      The sysdb entity containing test pattern counters and link status
   detailed: bool
      Indicates if we're using the detailed version of the counters command
   """
   model = XPM.InterfacesTransceiverTestPatternLaneCounters()
   model.linkState = prbsLaneStatus.locked
   model.bitErrors = prbsLaneStatus.totalErrors
   model.largestBurst = prbsLaneStatus.largestErrorBurst
   model.burstCount = prbsLaneStatus.changes
   # We want to show the timestamp as "never" to the customer unless it's been
   # updated at least once, instead of taking the default
   if prbsLaneStatus.lastChange != Tac.endOfTime:
      model.lastErrorTime = Tac.utcNow() - ( Tac.now() - prbsLaneStatus.lastChange )
   # If we're using the detailed version of the command, we populate a new
   # lane-indexed submodel containing extra information
   if detailed:
      model.detailed = XPM.InterfacesTransceiverTestPatternLaneCountersDetail()
      model.detailed.totalBits = prbsLaneStatus.totalBits
      model.detailed.ber = prbsLaneStatus.ber
      model.detailed.linkStateChanges = prbsLaneStatus.linkStateChanges
      if prbsLaneStatus.lastLinkStateChange != Tac.endOfTime:
         model.detailed.lastLinkStateChange = \
            Tac.utcNow() - ( Tac.now() - prbsLaneStatus.lastLinkStateChange )
   return model

def _populateCountersModel( intfName, side, detailed ):
   """
   This function parses the test pattern counters and link status
   for a given interface and side

   Parameters
   ----------
   intfName: Arnet::IntfId
      The interface whose counters are parsed
   side: String
      The physical side (line/media or host/system) of the counters that are parsed
   detailed: Bool
      Indicates if we're using the detailed version of the counters command
   """
   model = XPM.InterfacesTransceiverTestPatternCounters()

   prbsDir = getTestPatternStatusDir( intfName )
   if not prbsDir:
      return None

   prbsStatus = prbsDir.testPatternStatus.get( intfName )

   if not prbsStatus:
      return None

   # Get the number of channels to display from xcvr status attributes:
   # maxChannels for the host/electrical side, lineSideChannels for the
   # media/optical side.
   slotName = getXcvrSlotName( intfName )
   status = gv.xcvrStatusDir.xcvrStatus.get( slotName )
   if side == prbsCliHostKw:
      prbsSideStatus = prbsStatus.hostCheckerStatus
      numLanes = status.capabilities.maxChannels
   elif side == prbsCliMediaKw:
      prbsSideStatus = prbsStatus.mediaCheckerStatus
      numLanes = status.capabilities.lineSideChannels
   else:
      return None

   if not prbsSideStatus or not prbsSideStatus.testPatternLaneStatus:
      return None

   # If we're using the detailed version of the command, beyond adding new attributes
   # at the lane level, we also need to add the last reset time for prbs and the
   # current pattern in use.
   if detailed:
      model.lastClear = Tac.utcNow() - ( Tac.now() - prbsSideStatus.lastClear )
      operationalConfig = prbsSideStatus.operationalConfig
      if operationalConfig and operationalConfig.enabled:
         model.operationalPattern = operationalConfig.pattern

   for lane in range( numLanes ):
      prbsLaneStatus = prbsSideStatus.testPatternLaneStatus.get( lane )
      if prbsLaneStatus:
         model.lanes.append( _populateLaneCountersModel( prbsLaneStatus, detailed ) )
      else:
         model.lanes.append( None )
   return model

def showInterfacesTransceiverTestPatternCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detailed = detailKw in args
   baseModel = XPM.InterfacesTransceiverTestPatternCountersCollection()
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return baseModel
   baseModel.detailedIs( detailed )

   for intf in intfs:
      intfName = intf.name
      # Don't want secondary interfaces showing up in the output
      if intfCliReject( intf ):
         continue
      slotName = getXcvrSlotName( intfName )
      status = gv.xcvrStatusDir.xcvrStatus.get( slotName )
      if( not status or status.presence != PresenceEnum.xcvrPresent or
          not isCmisType( status.xcvrType ) ):
         continue
      systemModel = _populateCountersModel( intfName, prbsCliHostKw, detailed )
      lineModel = _populateCountersModel( intfName, prbsCliMediaKw, detailed )
      if systemModel:
         baseModel.systemInterfaces[ intfName ] = systemModel
      if lineModel:
         baseModel.lineInterfaces[ intfName ] = lineModel
   return baseModel

def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
