#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Logging
import os
import re

import BasicCliUtil
import Cell
import CliGlobal
import ConfigMount
import LazyMount
import Tac

from CliDynamicPlugin.XcvrSimulateRemoveCliLib import (
   transceiverDiagSimulateRemovedCmdHelper,
   getXcvrConfigCli )
from CliPlugin.XcvrRiserPowerCycleCli import hasIslRiser, hasEcbRiser
from XcvrLib import TRANSCEIVER_POWER_CYCLE

gv = CliGlobal.CliGlobal( entMibStatus=None, hwEcbDirConfig=None,
                          hwIslConfig=None, islSystemStatus=None,
                          pciDeviceStatusDir=None, powerFuseConfig=None,
                          scdStatusDir=None, xcvrConfigCliSliceDir=None )

def cardPciAddr( linecard: str ) -> str:
   '''
   Find the PCI address of the linecard SCD

   linecard : the linecard number such as "5"

   Returns the pci address
   '''
   pciDevices = {}
   for pciDevice in gv.pciDeviceStatusDir.pciDeviceStatus.values():
      pciAddr = pciDevice.addr
      pciDeviceName = pciDevice.name
      if 'pciFpga' in pciDeviceName:
         slotName = pciDeviceName.split( ':' )[ 0 ].lstrip( 'Slot' )
         pciId = f"{pciAddr.busNumber:02x}:{pciAddr.device:02x}.{pciAddr.function:x}"
         pciDevices[ slotName ] = pciId

   assert linecard in pciDevices, \
      f"slot {linecard} FPGA not found in PCI addresses"
   return pciDevices[ linecard ]

def riserFromPort( xcvrPort: int, linecard: str,
                   islPowerCycleEnabled: bool,
                   ecbPowerCycleEnabled: bool ):
   '''
   Verify the riser-port mapping standard and calculate the riser # from the port.

   xcvrPort : the port on the front of the card (1-36/48)

   linecard : the linecard number

   islPowerCycleEnabled : if the riser is controlled by isl

   ecbPowerCycleEnabled : if the riser is controlled by an ecb type chip

   Returns the riser number, the set of interfaces attached to the riser, and the
   riser card name
   '''
   lcName = f"Linecard{linecard}"
   lcPort = f"{linecard}/{xcvrPort}"
   riser = None
   intfs = {}
   riserName = ""

   # if ecb is controlling the riser cards, use ecb to derive the ports first
   if ecbPowerCycleEnabled:
      for ecb in gv.hwEcbDirConfig.card[ lcName ].ecb.values():
         if lcPort in ecb.intfs:
            riser = ecb.riserNum
            intfs = ecb.intfs
            riserName = ecb.name
   if islPowerCycleEnabled:
      # if ecb does not present, isl MUST present due to early filter
      # if both ecb and isl present, override with isl for later usage
      for powerController in gv.hwIslConfig.controller.values():
         if lcPort in powerController.intfs:
            riser = powerController.riserNum
            intfs = powerController.intfs
            riserName = powerController.name
   return riser, intfs, riserName

def isPowerCycleByScript( riserName: str ) -> bool:
   '''
   Check if the Isl chip power cycle process is enabled for script based method

   riserName : the name of the riser card

   return True if this Isl supports script based power cycle method
   '''
   powerCycleMethod = Tac.Type( "Hardware::PowerController::PowerCycleMethod" )
   return ( gv.hwIslConfig.controller[ riserName ].powerCycleMethod ==
            powerCycleMethod.powerCycleByScript )

def riserOffCheck( linecard: str, riser: str,
                   islPowerCycleEnabled: bool ) -> bool:
   '''
   Check if the given riser card is powered off. Procedure includes two parts:

   1) If SCD riser power good check pin is no good;
   2) (for SilverthroneMs) If the Isl rails are all offline.

   linecard : the linecard number like 5

   riser : the riser card number like 1

   islPowerCycleEnabled : if the riser is controlled by isl
   '''
   # check the Riser power good pin on SCD
   linecardName = "Linecard" + linecard
   # this will not work if there's a second SCD on the linecard, which is not the
   # case for Silverthrone & Wolverines
   scdName = "Scd" + linecard + ":0"
   scdStatus = gv.scdStatusDir[ linecardName ].scdStatus[ scdName ]
   riserPowerGoodStatus = scdStatus.riserPowerGoodStatus

   if ( 1 << ( int( riser ) - 1 ) ) & riserPowerGoodStatus != 0:
      # the riser power good pin is still on, riser is not off yet
      return False

   # if isl is controlling riser cards, check isl power too
   if islPowerCycleEnabled:
      riserName = linecardName + "Riser" + riser
      islControllerStatus = gv.islSystemStatus[ riserName ]
      if islControllerStatus.statusWord() & 0x0840 != 0x0840:
         return False

   return True

def validateCard( linecard: str ) -> bool:
   '''
   Validates that the linecard can support the recovery steps.

   1) Existence of the PCI addr of the linecard FPGA
   2) Existence of SCD

   linecard : the linecard number like 5
   '''

   # Currently only for check on availability of the PCI addr of the linecard FPGA
   cardPciAddr( linecard )

   entMibRoot = gv.entMibStatus.root
   if entMibRoot is None or entMibRoot.initStatus != "ok":
      print( "The system is not ready yet." )
      return False
   if int( linecard ) not in entMibRoot.cardSlot.keys():
      print( f">> Linecard {linecard} is not found on the switch" )
      return False
   card = entMibRoot.cardSlot[ int( linecard ) ].card
   if 0 not in card.chip or card.chip[ 0 ].tag != "scd-0":
      print( ">> Scd not available on the linecard." )
      return False
   return True

def xcvrRiserPowerCycleHandler( intfs: str, lc: str, riser: str,
                                impactedIntfs: list[ str ],
                                islPowerCycleEnabled: bool ) -> None:
   '''
   Main handler for Xcvr riser power cycling.
   Note here we use Smbus instead of the bash scd programming approach introduced in
   Xcvr/xcvrRiserPowerCycle for design purpose. The other approach is kept for the
   usage on Clearwater2.

   intfs : interface name in the form '<Ethernet><linecard>/<port>/1'
   lc : linecard number, e.g., some number like 5
   riser : the riser card that has the affected port
   impactedIntfs : all the affected ports on the riser
   islPowerCycleEnabled : if the riser is controlled by isl
   '''
   # The slotChecker has limitation verifying the validity of input linecard & port
   # So we first check on the correctness of the input
   assert validateCard( lc )

   # Steps
   # --------------------------------------------------------
   # (1)           Simulate removal of Xcvr
   # --------------------------------------------------------
   # (2)                Power off riser
   # --------------------------------------------------------
   # (3)             Wait for riser offline
   # --------------------------------------------------------
   # (4)                Power on riser
   # --------------------------------------------------------
   # (5)          Simulate insertion of Xcvr
   # --------------------------------------------------------
   #    State completion of power cycle recovery

   print( f">> Beginning recovery of {intfs}" )
   print( f">> Impacted interfaces: {impactedIntfs}" )

   powerCycleSuccess = True

   # (1) Simulate removal of Xcvr
   #     If there is a config already, record the related parameter values
   currentForceModuleRemoved = set()
   for intf in impactedIntfs:
      xcvrConfigCli = getXcvrConfigCli( intf, gv.xcvrConfigCliSliceDir )
      if xcvrConfigCli and xcvrConfigCli.forceModuleRemoved:
         currentForceModuleRemoved.add( intf )
      else:
         transceiverDiagSimulateRemovedCmdHelper(
            intf, True, gv.xcvrConfigCliSliceDir )

   # (2) Power off riser
   gv.powerFuseConfig.powerOffRequested[ f"Linecard{lc}Riser{riser}" ] = (
      "Riser power cycle" )

   # (3) Wait for riser offline
   #     tests on DUT indicate ~3.6s wait before riser card is offline, set to 10s to
   #     (i) ensure the riser is powered off, and (ii)the timeout is not too long
   #     (compared to the 10s delay of Clearwater2 power cycle).
   Tac.waitFor( lambda: riserOffCheck( lc, riser, islPowerCycleEnabled ), sleep=True,
                timeout=10, warnAfter=None )

   # (4) Power on riser
   del gv.powerFuseConfig.powerOffRequested[ f"Linecard{lc}Riser{riser}" ]

   # (5) Simulate insertion of Xcvr
   for intf in impactedIntfs:
      if intf not in currentForceModuleRemoved:
         transceiverDiagSimulateRemovedCmdHelper(
            intf, False, gv.xcvrConfigCliSliceDir )

   if powerCycleSuccess:
      print( ">> Recovery complete!" )
      Logging.log( TRANSCEIVER_POWER_CYCLE,
                   ", ".join( re.match( r'((Ethernet)(?:\d+)/(?:\d+))',
                                        intf ).group() for intf in impactedIntfs ) )

def xcvrPowerCycleCliHandler( mode, args ):
   slot = str( args[ "SLOT" ] )

   # Match the input returned by slotMatcher i.e. SlotX/Y.
   m = re.match( r"[^\d]+(\d+)/(\d+)", slot )
   if not m:
      mode.addError( f"Invalid slot {slot}" )
      return

   linecard = int( m.group( 1 ) )
   port = int( m.group( 2 ) )

   cardSlots = gv.entMibStatus.chassis.cardSlot

   if linecard not in cardSlots or not cardSlots[ linecard ].card:
      mode.addError( f"Invalid slot {slot}" )
      return

   if port not in cardSlots[ linecard ].card.xcvrSlot:
      mode.addError( f"Invalid slot {slot}" )
      return

   # check the chip type controlling the riser
   islPowerCycleEnabled = hasIslRiser( int( linecard ) )
   ecbPowerCycleEnabled = hasEcbRiser( int( linecard ) )

   if not islPowerCycleEnabled and not ecbPowerCycleEnabled:
      mode.addError( f"Error power cycling slot {slot}" )
      return

   # get the riser number and related info
   riser, intfs, riserName = riserFromPort( int( port ), str( linecard ),
                                            islPowerCycleEnabled,
                                            ecbPowerCycleEnabled )

   # there can be some logical error causing riser & intfs not found
   if not riser or not intfs:
      mode.addError( f"Error power cycling slot {slot}" )

   # The list of impacted interfaces follows from the script xcvrRiserPowerCycle.
   # It specifies that the /1 interfaces will be the ones specified since this is
   # only relevant to MSFT. To make this more generic, the logic contained below
   # must correctly enumerate all active broken-out interfaces.
   impactedIntfs = sorted( [ f"Ethernet{intf}/1" for intf in intfs ] )

   warning = (
      "WARNING\r\n"
      "This command will cause the following interfaces to flap: \r\n\r\n"
      + "\r\n".join( f"    {intf}" for intf in impactedIntfs ) +
      "\r\n"
   )
   mode.addWarning( warning )

   prompt = "Do you wish to proceed with this command? (y/[N])? "
   if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
      return

   # if the "Pass Riser Card Power Good to DPM" bit is on, the smbus approach is not
   # possible, and thus we need to use the old PCIe based approach
   # Here, this is only the case for Clearwater2 after the Cli command guard
   interfaceName = slot.lower().replace( "slot", "Ethernet" ) + "/1"
   if islPowerCycleEnabled and isPowerCycleByScript( riserName ):
      command = [
         "arista-python",
         "/usr/bin/xcvrRiserPowerCycle",
         # The script expects the following format:
         interfaceName
      ]

      if "SIMULATION_VMID" in os.environ:
         command.append( "--dryRun" )

      with open( "/var/tmp/xcvrPowerCycleLog", "w" ) as fd:
         try:
            Tac.run( command, stdout=fd, stderr=fd )
         except Tac.SystemCommandError:
            mode.addError( f"Failed to power cycle slot {slot}" )
   else:
      xcvrRiserPowerCycleHandler( interfaceName, str( linecard ), str( riser ),
                                  impactedIntfs, islPowerCycleEnabled )

def Plugin( entityManager ):
   cellId = Cell.cellId()

   gv.entMibStatus = LazyMount.mount( entityManager,
                                      "hardware/entmib",
                                      "EntityMib::Status",
                                      "r" )

   gv.hwEcbDirConfig = LazyMount.mount( entityManager,
                                        "hardware/ecb/config",
                                        "Hardware::Ecb::EcbDirConfig",
                                        "r" )

   gv.hwIslConfig = LazyMount.mount( entityManager,
                                     "hardware/powercontroller/config/isl6812x",
                                     "Hardware::Isl6812X::Config",
                                     "r" )

   gv.islSystemStatus = LazyMount.mount( entityManager,
                                         "hardware/archer/powercontroller/status/" +
                                         "system",
                                         "Tac::Dir",
                                         "r" )

   gv.pciDeviceStatusDir = LazyMount.mount( entityManager,
                                            f"cell/{cellId}/hardware/" +
                                            "pciDeviceStatusDir",
                                            "Hardware::PciDeviceStatusDir",
                                            "r" )

   gv.powerFuseConfig = ConfigMount.mount( entityManager,
                                           "power/fuse/config/admin",
                                           "Power::SoftwareFuse",
                                           "w" )

   gv.scdStatusDir = LazyMount.mount( entityManager,
                                      "hardware/archer/scd/status/slice",
                                      "Tac::Dir",
                                      "r" )

   gv.xcvrConfigCliSliceDir = ConfigMount.mount( entityManager,
                                                 "hardware/xcvr/cli/config/slice",
                                                 "Tac::Dir", "wi" )
