#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import Tac
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrConfigCli import ( mhzFreqToGhz )
import CliPlugin.IntfCli
import CliPlugin.EthIntfCli
import CliPlugin.EthIntfModel
from CliPlugin.XcvrConfigCli import getXcvrConfigCliDir
from CliPlugin.XcvrCliLib import getAllIntfsWrapper, isDualLaserModule
import CliGlobal
from XcvrLib import ( getAndFilterPrintableIntfNameByLane, getLineSideChannels,
                      getXcvrSlotName, slotHardwareControllerStatus,
                      getXcvrStatus, isCmisTransceiver )
from TypeFuture import TacLazyType

gv = CliGlobal.CliGlobal( entityManager=None, xcvrStatusDir=None, xgc=None )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )

XSHM = CliDynamicPlugin( 'XcvrShowHardwareModel' )

# -------------------------------------------------------------------------------
# The "show interfaces [<name>] transceiver hardware" command, in enable mode.
#   show interfaces [<name>] transceiver hardware
#   show interfaces module <modnum> transceiver hardware
# -------------------------------------------------------------------------------
def getWavelength( status ):
   # Wavelength precision can be in nanometers, picometers, or
   # not supported at all (eg, copper media).  The display is
   # based on what that precision is.
   # Returns tuple ( wavelength, if precision is nm )
   if status and status.optical:
      if status.laserWavelengthPrecision == 'laserWavelengthPrecisionNm':
         return status.laserWavelength / 1000.0, True
      elif status.laserWavelengthPrecision == 'laserWavelengthPrecisionPm':
         if status.xcvrType.lower() == "qsfpplus":
            # qsfp directly provides the wavelength in pm. Convert it to nm.
            # This introduces the decimal to get away from integer math and
            # ensures the first decimal is printed.
            return status.laserWavelength / 1000.0, False
         return status.laserWavelength / 1000.0, False
   return None, None

def getGridSpacingCap( status ):
   # in GHz
   gridSpacingCap = status.resolvedTuningStatus.gridSpacingCapabilities
   spacing = ""
   if gridSpacingCap.gridSpacing100000M:
      spacing += "100.0,"
   if gridSpacingCap.gridSpacing50000M:
      spacing += "50.0,"
   if gridSpacingCap.gridSpacing33000M:
      spacing += "33.0,"
   if gridSpacingCap.gridSpacing25000M:
      spacing += "25.0,"
   if gridSpacingCap.gridSpacing12500M:
      spacing += "12.5,"
   if gridSpacingCap.gridSpacing6250M:
      spacing += "6.25"
   return spacing.strip( ',' )

# Helper function to return the cable type based on the media type.
# Returns False for media which do not have a cable type.
def getCableType( mediaType ):
   cableType = None
   if mediaType in ( XcvrMediaType.xcvr50GBaseCr2N,
                     XcvrMediaType.xcvr100GBaseCr4N,
                     XcvrMediaType.xcvr200GBaseCr8N,
                     XcvrMediaType.xcvr400GBaseCr8N,
                     XcvrMediaType.xcvr25GBaseCrN ):
      cableType = 'CA-N'
   elif mediaType in ( XcvrMediaType.xcvr50GBaseCr2S,
                       XcvrMediaType.xcvr100GBaseCr4S,
                       XcvrMediaType.xcvr200GBaseCr8S,
                       XcvrMediaType.xcvr400GBaseCr8S,
                       XcvrMediaType.xcvr25GBaseCrS ):
      cableType = 'CA-S'
   elif mediaType in ( XcvrMediaType.xcvr50GBaseCr2L,
                       XcvrMediaType.xcvr100GBaseCr4,
                       XcvrMediaType.xcvr200GBaseCr8,
                       XcvrMediaType.xcvr400GBaseCr8,
                       XcvrMediaType.xcvr25GBaseCr ):
      cableType = 'CA-L'
   return cableType

def _isTxDisableSupportedModule( status ):
   # Currently, this field only applies to dual laser cfp2 dco modules
   return isDualLaserModule( status )

def _populateHardwareTuning( name, status ):
   tuningConstants = Tac.Value( "Xcvr::TuningConstants" )
   rts = status.resolvedTuningStatus
   sfpPlusPresent = status.xcvrType == 'sfpPlus'
   tunableLaserEnabled = status.ituTuningConfig.laserEnabled
   tuningFrequencyConfigured = bool(
      ( rts.operationalFrequency > 0 and
        rts.operationalFrequency == rts.configuredFrequency and
        tunableLaserEnabled ) or
      isDualLaserModule( status ) )

   # _tunableLaserEnabled should be true always for tunable SFP+ or when configuring
   # CFP2 with a valid tuning frequency or channel/grid
   model = XSHM.InterfacesTransceiverHardwareTuning(
      _outputSelect=tuningFrequencyConfigured,
      _tunableLaserEnabled=tunableLaserEnabled or sfpPlusPresent,
      _dualLaserModulePresent=isDualLaserModule( status ) )

   if tunableLaserEnabled or sfpPlusPresent:
      if tuningFrequencyConfigured:
         model.configuredFrequency = mhzFreqToGhz( rts.configuredFrequency )
      else:
         model.configuredChannel = rts.configuredChannel
         model.configuredChannelUnsupportedIs( sfpPlusPresent and
            rts.configuredChannel != rts.operationalChannel and
            rts.configuredFrequency == 0 )
         model.configuredGrid = mhzFreqToGhz( rts.configuredGrid )
         model.computedFrequency = mhzFreqToGhz( rts.computedFrequency )
         model.operationalChannel = rts.operationalChannel
         model.operationalGrid = mhzFreqToGhz( rts.operationalGrid )
         model.operationalGridDefaultIs(
            rts.operationalGrid == tuningConstants.defaultGrid )
         # (default) label operational channel only shown for 10GBASE-DWDM
         model.operationalChannelDefaultIs( sfpPlusPresent and
            rts.operationalChannel == tuningConstants.defaultChannel and
            rts.operationalGrid == tuningConstants.defaultGrid )
      model.computedWavelength = rts.computedWavelength
      model.operationalFrequency = mhzFreqToGhz( rts.operationalFrequency )
      model.operationalWavelength = rts.operationalWavelength
   else:
      # populate only configured attributes in model when misconfigured
      if rts.configuredFrequency > 0:
         model.configuredFrequency = mhzFreqToGhz( rts.configuredFrequency )
         model.computedWavelength = rts.computedWavelength
      elif ( rts.configuredChannel > 0 and rts.configuredGrid > 0 and
             not isDualLaserModule( status ) ):
         model.configuredChannel = rts.configuredChannel
         model.configuredGrid = mhzFreqToGhz( rts.configuredGrid )
         model.computedFrequency = mhzFreqToGhz( rts.computedFrequency )
         model.computedWavelength = rts.computedWavelength

   # Dual laser DCO module capable of tuning both tx and rx paths. Rx configuration
   # and status should always be displayed for this module.
   if isDualLaserModule( status ):
      rrts = status.cfp2DcoNewStatus.resolvedRxTuningStatus
      model.configuredRxFrequency = mhzFreqToGhz( rrts.configuredFrequency )
      model.configuredRxFineFrequency = mhzFreqToGhz(
         rrts.configuredFineFrequency )
      model.computedRxWavelength = rrts.computedWavelength
      model.operationalRxFrequency = mhzFreqToGhz( rrts.operationalFrequency )
      model.operationalRxWavelength = rrts.operationalWavelength
   # Grid spacing capabilites for tunable transceivers only (i.e. CFPX-100G-DWDM,
   # 100G-DWDM-E, 10GBASE-DWDM)
   if rts.gridSpacingCapabilities:
      gridSpacingCap = getGridSpacingCap( status )
      model.gridSpacingCapabilities = gridSpacingCap
      model.gridSpacingCapabilitiesUnknownIs( gridSpacingCap == "" )

   return model

def _populateHardwarePower( status, cap, name ):
   model = XSHM.InterfacesTransceiverHardwarePower(
      _tunableLaserEnabled=status.ituTuningConfig.laserEnabled,
      _dualLaserModulePresent=isDualLaserModule( status ) )

   if cap.configurableTxPower:
      txPowerConfig = gv.xgc.txPowerConfig.get( name, None )
      if isCmisTransceiver( status ):
         if txPowerConfig:
            txPower = txPowerConfig.signalPower
         else:
            txPower = status.prgOutputPowerCapabilities.defaultPower
         model.configuredTxPower = txPower
         model.configuredTxPowerDefaultIs(
            txPower == status.prgOutputPowerCapabilities.defaultPower )
         lane = 0
         operationalTxPower = status.prgOutputPowerStatus[ lane ].configuredPower
         model.operationalTxPower = operationalTxPower
      else:
         powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
         if txPowerConfig:
            txPower = txPowerConfig.signalPower
         else:
            txPower = powerConstants.defaultTxPower
         model.configuredTxPower = txPower
         model.configuredTxPowerDefaultIs( txPower == powerConstants.defaultTxPower )

   if cap.configurableRxPower:
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      rxPowerConfig = gv.xgc.rxPowerConfig.get( name, None )
      rxPower = None
      if rxPowerConfig:
         rxPower = rxPowerConfig.signalPower
      model.configuredRxPower = rxPower
      # some transceivers, such as CFP2-DCO, do not support default Rx power and
      # do not provide operational Rx attenuation
      if not cap.directRxPowerConfig:
         model.configuredRxPowerDefaultIs( rxPower == powerConstants.defaultRxPower )
         model.operationalRxAttenuation = status.currRxAttenuation

   # BUG773150
   return model if model != XSHM.InterfacesTransceiverHardwarePower() else None

def _populateHardware( status, slotStatus, intfName, hardwareStatusDir ):
   """
   Parameters
   ----------
   status : Xcvr::XcvrNewStatus derived object
      The XcvrStatus belonging to the currently inserted transceiver

   slotStatus : Xcvr::XcvrNewStatus derived object
      The XcvrStatus belonging to the slot

   intfName : str
      The interface name we print on the CLI.
      For example, Ethernet16/3/1 or Ethernet24/1
   """
   # Uses helper functions to drive the logic for conditonally populating certain
   # attributes. Attributes which should be omitted from Cli output are set to None.
   cap = status.capabilities
   model = XSHM.InterfacesTransceiverHardware()
   model.modulePresent = getSlotControllerPresence( status, hardwareStatusDir,
                                                    slotStatus )
   model.mediaType, model.detectedMediaType = getHardwareMediaTypeStrings(
      model.modulePresent, status )
   model.cableType = getCableType( status.mediaType )
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   powerIgnoreState = xcvrConfigCli and xcvrConfigCli.modulePowerIgnore
   model.modulePowerIgnoredIs( powerIgnoreState )

   model.maxPowerSupportedCharacteristic = getHardwareSlotMaxPowerCharacteristic(
      slotStatus )
   model.powerCharacteristic = getHardwareMaxPowerCharacteristic(
      status.powerCharacteristic, model.modulePresent )
   wavelengthSupported = status.isOpticalDomSupported( status.mediaType )
   if wavelengthSupported and not cap.tunableWavelength:
      ( wavelength, precision ) = getWavelength( status )
      model.wavelength = wavelength
      model.wavelengthPrecNmIs( precision )
   model.power = _populateHardwarePower( status, cap, intfName )
   if cap.tunableWavelength:
      model.tuning = _populateHardwareTuning( intfName, status )
   if _isTxDisableSupportedModule( status ):
      model.txDisabled = (
         status.resolvedLaserTxOutputState !=
         Tac.Type( 'Xcvr::LaserTxOutputState' ).laserTxOutputEnabled )
   return model

def getHardwareSlotMaxPowerCharacteristic( slotStatus ):
   """
   Returns maxPowerSupportedCharacteristic unless slot type is an rj45

   Parameters
   ----------
   slotStatus
   """
   if slotStatus.xcvrType != XcvrType.rj45:
      # Omit power details for rj45 media
      return slotStatus.maxPowerSupportedCharacteristic
   else:
      return None

def getHardwareMaxPowerCharacteristic( powerCharacteristic: int,
                                       modulePresent: bool ):
   """
   Returns powerCharacteristic if a slot has a module present
   """
   if modulePresent:
      return powerCharacteristic
   else:
      # Module is either not present or has no detection capabilities (i.e rj45)
      # In either case we don't have a transceiver rated power characteristic.
      return None

def getHardwareMediaTypeStrings( modulePresent: bool, status ):
   """
   Returns both media strings if a module is present.
   Parameters
   ----------
   status
   slotStatus
   """
   # modulePresent is True/False for presence/absense, and None in the case of rj45
   if modulePresent:
      return status.mediaTypeString, status.detectedMediaTypeString
   elif modulePresent is False:
      # We know there is no media inserted.
      return None, None
   else:
      # modulePresence is None - (rj45 slots don't have presence pins)
      return status.mediaTypeString, status.detectedMediaTypeString

def getSlotControllerPresence( status, hardwareStatusDir, slotStatus ):
   """
   Uses the presence pin status from slot status controllers to determine whether or
   not a transceiver is present in the slot.
   Parameters
   ----------
   status: XcvrNewStatus
   hardwareStatusDir: Hardware::XcvrController::AllStatusDir
      Aggregator SM containing all transceiver hardware status entities
   slotStatus: slotStatus entity with xcvrType and name properties of interface
   """
   # We resolve redondo whisper mode to be present as there is no reliable pin status
   if getattr( status, "whisperConnectorMode", False ):
      return True
   slotHwControllerStatus = slotHardwareControllerStatus( hardwareStatusDir,
                                                          slotStatus.xcvrType,
                                                          slotStatus.name )
   if ( modPresent:= getattr( slotHwControllerStatus, 'modulePresent', None ) ) is \
      not None:
      return modPresent == 1
   elif ( modAbs:= getattr( slotHwControllerStatus, 'modAbs', None ) ) is not None:
      return modAbs == 0
   else:
      return None

def showInterfacesXcvrHardware( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = XSHM.InterfacesTransceiverHardwareBase()
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model
   xcvrStatus = gv.xcvrStatusDir.xcvrStatus
   hardwareStatusDir = CliPlugin.XcvrHardwareAllStatusDir.allXcvrHwStatusDir(
                       gv.entityManager )
   for intfName in intfNames:
      slotName = getXcvrSlotName( intfName )
      slotStatus = xcvrStatus.get( slotName )
      status = getXcvrStatus( slotStatus )
      if not status:
         continue

      lineSideChannels = getLineSideChannels( status )
      for laneId in range( lineSideChannels ):
         # gyorgym:
         # Why does this use lineSideChannels?  I'll never know.
         name = getAndFilterPrintableIntfNameByLane( laneId, status,
                                                     lineSideChannels, intfNames )
         if not name:
            continue
         model.interfaces[ name ] = _populateHardware( status, slotStatus, name,
                                                       hardwareStatusDir )
   return model

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.entityManager = em
   gv.xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
