# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliGlobal
import ConfigMount
import LazyMount
import natsort
import Tac
from CliDynamicSymbol import CliDynamicPlugin
from TypeFuture import TacLazyType

import CliPlugin.EthIntfCli
import CliPlugin.EthIntfModel
import CliPlugin.IntfCli
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
from CliPlugin.XcvrConfigCli import getXcvrConfigCliDir
from XcvrLib import getXcvrStatus, isCmisTransceiver, isCmisType

gv = CliGlobal.CliGlobal( entityManager=None,
                          XcvrStatusDir=None,
                          xcvrSxcvrCliConfigSliceDirtatusDir_=None )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
XcvrShowIdpromModel = CliDynamicPlugin( 'XcvrShowIdpromModel' )

# ----------------------------------------------------------------------------
#
# "show idprom transceiver [ <interface> ] [ extended ]"
#
# ----------------------------------------------------------------------------

# Helper collections
#    Headers: headers used in extended dump formatting
#    Contents: the attribute in xcvrStatus that contains the eeprom memory map
#    Pages: the attributes in contents that contain the page data we want
qsfpPageNums = [ "lower0",
                 "upper0",
                 "upper2",
                 "upper3",
                 "upper32",
                 "upper33" ]
# Note that page20h and page21h only exist when enhanced DOM is supported
# by the module. When not supported, these pages are empty and we don't
# print them out.
qsfpEepromPages = [ "eepromPage00LowerData",
                    "eepromPage00UpperData",
                    "eepromPage02UpperData",
                    "eepromPage03UpperData",
                    "eepromPage20hUpperData",
                    "eepromPage21hUpperData" ]

sfpPageNums = [ "A0", "A2" ]
sfpEepromPages = [ "buf", "eepromPageA2Data" ]

# Static pages
_upper00h = "upper0"
_upper01h = "upper1"
_upper02h = "upper2"
_upperC1h = "upper193" # Extra page for Arista Custom LPO
_upperC2h = "upper194" # Extra page for Arista Custom LPO
# All OSFP pages
osfpPageNums = [ "lower0",
                 _upper00h,
                 _upper01h,
                 _upper02h,
                 "upper16",
                 "upper17",
                 "upper19",
                 _upperC1h,
                 _upperC2h ]
_osfpBank = Tac.Type( "Xcvr::CmisBank" )
_osfpPage = Tac.Type( "Xcvr::CmisPage" )

def osfpPageBankFactory( page, bank=_osfpBank.bank0 ):
   return Tac.Value( "Xcvr::CmisBankAndPage", bank, page )

osfpEepromPages = [
   osfpPageBankFactory( _osfpPage.lowerPage00, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPage00, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPage01, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPage02, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPage10 ),
   osfpPageBankFactory( _osfpPage.upperPage11 ),
   osfpPageBankFactory( _osfpPage.upperPage13 ),
   osfpPageBankFactory( _osfpPage.upperPageC1, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPageC2, _osfpBank.bankNa ), ]

# Extra pages for tunable wavelength
osfpTunableWavelengthPageNums = [ "upper4",
                                  "upper18" ]

osfpTunableWavelengthEepromPages = [
   osfpPageBankFactory( _osfpPage.upperPage04, _osfpBank.bankNa ),
   osfpPageBankFactory( _osfpPage.upperPage12 ), ]

# Extra pages for VDM Group 1
osfpVersatileDiagsMonitoringCommonPageNums = [ "upper32",
                                               "upper36",
                                               "upper40",
                                               "upper44",
                                               "upper45",
                                               "upper47" ]
osfpVersatileDiagsMonitoringCommonPages = [
   osfpPageBankFactory( _osfpPage.upperPage20 ),
   osfpPageBankFactory( _osfpPage.upperPage24 ),
   osfpPageBankFactory( _osfpPage.upperPage28 ),
   osfpPageBankFactory( _osfpPage.upperPage2C ),
   osfpPageBankFactory( _osfpPage.upperPage2D ),
   osfpPageBankFactory( _osfpPage.upperPage2F ), ]

# Extra pages for VDM Group 2
osfpVersatileDiagsMonitoringGroup2PageNums = [ "upper33",
                                               "upper37",
                                               "upper41" ]
osfpVersatileDiagsMonitoringGroup2Pages = [
   osfpPageBankFactory( _osfpPage.upperPage21 ),
   osfpPageBankFactory( _osfpPage.upperPage25 ),
   osfpPageBankFactory( _osfpPage.upperPage29 ), ]

# Extra pages for VDM Group 3
osfpVersatileDiagsMonitoringGroup3PageNums = [ "upper34",
                                               "upper38",
                                               "upper42" ]
osfpVersatileDiagsMonitoringGroup3Pages = [
   osfpPageBankFactory( _osfpPage.upperPage22 ),
   osfpPageBankFactory( _osfpPage.upperPage26 ),
   osfpPageBankFactory( _osfpPage.upperPage2A ), ]

# Extra pages for VDM Group 4
osfpVersatileDiagsMonitoringGroup4PageNums = [ "upper35",
                                               "upper39",
                                               "upper43" ]
osfpVersatileDiagsMonitoringGroup4Pages = [
   osfpPageBankFactory( _osfpPage.upperPage23 ),
   osfpPageBankFactory( _osfpPage.upperPage27 ),
   osfpPageBankFactory( _osfpPage.upperPage2B ), ]

# Extra pages for DWDM Coherent (ZR)
osfpCoherentDiagsMonitoringPageNums = [ "upper48",
                                        "upper50",
                                        "upper51",
                                        "upper53",
                                        "upper66",
                                        "upper68" ]

osfpCoherentDiagsMonitoringPages = [
   osfpPageBankFactory( _osfpPage.upperPage30 ),
   osfpPageBankFactory( _osfpPage.upperPage32 ),
   osfpPageBankFactory( _osfpPage.upperPage33 ),
   osfpPageBankFactory( _osfpPage.upperPage35 ),
   osfpPageBankFactory( _osfpPage.upperPage42 ),
   osfpPageBankFactory( _osfpPage.upperPage44 ) ]

# Extra pages for POLS
polsPageNums = [ "upper192",
                 "upper193",
                 "upper208",
                 "upper209",
                 "upper216",
                 "upper217" ]
polsEepromPages = [ Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageC0 ),
                    Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageC1 ),
                    Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD0 ),
                    Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD1 ),
                    Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD8 ),
                    Tac.Value( "Xcvr::CmisBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD9 ), ]

# Prepare function to create idprom-dump request. "show idprom transceiver extended"
# works in two parts:
#   1. Prepare the idprom dump. This is done by writing to XcvrConfigCli (i.e.
#      eepromDumpRequest), which the XcvrCtrl SMs react to and read additional
#      pages as needed. The preparation is done in createIdpromDumpRequest()
#   2. Print the results of the extended idprom read. This is done primarily
#      doExtendedIdpromDump(), which waits for the extended read to complete
#      if necessary, then reads attributes in xcvrStatus where the idprom contents
#      is stored.
def createIdpromDumpRequest( mode, intf=None, mod=None, includeAuxiliary=False ):
   ( _, intfNames ) = getAllIntfsWrapper( mode, intf, mod, includeAuxiliary )
   if not intfNames:
      return
   # Filter out interface names that are valid to create XcvrConfigCli objs with.
   # Valid names are the primary interface name. If the user specified the dump on
   # any lane of a multi-lane interface, request a dump as if they specified the
   # primary lane.
   xcvrStatuses = gv.XcvrStatusDir.xcvrStatus
   primaryLaneId = 0
   primaryIntfNames = [ status.xcvrConfig.intfName[ primaryLaneId ] for status in
                        xcvrStatuses.values() if
                        any( intfName in intfNames for intfName in
                             status.xcvrConfig.intfName.values() ) ]
   if includeAuxiliary:
      primaryIntfNames += _getAuxiliarySlotsFromEthernet( xcvrStatuses, intfNames )

   for intfName in Arnet.sortIntf( primaryIntfNames ):
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      # Write a snapshot request
      if intfName not in xcvrConfigCliDir.xcvrConfigCli:
         xcvrConfigCliDir.newXcvrConfigCli( intfName )
      xcvrConfigCliDir.xcvrConfigCli[ intfName ].eepromDumpRequest = Tac.now()

# Helper function. When called, checks whether we have response for our dump-request
def _dumpRequestCompleted( xcvrStatus, xcvrDir ):
   primaryLaneId = 0
   primaryIntfName = xcvrStatus.name
   if "Auxiliary" not in xcvrStatus.name:
      primaryIntfName = xcvrStatus.xcvrConfig.intfName[ primaryLaneId ]
   xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ primaryIntfName ]
   # If timestamp is newer than our request, eeprom is printable
   if( ( xcvrStatus.idEepromDumpTimestamp >= xcvrConfigCli.eepromDumpRequest ) or
       xcvrStatus.name not in xcvrDir ):
      return True
   return False

# Helper function. Prints msg informing user that xcvr is not there
def _printNotPresentMsg( mode, name, lenIntfs ):
   if lenIntfs == 1:
      # Only show "transceiver not present" message if the user requested
      # idprom of a single xcvr. Otherwise, when dumping idproms of all
      # xcvrs, don't pollute the output with such messages.
      mode.addError( f"{name}: transceiver not present" )

# Helper function. Factors out code for issuing a basic IDPROM data dump.
def _doNormalIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model ):
   intfName = xcvrStatus.name

   # The non-extended version of the command only prints if the xcvr is
   # identified as present, so we want to preserve that behavior.
   if xcvrStatus.presence != "xcvrPresent":
      _printNotPresentMsg( mode, intfName, len( intfs ) )
      return

   model.interfaces[ intfName ] = \
      XcvrShowIdpromModel.InterfacesTransceiverIdpromPages()
   intfModel = model.interfaces[ intfName ]
   if xcvrStatus.xcvrType.lower() == "qsfpplus":
      intfModel.pages[ qsfpPageNums[ 1 ] ] = \
         XcvrShowIdpromModel.InterfacesTransceiverIdpromData()
      intfModel.pages[ qsfpPageNums[ 1 ] ].dataIs(
                           xcvrStatus.sff8436IdEepromContents.eepromPage00UpperData )
      intfModel.pages[ qsfpPageNums[ 1 ] ].regWidth = 8
   elif xcvrStatus.xcvrType.lower() == "sfpplus":
      intfModel.pages[ sfpPageNums[ 0 ] ] = \
         XcvrShowIdpromModel.InterfacesTransceiverIdpromData()
      intfModel.pages[ sfpPageNums[ 0 ] ].regWidth = 8
      intfModel.pages[ sfpPageNums[ 0 ] ].dataIs(
                                             xcvrStatus.sff8472IdEepromContents.buf )
   elif isCmisTransceiver( xcvrStatus ):
      # Create a temporary mapping from page numbers to OSFP EEPROM page objects
      cmisPageNumsToEeprom = dict( zip( osfpPageNums, osfpEepromPages ) )

      def _populateModel( pageNum ):
         intfModel.pages[ pageNum ] = \
            XcvrShowIdpromModel.InterfacesTransceiverIdpromData()
         intfModel.pages[ pageNum ].regWidth = 8
         data = xcvrStatus.rawEeprom.memory.get( cmisPageNumsToEeprom[ pageNum ] )
         # If the data for a page does not exist in our memory shadow,
         # then we use an empty string to represent that data.
         data = data if data is not None else ""
         intfModel.pages[ pageNum ].dataIs( data )

      # UpperPage00h, UpperPage01h, and UpperPage02h are static pages
      # and therefore do not require another SMBus read.  Whether or not
      # transceiver has page 01h and 02h implemented is determined by the
      # flatMem bit in lowerPage00h. Populate only the pages which are implemented.
      _populateModel( _upper00h )
      if xcvrStatus.writableEeprom():
         for pageNum in [ _upper01h, _upper02h ]:
            _populateModel( pageNum )

   elif xcvrStatus.xcvrType.lower() == "cfp2":
      if not xcvrStatus.initialized:
         # Avoid printing the spacing line for a non-initialized cfp2
         return
      memory = xcvrStatus.cfpMsaMisIdEepromContents.memory
      for tableName in sorted( memory.keys() ):
         table = memory[ tableName ]
         if not table.contents:
            # Skip empty memory pages
            continue
         intfModel.pages[ tableName ] = \
            XcvrShowIdpromModel.InterfacesTransceiverIdpromData()
         intfModel.pages[ tableName ].dataIs( table.contents )
         intfModel.pages[ tableName ].regWidth = table.registerWidth

# Helper function for the extended idprom dump command.
# Returns a list of tuples of (page number, page data) of type (string, string) when
# the EEPROM data is valid, otherwise returns None.
def getIdpromData( xcvrStatus ):
   xcvrType = xcvrStatus.xcvrType.lower()

   # Different form-factors have different software EEPROM models,
   # so handle them differently
   if xcvrType in [ 'qsfpplus', 'sfpplus' ]:
      if xcvrType == 'qsfpplus':
         pageNums = qsfpPageNums
         eepromAttrName = "sff8436IdEepromContents"
         pages = qsfpEepromPages
      else:
         pageNums = sfpPageNums
         eepromAttrName = "sff8472IdEepromContents"
         pages = sfpEepromPages

      eeprom = getattr( xcvrStatus, eepromAttrName )
      # Check the validity of the EEPROM data
      if eeprom.lastValidData < xcvrStatus.idEepromDumpTimestamp:
         return None

      data = [ getattr( eeprom, page ) for page in pages ]
      return list( zip( pageNums, data ) )

   elif isCmisTransceiver( xcvrStatus ):
      # Check the validity of the EEPROM data
      if xcvrStatus.rawEeprom.lastValidData < xcvrStatus.idEepromDumpTimestamp:
         return None

      rawPageData = []
      for page in osfpEepromPages:
         pgData = xcvrStatus.rawEeprom.memory.get( page )
         rawPageData.append( pgData if pgData is not None else '' )

      idpromData = list( zip( osfpPageNums, rawPageData ) )

      # For transceivers with tunable wavelength, show advertise and control pages
      rawTunableWlPageData = []
      if xcvrStatus.capabilities.tunableWavelength:
         for page in osfpTunableWavelengthEepromPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawTunableWlPageData.append( pgData if pgData is not None else '' )

         idpromData += list( zip( osfpTunableWavelengthPageNums,
                                  rawTunableWlPageData ) )

      def rawPageDataList( pages ):
         rawPageData = []
         for page in pages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawPageData.append( pgData if pgData is not None else '' )
         return rawPageData

      # For transceiver that support Versatile Diagnostics Monitoring
      if xcvrStatus.vdmCapabilities.vdmSupported:
         if xcvrStatus.vdmCapabilities.maxPageGroup >= 2:
            idpromData += list( zip(
               osfpVersatileDiagsMonitoringGroup2PageNums,
               rawPageDataList( osfpVersatileDiagsMonitoringGroup2Pages ) ) )

         if xcvrStatus.vdmCapabilities.maxPageGroup >= 3:
            idpromData += list( zip(
               osfpVersatileDiagsMonitoringGroup3PageNums,
               rawPageDataList( osfpVersatileDiagsMonitoringGroup3Pages ) ) )

         if xcvrStatus.vdmCapabilities.maxPageGroup >= 4:
            idpromData += list( zip(
               osfpVersatileDiagsMonitoringGroup4PageNums,
               rawPageDataList( osfpVersatileDiagsMonitoringGroup4Pages ) ) )

         # Append VDM Group 1 and common pages
         idpromData += list( zip(
            osfpVersatileDiagsMonitoringCommonPageNums,
            rawPageDataList( osfpVersatileDiagsMonitoringCommonPages ) ) )

      # For Coherent CMIS modules
      if xcvrStatus.isCoherent( xcvrStatus.mediaType ):
         idpromData += list( zip(
            osfpCoherentDiagsMonitoringPageNums,
            rawPageDataList( osfpCoherentDiagsMonitoringPages ) ) )

      # POLS follow CMIS4.0 but has more customized pages
      rawPageData = []
      if xcvrStatus.mediaType == XcvrMediaType.xcvrAmpZr:
         for page in polsEepromPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawPageData.append( pgData if pgData is not None else '' )

         idpromData += list( zip( polsPageNums, rawPageData ) )
      # Pages could be added in any order, sort them by page number
      return natsort.natsorted( idpromData )

   else:
      assert 0, f"Unexpected xcvrType {xcvrType} in Idprom dump"
      return None

# Helper function. Factors out common code for issuing an extended IDPROM dump rather
# than a basic one.
def doExtendedIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model ):
   intfName = xcvrStatus.name
   xcvrType = xcvrStatus.typeInUse.lower()

   if xcvrType == 'cfp2':
       # Cfp2 doesn't need to create a request for extended, so print "normal" dump
      _doNormalIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model )
      return

   # Try to print. If the request becomes stale, timeout
   try:
      # Non-Arista hardware mostly lacks fancy SMBus accelerators.
      # We'll be more lenient with the timeout on non-Arista hardware.
      timeout = 120 if runningOnWhiteboxHardware() else 15

      ds = f"idprom-dump on intf {intfName}"
      Tac.waitFor( lambda: _dumpRequestCompleted( xcvrStatus, xcvrDir ),
                   sleep=True, timeout=timeout, description=ds, maxDelay=1.0 )

      # Check to see if the linecard has been removed/the name is valid
      if xcvrStatus.name not in xcvrDir:
         _printNotPresentMsg( mode, intfName, len( intfs ) )
         return

      # Get the actual EEPROM data from xcvrStatus
      eeprom = getIdpromData( xcvrStatus )

      # Check if there was an error during the read. The value of 'eeprom' will be
      # None if the lastValidData timestamp predates the dump request timestamp.
      if xcvrStatus.presence == "xcvrPresent" and eeprom is None:
         print( f"{intfName} IDPROM: error retrieving data; read failed" )
      # The extended version will try to read as much as it can, regardless of xcvr
      # presence for sfp+, qsfp+, osfp and qsfpDd. Example: if we insert a faulty or
      # third-party xcvr, I believe status.presence gets stuck at xcvrPresenceUnknown
      else:
         anyData = eeprom and any( pageData for _, pageData in eeprom )
         if not anyData:
            if len( intfs ) == 1:
               print( f"{intfName}: no data to print\n" )
            return
         else:
            model.interfaces[ intfName ] = \
               XcvrShowIdpromModel.InterfacesTransceiverIdpromPages()
            intfModel = model.interfaces[ intfName ]
            # Add the partial dump error flag for QSFP modules in case of error
            if xcvrType == 'qsfpplus' and xcvrStatus.idEepromPartialDumpError:
               intfModel.partialIdpromDumpError = True
            for ( pageNum, pageData ) in eeprom:
               # Check that the page has data loaded
               if pageData:
                  intfModel.pages[ pageNum ] = \
                     XcvrShowIdpromModel.InterfacesTransceiverIdpromData()
                  intfModel.pages[ pageNum ].dataIs( pageData )
                  intfModel.pages[ pageNum ].regWidth = 8

   except Tac.Timeout:
      print( f"{intfName}: Timeout" )

def idpromDumpSupported( xcvrStatus ):
   """
   Parameters
   ----------
   xcvrStatus : Xcvr::XcvrNewStatus derived object

   Returns
   -------
   bool
      True when the passed in xcvrStatus represents a transceiver which supports
      EEPROM dumps, false otherwise.
   """
   typeInUse = xcvrStatus.typeInUse
   if typeInUse == XcvrType.qsfpPlus:
      # All QSFPs support EEPROM dump except for Whisper Connectors which are
      # special QSFP-like copper connectors with no memory to dump.
      return not xcvrStatus.whisperConnectorMode
   # CMIS, SFP (SFF8472), QSFP (SFF8436), and CFP2 transceivers support memory dump
   return ( isCmisType( typeInUse ) or
            typeInUse in [ XcvrType.sfpPlus, XcvrType.qsfpPlus, XcvrType.cfp2 ] )

def _getAuxiliarySlotsFromEthernet( xcvrStatuses, intfNames: list[ str ] ) \
      -> list[ str ]:
   """
   Finds the `intfName` names corresponding to the statuses created for
   auxiliary slots.

   Auxiliary slot statuses are named "AuxiliaryX" in the xcvrStatus collection.
   This utility method finds any auxiliary statuses named like "EthernetX", in
   the range.
   """
   return [ xcvr.name for xcvr in xcvrStatuses.values() if
            "Auxiliary" in xcvr.name and
            any( xcvr.name.replace( "Auxiliary", "Ethernet" ) in
            intfName for intfName in intfNames ) ]

# Dump IDPROM contents.
def xcvrIdpromDumpFunc( mode, func, model, intf=None, mod=None,
                        includeAuxiliary=False ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod, includeAuxiliary )
   if not intfNames:
      return
   xcvrStatuses = gv.XcvrStatusDir.xcvrStatus

   # Filter the xcvr names based on the specified interfaces
   filteredXcvrNames = [ xcvr.name for xcvr in xcvrStatuses.values()
                         if any( i in intfNames
                                 for i in xcvr.xcvrConfig.intfName.values() )
                         or xcvr.name in intfNames ]
   if includeAuxiliary:
      # Concatenate auxiliary ports if they are intentended to be included.
      filteredXcvrNames += _getAuxiliarySlotsFromEthernet( xcvrStatuses, intfNames )
   for xcvrName in Arnet.sortIntf( filteredXcvrNames ):
      # Call our library function to get the nested-most XcvrStatus object.
      xcvrStatus = getXcvrStatus( xcvrStatuses.get( xcvrName ) )

      # Check to see if the linecard has been removed
      if not xcvrStatus or xcvrName not in xcvrStatuses:
         _printNotPresentMsg( mode, xcvrName, len( intfs ) )
         continue

      # Check that the xcvr is supported
      if not idpromDumpSupported( xcvrStatus ):
         continue

      func( mode, xcvrStatus, xcvrStatuses, intfs, model )

def runningOnWhiteboxHardware():
   # pkgdeps: library HwPlutoLibrary
   hwPlutoLibConfig = LazyMount.mount( gv.entityManager,
                                       'hardware/pluto/library/config',
                                       'Hardware::PlutoLibrary::Config', 'r' )
   return hwPlutoLibConfig.isWhitebox

def showInterfaceXcvrIdprom( mode, args ):
   intf = args.get( 'INTFS' )
   model = XcvrShowIdpromModel.InterfacesTransceiverIdpromBase()
   if args.get( 'extended' ):
      createIdpromDumpRequest( mode, intf=intf )
      xcvrIdpromDumpFunc( mode, doExtendedIdpromDump, model, intf=intf )
   else:
      xcvrIdpromDumpFunc( mode, _doNormalIdpromDump, model, intf=intf )
   return model

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.entityManager = em
   gv.xcvrSxcvrCliConfigSliceDirtatusDir_ = ConfigMount.mount( em,
      "hardware/xcvr/cli/config/slice", "Tac::Dir", "wi" )
   gv.XcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
