# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import ( Dict, OrderedDict, Model, Int, List, Bool )
import Hexdump
from IntfModels import Interface
import Tac
import re
from TypeFuture import TacLazyType

SUPPORTED_FORMATS = frozenset( [ "default", "csv" ] )

CmisModuleState = TacLazyType( "Xcvr::CmisModuleState" )
DataPathState = TacLazyType( "Xcvr::DataPathState" )
MediaType = TacLazyType( "Xcvr::MediaType" )
TributaryId = TacLazyType( 'Xcvr::TributaryId' )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

# --------------------------------------------------------------------------------
#
# Models for "show idprom transceiver [<interfaces>]"
#
# --------------------------------------------------------------------------------

class InterfacesTransceiverIdpromData( Model ):
   registers = List( valueType=int, help='Eeprom data as an integer' )
   regWidth = Int( help="Eeprom register width in bits" )

   def dataIs( self, dataStr ):
      # pylint: disable-next=unnecessary-comprehension
      self.registers = [ x for x in iter( dataStr ) ]

   def renderModel( self ):
      # Don't condense repeating rows.
      print( Hexdump.hexdump( bytes( bytearray( self.registers ) ),
             condense=False ) )

class InterfacesTransceiverIdpromPages( Model ):
   partialIdpromDumpError = Bool( help="Flag indicating error while reading some " +
                                       "Idprom pages", optional=True )
   pages = OrderedDict( keyType=str, valueType=InterfacesTransceiverIdpromData,
                        help="Mapping of page name to register" )

   def renderModel( self, intfName, ):
      if self.partialIdpromDumpError:
         print( f"{intfName} IDPROM: partial error retrieving data" )
      for ( pageNum, page ) in self.pages.items():  # pylint: disable=no-member
         # pylint: disable-next=consider-using-f-string
         header = f"{intfName} IDPROM Page "
         if pageNum.startswith( 'upper' ) or pageNum.startswith( 'lower' ):

            m = re.match( r"(?P<upperOrLower>(upper|lower))(?P<address>\d+)",
                           pageNum )
            address = int( m.groupdict()[ 'address' ] )
            address = format( address, 'x' ).rjust( 2, '0' )
            header += "%sh %s" % ( address, # pylint: disable=consider-using-f-string
                                   m.groupdict()[ 'upperOrLower' ].title() )
         else:
            header += pageNum
         if page.regWidth:
            # pylint: disable-next=consider-using-f-string
            header += " Register Width %u" % page.regWidth
         print( header )
         page.renderModel()

class InterfacesTransceiverIdpromBase( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfacesTransceiverIdpromPages,
                      help='Mapping between interface name and the eeprom' )

   def render( self ):
      if not self.interfaces:
         return
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )
         print()
