# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
from typing import Optional, Literal

import CliGlobal
import ConfigMount
import EthIntfLib
import LazyMount
import Tac
from TypeFuture import TacLazyType

import CliPlugin.XcvrAllStatusDir
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.XcvrCliLib import getAllIntfsWrapper, isModular
from CliPlugin.XcvrModel import InterfaceTransceiverDetailThresholds
from CliPlugin.XcvrShowDomModel import InterfaceTransceiverDomParameter
from XcvrLib import isCmisTransceiver

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
DomParamType = TacLazyType( 'Xcvr::EnhancedDomParamType' )
DomParamScope = TacLazyType( 'Xcvr::EnhancedDomParamScope' )

gv = CliGlobal.CliGlobal( xgc=None, xcvrPerfMonSliceDir=None, xcvrStatusDir=None )

XSPFMM = CliDynamicPlugin( 'XcvrShowPerfMonModel' )

PerfMonConfigState = TacLazyType(
   'Xcvr::PerformanceMonitoringConfigState' )

RxLosOrTxFaultImplemented = namedtuple( 'rxLosOrTxFaultImplemented',
                                        'rx_los tx_fault' )
CoherentClientStatusAttr = namedtuple( 'ClientPcsAlarmStatusAttr',
                                       'name trueState falseState' )
# Following parameters are exceeded counters.
# These parameters don't have any thresholds.
exceededCountersList = [ 'preFECBERHighAlarmExceeded', 'preFECBERHighWarnExceeded',
      'uncorrectedBERHighAlarmExceeded', 'uncorrectedBERHighWarnExceeded',
      'tempAlarmExceeded', 'tempWarnExceeded' ]
# These parameters are BERs.
berParametersList = [ 'preFecBERCurr', 'uncorrectedBERCurr', 'preFECBERAvg',
      'preFECBERMax', 'preFECBERMin', 'uncorrectedBERAvg', 'uncorrectedBERMax',
      'uncorrectedBERMin' ]

# --------------------------------------------------------------------------------
#
# Models for "show int [ <interface> ] transceiver performance-monitoring [ raw ]
#                [ thresholds ]"
#
# --------------------------------------------------------------------------------

def getXcvrPerfMonDir( intfName ):
   # Supervisors having xcvr slots will be present in FixedSystem slice
   sliceName = 'FixedSystem'
   if isModular() and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
   return gv.xcvrPerfMonSliceDir[ sliceName ][ 'xcvr' ]

def _isQsfpPmSupported( status ) -> bool:
   """
   Note: the status object provided must be the child-most transceiver
   status.
   """
   return ( isinstance( status, Tac.Type( "Xcvr::QsfpNewStatus" ) ) and
            status.performanceMonitoringSupported )

def _isPerformanceMonitoringSupported( status ) -> bool:
   '''
   Helper method for determining if performance monitoring is supported on this
   transceiver. Note: the status object provided must be the child-most transceiver
   status.
   '''
   # CMIS and CFP2-DCO modules
   if ( hasattr( status, 'performanceMonitoringStatus' ) and
        status.performanceMonitoringStatus ):
      return True
   return _isQsfpPmSupported( status )

def _showInterfacesXcvrPerformanceMonitoring( mode, intf: Optional[ str ] = None,
                                              mod=None, rawFormat: bool = False,
                                              thresholds: bool = False ):
   anyQsfpPmPresent = False
   for status in gv.xcvrStatusDir.xcvrStatus.values():
      # The following logic is only for when at least one PM capable QSFP
      # (i.e. 100GE-DWDM2) module is inserted and configured for
      # performance-monitoring on the switch.
      xcvrStatus = status.statusInUse()
      if ( _isQsfpPmSupported( xcvrStatus ) and
           xcvrStatus.performanceMonitoringConfigured ):
         anyQsfpPmPresent = True
         break
   res = XSPFMM.InterfacesTransceiverPerformanceMonitoring(
      _detailed=thresholds, _qsfpPmPresent=anyQsfpPmPresent,
      _rawFormat=rawFormat )
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return XSPFMM.InterfacesTransceiverPerformanceMonitoring(
         _qsfpPmPresent=anyQsfpPmPresent, _rawFormat=rawFormat )

   isPerformanceMonitoringConfigured = False

   for status in gv.xcvrStatusDir.xcvrStatus.values():
      # Get the child-most XcvrStatus instance.
      status = status.statusInUse()

      if not _isPerformanceMonitoringSupported( status ):
         continue

      name = status.xcvrConfig.intfName.get( 0 )
      if name not in intfNames:
         # Only display performance monitoring for primary interfaces.
         continue

      # The following logic is only for QSFP optics which support enhanced DOM.
      if _isQsfpPmSupported( status ):
         if ( not isPerformanceMonitoringConfigured and
              status.performanceMonitoringConfigured ):
            isPerformanceMonitoringConfigured = True
            # At this point, PM is configured on this interface.
            # Store the PM period in the model irrespective of whether PM is
            # enabled or not.
            res.performanceMonitoringPeriodSeconds = \
                  gv.xgc.performanceMonitoringPeriod.valueInSeconds

         if not status.performanceMonitoringEnabled:
            continue

         res.interfaces[ name ] = _xcvrQsfpPmPerformanceMonitoring( status,
                                                                    thresholds )

      else:
         perfMonStatus = status.performanceMonitoringStatus
         xcvrPerfMonDir = getXcvrPerfMonDir( name )
         if ( not isPerformanceMonitoringConfigured and
              ( perfMonStatus.performanceMonitoringConfiguration ==
                PerfMonConfigState.performanceMonitoringConfigValid ) ):
            isPerformanceMonitoringConfigured = True
            # At this point, PM is configured on this interface.
            # Store the PM period in the model irrespective of whether PM is
            # enabled or not.
            res.performanceMonitoringPeriodSeconds = \
               int( xcvrPerfMonDir.performanceMonitoringPeriod )

         if not perfMonStatus.performanceMonitoringEnabled:
            continue

         xcvrName = status.xcvrConfig.name
         if xcvrName in xcvrPerfMonDir.pmCurrentDir:
            currentPmDir = (
               xcvrPerfMonDir.pmCurrentDir[ xcvrName ].perfMonParameterDirEntry )
            historyDir = xcvrPerfMonDir.pmHistoryDir.get( status.xcvrConfig.name )
            previousPmDir = historyDir.getPreviousSnapshot()
            res.interfaces[ name ] = _populatePerformanceMonitoring(
               status, currentPmDir, previousPmDir, rawFormat, thresholds,
               anyQsfpPmPresent )
            res.interfaces[ name ].displayName = name

   # pylint: disable-msg=W0212
   res._performanceMonitoringConfigured = isPerformanceMonitoringConfigured
   return res

def insertParameterInDict( parameters: dict[ str, InterfaceTransceiverDomParameter ],
                           paramType: str,
                           unit: Literal[ "C", "V", "dBm" ], channel: str,
                           value: Optional[ float ], threshold:
                           Optional[ InterfaceTransceiverDetailThresholds ] = None )\
                              -> tuple[ dict, bool ]:
   '''
   Helper function to insert a parameter in a dict
   If parameter exists, we are probably adding value for another channel
   If parameter doesn't exist, create one.
   '''
   # Set value to None if is "-inf". This prevents -inf from entering into the model.
   value = None if value == float( "-inf" ) else value
   if paramType in parameters:
      parameters[ paramType ].channels[ channel ] = value
   else:
      newParam = InterfaceTransceiverDomParameter()
      newParam.unit = unit
      newParam.channels[ channel ] = value
      if threshold is not None:
         newParam.threshold = threshold
      parameters[ paramType ] = newParam
   return ( parameters, len( parameters[ paramType ].channels ) > 1 )

def _getParameters( period, thresholds: bool ):
   newParam = {}
   paramList = []
   _threshold = None
   pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
   for key in sorted( period.parameter.keys() ):
      thresh = period.parameter[ key ]
      paramType = key
      # Check if this paramType is not already in our list. Well, this is a little
      # inefficient if the list is long but the sample space is quite small
      # here (in tens) so it doesn't hurt much to do a lookup everytime.
      if key not in paramList:
         paramList.append( key )
      paramUnit = ''
      # If the parameter is a BER and it's value is the max possible value for this
      # parameter (which is the initialized value), use value 0 instead.
      paramValue = period.parameter[ key ].value
      if( ( paramType in berParametersList ) and ( paramValue == pm.berMaxValue ) ):
         paramValue = 0.0
      paramChannel = \
            str( period.parameter[ key ].channel ) \
            if period.parameter[ key ].channelSpecific else '-'
      if thresholds:
         # Exceeded counts don't have any thresholds, other BER counts only have
         # high alarm and high warn thresholds.
         if paramType in exceededCountersList:
            _threshold = InterfaceTransceiverDetailThresholds()
         else:
            _threshold = InterfaceTransceiverDetailThresholds(
                  highAlarm=thresh.highAlarm, highWarn=thresh.highWarn )
      ( newParam, _ ) = \
            insertParameterInDict( newParam, paramType,
                                   paramUnit, paramChannel, paramValue, _threshold )

   return newParam, paramList

def _getPerformanceMonitoringParameters( xcvrStatus, parameterDir,
                                         rawFormat=False, thresholds=False ) -> list:
   """
   Helper function for providing a set of ordered lists of EnhancedDomParamTypes for
   which to print stats for performance-monitored parameters.

   Parameters
   ----------
   xcvrStatus : XcvrNewStatus object for the requested ethernet interface.
   parameterDir : PerfMonParameterDirEntry, directory of all currently
      performance monitored parameters. This object will be empty until the first
      round of polling has completed.
   rawFormat : bool
      Indicates whether to order all parameters at the same level with lane
      information, regardless of parameter scope. Note when this is true, the
      returned paramList will account for module, host, and media scoped parameters
      instead of just media-scoped parameters.
   thresholds : bool
      Indicates whether or not to populate the CAPI model with threshold information
      for monitored parameters.

   Returns
   -------
   newParam : dictionary of all parameters that have been inserted into the CAPI
      model
   modParamList : list
      ordered list of any monitored DOM paramTypes at the module scope
   dpParamList : list
      ordered list of any monitored DOM paramTypes at the datapath (host electrical)
      scope.
   paramList : list
      ordered list of either: 1) any monitored DOM paramTypes at the media scope or
                              2) all monitored DOM paramTypes regardless of scope if
                                 rawFormat is set to True
   """
   enhancedDomParamHelper = Tac.Value( "Xcvr::EnhancedDomParameterHelper",
                                       xcvrStatus.xcvrType )
   newParam = {}
   modParamList = []
   dpParamList = []
   paramList = []
   _threshold = None
   moduleScopeParams = []
   hostScopeParams = []
   mediaScopeParams = []
   for param in parameterDir.parameter.values():
      if param.scope == DomParamScope.enhancedDomScopeModule:
         moduleScopeParams.append( param )
      elif param.scope == DomParamScope.enhancedDomScopeHost:
         hostScopeParams.append( param )
      else:
         mediaScopeParams.append( param )
   isCmisVdmSupported = False
   if isCmisTransceiver( xcvrStatus ):
      isCmisVdmSupported = xcvrStatus.vdmCapabilities.vdmSupported

   def _getParamChannel( xcvrStatus, parameter, channel: int ) -> str:
      """
      Helper function to return the proper formatted channel/lane for the provided
      enhanced DOM parameter.
      """
      paramChannel = str( channel )
      if ( channel == 15 or
           parameter.scope == DomParamScope.enhancedDomScopeModule or
           ( xcvrStatus.xcvrType == XcvrType.cfp2 and
             xcvrStatus.mediaType == XcvrMediaType.xcvr400GDwdmDco ) ):
         # Use '-' to indicate module parameter is lane agnostic, or module wide.
         # Note: all CFP2 DP04 dom parameters are module wide.
         paramChannel = "-"
      return paramChannel

   # BUG540999: Add parameter order map for determining output format per module.
   # For now, order parameters using scope: module, host, media.
   for parameters, paramOrder in ( ( moduleScopeParams, modParamList ),
                                   ( hostScopeParams, dpParamList ),
                                   ( mediaScopeParams, paramList ) ):
      for parameter in parameters:
         # Order multilane parameters from smallest to largest.
         for lane, current in sorted( parameter.current.items(),
                                      key=lambda x: x[ 0 ] ):
            paramChannel = _getParamChannel( xcvrStatus, parameter, lane )
            for paramId, paramValue in (
               ( parameter.paramId.minStatsParamId, current.min ),
               ( parameter.paramId.maxStatsParamId, current.max ),
               ( parameter.paramId.avgStatsParamId, current.avg ) ):
               # Note: There may be cases where modules report PM
               # data for more parameters than we expose in the CLI, so skip
               # those params for now.
               if not enhancedDomParamHelper.isPerfMonParamReadable(
                  paramId, isCmisVdmSupported ):
                  continue
               if parameter not in paramOrder:
                  paramOrder.append( paramId )
               if thresholds:
                  _threshold = InterfaceTransceiverDetailThresholds()
               ( newParam, _ ) = insertParameterInDict( newParam, paramId,
                                                        parameter.units,
                                                        paramChannel,
                                                        paramValue,
                                                        _threshold )
   if rawFormat:
      paramList = modParamList + dpParamList + paramList
      modParamList = []
      dpParamList = []
   return newParam, modParamList, dpParamList, paramList

def _getPerformanceMonitoringInterval( startTime, updateTime, params ):
   interval = XSPFMM.InterfaceTransceiverPerformanceMonitoringInterval()
   interval.intervalStartTime = \
         Tac.utcNow() - ( Tac.now() - startTime )
   interval.updateTime = \
      Tac.utcNow() - ( Tac.now() - updateTime )
   interval.parameters = params
   return interval

def _populatePerformanceMonitoring( xcvrStatus, currentPmDir, previousPmDir,
                                    rawFormat, thresholds, anyQsfpPmPresent=False ):
   """
   Gather parameter information from each of the currently monitored parameters for
   this module.

   Parameters
   ----------
   xcvrStatus : XcvrNewStatus object for the requested ethernet interface.
   currentPmDir : PerfMonParameterDirEntry, directory of all currently monitored
      performance monitoring parameters. This object will be empty until the first
      round of polling has been reported.
   previousPmDir : PerfMonParameterDirEntry, directory of all PM parameters that
      were monitored for the duration of the previous PM interval, if it was
      completed. Otherwise, this object will return None.
   rawFormat : boolean that indicates whether or not PM parameters should be
      rendered at same level regardless of scope with channel information reported
      by the module's EEPROM.
   thresholds : boolean that indicates whether or not PM thresholds should be
      populated in the resulting CliModel.
   anyQsfpPmPresent : bool
      A flag, which indicates to use the same Cli format as is currently implemented
      for PM capable QSFP modules.
   """
   model = XSPFMM.InterfaceTransceiverPerformanceMonitoring(
      _qsfpPmPresent=anyQsfpPmPresent, _rawFormat=rawFormat )
   intervals = {}
   paramList = []
   if currentPmDir and currentPmDir.parameter:
      newParamCurrent, modParamList, dpParamList, paramList = \
         _getPerformanceMonitoringParameters( xcvrStatus, currentPmDir,
                                              rawFormat, thresholds )
      intervals[ 0 ] = _getPerformanceMonitoringInterval( currentPmDir.startTime,
                                                          currentPmDir.updateTime,
                                                          newParamCurrent )
   # Use ordering of EnhancedDomParamType enum for all PM parameters.
   model.moduleParamOrderIs( modParamList )
   model.datapathParamOrderIs( dpParamList )
   model.paramOrderIs( paramList )
   # Only add snapshot if data for all parameters has been recorded.
   if len( previousPmDir.parameter ) == len( currentPmDir.parameter ):
      newParamPrevious, _, _, _ = _getPerformanceMonitoringParameters( xcvrStatus,
                                                                       previousPmDir,
                                                                       rawFormat,
                                                                       thresholds )
      intervals[ 1 ] = _getPerformanceMonitoringInterval( previousPmDir.startTime,
                                                          previousPmDir.updateTime,
                                                          newParamPrevious )
   model.performanceMonitoringIntervals = intervals
   return model

def _xcvrQsfpPmPerformanceMonitoring( status, thresholds ):
   """
   This method for populating performance monitoring status information is only
      supported by the 100GE-DWDM2 Madion module. All other modules should
      instead utilize the _populatePerformanceMonitoring method above.

   Parameters
   ----------
   status : QsfpNewStatus object for the provided module.
   thresholds : boolean that indicates whether or not PM thresholds should be
      populated in the resulting CliModel.
   """
   model = XSPFMM.InterfaceTransceiverPerformanceMonitoring( _qsfpPmPresent=True )

   intervals = {}
   newParamCurrent = {}
   newParamPrevious = {}
   paramList = []
   currPeriod = status.performanceMonitoring.currentPeriod
   prevPeriod = status.performanceMonitoring.previousPeriod
   if currPeriod.isValid:
      newParamCurrent, paramList = _getParameters( currPeriod, thresholds )
      intervals[ 0 ] = _getPerformanceMonitoringInterval( currPeriod.startTime,
            currPeriod.updateTime, newParamCurrent )

   model.paramOrderIs( paramList )

   if prevPeriod.isValid:
      newParamPrevious, paramList = _getParameters( prevPeriod, thresholds )
      intervals[ 1 ] = _getPerformanceMonitoringInterval( prevPeriod.startTime,
            prevPeriod.updateTime, newParamPrevious )

   model.performanceMonitoringIntervals = intervals
   return model

def showInterfacesXcvrPerformanceMonitoring( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   raw = 'raw' in args
   thresholds = 'thresholds' in args
   return _showInterfacesXcvrPerformanceMonitoring( mode, intf, mod,
                                                    rawFormat=raw,
                                                    thresholds=thresholds )

# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   gv.xcvrPerfMonSliceDir = LazyMount.mount( em, "hardware/perfmon/slice",
                                             "Tac::Dir", "ri" )
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
