# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCli import getAllIntfsWrapper
from CliPlugin.XcvrConfigCli import ( getXcvrConfigCliDir,
                                      getXcvrConfigCliForConfigCommand,
                                      xcvrConfigCliDelete )
from XcvrLib import getXcvrSlotName, getXcvrStatus, isPrimaryIntf
import CliGlobal

gv = CliGlobal.CliGlobal( xcvrStatusDir=None )

XSM = CliDynamicPlugin( 'XcvrSquelchModel' )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver receiver squelch auto [ disabled ]
# --------------------------------------------------------------------------------

def transceiverRxSquelchAutoCmdHandler( mode, args ):
   intfName = mode.intf.name
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   disabled = 'disabled' in args

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     not noOrDefault )
   if xcvrConfigCli is None:
      assert noOrDefault
      return

   config = Tac.Value( 'Xcvr::SquelchConfig',
                       True,                              # valid
                       not noOrDefault and not disabled,  # enabled
                       noOrDefault )                      # moduleDefault
   xcvrConfigCli.rxAutoSquelch = config

   if noOrDefault and xcvrConfigCliDelete( xcvrConfigCli ):
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

# --------------------------------------------------------------------------------
# show transceiver squelch [ interface INTF ]
# --------------------------------------------------------------------------------

def _populateXcvrSquelch( slotName ):
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( slotName ) )
   # Only accept ZR and ZR+
   if not xcvrStatus or not xcvrStatus.isCoherent( xcvrStatus.mediaType ):
      return None

   primaryLaneId = 0
   primaryIntfName = xcvrStatus.xcvrConfig.intfName[ primaryLaneId ]
   xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( primaryIntfName )
   implControlsAdvertisement = Tac.Value( 'Xcvr::CmisImplControlsAdvertisement',
                               xcvrStatus.eepromContents.implControlsAdvertisement )

   model = XSM.TransceiverSquelchTypes()

   # Populate Rx Auto Squelch fields of model
   rxAutoSquelch = XSM.TransceiverSquelchType()

   rxAutoSquelch.supported = bool( implControlsAdvertisement.rxSquelchDisableImpl )
   for lane in range( xcvrStatus.capabilities.maxChannels ):
      setting = XSM.TransceiverLaneSquelchSetting()
      setting.default = xcvrStatus.defaultRxAutoSquelch[ lane ]
      setting.operational = xcvrStatus.currentRxAutoSquelch[ lane ]
      rxAutoSquelch.lanes[ lane + 1 ] = setting
   rxAutoSquelch.configured = 'moduleDefault'

   if xcvrConfigCli and xcvrConfigCli.rxAutoSquelch:
      config = xcvrConfigCli.rxAutoSquelch
      if config.valid and not config.moduleDefault:
         rxAutoSquelch.configured = 'enabled' if config.enabled else 'disabled'

   model.rxAutoSquelch = rxAutoSquelch
   return model

def _showXcvrSquelchInterfaces( mode, intf, mod ):
   model = XSM.TransceiverSquelchSlots()

   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   for intfName in intfNames:
      # Only accept primary intfs since we process by slot.
      if not isPrimaryIntf( intfName ):
         continue

      slotName = getXcvrSlotName( intfName )
      xcvrSquelchModel = _populateXcvrSquelch( slotName )
      if not xcvrSquelchModel:
         continue
      model.slots[ slotName ] = xcvrSquelchModel

   return model

def showXcvrSquelchInterfaces( mode, args ):
   intf = args.get( 'INTFS' )
   mod = args.get( 'MOD' )
   return _showXcvrSquelchInterfaces( mode, intf, mod )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
