# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from CliDynamicSymbol import CliDynamicPlugin
import CliGlobal
import hashlib
import LazyMount
import os

XcvrTuningExceptionModel = CliDynamicPlugin( 'XcvrTuningExceptionModel' )
gv = CliGlobal.CliGlobal( xcvrTuningExceptionSliceDir=None, agentHealthStatus=None,
                          xgc=None )

# --------------------------------------------------------------------------------
# show transceiver status electrical tuning file
# --------------------------------------------------------------------------------

def _getAllActiveXcvrAgents():
   # Goes through all "XcvrAgents" under agent/health/status and return the process
   # names of active XcvrAgents. ex. XcvrAgent-Linecard3, XcvrAgent-Linecard4 ...
   allAgents = gv.agentHealthStatus.agent.keys()
   activeXcvrAgents = list( filter(
      lambda agent: agent.startswith( "XcvrAgent" ) and
      gv.agentHealthStatus.agent[ agent ].active, allAgents ) )
   activeXcvrAgents.sort()

   return activeXcvrAgents

def _translateXcvrAgentProcessToSlice( activeXcvrAgents ):
   # Takes in a list of XcvrAgent process names and returns the sliceName for the
   # process.
   #
   # ex. XcvrAgent -> returns "FixedSystem"
   #     XcvrAgent-Linecard3 -> returns "Linecard3"
   sliceNames = []
   for activeXcvrAgent in activeXcvrAgents:
      sliceName = ""
      if activeXcvrAgent == "XcvrAgent":
         sliceName = "FixedSystem"
      else:
         sliceName = activeXcvrAgent.replace( "XcvrAgent-", "" )
      sliceNames.append( sliceName )
   return sliceNames

def _getCurrentlyConfiguredOverrideFileInfo():
   # This function will return the fileSrc and md5Hash of the currently configured
   # override file in xgc. If no file is configured, an empty string for the hash
   # is returned.
   fileSrc = gv.xgc.overrideTuningExceptionSrc
   md5Hash = ""
   if fileSrc and os.path.isfile( fileSrc ):
      with open( fileSrc, 'rb' ) as f:
         md5Hash = hashlib.md5( f.read() ).hexdigest()
   return fileSrc, md5Hash

def _verifyTuningExceptionLoadStatusSlices( sliceNames, configuredSrc,
                                            configuredHash ):
   # Given a list of slices, this function will look at the TuningExceptionLoadStatus
   # for each string and ensure that each loadStatus's file src and hash matches the
   # the file in xgc.
   #
   # The slices that match both the configuredSrc and configuredHash will be returned
   # in synchronizedSlice while the ones that do not are returned in
   # nonSynchronizedSlices

   synchronizedSlices = []
   nonSynchronizedSlices = []
   for sliceName in sliceNames:
      loadStatus = gv.xcvrTuningExceptionSliceDir.get( sliceName )
      if ( configuredSrc != loadStatus.lastAttemptedLoadSrc or
           configuredHash != loadStatus.lastLoadHash ):
         nonSynchronizedSlices.append( sliceName )
      else:
         synchronizedSlices.append( sliceName )

   return synchronizedSlices, nonSynchronizedSlices

def showTransceiverElectricalTuningFileHandler( mode, args ):
   model = XcvrTuningExceptionModel.TransceiverShowTuningExceptionModel()

   # 1. Grab all slices with a XcvrAgent running on it
   activeXcvrAgents = _getAllActiveXcvrAgents()
   sliceNames = _translateXcvrAgentProcessToSlice( activeXcvrAgents )

   if not sliceNames:
      # There's no active slices to look at. This means there's no XcvrAgents running
      # so return an empty model.
      return model

   # 2. Determine which slices have synchronized TuningExceptionLoadStatuses and
   #    which do not.
   configuredFileSrc, md5Hash = _getCurrentlyConfiguredOverrideFileInfo()
   synchronizedSlices, nonSynchronizedSlices =\
      _verifyTuningExceptionLoadStatusSlices( sliceNames, configuredFileSrc,
                                              md5Hash )

   # 3. Use one of the synchronized load statuses to populate the CLI model. Attempt
   #    to choose one from synchronizedSlices. But if there are none, then grab one
   #    from unsynchronized slices (since more time might be needed to populate load
   #    status).
   loadStatus = None
   if synchronizedSlices:
      loadStatus = gv.xcvrTuningExceptionSliceDir.get( synchronizedSlices[ 0 ] )
   else:
      loadStatus = gv.xcvrTuningExceptionSliceDir.get( nonSynchronizedSlices[ 0 ] )

   model.configured = loadStatus.lastAttemptedLoadSrc

   # loadSuccess can only be None if there is no file configured. Otherwise a
   # load must have been attempted - which can only be a success or a fail.
   if model.configured:
      model.loadSuccess = loadStatus.overrideFileActive
      model.synchronized = len( nonSynchronizedSlices ) == 0
      model.waitingFor = [ "Supervisor" if sliceName == "FixedSystem" else sliceName
                              for sliceName in nonSynchronizedSlices ]

   model.version = loadStatus.lastSuccessfulLoadVersion
   model.lastChange = loadStatus.lastChange

   return model

# --------------------------------------------------------------------------------
# transceiver electrical tuning file reload
# --------------------------------------------------------------------------------

def transceiverTuningExceptionReloadHandler( mode, args ):
   # When the handler is called without a override exception file configured, the
   # default exception file will be reloaded. When this called with a override
   # exception file configured, that exception file will be reloaded.

   gv.xgc.overrideTuningExceptionGenerationId += 1

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrTuningExceptionSliceDir = LazyMount.mount( em,
      "hardware/xcvr/status/tuningException/slice", "Tac::Dir", "ri" )
   gv.agentHealthStatus = LazyMount.mount( em,
                                           Cell.path( 'agent/health/status' ),
                                           "Agent::Health::Status", "r" )
   gv.xgc = LazyMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "r" )
