# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Tac
from CliModel import Bool, Float, List, Model, Str
from CliPlugin.XcvrCliLib import isModular

# ------------------------------------------------------------------------------
#
# Model for "show transceiver status electrical tuning file"
#
# ------------------------------------------------------------------------------
class TransceiverShowTuningExceptionModel( Model ):
   configured = Str( optional=True, help="Path to file configured" )
   loadSuccess = Bool( optional=True,
                       help="Loaded configured file" )
   synchronized = Bool( optional=True, help="Synced to all transceiver managers" )
   waitingFor = List( valueType=str,
                      help="Unsynced transceiver managers" )
   version = Str( optional=True, help="Version of configured file" )
   lastChange = Float( optional=True, help="Last update time in UTC" )

   def renderTime( self ):
      renderedTime = 'none'
      if self.lastChange:
         renderedTime = Ark.timestampToStr( Ark.utcToSwitchTime( self.lastChange ),
                                            relative=False )
      return renderedTime

   def renderLoadSuccess( self ):
      renderedLoadSuccess = 'none'
      if self.loadSuccess is not None:
         renderedLoadSuccess = "success" if self.loadSuccess else "fail"
      return renderedLoadSuccess

   def renderSynchronized( self ):
      renderedSynchronized = 'none'
      if self.synchronized is not None:
         if len( self.waitingFor ) > 0:
            renderedSynchronized =\
               f"no ({ ', '.join( self.waitingFor ) } waiting)" if isModular() else\
               "no"
         else:
            renderedSynchronized = "yes"
      return renderedSynchronized

   def render( self ):
      print( f"Configured: { self.configured or 'none' }" )
      print( f"Load result: { self.renderLoadSuccess() }" )
      print( f"Synchronized: { self.renderSynchronized() }" )
      print( f"Version: { self.version or 'none' }" )
      print( f"Last change: { self.renderTime() }" )
