# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import importlib
import os
import sys
import threading

import Ark
from ArPyUtils.LazyModule import LazyModule
import Plugins
import Tracing
import _Tac

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2

# For test purposes to test lazy handlers are valid without running all commands
__cliHandlerAutoresolve__ = os.environ.get( "CLI_HANDLER_AUTORESOLVE" )

__cliEntityManager__ = None

# Some btest spawn multiple TestCli in different entity manager namespaces, and then
# also respawn them between testlets, so returning a module that was loaded with
# another em is not a healthy thing, as the CliPlugin might have written to namespace
# X, but the DynamicCliPlugin will wait on status in namespace Y (say for  config
# mounts for example). See OpenConfigTest/test/OcProxyDutLibTests.py
# So if we detect multiple em in this process, zap the dynamic module cache set by
# previous testlets, or even current testlet, and thus also silence the complaints
# triggered when the cache does not know about a file that is already loaded.
__directImportCheckDisabled__ = False

loadDynamicPluginLock_ = threading.RLock()

@Ark.synchronized( loadDynamicPluginLock_ )
def setEntityManager( em ):
   global __cliEntityManager__
   if __cliEntityManager__ and __cliEntityManager__ != em:
      global __loadedPlugins__
      __loadedPlugins__ = {}
      global __directImportCheckDisabled__
      __directImportCheckDisabled__ = True
   __cliEntityManager__ = em

__loadedPlugins__ = {} # module name -> module

@Ark.synchronized( loadDynamicPluginLock_ )
def loadDynamicPlugin( moduleName ):
   module = __loadedPlugins__.get( moduleName )
   if module:
      return module
   fullModuleName = "CliDynamicPlugin." + moduleName

   if sys.modules.get( fullModuleName ):
      if os.environ.get( "A4_CHROOT" ):
         assert __directImportCheckDisabled__, (
            f"CliDynamic plugin {moduleName} was imported directly without being "
            "loaded as a plugin; if there is code needed by another file, please "
            "move it to a separate file that does not require to be loaded as a "
            "CliDynamic plugin" )

   # Debugging: we should not have held activity lock (e.g., from CliSave)
   assert _Tac.activityLockOwner() != threading.get_native_id(), \
      "Cannot load dynamic plugin while holding activity lock"

   module = importlib.import_module( fullModuleName )
   # record the module first as the Plugin function can raise exceptions
   __loadedPlugins__[ moduleName ] = module

   # some tests use mock without setting entity manager
   if __cliEntityManager__:
      pluginFunc = getattr( module, "Plugin", None )
      if pluginFunc:
         pluginFunc( __cliEntityManager__ )

   return module

def maybeLoadAllDynamicCliPlugins():
   if os.environ.get( "LOAD_DYNAMIC_PLUGIN" ):
      dirname = "CliDynamicPlugin"
      candidateDirs = ( os.path.join( d, dirname ) for d in sys.path )
      paths = list( filter( os.path.isdir, candidateDirs ) )
      for d in paths:
         files = os.listdir( d )
         for fname in files:
            if fname.startswith( '.' ):
               continue
            name, ext = os.path.splitext( fname )
            if ext in ( '.py', '.pyc' ):
               if name == '__init__':
                  continue
               try:
                  loadDynamicPlugin( name )
               except Plugins.SkipPluginModule:
                  pass

def resolveSymbol( name ):
   moduleDotFunc = name.split( '.' )
   obj = loadDynamicPlugin( moduleDotFunc[ 0 ] )
   for attrName in moduleDotFunc[ 1 : ]:
      obj = getattr( obj, attrName )
   return obj

class LazyCallback:
   def __init__( self, handlerName ):
      self.handlerName = handlerName
      self.handlerFunc = None
      if __cliHandlerAutoresolve__:
         self.resolve()

   def __call__( self, *args, **kwargs ):
      if not self.handlerFunc:
         self.resolve()
      return self.handlerFunc( *args, **kwargs )

   def resolve( self ):
      handler = resolveSymbol( self.handlerName )
      self.handlerFunc = handler

class CommandHandler( LazyCallback ):
   """ All commands have only mode and args parameters."""

   # pylint: disable-msg=W0221
   def __call__( self, mode, args ):
      if not self.handlerFunc:
         self.resolve()
      return self.handlerFunc( mode, args )

class CliDynamicPlugin( LazyModule ):
   """
   This class provides deferred loading of CliDynamicPlugins, similar to TacLazyType.

   Users should never import a CliDynamicPlugin directly. Instead they should use
   this wrapper, which ensures that the plugin is loaded in the proper fashion.

   So instead of

   import <MODULE>

   do

   MODULE = CliDynamicSymbol.CliDynamicPlugin( '<MODULE>' )
   """

   def __init__( self, name ):
      LazyModule.__init__( self, name )
      if __cliHandlerAutoresolve__:
         self._getmod()

   def _loadmod( self ):
      return loadDynamicPlugin( self.__lazy__ )
