# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
import fnmatch
import importlib
import inspect
import os
import shlex
import time
import traceback
import yaml

import AgentCommandRequest
import BasicCli
import CliCommand
import CliExtensionConsts
import CliExtensionMatchers
import CliExtensionValidator
import CliGlobal
import CliMatcher
import CliMode
import CliModel
from CliPlugin import TechSupportCli
import CliSave
import ConfigMount
import EbnfParser
import LazyMount
import LauncherDaemonConstants
import LauncherLib
import LauncherUtil
import ShowCommand
import TableOutput
from TypeFuture import TacLazyType
from io import StringIO

AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )
registeredCommands = {}
loadedDaemons = {}
loadedModes = {}
loadedCommands = {}

DEFAULT_EOSSDK_DAEMON_HEARTBEAT = 0

gv = CliGlobal.CliGlobal( {
   'extensionsLoaded': False,
   'useLocalMounts': False,
   'cliExtensionConfig': None,
   'daemonConfigDir': None,
   'daemonStatusDir': None,
   'aclConfigDir': None,
   'launcherConfifgDir': None,
} )

class CliExtensionLoadError( Exception ):
   pass

class CliCommandClass:
   pass

class ShowCommandClass:
   def handler( self, ctx ):
      raise NotImplementedError( 'Handler function must be defined' )

   def render( self, data ):
      print( data )

class TableShowCommandClass( ShowCommandClass ):
   HEADERS = []

   def __init__( self ):
      assert self.HEADERS, 'HEADERS must be present'
      assert len( self.HEADERS ) == len( set( self.HEADERS ) ), ( 'HEADERS must '
                                                                  'be unique' )

   def getData( self, ctx ):
      raise NotImplementedError( 'getData function must be defined' )

   def handler( self, ctx ):
      result = []
      for data in self.getData( ctx ):
         assert len( data ) == len( self.HEADERS ), ( 'Data length must match '
                                                      'headers length' )
         result.append( collections.OrderedDict( zip( self.HEADERS, data ) ) )

      # TODO: we should change this to a yield at some point, so we could support
      # streamed output from the customers function
      return result

   def render( self, data ):
      table = TableOutput.createTable( self.HEADERS )
      for row in data:
         table.newRow( *list( row.values() ) )
      print( table.output() )

class CliExtensionCtx:
   def __init__( self, mode, args ):
      self.mode_ = mode
      self.args_ = args

   @property
   def args( self ):
      return self.args_

   def addWarning( self, warning ):
      self.mode_.addWarning( warning )

   def addError( self, error ):
      self.mode_.addError( error )

   def isStartupConfig( self ):
      return self.mode_.session_.startupConfig()

   def isNoOrDefaultCmd( self ):
      return CliCommand.isNoOrDefaultCmd( self.args_ )

   def isNoCmd( self ):
      return CliCommand.isNoCmd( self.args_ )

   def isDefaultCmd( self ):
      return CliCommand.isDefaultCmd( self.args_ )

   @property
   def config( self ):
      # first try to get the config from a daemon
      daemon = self.daemon
      if daemon is not None:
         return self.daemon.config

      if not hasattr( self.mode_, 'getModeName' ):
         return None
      modeName = self.mode_.getModeName()
      return self.getModeConfig( modeName )

   @property
   def status( self ):
      # first try to get the config from a daemon
      daemon = self.daemon
      if daemon is not None:
         return self.daemon.status

      # no other status for source so return None
      return None

   def getModeConfig( self, modeName ):
      if modeName not in gv.cliExtensionConfig.config:
         return None
      return ConfigAccessor( gv.cliExtensionConfig.config, modeName )

   @property
   def daemon( self ):
      if not hasattr( self.mode_, 'getDaemonName' ):
         return None
      daemonName = self.mode_.getDaemonName()
      return self.getDaemon( daemonName )

   def getDaemon( self, daemonName ):
      if daemonName not in loadedDaemons:
         return None
      return DaemonAccessor( daemonName )

class ConfigAccessor:
   def __init__( self, configDir, name ):
      self.configDir_ = configDir
      self.name_ = name

   def configSet( self, key, value='' ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.option[ key ] = str( value )

   def configDel( self, key=None ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      del config.option[ key ]

   def config( self, key, defaultVal=None ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return defaultVal

      return config.option.get( key, defaultVal )

   def configIter( self, pattern='*' ):
      """Returns an iterator over the config for matching entries"""
      config = self.configDir_.get( self.name_ )
      if config is None:
         return

      for k, v in sorted( config.option.items() ):
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

   def isEnabled( self ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return False
      return config.enabled

   def enable( self ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.enabled = True

   def disable( self ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.enabled = False

class StatusAccessor:
   def __init__( self, statusDir, name ):
      self.statusDir_ = statusDir
      self.name_ = name

   def status( self, key, defaultVal=None ):
      status = self.statusDir_.get( self.name_ )
      if status is None:
         return defaultVal

      return status.data.get( key, defaultVal )

   def statusIter( self, pattern='*' ):
      """Returns an iterator over the status for matching entries"""
      status = self.statusDir_.get( self.name_ )
      if status is None:
         return

      for k, v in sorted( status.data.items() ):
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

class DaemonAccessor:
   def __init__( self, daemonName ):
      self.daemonName_ = daemonName

   @property
   def config( self ):
      return ConfigAccessor( gv.daemonConfigDir, self.daemonName_ )

   @property
   def status( self ):
      return StatusAccessor( gv.daemonStatusDir, self.daemonName_ )

def _generateData( syntax, cmdData, isShow=False ):
   data = {}
   for token in EbnfParser.tokenize( syntax ):
      if isShow and token.lower() == 'show':
         continue

      if token == '...':
         # this treated as trailing garbage by the parser we shouldn't
         # try to populate it
         continue

      if token in data:
         # this means that a token is repeated twice in the syntax
         continue

      if token not in cmdData:
         # this as a keyword
         matcher = CliMatcher.KeywordMatcher( token, helpdesc=token )
      else:
         # this assert is validated at a higher level
         assert len( cmdData[ token ] ) == 1
         matcherType = next( iter( cmdData[ token ] ) )
         matcherArgs = cmdData[ token ][ matcherType ]
         if matcherArgs is None:
            matcherArgs = {}
         matcherGenFunc = CliExtensionMatchers.MATCHERS[ matcherType ][ 'func' ]
         matcher = matcherGenFunc( token, **matcherArgs )

      # we set canMerge=False because when we merge we also ensure that the
      # helpdescs are the same. If they are not we throw an error. However
      # for customer plugins we don't want them to be beholden to our
      # potentially changing helpdesc. This does lower performance a bit,
      # but customers won't be adding a ton of commands.
      if isinstance( matcher, CliCommand.SetEnumMatcher ):
         data[ token ] = matcher
      else:
         data[ token ] = CliCommand.Node( matcher=matcher, canMerge=False )

   return data

def _registerShowCommand( name, cmdInfo, cmdClass ):
   cmdClassInstance = cmdClass()

   class CustomerShowCmdModel( CliModel.Model ):
      #pylint: disable-next=protected-access
      _ext_data = CliModel._AttributeType( help='.', optional=True )
      def toDict( self, revision=None, includePrivateAttrs=False, streaming=False ):
         return self._ext_data

      @property
      def __dict__( self ):
         # compat for Cli/CliApi handling modified in follow-up mut
         return { '__data': self._ext_data }

      def render( self ):
         cmdClassInstance.render( self._ext_data )

   hasSchema = 'outputSchema' in cmdInfo

   class CliExtensionShowCmd( ShowCommand.ShowCliCommandClass ):
      syntax = cmdInfo[ 'syntax' ]
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ),
            isShow=True )
      privileged = cmdInfo[ 'mode' ] != 'Unprivileged'
      hidden = cmdInfo.get( 'hidden', False )
      cliModel = CustomerShowCmdModel if hasSchema else None

      @staticmethod
      def handler( mode, args ):
         ctx = CliExtensionCtx( mode, args )
         # TODO: should we capture output if we a schema?
         data = cmdClassInstance.handler( ctx )
         if hasSchema:
            if data is None:
               mode.addError( 'Handler has returned no data, but data was '
                              'expected because a schema is registered' )
               return None
            else:
               # TODO: we should validate the data against the schema provided
               model = CustomerShowCmdModel()
               model._ext_data = data  # pylint: disable=protected-access
               return model
         else:
            # handle a command that doesn't have a defined schema
            if data is not None:
               mode.addError( 'Handler has no output schema but return data' )
               return None
         return None

   BasicCli.addShowCommandClass( CliExtensionShowCmd )

def _getHandlerFn( name, cmdClassInstance ):
   def wrapHandler( handler ):
      def wrappedHandler( mode, args ):
         ctx = CliExtensionCtx( mode, args )
         return handler( ctx )
      return wrappedHandler

   if hasattr( cmdClassInstance, name ):
      return wrapHandler( getattr( cmdClassInstance, name ) )
   return None

def _registerNonShowCommand( name, cmdInfo, cmdClass ):
   cmdClassInstance = cmdClass()
   cmdHandler = _getHandlerFn( 'handler', cmdClassInstance )
   cmdNoHandler = _getHandlerFn( 'noHandler', cmdClassInstance )
   cmdDefaultHandler = _getHandlerFn( 'defaultHandler', cmdClassInstance )

   # if we have a no handler, but a default handler wasn't supplied
   # then we assume that the default handler is the same as the no handler
   if cmdNoHandler and cmdDefaultHandler is None:
      cmdDefaultHandler = cmdNoHandler

   if 'syntax' in cmdInfo and cmdHandler is None:
      raise CliExtensionLoadError(
            'Command Definition %s: Syntax field was defined but no '
            'command handler was defined' % name )
   if 'noSyntax' in cmdInfo and cmdNoHandler is None:
      raise CliExtensionLoadError(
            'Command Definition %s: NoSyntax field was defined but no '
            'command handler was defined' % name )

   class CliExtensionCmd( CliCommand.CliCommandClass ):
      syntax = cmdInfo.get( 'syntax' )
      noOrDefaultSyntax = cmdInfo.get( 'noSyntax' )
      handler = cmdHandler
      noHandler = cmdNoHandler
      defaultHandler = cmdDefaultHandler
      # TODO: we shouldn't rely on syntax always being there
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ) )
      hidden = cmdInfo.get( 'hidden', False )

   mode = loadedModes[ cmdInfo[ 'mode' ] ]
   mode.addCommandClass( CliExtensionCmd )

class RunningConfigGeneratorCtx:
   def __init__( self, configAccessor ):
      self.configAccessor_ = configAccessor

   @property
   def config( self ):
      return self.configAccessor_

def _defaultRunningConfigGenerator( ctx ):
   cmds = []
   for k, v in ctx.config.configIter():
      cmd = f'{k} {v}'
      cmds.append( cmd.strip() )

   if ctx.config.isEnabled():
      cmds.append( 'no disabled' )

   return cmds

def _createModeCliSavePlugin( modeName, ExtensionCliSaveMode,
      runningConfigGenerator ):
   @CliSave.saver( 'CliExtension::Config', 'cli/extension' )
   # pylint: disable-msg=unused-variable
   def saveExtensionCliConfig( entity, root, sysdbRoot, options ):
      extensionConfig = entity.config.get( modeName )
      configAccessor = ConfigAccessor( entity.config, modeName )
      if extensionConfig is None:
         return

      if not extensionConfig.option and not extensionConfig.enabled:
         # if the mode is empty don't add any commands
         return

      CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
      ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )
      mode = root[ ExtensionCliSaveMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'ExtensionCli.config' ]
      ctx = RunningConfigGeneratorCtx( configAccessor )
      for cmd in runningConfigGenerator( ctx ):
         cmds.addCommand( cmd )
   # pylint: enable-msg=unused-variable

def _createDaemonCliSavePlugin( daemonName, ExtensionCliSaveMode,
      runningConfigGenerator ):
   # pylint: disable-msg=unused-variable
   @CliSave.saver( 'Tac::Dir', 'daemon/agent/config' )
   def saveExtensionCliConfig( entity, root, sysdbRoot, options ):
      extensionConfig = entity.get( daemonName )
      configAccessor = ConfigAccessor( entity, daemonName )
      if extensionConfig is None:
         return

      if not extensionConfig.option and not extensionConfig.enabled:
         # if the mode is empty don't add any commands
         return

      CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
      ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )
      mode = root[ ExtensionCliSaveMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'ExtensionCli.config' ]
      ctx = RunningConfigGeneratorCtx( configAccessor )
      for cmd in runningConfigGenerator( ctx ):
         cmds.addCommand( cmd )
   # pylint: enable-msg=unused-variable

def _createCliSavePlugin( modeName, modeInfo, ExtensionCliSaveMode ):
   daemonName = modeInfo.get( 'daemon' )
   runningConfigGenerator = modeInfo.get( 'runningConfigGenerator',
         _defaultRunningConfigGenerator )
   if daemonName is not None:
      _createDaemonCliSavePlugin( daemonName, ExtensionCliSaveMode,
            runningConfigGenerator )
   else:
      _createModeCliSavePlugin( modeName, ExtensionCliSaveMode,
            runningConfigGenerator )

def _createDaemonLauncherConfig( mode, daemonName ):
   ''' Programs into LauncherConfig '''
   assert daemonName not in gv.launcherConfifgDir.agent
   daemonInfo = loadedDaemons[ daemonName ]
   daemonConfig = gv.launcherConfifgDir.newAgent( daemonName )
   daemonConfig.userDaemon = True
   daemonConfig.heartbeatPeriod = daemonInfo.get( 'heartbeatPeriod',
         DEFAULT_EOSSDK_DAEMON_HEARTBEAT )
   daemonConfig.useEnvvarForSockId = True # True for EosSdk
   daemonConfig.oomScoreAdj = daemonInfo.get( 'oomScore',
         LauncherDaemonConstants.DEFAULT_OOM_SCORE_ADJ )
   daemonConfig.exe = daemonInfo[ 'exe' ]
   argv = shlex.split( daemonInfo.get( 'argv', '' ) )
   for i, arg in enumerate( argv ):
      daemonConfig.argv[ i ] = arg

   if not LauncherUtil.isExecutable( daemonInfo[ 'exe' ] ):
      mode.addWarning( 'The executable %s does not exist '
                       '(or is not executable)' % daemonInfo[ 'exe' ] )

   # Set up the runnability criteria, so the agent only runs when
   # the genericAgentCfg.enabled is True on the application. Meaning only if the
   # application is enabled do we run the daemon
   # (even if we try to start it anyway)
   daemonConfig.runnability = ( 'runnability', )
   daemonConfig.runnability.qualPath = ( 'daemon/agent/runnability/%s' % daemonName )

   for redProto in LauncherUtil.allRedProtoSet:
      daemonConfig.criteria[ redProto ] = LauncherUtil.activeSupervisorRoleName
   daemonConfig.stable = True

def _maybeCreateDaemonConfig( mode, daemonName ):
   genericAgentCfg = gv.daemonConfigDir.get( daemonName )
   if genericAgentCfg is None:
      genericAgentCfg = gv.daemonConfigDir.newEntity(
            'GenericAgent::Config', daemonName )
      genericAgentCfg.agentType = AGENT_TYPE.cliExtensionAgent
      gv.aclConfigDir.newEntity( 'Acl::ServiceAclTypeVrfMap', daemonName )
      _createDaemonLauncherConfig( mode, daemonName )
   else:
      if genericAgentCfg.agentType != AGENT_TYPE.cliExtensionAgent:
         mode.addError( 'Unable to create daemon \'%s\' because CLI daemon '
                        '\'%s\' already exists. Please delete the daemon'
                        % ( daemonName, daemonName ) )

def _createGotoModeCommandClass( modeName, modeInfo, ExtensionCliMode ):
   cmdInfo = modeInfo[ 'command' ]
   daemonName = modeInfo.get( 'daemon' )

   class GotoExtensionModeCmd( CliCommand.CliCommandClass ):
      syntax = cmdInfo[ 'syntax' ]
      noOrDefaultSyntax = cmdInfo[ 'noSyntax' ]
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ) )

      @staticmethod
      def handler( mode, args ):
         if daemonName is not None:
            _maybeCreateDaemonConfig( mode, daemonName )
         else:
            if modeName not in gv.cliExtensionConfig.config:
               gv.cliExtensionConfig.config.newMember( modeName )
         childMode = mode.childMode( ExtensionCliMode )
         mode.session_.gotoChildMode( childMode )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         if daemonName is not None:
            genericAgentCfg = gv.daemonConfigDir.get( daemonName )
            if ( genericAgentCfg is None or
                 genericAgentCfg.agentType != AGENT_TYPE.cliExtensionAgent ):
               return
            gv.daemonConfigDir.deleteEntity( daemonName )
            gv.aclConfigDir.deleteEntity( daemonName )
            del gv.launcherConfifgDir.agent[ daemonName ]
         else:
            del gv.cliExtensionConfig.config[ modeName ]

   BasicCli.GlobalConfigMode.addCommandClass( GotoExtensionModeCmd )

def _createCliMode( modeName, modeInfo ):
   modeKey = modeInfo[ 'modeKey' ]
   longModeKey = modeInfo.get( 'longModeKey', modeKey )

   class ExtensionBaseMode( CliMode.ConfigMode ):
      def __init__( self, param ):
         self.modeKey = modeKey
         self.longModeKey = longModeKey
         CliMode.ConfigMode.__init__( self, param )

      def enterCmd( self ):
         return modeInfo[ 'command' ][ 'syntax' ]

   class ExtensionCliMode( ExtensionBaseMode, BasicCli.ConfigModeBase ):
      name = longModeKey

      def __init__( self, parent, session ):
         ExtensionBaseMode.__init__( self, None )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def getDaemonName( self ):
         return modeInfo.get( 'daemon' )

      def getModeName( self ):
         return modeName

   class ExtensionCliSaveMode( ExtensionBaseMode, CliSave.Mode ):
      def __init__( self, param ):
         ExtensionBaseMode.__init__( self, param )
         CliSave.Mode.__init__( self, param )

   CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
   ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )

   _createCliSavePlugin( modeName, modeInfo, ExtensionCliSaveMode )
   _createGotoModeCommandClass( modeName, modeInfo, ExtensionCliMode )

   return ExtensionCliMode

def loadSyntaxYaml( yamlContents, checkVendor=False, verifyOnly=False ):
   extContentsJson = yaml.load( yamlContents, yaml.Loader )

   if 'techSupportCommands' in extContentsJson:
      techSupportCmds = extContentsJson[ 'techSupportCommands' ]
      CliExtensionValidator.validateTechSupportCommands( techSupportCmds )
      registerTechSupport( techSupportCmds )

   if checkVendor:
      if 'vendor' not in extContentsJson:
         raise CliExtensionLoadError( 'Vendor information is missing' )
      CliExtensionValidator.valdiateVendorInfo( extContentsJson[ 'vendor' ] )

   for daemonName, daemonInfo in extContentsJson.get( 'daemons', {} ).items():
      if daemonName in loadedDaemons:
         raise CliExtensionLoadError(
               'Daemon Definition %s: Duplicate daemon name' %
               daemonName )
      CliExtensionValidator.validateDaemon( daemonName, daemonInfo )
      loadedDaemons[ daemonName ] = daemonInfo

   namespace = extContentsJson.get( 'namespace' )
   CliExtensionValidator.validateNamespace( namespace )

   loadedModes[ 'Privileged' ] = BasicCli.EnableMode
   loadedModes[ 'Unprivileged' ] = BasicCli.UnprivMode
   modesInfo = extContentsJson.get( 'modes', {} )
   for modeName, modeInfo in sorted( modesInfo.items() ):
      if modeName in loadedModes:
         raise CliExtensionLoadError(
               'Mode Definition %s: Mode name has already been seen' % modeName )
      CliExtensionValidator.validateMode( modeName, modeInfo )
      if not verifyOnly:
         loadedModes[ modeName ] = _createCliMode( modeName, modeInfo )

   for cmdName, cmdInfo in extContentsJson.get( 'commands', {} ).items():
      if cmdName in loadedCommands.get( namespace, {} ):
         raise CliExtensionLoadError(
               'Command Definition %s: Command name has already been seen' %
               cmdName )

      if ( ( namespace not in registeredCommands and not verifyOnly ) or
           ( cmdName not in registeredCommands[ namespace ] and not verifyOnly ) ):
         raise CliExtensionLoadError(
               'Command Definition %s: Command is missing a command handler' %
               cmdName )

      CliExtensionValidator.validateCmd( cmdName, cmdInfo )
      if namespace not in loadedCommands:
         loadedCommands[ namespace ] = {}
      loadedCommands[ namespace ][ cmdName ] = cmdInfo # To keep track of duplicates

      if verifyOnly:
         return

      # register the command with the parser!
      cmdClass = registeredCommands[ namespace ][ cmdName ]
      if issubclass( cmdClass, ShowCommandClass ):
         _registerShowCommand( cmdName, cmdInfo, cmdClass )
      else:
         _registerNonShowCommand( cmdName, cmdInfo, cmdClass )

def _loadSyntaxFile( path ):
   try:
      with open( path ) as f:
         contents = f.read()
   except OSError as e:
      print( f'Error: Unable to load syntax file {path} due to: {e}' )
      return

   try:
      loadSyntaxYaml( contents, checkVendor=True )
   except ( yaml.YAMLError,
            CliExtensionValidator.CliExtensionValidationError,
            CliExtensionLoadError ) as e:
      print( f'Error: Unable to parse syntax file {path} due to: {e}' )
      traceback.print_exc()

def initCliExtensions( entityManager ):
   if gv.extensionsLoaded:
      return
   gv.extensionsLoaded = True

   if not gv.useLocalMounts:
      gv.cliExtensionConfig = ConfigMount.mount( entityManager,
            'cli/extension', 'CliExtension::Config', 'wi' )
      gv.daemonConfigDir = ConfigMount.mount( entityManager,
            'daemon/agent/config', 'Tac::Dir', 'wi' )
      gv.daemonStatusDir = LazyMount.mount( entityManager,
            'daemon/agent/status', 'Tac::Dir', 'ri' )
      gv.aclConfigDir = ConfigMount.mount( entityManager,
            'daemon/acl/config', 'Tac::Dir', 'wi' )
      gv.launcherConfifgDir = ConfigMount.mount( entityManager,
            LauncherLib.agentConfigCliDirPath, 'Launcher::AgentConfigDir', 'wi' )
   else:
      gv.cliExtensionConfig = entityManager.root()[ 'cli' ][ 'extension' ]
      gv.daemonConfigDir = entityManager.root()[ 'daemon' ][ 'agent' ][ 'config' ]
      gv.daemonStatusDir = entityManager.root()[ 'daemon' ][ 'agent' ][ 'status' ]
      gv.aclConfigDir = entityManager.root()[ 'daemon' ][ 'acl' ][ 'config' ]
      gv.launcherConfifgDir = LauncherLib.agentConfigCliDir( entityManager.root() )

   _loadCliExtensions()

def _extensionFilesToLoad():
   filePaths = set()
   cliExtensionDir = os.environ.get( 'CLI_EXTENSION_DIR',
         CliExtensionConsts.CLI_EXTENSION_DIR )

   if not os.path.isdir( cliExtensionDir ):
      # no extensions: nothing to do
      return filePaths

   pluginTree = ( importlib.import_module( "PluginTree" ).PluginTree(
            "/bld/deps", "DataPlugin" )
         if importlib.util.find_spec( "PluginTree" ) else None )

   for f in os.listdir( cliExtensionDir ):
      if not f.endswith( '.yaml' ):
         continue

      # BUG642483: In the build only load up yaml files that are relevent
      if pluginTree is None or pluginTree.canLoad( cliExtensionDir, f ):
         filePaths.add( os.path.join( cliExtensionDir, f ) )

   return filePaths

def _loadCliExtensions():
   for filename in _extensionFilesToLoad():
      try:
         _loadSyntaxFile( filename )
      except Exception as e: # pylint: disable-msg=broad-except
         # ideally we should never hit here, but alas we can't always get what we
         # want
         print( 'Exception', e )
         traceback.print_exc()

def registerCommand( name, cmdClass, namespace=None ):
   if not isinstance( name, str ):
      raise CliExtensionLoadError(
            'Unable to add Command Class %r'
            'due to invalid name %r' % ( cmdClass, name ) )
   CliExtensionValidator.validateNamespace( namespace )

   if ( not inspect.isclass( cmdClass ) or
        ( not issubclass( cmdClass, ShowCommandClass ) and
          not issubclass( cmdClass, CliCommandClass ) ) ):
      raise CliExtensionLoadError(
            'Unable to register command "%s" because the Command Class "%s" does '
            'not inherit from "CliExtension.ShowCommandClass" or '
            '"CliExtension.CliCommandClass"' % ( name, cmdClass ) )

   if namespace not in registeredCommands:
      registeredCommands[ namespace ] = {}

   if name in registeredCommands[ namespace ]:
      if cmdClass == registeredCommands[ namespace ][ name ]:
         return
      raise CliExtensionLoadError(
            'Unable to register command "%s" because Command Class "%s" already '
            'registered with the same name' %
            ( name, registeredCommands[ namespace ][ name ] ) )

   registeredCommands[ namespace ][ name ] = cmdClass

def registerTechSupport( cmds, cmdsGuard=None ):
   if cmdsGuard is not None:
      assert callable( cmdsGuard ), 'cmdsGuard %s is not callable' % cmdsGuard
   # use the current state so that customer 'show tech' commands always appear at
   # the  of the 'show tech'. This feels like an internal implementation detail
   # that shouldn't be in a customer facing API
   timestamp = time.strftime( '%Y-%m-%d %H:%M:%S' )
   TechSupportCli.registerShowTechSupportCmd( timestamp, cmds,
         cmdsGuard=cmdsGuard )

def agentRpc( ctx, agentName, command ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand(
         ctx.mode_.entityManager, # entityManager
         f'{agentName}-{agentName}', # dirName
         "EosSdkAgent", # commandType
         command, # command
         stringBuff=buff )

   return buff.getvalue()
