# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import types

import CliCommand
import CliMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpGenAddrMatcher
from CliPlugin import MacAddr
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
from Intf import IntfRange
import MultiRangeRule

# pylint: disable=redefined-builtin

MATCHERS = {}

class CliExtensionMatcherError( Exception ):
   pass

def _checkArgs( args ):
   if not isinstance( args, dict ):
      raise CliExtensionMatcherError( 'Args %r was expected to be a dict' % args )
   for arg, requiredType in args.items():
      if not isinstance( arg, str ):
         raise CliExtensionMatcherError( 'Arg %r was expected to be a string' % arg )
      if not isinstance( requiredType, ( type, tuple ) ):
         raise CliExtensionMatcherError(
               'Arg %r value was expected to be a type or tuple' % arg )
      if isinstance( requiredType, tuple ):
         for i in requiredType:
            if not isinstance( i, type ):
               raise CliExtensionMatcherError(
                     f'Arg {arg!r} value {i!r} was expected to be a type' )

def registerMatcher( matcherType, matcherGenFunc, requiredArgs=None,
      optionalArgs=None ):
   if requiredArgs is None:
      requiredArgs = {}
   if optionalArgs is None:
      optionalArgs = {}

   for arg in requiredArgs:
      if arg in optionalArgs:
         raise CliExtensionMatcherError(
               'Argument %r is both in requiredArgs and optionalArgs' % arg )

   _checkArgs( requiredArgs )
   _checkArgs( optionalArgs )

   if not callable( matcherGenFunc ):
      raise CliExtensionMatcherError(
            'matcherGenFunc %r is not callable' % matcherGenFunc )

   if matcherType in MATCHERS:
      raise CliExtensionMatcherError(
            'Matcher %r has already been defined' % matcherType )

   MATCHERS[ matcherType ] = {
      'func': matcherGenFunc,
      'requiredArgs': requiredArgs,
      'optionalArgs': optionalArgs
   }

#-------------------------------------------------------------------------------
# keyword matcher
#-------------------------------------------------------------------------------
def _generateKeywordMatcher( token, help=None ):
   helpdesc = token if help is None else help
   return CliMatcher.KeywordMatcher( token, helpdesc=helpdesc )

registerMatcher(
   matcherType='keyword',
   matcherGenFunc=_generateKeywordMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str
   }
)

#-------------------------------------------------------------------------------
# enum matcher
#-------------------------------------------------------------------------------
# we define a seperate context to send to the dynamic function so that customers
# don't directly use the mode object. This will allow us to extend what is sent
# to the function over time without causing any incompatabilities
class DynamicKeywordMatcherContext:
   def __init__( self, mode ):
      self.mode_ = mode

def _generateEnumMatcher( token, values, alwaysMatchInStartupConfig=False,
      regex=None ):
   def _keywordsFn( mode ):
      ctx = DynamicKeywordMatcherContext( mode )
      return values( ctx )

   def _valuesFn( mode ):
      return values

   if regex:
      namesFn = _keywordsFn if callable( values ) else _valuesFn
      return CliMatcher.DynamicNameMatcher( namesFn, helpdesc=token, pattern=regex )

   if callable( values ):
      return CliMatcher.DynamicKeywordMatcher( _keywordsFn,
            alwaysMatchInStartupConfig=alwaysMatchInStartupConfig )
   else:
      return CliMatcher.EnumMatcher( values )

registerMatcher(
   matcherType='enum',
   matcherGenFunc=_generateEnumMatcher,
   requiredArgs={
      'values': ( dict, types.FunctionType )
   },
   optionalArgs={
      # ignored for the static case. Just meant for the DynamicKeywordMatcher case
      'alwaysMatchInStartupConfig': bool,
      'regex': str
   }
)

#-------------------------------------------------------------------------------
# setEnum matcher
#-------------------------------------------------------------------------------
def _generateSetEnumMatcher( token, values ):
   return CliCommand.SetEnumMatcher( values )

registerMatcher(
   matcherType='setEnum',
   matcherGenFunc=_generateSetEnumMatcher,
   requiredArgs={
      'values': dict
   },
   optionalArgs=None
)

#-------------------------------------------------------------------------------
# regex matcher
#-------------------------------------------------------------------------------
def _generateRegexMatcher( token, regex, help=None ):
   helpdesc = token if help is None else help
   return CliMatcher.PatternMatcher( regex, helpname=token, helpdesc=helpdesc )

registerMatcher(
   matcherType='regex',
   matcherGenFunc=_generateRegexMatcher,
   requiredArgs={
      'regex': str
   },
   optionalArgs={
      'help': str
   }
)

#-------------------------------------------------------------------------------
# integer matcher
#-------------------------------------------------------------------------------
# we define a seperate context to send to the dynamic function so that customers
# don't directly use the mode object. This will allow us to extend what is sent
# to the function over time without causing any incompatabilities
class DynamicNumberContext:
   def __init__( self, mode ):
      self.mode_ = mode

def _getBoundFunc( lbound, ubound ):
   def getBound( ctx, bound ):
      return bound( ctx ) if callable( bound ) else bound

   def rangeFn( mode, context ):
      ctx = DynamicNumberContext( mode )
      return ( getBound( ctx, lbound ),
               getBound( ctx, ubound ) )
   return rangeFn

def _generateIntegerMatcher( token, help='Integer value', min=None, max=None ):
   lbound = min if min is not None else -sys.maxsize - 1
   ubound = max if max is not None else sys.maxsize
   if callable( lbound ) or callable( ubound ):
      boundFunc = _getBoundFunc( lbound, ubound )
      return CliMatcher.DynamicIntegerMatcher( boundFunc, helpdesc=help )

   return CliMatcher.IntegerMatcher( lbound, ubound, helpdesc=help )

registerMatcher(
   matcherType='integer',
   matcherGenFunc=_generateIntegerMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'min': ( int, types.FunctionType ),
      'max': ( int, types.FunctionType )
   }
)

#-------------------------------------------------------------------------------
# integerList matcher
#-------------------------------------------------------------------------------
def _getRangeFn( lbound, ubound ):
   def getRange( ctx, bound ):
      return bound( ctx ) if callable( bound ) else bound

   def rangeFn():
      ctx = DynamicNumberContext( None )
      return ( getRange( ctx, lbound ),
               getRange( ctx, ubound ) )
   return rangeFn

def _generateIntegerListMatcher( token, help='Integer value', min=None, max=None,
      maxRanges=0 ):
   lbound = min if min is not None else 0
   ubound = max if max is not None else sys.maxsize
   rangeFn = _getRangeFn( lbound, ubound )

   return MultiRangeRule.MultiRangeMatcher(
         rangeFn,
         noSingletons=False,
         helpdesc=help,
         maxRanges=maxRanges,
         value=lambda mode, grList: grList.ranges() )

registerMatcher(
   matcherType='integerList',
   matcherGenFunc=_generateIntegerListMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'min': ( int, types.FunctionType ),
      'max': ( int, types.FunctionType ),
      'maxRanges': int
   }
)

#-------------------------------------------------------------------------------
# float matcher
#-------------------------------------------------------------------------------
def _generateFloatMatcher( token, help='Floating-point value', min=None, max=None ):
   lbound = min if min is not None else -sys.float_info.max
   ubound = max if max is not None else sys.float_info.max
   if callable( lbound ) or callable( ubound ):
      return CliMatcher.DynamicFloatMatcher( _getBoundFunc( lbound, ubound ),
            helpdesc=help, precisionString='%.2g' )

   return CliMatcher.FloatMatcher( lbound, ubound, helpdesc=help,
         precisionString='%.2g' )

registerMatcher(
   matcherType='float',
   matcherGenFunc=_generateFloatMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'min': ( float, types.FunctionType ),
      'max': ( float, types.FunctionType )
   }
)

#-------------------------------------------------------------------------------
# ipv4Address matcher
#-------------------------------------------------------------------------------
def _generateIpv4AddressMatcher( token, help='IPv4 Address' ):
   return IpAddrMatcher.IpAddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='ipv4Address',
   matcherGenFunc=_generateIpv4AddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# ipv6Address matcher
#-------------------------------------------------------------------------------
def _generateIpv6AddressMatcher( token, help='IPv6 Address' ):
   return Ip6AddrMatcher.Ip6AddrMatcher( helpdesc=help,
         value=lambda mode, result: str( result ) )

registerMatcher(
   matcherType='ipv6Address',
   matcherGenFunc=_generateIpv6AddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# ipAddress matcher
#-------------------------------------------------------------------------------
def _generateIpAddressMatcher( token, help='IPv4 or IPv6 Address' ):
   return IpGenAddrMatcher.IpGenAddrMatcher( helpdesc=help,
         value=lambda mode, result: str( result ) )

registerMatcher(
   matcherType='ipAddress',
   matcherGenFunc=_generateIpAddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# macAddress matcher
#-------------------------------------------------------------------------------
def _generateMacAddressMatcher( token, help='Ethernet address' ):
   return MacAddr.MacAddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='macAddress',
   matcherGenFunc=_generateMacAddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# Interface matcher
#-------------------------------------------------------------------------------
SUPPORTED_INTERFACE_TYPES = (
   "Ethernet",
   "Cpu",
   "Application",
   "Management",
)

def _isSupportedInterfaceType( interfaceTypes=None ):
   for type_ in interfaceTypes:
      if type_ not in SUPPORTED_INTERFACE_TYPES:
         raise CliExtensionMatcherError(
            f'Invalid interface type specified: {type_}' )

def _generateIntfMatcher( token, interfaceTypes=None ):
   # fail if unsupported
   _isSupportedInterfaceType( interfaceTypes )

   # define vanilla value function
   def valueFunc_( mode, result ):
      return result

   matchers = IntfMatcher()
   for intfType in interfaceTypes:
      matchers |= PhysicalIntfMatcher( intfType, value=valueFunc_ )

   return matchers

registerMatcher(
   matcherType='interface',
   matcherGenFunc=_generateIntfMatcher,
   requiredArgs={
      'interfaceTypes': list,
   },
   optionalArgs=None
)

# -------------------------------------------------------------------------------
# Interface range matcher
# -------------------------------------------------------------------------------

def _generateIntfRangeMatcher( token, interfaceTypes=None ):
   _isSupportedInterfaceType( interfaceTypes )
   enExplicitIntfTypes = [
      t for k, t in IntfRange.allIntfTypes_.items() if k in interfaceTypes
   ]

   def valueFunc_( mode, result ):
      return list( result )

   return IntfRange.IntfRangeMatcher( explicitIntfTypes=enExplicitIntfTypes,
                                     value=valueFunc_ )


registerMatcher(
   matcherType='interfaceRange',
   matcherGenFunc=_generateIntfRangeMatcher,
   requiredArgs={
      'interfaceTypes': list,
   },
   optionalArgs=None
)
