#!/usr/bin/env python3
# Copyright (c) 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Todo: It might be good to ensure that the arglist of all extensions added
# to a given hook are consistent (see inspect.getargspec)
class CliHook:
   """A hook to allow clients to modify the behavior of the Cli, possibly in
   a manner that is independent of package dependencies."""
   def __init__( self ):
      self.extension_ = []

   def extensions( self ):
      """Returns a list of the extensions attached to this hook."""
      return list( self.extension_ )

   def addExtension( self, extension ):
      """Add an extension to this CliHook.  It is up to the client to ensure
      that the extension is compatible with the CliHook.  API of extension
      should be provided in documentation at point of definition of
      each CliHook."""
      self.extension_.append( extension )

   def notifyExtensions( self, *args, **kwargs ):
      """Notify all registered extentions by the passing arguments."""
      for cb in self.extension_:
         cb( *args, **kwargs )

   def any( self, *args, **kwargs ):
      """Wrapper to the `any` builtin with optional arguments."""
      return any( cb( *args, **kwargs ) for cb in self.extension_ )

   def all( self, *args, **kwargs ):
      """Wrapper to the `all` builtin with optional arguments."""
      return all( cb( *args, **kwargs ) for cb in self.extension_ )

def executeCliHookBefore( cliHook ):
   """ @brief It basically executes the method `notifyExtensions` of the
              hook passed as the argument and it after calls the
              decorated function.
              All parameters of the funtion are forwarded to the hook.
   """
   def cliHandlerDecorator( cliHandlerFn ):
      def inner( *args, **kwargs ):
         cliHook.notifyExtensions( *args, **kwargs )
         cliHandlerFn( *args, **kwargs )
      return inner
   return cliHandlerDecorator
