# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import io
import itertools
from tempfile import NamedTemporaryFile

import Ark
import ArPyUtils
import CliCommon
from CliSave import ( saveSessionDiffs,
                      getRunningConfigSaveBlock,
                      getCleanConfigSaveBlock,
                      getSessionSaveBlock )
from CliSaveBlock import (
      CliMergeConflict,
      SaveBlockModelRenderer,
      ShowRunningConfigRenderOptions )
import ConfigMount
import CliSession as CS
from FileCliUtil import copyFile
import FileUrl
from FileUrlDefs import MANAGED_CONFIG_DIR
import MainCli
import SessionUrlUtil
import Tac
import Tracing
import GnmiSetCliSession
from Url import urlArgsFromMode, parseUrl
from UrlPlugin.FlashUrl import flashFileUrl

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t8 = Tracing.trace8
t9 = Tracing.trace9

fragmentStatus = None
fragmentConfig = None
sessionConfig = None
sessionStatus = None
fragmentSessionDir = None
fragmentEm = None
sessionCliConfig = None
# The cached SaveBlock keyed by ( fragName, pending )
# The 'pending' flag indicate if this is pending or committed version
cachedSaveBlock = {}
sysdbStatus = None

fragmentSessionPrefix = "_fragmentSession_"

# Get cached SaveBlock of the fragment. If pending=True, we use pending version
# if available
@Ark.synchronized( CS.sessionLock )
def getCachedSaveBlock( fragmentName, em, pending=False ):
   t9( f'getCachedSaveBlock for ({fragmentName},{pending})' )
   status = fragmentStatus.cliFragmentDir.status.get( fragmentName )
   assert status, f"Fragment {fragmentName} does not exist!"
   t9( f"Fragment {fragmentName} pending: {status.config.pendingVersion} "
       f"committed: {status.committedVersion}" )
   # If pending is set, use pending version if available
   pending = pending and status.config.pendingVersion > 0
   key = ( fragmentName, pending )
   if key not in cachedSaveBlock:
      sessionName = status.pendingSessionName if pending \
                        else status.committedSessionName
      cachedSaveBlock[ key ] = getSessionSaveBlock( em, sessionName )
   return cachedSaveBlock[ key ]

@Ark.synchronized( CS.sessionLock )
def removeCachedSaveBlock( fragmentNames ):
   # This is called from prepare/abort/delete handlers
   t9( f'removeCachedSaveBlock for {fragmentNames}' )
   fragmentNames = fragmentNames if isinstance( fragmentNames, list ) \
         else [ fragmentNames ]
   for fragmentName, pending in itertools.product(
         fragmentNames, [ True, False ] ):
      key = ( fragmentName, pending )
      if key in cachedSaveBlock:
         t9( f"Invalidate SaveBlock cache for {key}" )
         cachedSaveBlock.pop( key )
      else:
         t9( f"{key} cache does not exist" )

def waitForFragment( fragmentName, present=True ):
   sleep = not Tac.activityManager.inExecTime.isZero
   expected = "created" if present else "deleted"
   Tac.waitFor( lambda: ( ( fragmentName in fragmentStatus.cliFragmentDir.status )
                          == present ),
                sleep=sleep,
                description=f"CliFragment {fragmentName} to be {expected}" )

# Internal operation of commit and delete fragments
def requestCommitFragments( request, fragNames ):
   pendingSessions = {}
   for fragmentName in fragNames:
      frag = fragmentConfig.cliFragmentDir.config.get( fragmentName )
      msg = f"No pending session while committing fragment '{fragmentName}'"
      assert frag and frag.pendingVersion > 0, msg
      pendingSessions[ fragmentName ] = frag.pendingVersion
      request.fragment[ fragmentName ] = frag.pendingVersion
   return pendingSessions

def requestDeleteFragments( request, deletedSessions ):
   committedSessions = {}
   for sessionName in deletedSessions:
      version = fragmentConfig.extractVersion( sessionName )
      committedSessions[ sessionName ] = version
      request.fragment[ sessionName ] = version
   return committedSessions

# This is a temporary implementation, until we decide how saveSessionDiffs API should
# specify/handle clean-config as source or dest.
__cleanConfigContents__ = None

def fillCleanConfigContents( mode ):
   global __cleanConfigContents__
   __cleanConfigContents__ = "@cleanConfigForFragments"
   sname = __cleanConfigContents__
   # If we're in the middle of a session, we'll want to re-enter. Save name
   existingSession = CS.currentSession( fragmentEm )
   try:
      CS.enterSession( sname, entityManager=fragmentEm, transient=False )
      CS.rollbackSession( fragmentEm )
   finally:
      if sname == CS.currentSession( fragmentEm ):
         CS.exitSession()
      # Re-enter session, if relevant
      if existingSession:
         CS.enterSession( existingSession, entityManager=fragmentEm )
# end of temporary clean-config code

# Create a config that will modify a config that contains 'currentSession'
# to an otherwise equivalent one, with the contents of `currentSession`
# replaced by the contents of `desiredSession`.
# This handles deletions from performCommit and case five from
# copyFragmentsIntoSession()
# We have a fragment in fragNames, with a committed version.
# Deletion case is when delete == True, and the fragment is in fragNames:
#     we use saveSessionDiffs( committedVersion, clean-config ) to change
#     the temp session (`sessionName` in `context`)
# Fifth case (committing a pending with an existing committed)
#     use saveSessionDiffs( committedVersion, pendingVersion ) to change
#     temp session
def configureDesiredFromCurrent( currentSession, desiredSession, context ):
   mode, toUrl, sessionName = context
   t8( "configureDesiredFromCurrent: "
       f"config {currentSession} to {desiredSession} through {sessionName}" )
   result = ""
   if desiredSession is None:
      if __cleanConfigContents__ is None:
         fillCleanConfigContents( mode )
      desiredSession = __cleanConfigContents__
   try:
      fname = ""
      try:
         with NamedTemporaryFile( mode='w+', delete=False ) as tf:
            fname = tf.name
            saveSessionDiffs( fragmentEm, tf,
                              currentSession, 'session-config',
                              'session-config', diffType='cli',
                              mySessionName=desiredSession )
            tf.seek( 0 )
            t9( f"diff: {tf.read()}" )
         toUrl.put( fname )
         return result
      finally:
         if fname:
            os.unlink( fname )
   # Catch *all* exceptions (W0703=broad-except)
   except Exception as e: # pylint: disable-msg=W0703
      exceptStr = str( e )
      if result:
         result = f"Exception {exceptStr}, after: {result}"
         return result
      else:
         result = f"Exception {exceptStr}"
         return result
   finally:
      if CS.currentSession( fragmentEm ) != sessionName:
         response = CS.enterSession(
            sessionName, entityManager=fragmentEm, transient=True )
         if response:
            msg = f"While returning to temp session for commit: {response}"
            # disable pylint complaint about return, because we incorporate
            # msg from exception, so it isn't getting ignored..
            if result:
               msg = f"Failed with {msg}, after: {result}"
               return msg # pylint: disable-msg=W0150
            return msg # pylint: disable-msg=W0150

# This copies committed sessions from fragments (and pending sessions from
# fragNames) into session sessionName (which should be the currently open
# session).
# In the case of deletions, fragNames should be [] (and no deleted fragments
# should be in cliFragmentNames())
# This is used by performCommit() and will be used by the
# "show config fragments combined" Cli command.
def copyFragmentContentsIntoSession( fragNames, sessionName=None, mode=None ):
   ( em, disableAaa, cliCmdSession ) = ( urlArgsFromMode( mode ) if mode
                                         else ( fragmentEm, True, None ) )
   for fragmentName in cliFragmentNames( em ):
      frag = fragmentConfig.cliFragmentDir.config.get( fragmentName )
      fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
      hasCommitted = fragStat and fragStat.committedVersion > 0
      # There are 5 cases to handle.
      # First, a fragment not in fragNames, and no committed version:
      # Second, a fragment being deleted, that had no committed version.
      # Do nothing, it is not yet part of running-config.
      if fragmentName not in fragNames and not hasCommitted:
         continue
      # At this point we know the fragment is going to be part of
      # running-config, and it can never be part of a deletion.
      # There are still some cases to cover.
      # Third, a fragment not in fragNames, but with a committed version:
      #   Should already be in running-config, but to handle the case, that
      #   a user manually removed/added some config in running-config, we must
      #   do a copyFile from fragment's session to new temp session to
      #   override those (temporary) changes.
      # Fourth, a fragment *in* fragNames, but with *no* committed version to
      # undo:
      #   do a copyFile from fragment's pending session to new temp session
      # Fifth case (committing a pending with an existing committed):
      #   put the diff between the previously committedVersion and the
      #   current pendingVersion into the temp session (sessionName)
      #   But, first, we also need to restore the previous committed version
      #   into the workspace (as above) in case the user (temporarily)
      #   overrode the config from a fragment.

      # In cases 3 and 5 we start with committedSession, in 4 we start with
      # pending:
      fromSession = None
      if fragmentName not in fragNames:
         # In the case of delete, fragmentName is always 'not in fragNames',
         # and in any case, hasCommitted must be True if we reached here and
         # fragment not in fragNames:
         fromSession = fragStat.committedSessionName
      else:
         fromSession = frag.pendingSessionName
      fromUrl = SessionUrlUtil.sessionConfig(
         em, disableAaa, cliCmdSession, sessionName=fromSession )
      toUrl = SessionUrlUtil.sessionConfig(
         em, disableAaa, cliCmdSession, sessionName=sessionName )
      result = None
      # In all remaining cases we start with a copy into the workspace, in
      # case something was removed from
      result = copyFile( None, mode, fromUrl, toUrl,
                         commandSource="configure replace",
                         ignoreENOENT=True )
      if result:
         return f"While copying {fragmentName} into session: {result}"
      # In case 5 we *also* now have to apply the diff between committed and
      # pending session:
      if fragmentName in fragNames and hasCommitted:
         # in the case of delete, no deleted fragments would be in [],
         # so delete = False
         ctx = ( mode, toUrl, sessionName )
         result = configureDesiredFromCurrent(
            fragStat.committedSessionName, frag.pendingSessionName, ctx )
         if result:
            return f"While reflecting {fragmentName} into session: {result}"
   return ""

# Get the fragNames to be committed. See comments on performCommit() for
# 'fragNames' and 'delete' arguments.
def getToBeCommittedFragNames( fragNames, delete ):
   frags = set()
   for fragmentName in fragmentStatus.cliFragmentDir.status:
      fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
      if fragStat.committedVersion > 0:
         frags.add( fragmentName )
   if delete:
      frags = frags - fragNames
   else:
      frags |= fragNames
   return list( frags )

def commitReplace( em, unionSaveBlock ):
   runningConfigSaveBlock = getRunningConfigSaveBlock( em )
   with NamedTemporaryFile( mode='w+' ) as stream:
      # 1. generate the diff between unionSaveBlock and running-config
      SaveBlockModelRenderer.renderDiffCliCommands( stream,
            ShowRunningConfigRenderOptions(),
            '', '', runningConfigSaveBlock, unionSaveBlock,
            # workaround for BUG972060
            forceSortBlock=True )
      stream.flush()
      stream.seek( 0 )
      t9( f"CommitReplace: applying\n{stream.read()}{stream.seek(0)}" )
      # 2. apply the merged config to the current session
      with ConfigMount.ConfigMountDisabler( disable=False ):
         with ArPyUtils.StdoutAndStderrInterceptor() as out:
            # All commands should have been authorized and accepted by
            # now, so we disable AAA and guards.
            MainCli.loadConfigFromFile( stream,
                                        em,
                                        privLevel=CliCommon.MAX_PRIV_LVL,
                                        disableAaa=True,
                                        disableGuards=True,
                                        skipConfigCheck=True )
      lines = out.contents().splitlines()
      if any( line.startswith( '%' ) for line in lines ):
         t9( f"Error: {out.contents()}" )
         return f"While applying fragments to the session:\n{out.contents()}"
   return ""

# If delete is False, then fragNames is a list of fragments. If delete == True
# then fragNames is actually a list of the committedSessions we are deleting.
# (in the delete case, the fragments should already be gone).
# If 'replace' is true, replace the running-config.
# Otherwise, merge with the running-config."""
def performCommit( fragOrSessionNames, mode=None, replace=True ):
   t8( f"performCommit {fragOrSessionNames}" )
   gid = None
   # Use an internal session name (starting with "@") so we don't cause any
   # other sessions to get deleted because of limits.
   # Use a well-known prefix so we can detect orphaned sessions at startup.
   # (We use an internal session here so that we don't bump up against
   # CliSession limits --- the issue is that this session goes idle (and
   # therefore deletable by CliSessionMgrAgent) when we construct diffs during
   # a session commit by going into another temporary session.
   sessionName = CS.uniqueSessionName( prefix=f"@{fragmentSessionPrefix}-" )
   deletedSessions = set()
   fragNames = set( fragOrSessionNames )
   # Now add in any attempted deletions that didn't complete before,
   # either because of ConfigAgent crashing, or errors in the command
   # or operation:
   for sname, session in fragmentConfig.sessionToProcess.items():
      if ( session.fragmentName and sname not in deletedSessions ):
         processed = fragmentStatus.processedSession.get( sname )
         if not ( processed and session == processed.config ):
            deletedSessions.add( sname )
   try:
      # prepareCommit allocates a request from fragmentConfig.requestPool,
      # with a new groupCommitId. It creates the session for the commit,
      # but doesn't enter it).
      request = fragmentConfig.prepareCommit( sessionName )
      # request2 will be assigned, if there are any leftover deletions
      request2 = None
      gid = request.id
      request.deletion = False
      gidsToProcess = [ gid ]
      sessions = {}
      sessions.update( requestCommitFragments( request, fragNames ) )
      if deletedSessions:
         request2 = fragmentConfig.prepareCommit( sessionName )
         request2.deletion = True
         gidsToProcess.append( request2.id )
         sessions = requestDeleteFragments( request2, deletedSessions )
      sessions.update( requestCommitFragments( request, fragNames ) )
      t9( f"Preparing to commit fragments {fragNames}, gid: {gid}" )
      response = CS.enterSession(
         sessionName, entityManager=fragmentEm, transient=True )
      if response:
         return f"While entering session for fragment commit: {response}"
      if not ( CS.currentSession( fragmentEm ) and
               sessionName == CS.currentSession( fragmentEm ) ):
         return "Could not create Cli session for commit"

      # We're going to commit these pending sessions, so we want to close
      # them. That way no one will/can modify them post-commit.
      for fragmentName in fragNames:
         fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
         if fragStat.currentPending:
            if CS.currentSession( fragmentEm ):
               CS.exitSession( fragmentEm )
            psession = fragStat.pendingSessionName
            CS.enterSession( psession, entityManager=fragmentEm )
            n = CS.closeSession( fragmentEm, psession, mode=mode )
            if n:
               msg = f"failed to close pending {psession}"
               return f"{msg} for {fragmentName}: {n}"
      if sessionName != CS.currentSession( fragmentEm ):
         # Need to re-enter the session we are using for commit.
         response = CS.enterSession(
            sessionName, entityManager=fragmentEm, transient=True )
         if response:
            return f"While re-entering session for fragment commit: {response}"
      pcsDir = CS.sessionStatus.pendingChangeStatusDir
      result = fragmentConfig.commitPendingFragments( request, pcsDir )
      if result:
         return result
      ( em, disableAaa, cliCmdSession ) = ( urlArgsFromMode( mode ) if mode
                                            else ( fragmentEm, True, None ) )

      t9( "Check for conflicts in fragment commit" )
      # Check for conflicts between the fragments --- and
      # only between the fragments, not with the current running-config.
      # The unionSaveBlock is the union of all to-be-committed fragments, which are:
      # - fragments in fragNames: use pending version if available
      # - fragments *not* in fragNames but already committed: use committed version
      cleanSaveBlock = getCleanConfigSaveBlock( fragmentEm )
      frags = getToBeCommittedFragNames( fragNames, False )
      for i, fragmentName in enumerate( frags ):
         t9( f"Check {fragmentName}" )
         if i == 0:
            # First fragment always has no conflict
            unionSaveBlock = getCachedSaveBlock( fragmentName, fragmentEm,
                                 pending=fragmentName in fragNames )
            continue
         try:
            fragmentSaveBlock = getCachedSaveBlock( fragmentName, fragmentEm,
                                 pending=fragmentName in fragNames )
            unionSaveBlock = fragmentSaveBlock.getMergedModel(
                                 cleanSaveBlock, unionSaveBlock )
         except CliMergeConflict as e:
            stream = io.StringIO()
            stream.write( f"Conflicts detected with fragment {fragmentName}\n" )
            stream.write( f"Error: {e}\n" )
            e.renderConflict( stream, 'cleanConfig', 'mergedConfig',
                              'fragmentConfig' )
            return stream.getvalue()

      msg = "Fragment 'commit merge' is no longer supported"
      assert replace, msg
      # fragments will always override any conflict with running-config.
      t9( f"Load pending/committed fragments into commit session {sessionName}" )
      if replace:
         # 'commit replace': Replace running-config with union config of fragments
         if sessionName != CS.currentSession( fragmentEm ):
            # Need to re-enter the session we are using for commit.
            result = CS.enterSession(
               sessionName, entityManager=fragmentEm, transient=True )
            if result:
               return f"While re-entering session for loading fragments: {result}"

         result = commitReplace( fragmentEm, unionSaveBlock )
         if result:
            return result
      else:
         # 'commit merge'
         # For listed fragments on delete, undo any configuration in running-config
         # that came from that fragment.
         # For listed fragments on commit, just copy from pending session into
         # working Cli session (sessionName).
         # For unlisted fragments, copy in the previous committed session. We need
         # to copy the already-committed session because
         # we don't know whether the user has (temporarily) overwritten or undone
         # any configuration that came from that fragment.
         toUrl = SessionUrlUtil.sessionConfig(
            em, disableAaa, cliCmdSession, sessionName=sessionName )
         t9( "Undo deleted committed sessions" )
         # Undo deleted sessions, if any, from running-config, first:
         for deletedSession in deletedSessions:
            t9( f"Handle diff of {deletedSession} to []" )
            ctx = ( mode, toUrl, sessionName )
            result = configureDesiredFromCurrent( deletedSession, None, ctx )
            if result:
               return f"While deleting {deletedSession} into session: {result}"
         # Now deal with configuration that actually belongs in running-config:
         copyFragmentContentsIntoSession(
            fragNames, sessionName=sessionName, mode=mode )

      # At some point, this might be done in FragmentMgr by requesting CliSessionMgr
      # to commit, directly::
      t9( "Copy to running-config" )
      result = CS.commitSession( fragmentEm, sessionName=sessionName, mode=mode )
      if result:
         return f"While commiting {fragNames}: {result}"

      # send request to fragmentMgr (key is gid). Reactor will delete session,
      # so must already be committed to running-config.
      if request2:
         fragmentConfig.request.addMember( request2 )
      fragmentConfig.request.addMember( request )
      Tac.waitFor( lambda: fragmentSessionDir.response.get( gid ),
                   description="response from CliFragmentMgr" )
      response = fragmentSessionDir.response.get( gid )
      if response:
         del fragmentConfig.request[ gid ]
      if not response.success:
         return f"CliFragmentMgr failed to commit, reason: {response.reason}"
      CS.exitSession( fragmentEm )
      if request2 and fragmentConfig.request.get( request2.id ):
         Tac.waitFor( lambda: fragmentSessionDir.response.get( request2.id ),
                      description="response from CliFragmentMgr" )
         response2 = fragmentSessionDir.response.get( request2.id )
         if response2:
            del fragmentConfig.request[ request2.id ]
            if not response2.success:
               # No need to fail, here (it is the background request that failed),
               # but let's trace a warning.
               t1( f"Pending fragment deletion failed because: {response2.reason}" )
      t9( "Clear pending, now that they are committed" )
      for fragmentName in fragNames:
         pendingVersion = sessions.get( fragmentName )
         fragConfig = fragmentConfig.cliFragmentDir.config.get( fragmentName )
         fragConfig.clearPendingSession( pendingVersion )
      if deletedSessions:
         for sessionName in fragmentConfig.sessionToProcess.keys():
            if sessionName in fragmentStatus.processedSession:
               del fragmentConfig.sessionToProcess[ sessionName ]
      t9( "Maybe copy running-config to startup-config" )
      if sessionCliConfig.saveToStartupConfigOnCommit:
         result = copyFile( None, mode,
                            FileUrl.localRunningConfig( fragmentEm, True ),
                            FileUrl.localStartupConfig( fragmentEm, True ),
                            commandSource="configure replace",
                            # Ignorable, in the case of a test
                            ignoreENOENT=( mode is None ) )
         # That should have implicitly caused the fragment files to be written
         # to managed-config directory.
   finally:
      if gid and gid in fragmentConfig.request:
         del fragmentConfig.request[ gid ]
      if gid and gid in fragmentConfig.requestPool:
         del fragmentConfig.requestPool[ gid ]
      if gid:
         for fragmentName in fragNames:
            frag = fragmentConfig.cliFragmentDir.config.get( fragmentName )
            if frag and frag.commitGroup == gid:
               frag.commitGroup = None
      if sessionName:
         if CS.currentSession( fragmentEm ) == sessionName:
            CS.exitSession( fragmentEm )
         if sessionName in fragmentStatus.processedSession:
            del fragmentConfig.sessionToProcess[ sessionName ]
      if sessionName: # internal session, so needs to be managed explicitly
         if sessionConfig.pendingChangeConfigDir.config.get( sessionName ):
            del sessionConfig.pendingChangeConfigDir.config[ sessionName ]
      GnmiSetCliSession.resetGnmiSetState()
   # (result may be None)
   return result or ""

# Basic Fragment operations:
def addFragment( fragmentName, waitForStatus=True ):
   t8( "addFragment" )
   waitForFragmentStatusInitialized( fragmentEm )
   if fragmentName not in fragmentConfig.cliFragmentDir.config:
      fragmentConfig.cliFragmentDir.config.newMember( fragmentName )
   if waitForStatus:
      waitForFragment( fragmentName, True )

def deleteFragments( fragmentNames, mode=None, waitForStatus=True ):
   result = ""
   t8( f"deleteFragments {fragmentNames}" )
   waitForFragmentStatusInitialized( fragmentEm )
   committedSessions = []
   for fragmentName in fragmentNames:
      pendingSession = ""
      committedSession = ""
      fragCfg = fragmentConfig.cliFragmentDir.config.get( fragmentName )
      fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
      if fragCfg and fragCfg.pendingVersion > 0:
         pendingSession = fragCfg.pendingSessionName
      if fragStat and fragStat.committedVersion > 0:
         committedSession = fragStat.committedSessionName
      if fragCfg:
         fragCfg.commitGroup = None
      if committedSession:
         committedSessions.append( committedSession )
      if pendingSession:
         fragCfg.clearPendingSession( fragCfg.pendingVersion )
         del sessionConfig.pendingChangeConfigDir.config[ pendingSession ]
   try:
      for fragmentName in fragmentNames:
         del fragmentConfig.cliFragmentDir.config[ fragmentName ]
      if waitForStatus:
         for fragmentName in fragmentNames:
            waitForFragment( fragmentName, False )
         # (waitForStatus=False is for testing purposes, only)
      removeCachedSaveBlock( fragmentNames )
      return result
   finally:
      for committedSession in committedSessions:
         if committedSession in fragmentStatus.processedSession:
            del fragmentConfig.sessionToProcess[ committedSession ]

def cliFragmentNames( entityManager=None ):
   waitForFragmentStatusInitialized( entityManager or fragmentEm )
   # Get names of fragments. We use both config and status, even though they
   # are usually an identical set of names, to capture those that have been
   # requested for deletion, but haven't been deleted yet, and those that are
   # requested for creation, but haven't been fully created yet.
   return set( list( fragmentConfig.cliFragmentDir.config ) +
               list( fragmentStatus.cliFragmentDir.status ) )

def waitForPendingSession( fragmentName, present=True ):
   waitForFragment( fragmentName, present=True )
   fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: bool( fragStat.currentPending ) == bool( present ),
                sleep=sleep, warnAfter=5,
                description="fragment status to ack pendingSession" )

# Temporary skeleton, just for basic testing.
# Copy == True means copy previous session; copy == false means use
# clean-config
def prepareFragmentSession( fragmentName, copy, mode=None ):
   t8( f"prepareFragmentSession fragment {fragmentName}" )
   # ensure this fragment exists
   if fragmentName == fragmentSessionPrefix:
      errorMsg = f"{fragmentName} is a reserved name"
      return f"{errorMsg}: Cannot be used to name a fragment"
   addFragment( fragmentName, waitForStatus=True )
   frag = fragmentConfig.cliFragmentDir.config.get( fragmentName )
   assert frag, f"Fragment {fragmentName} not created by addFragment()!"
   # fragStat must exist, else addFragment( wait=True )  would have failed
   fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
   committedName = ""
   # copy=True only makes sense if there *is* a committed version.
   if copy and fragStat.committedVersion == 0:
      copy = False
   if copy:
      committedName = fragStat.committedSessionName
   # clear out old pending session first. No way of knowing whether it is
   # empty, or has the correct contents, or if already completed.
   frag.pendingVersion = 0
   frag.pendingSession = None
   # allocate new one
   fragmentConfig.openPendingFragment( fragmentName )
   waitForPendingSession( fragmentName, present=True )
   sessionName = fragStat.pendingSessionName
   try:
      CS.enterSession( sessionName, CS.sessionEm )
      if copy:
         response = CS.copyFromSession( CS.sessionEm, committedName )
      else:
         response = CS.rollbackSession( CS.sessionEm, sessionName=sessionName )
      if response:
         return response
   finally:
      CS.exitSession( CS.sessionEm )
   # Remove the cache only if the fragment is successfully prepared
   removeCachedSaveBlock( fragmentName )
   return ""

# Load the config contained in configUrl into the pending session of fragment.
# The URL is typically the terminal, or a file (during btests it is most likely
# a completed session).
def loadFragmentSession( mode, fragmentName, configUrl ):
   t8( f"loadFragmentSession fragment {fragmentName} {str(configUrl)}" )
   if not configUrl.exists():
      return ( f"Contents of URL {str(configUrl)} doesn't exist, "
               f"for fragment {fragmentName}" )
   # ensure pending fragment exists and is open
   fragStat = fragmentStatus.cliFragmentDir.status.get( fragmentName )
   if not fragStat:
      return f"Fragment {fragmentName} does not exist"
   if not fragStat.currentPending:
      return f"Fragment {fragmentName} has no prepared pending session"
   fSessionName = fragStat.pendingSessionName
   t9( f" Session name {fSessionName} for fragment {fragmentName}" )
   sstat = CS.sessionStatus.pendingChangeStatusDir.status.get( fSessionName )
   if not sstat:
      return f"Fragment {fragmentName} has no open session"
   if sstat.completed:
      return f"Fragment {fragmentName} pending session already completed"
   try:
      # copy from session
      CS.enterSession( fSessionName, mode.entityManager )
      fragUrl = SessionUrlUtil.sessionConfig( mode.entityManager, True,
                                              cliSession=mode.session,
                                              sessionName=fSessionName )
      fragUrl.abortOnError = True
      errorStr = copyFile( None, mode, configUrl, fragUrl,
                           commandSource="configure replace", ignoreENOENT=True )
      if errorStr:
         return errorStr
   finally:
      CS.exitSession( CS.sessionEm )
   return ""

# This is an implementation, only used for testing, in tests where no Cli is
# running.
# For the moment we'll use the session lock to avoid implementing this in
# the CliFragmentReactor.
# Could also implement this as a method in CliFragmentSysdb.tin and then would
# be atomic (would hold the activity lock and wouldn't yield to attrlog)??
# But that's pointless, because we are going to move this to CliFragmentMgr
# shortly.
#
# Must pass in pendingVersion to know which version was actually
# committed (a new pending session may have been opened up in another
# cli thread, after trying to commit the original).
# Return False if we don't change committedVersion, and True if we do.
@Ark.synchronized( CS.sessionLock )
def commitPendingSession( fragConfig, fragStat, pendingVersion ):
   t9( f"commitPendingSession {fragConfig.name} "
       f"version: {pendingVersion},{fragStat.committedVersion}" )
   if fragStat.committedVersion and fragStat.committedVersion >= pendingVersion:
      return False
   oldCommittedSession = ( fragStat.committedVersion and
                           fragStat.committedSessionName )
   pSessionName = fragConfig.encodeSessionName( fragConfig.name, pendingVersion )
   pcs = sessionStatus.pendingChangeStatusDir.status.get( pSessionName )
   if not pcs:
      return False
   if not pcs.completed:
      # Close session we're committing, to make sure no future modifications
      # are allowed.
      errorStr = CS.closeSession( CS.sessionEm, sessionName=pSessionName )
      if errorStr:
         return False
      try:
         CS.runPreCommitHandlers( CS.sessionEm, sessionName=pSessionName )
      except CS.PreCommitHandlerError as e:
         return e.recordAndResponse()
   # Must set committedVersion before pendingVersion, so reactor doesn't delete
   # the session before we register it in committedVersion.
   t9( f" updating committedVersion to {pendingVersion}" )
   fragStat.committedVersion = pendingVersion
   if fragConfig.pendingVersion == pendingVersion:
      fragConfig.pendingVersion = 0
      # Even if pending session doesn't match, it would have to be obsolete,
      # otherwise pendingVersion would match, too. So clear it regardless:
      fragConfig.pendingSession = None
   # Get rid of session that is no longer used (will eventually be done in
   # CliFragmentMgr reactor).
   if ( oldCommittedSession and
        oldCommittedSession in sessionConfig.pendingChangeConfigDir.config ):
      del sessionConfig.pendingChangeConfigDir.config[ oldCommittedSession ]
   return True

def commitFragmentInternal( frag ):
   fragConfig = fragmentConfig.cliFragmentDir.config.get( frag )
   fragStatus = fragmentStatus.cliFragmentDir.status.get( frag )
   t8( f"commitFragmentInternal( {frag} )" )
   if fragConfig and fragStatus:
      pendingVersion = fragConfig.pendingVersion
      ret = commitPendingSession( fragConfig, fragStatus, pendingVersion )
      return "" if ret else f"commitPendingSession failed for {frag}"
   else:
      return f"fragConfig and fragStatus must both exist for fragment '{frag}'"

def abortFragments( fragNames, mode=None ):
   """Removes fragments from set of to-be-commited fragments, and aborts
   any open sessions"""
   t8( f"abortFragments {fragNames}" )
   if isinstance( fragNames, str ):
      fragNames = [ fragNames ]
   for fragmentName in fragNames:
      frag = fragmentConfig.cliFragmentDir.config.get( fragmentName )
      if not frag:
         continue
      session = frag.pendingSession
      if not session:
         continue
      sessionStat = sessionStatus.pendingChangeStatusDir.status.get( session.name )
      if ( not sessionStat or
           ( sessionStat.completed and not sessionStat.success ) ):
         continue
      try:
         if not sessionStat.completed:
            resp = CS.abortSession( fragmentEm, sessionStat.name )
            if resp != "":
               return resp
      finally:
         frag.pendingVersion = 0
         frag.pendingSession = None
         resp = CS.deleteSession( fragmentEm, sessionStat.name )
         if resp:
            return resp # pylint: disable-msg=W0150
         removeCachedSaveBlock( fragmentName )
   return ""

def commitFragments( fragNames, mode=None, replace=True ):
   """Takes a set of fragments with pending sessions and commits those,
   moving the pending sessions to committed sessions, and replacing the
   running-config with the union of all the new fragments.
   Returns empty string if commit was a success. Returns error/explanation
   if commits did not succeed.
   mode will be None only when this is not called from the Cli --- which should
   only happen in tests. Without mode, the current version of
   copy-to-startup-config cannot work.
   Replace the running-config."""
   t8( f"commitFragments {fragNames}" )
   # 'replace' should always be true, but protect against some out-of-date
   # internal callers that may use replace=False
   assert replace, "commit merge is no longer supported."
   if isinstance( fragNames, str ):
      fragNames = [ fragNames ]
   waitForFragmentStatusInitialized( fragmentEm )
   return performCommit( fragNames, mode=mode )

def loadFragmentFiles( cliMode ):
   """
   This method loads any fragment config files stored under managed-configs
   into Sysdb. It does not affect running-config, but it does populate
   cliFragmentConfig and cliFragmentStatus and fills in the associated
   sessions.
   It mimics the behavior of LoadFragmentFiles, but can be called from a btest.
   It can only be called in a context in which you can enter the activity
   loop, so it is called via PyClient using the separate thread execMode, or
   directly from the top-level of a test.
   """
   t8( "loadFragmentFiles()" )
   fragments = set( [] )
   loadedFragments = set( [] )

   fragmentConfigDir = flashFileUrl( MANAGED_CONFIG_DIR )
   fragmentConfigDirUrl = parseUrl( fragmentConfigDir, None )
   if not fragmentConfigDirUrl.exists():
      msg = f"Nonexistent fragment config dir: {fragmentConfigDirUrl}"
      t1( msg )
      return msg
   fragmentStatus.fragmentConfigStatus = "inProgress"
   for filename in fragmentConfigDirUrl.listdir():
      if filename.endswith( ".fragment" ):
         frag = filename[ 0 : ( - len( ".fragment" ) ) ]
         fragments.add( frag )
   t1( f"Found {len( fragments )} existing fragments." )
   try:
      for frag in fragments:
         filename = fragmentConfigDir + "/" + frag + ".fragment"
         filenameUrl = parseUrl( filename, None )
         ret = prepareFragmentSession( frag, False )
         if ret:
            t9( f"prepare {frag}: {ret}" )
            return ret
         ret = loadFragmentSession( cliMode, frag, filenameUrl )
         if ret:
            t9( f"load {frag}: {ret}" )
            return ret
         ret = commitFragmentInternal( frag )
         if ret:
            t9( f"commit {frag}: {ret}" )
            fragmentStatus.fragmentConfigErrorCount += 1
            return ret
         else:
            t2( f"Loaded fragment {frag} " )
            loadedFragments.add( frag )
   except Exception as e: # pylint: disable=broad-except
      t1( f"Exception raised in loadFragmentFiles(): {str(e)}" )
      fragmentStatus.fragmentConfigErrorCount += 1
      return str( e )
   finally:
      # Write to cli/fragment/status that we're done loading the fragment
      # configs
      fragmentStatus.fragmentConfigStatus = ( "completed"
                                              if fragments == loadedFragments
                                              else "incomplete" )
   return ""

def fragmentStatusInitialized( em ):
   # this test is needed for tests using Simple EntityManager
   if not ( fragmentConfig and fragmentStatus and
            sessionConfig and sessionStatus ):
      return False
   if ( not ( isinstance( fragmentStatus,
                          Tac.Type( "Cli::Session::FragmentStatus" ) ) and
              fragmentStatus.initialized ) ):
      return False
   # This isinstance is probably unnecessary unless someone uses this
   # in a multi-process test
   if not isinstance( fragmentConfig,
                      Tac.Type( "Cli::Session::FragmentConfig" ) ):
      return False
   return bool( fragmentConfig.pendingChangeConfigDir )

def waitForFragmentStatusInitialized( em ):
   CS.waitForSessionStatusInitialized( em )
   sleep = not Tac.activityManager.inExecTime.isZero
   Tac.waitFor( lambda: fragmentStatusInitialized( em or fragmentEm ),
                sleep=sleep, warnAfter=0,
                description="cli Fragment status to be initialized" )

def doMountsInternal( em, mg ):
   global fragmentConfig, fragmentStatus, fragmentSessionDir
   global sessionConfig, sessionStatus, fragmentEm, sessionCliConfig
   global sysdbStatus
   t8( "CliFragment: starting doMountsInternal" )
   # clear previous data
   fragmentConfig = mg.mount( "cli/fragment/config",
                              "Cli::Session::FragmentConfig", "wi" )
   fragmentStatus = mg.mount( "cli/fragment/status",
                              "Cli::Session::FragmentStatus", "wi" )
   fragmentSessionDir = mg.mount( "cli/fragmentDir",
                                  "Cli::Session::FragmentSessionDir", "w" )
   # Mount sessionConfig writably, because we will need to add and
   # delete pending session configs.
   sessionConfig = mg.mount( "cli/session/config",
                             "Cli::Session::Config", "wi" )
   sessionStatus = mg.mount( "cli/session/status",
                             "Cli::Session::Status", "ri" )
   sessionCliConfig = mg.mount( "cli/session/input/config",
                                "Cli::Session::CliConfig", "r" )
   sysdbStatus = mg.mount( "sysdb/status", "Sysdb::Status", "r" )
   fragmentEm = em

def doMountsComplete( callback=None ):
   t0( "initialize fragmentConfig" )
   fragmentConfig.pendingChangeConfigDir = sessionConfig.pendingChangeConfigDir
   fragmentConfig.cliFragmentDir = ( "cliFragmentDir", )
   fragmentStatus.cliFragmentDir = ( "cliFragmentDir", )
   fragmentStatus.initialized = True

   if callback:
      t9( "CliFragment.doMountsComplete calling callback" )
      callback()

def doMounts( em, block=True, callback=None ):
   t1( "doMounts(", em.sysname(), ", block", block, ")" )
   if ( fragmentEm == em and
        CS.sessionStatusInitialized( em ) and
        fragmentStatusInitialized( em ) ):
      t2( "sysname", em.sysname(), "already mounted for fragmentStatus" )
      if callback:
         callback()
      return

   mg = em.mountGroup()
   doMountsInternal( em, mg )
   mg.close( callback=lambda:doMountsComplete( callback ),
             blocking=False )
   if block:
      waitForFragmentStatusInitialized( em )

def cleanupFragmentMgrState():
   global fragmentStatus, fragmentConfig, fragmentSessionDir
   global sessionConfig, sessionStatus, fragmentEm, cachedSaveBlock

   fragmentStatus = None
   fragmentConfig = None
   fragmentSessionDir = None
   sessionConfig = None
   sessionStatus = None
   fragmentEm = None
   cachedSaveBlock = {}
