#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import CliFragment as CF
from CliSession import sessionLock
import Tac
import Tracing

t0 = Tracing.trace0

# FragmentMgr is enabled by default in the system, for now. If during
# development, this causes the Cli or ConfigAgent to fail, set this to
# False. If you then need/want to use this in a test, call
# enableFragmentMgr() before launching the CliSessionMgrAgent or
# entire ConfigAgent (in the same process).
# When this feature is complete, we will remove enable/disableFragmentMgr()
_fragmentMgrEnabled = True

@Ark.synchronized( sessionLock )
def enableFragmentMgr():
   global _fragmentMgrEnabled
   _fragmentMgrEnabled = True

@Ark.synchronized( sessionLock )
def disableFragmentMgr():
   global _fragmentMgrEnabled
   _fragmentMgrEnabled = False

def fragmentMgrEnabled():
   return _fragmentMgrEnabled

def getAgentDir( em, create=True ):
   parent = em.root().parent
   mgr = parent.get( 'CliFragmentMgr' )
   if mgr or not create:
      return mgr
   return parent.newEntity( "Tac::Dir", "CliFragmentMgr" )

def getAgent( em, create=True ):
   agentDir = getAgentDir( em, create )
   if not agentDir:
      return None
   agent = agentDir.get( "root" )
   if agent or not create:
      return agent
   return agentDir.newEntity( "Cli::Session::Agent::FragmentSmCreator",
                              "root" )

def startFragmentMgrSm():
   t0( "FragmentMgr SM starting" )
   assert CF.fragmentEm, "FragmentMgr started before fragmentEm is set"
   agent = getAgent( CF.fragmentEm )
   agent.cliFragmentSm = ( CF.fragmentConfig,
                           CF.fragmentStatus,
                           CF.fragmentSessionDir,
                           CF.sessionConfig.pendingChangeConfigDir,
                           CF.sessionStatus.pendingChangeStatusDir,
                           CF.sessionStatus.sessionDir )

def startFragmentSessionMgrSm():
   t0( "FragmentSessionMgr SM starting" )
   assert CF.fragmentEm, "FragmentSessionMgr started before fragmentEm is set"
   agent = getAgent( CF.fragmentEm )
   smControl = Tac.Value( "Arx::SmControl" )
   agent.cliFragmentSessionSm = ( CF.fragmentConfig,
                                  CF.fragmentStatus,
                                  CF.fragmentConfig.cliFragmentDir,
                                  CF.fragmentStatus.cliFragmentDir,
                                  CF.sessionConfig.pendingChangeConfigDir,
                                  smControl )

def startFragmentMgr( em, callback=None, block=True ):
   """Requires CliSessionMgr agent to be running"""
   t0( "startFragmentMgr( sysname", em.sysname(), ")" )

   def mountsComplete():
      startFragmentMgrSm()
      if callback:
         callback()
   CF.doMounts( em, block=block, callback=mountsComplete )

def stopFragmentMgr( em ):
   agentDir = getAgentDir( em )
   agent = agentDir.get( 'root' )
   if not agent:
      return
   agent.cliFragmentSm = None
   agent = None
   agentDir.deleteEntity( "root" )
