# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import pickle

import FastServUtil

class CliInterface:
   def tabComplete( self, cmd ):
      raise NotImplementedError

   def printHelp( self, cmd ):
      raise NotImplementedError

   def prompt( self, promptPrefix ):
      raise NotImplementedError

   def getHistoryKeys( self ):
      raise NotImplementedError

   def getOrigModeHistoryKeys( self ):
      raise NotImplementedError

   def getParentHistoryKeys( self ):
      raise NotImplementedError

   def exitConfigMode( self ):
      raise NotImplementedError

   def runCmd( self, cmd, aaa=True, expandAliases=False ):
      raise NotImplementedError

   def handleCliException( self, exc_info, cmd, lineNo=None,
                           handleKeyboardInterrupt=True ):
      raise NotImplementedError

   def endSession( self ):
      raise NotImplementedError

class LocalCliInterface( CliInterface ):
   def __init__( self, session ):
      self.session_ = session

   def tabComplete( self, cmd ):
      return self.session_.tabComplete( cmd )

   def printHelp( self, cmd ):
      return self.session_.printHelp( cmd )

   def prompt( self, promptPrefix ):
      return self.session_.prompt( promptPrefix )

   def getHistoryKeys( self ):
      return self.session_.getHistoryKeys()

   def getOrigModeHistoryKeys( self ):
      return self.session_.getOrigModeHistoryKeys()

   def getParentHistoryKeys( self ):
      return self.session_.getParentHistoryKeys()

   def exitConfigMode( self ):
      return self.session_.exitConfigMode()

   def runCmd( self, cmd, aaa=True, expandAliases=False ):
      return self.session_.runCmd( cmd, aaa, expandAliases=expandAliases )

   def handleCliException( self, exc_info, cmd, lineNo=None,
                           handleKeyboardInterrupt=True ):
      return self.session_.handleCliException( exc_info, cmd, lineNo,
                                               handleKeyboardInterrupt )

   def endSession( self ):
      pass

class _RemoteRequest:
   def __init__( self, method, args, kwargs ):
      self.method_ = method
      self.args_ = args
      self.kwargs_ = kwargs

class _RemoteAttr:
   def __init__( self, method, cliShell, sessionSock ):
      self.method_ = method
      self.cliShell_ = cliShell
      self.sessionSock_ = sessionSock

   def __call__( self, *args, **kwargs ):
      try:
         self.cliShell_.disableIsig()
         request = _RemoteRequest( self.method_, args, kwargs )
         # pylint: disable=no-member
         requestData = pickle.dumps(
            request,
            protocol=FastServUtil.PICKLE_PROTO )
         FastServUtil.writeBytes( self.sessionSock_, requestData )
         responseData = FastServUtil.readBytes( self.sessionSock_ )
         if not responseData:
            return None
         response = pickle.loads( responseData )
         if ( issubclass( type( response ), Exception ) or
              isinstance( response, SystemExit ) ):
            raise response
         return response
      finally:
         self.cliShell_.enableIsig()
      return None

class RemoteCliInterface( CliInterface ): # pylint: disable-msg=W0223
   def __init__( self, cliShell, sessionSock ):
      self.cliShell_ = cliShell
      self.sessionSock_ = sessionSock

   def __getattribute__( self, name ):
      # pylint: disable-next=consider-using-in
      if name == 'sessionSock_' or name == 'cliShell_':
         return super().__getattribute__( name )
      return _RemoteAttr( name, self.cliShell_, self.sessionSock_ )
