# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliMode
import BasicCli

Constants = Tac.Type( "Dps::DpsConstants" )
constants = Constants()
OutlierThreshold = Tac.Type( "Avt::AvtPathOutlierThreshold" )
outlierThresh = OutlierThreshold()
DEFAULT_OUTLIER_THRESH_JITTER = outlierThresh.jitter
DEFAULT_OUTLIER_THRESH_LATENCY = outlierThresh.latency
DEFAULT_OUTLIER_THRESH_LOSSRATE = outlierThresh.loss
DEFAULT_OUTLIER_THRESH_LOAD = outlierThresh.load

class RouterAdaptiveVirtualTopologyModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'router adaptive-virtual-topology'

   def __init__( self ):
      self.modeKey = 'avt'
      self.longModeKey = 'adaptive-virtual-topology'
      CliMode.ConfigMode.__init__( self, None, parentMode=CliMode.ConfigMode )

class RouterAdaptiveVirtualTopologyConfigMode (
         RouterAdaptiveVirtualTopologyModeBase,
         BasicCli.ConfigModeBase ):
   ''' Configuration mode for router AVT'''
   name = 'router adaptive-virtual-topology'

   def __init__( self, parent, session ):
      RouterAdaptiveVirtualTopologyModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterAvtPolicyConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'policy %s' % self.policyName

   def __init__( self, name ):
      self.policyName = name
      self.modeKey = 'policy'
      self.longModeKey = 'policy-%s' % self.policyName
      CliMode.ConfigMode.__init__( self, self.policyName,
            parentMode=RouterAdaptiveVirtualTopologyConfigMode )

class RouterAvtPolicyConfigMode( RouterAvtPolicyConfigModeBase,
                                 BasicCli.ConfigModeBase ):
   ''' Configuration mode for setting AVT policy '''
   name = 'Policy configuration for AVT'
   ruleChanged = False

   def __init__( self, parent, session, context ):
      self.context = context
      self.policyConfig = context.policyConfig()
      RouterAvtPolicyConfigModeBase.__init__( self, context.policyName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if self.ruleChanged:
         self.policyConfig.configVersion = self.policyConfig.configVersion + 1
         self.policyConfig.defaultRuleCfgd = True

class RouterAvtPolicyAppProfileConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'match application-profile %s' % self.profileName

   def __init__( self, policyName, ruleKey, appProfile ):
      self.policy = policyName
      self.ruleKey = ruleKey
      self.policyName = policyName
      self.profileName = appProfile
      self.modeKey = 'match'
      self.longModeKey = 'match-application-profile-%s' % self.profileName
      CliMode.ConfigMode.__init__( self, self.profileName,
            parentMode=RouterAvtPolicyConfigMode )

   def commentKey( self ):
      # we need to take into account the policy name
      # perhaps we should just change longModeKey
      return f"{self.longModeKey}:{self.policyName}"

class RouterAvtPolicyAppProfileConfigMode(
         RouterAvtPolicyAppProfileConfigModeBase,
         BasicCli.ConfigModeBase,
         ):
   # pylint: disable=pointless-string-statement
   ''' Configuration mode for setting application profile name and its '''
   ''' ordering position '''
   name = 'Application profile configuration for AVT'

   def __init__( self, parent, session, context ):
      self.context = context
      RouterAvtPolicyAppProfileConfigModeBase.__init__( self, context.policyName(),
                                          context.ruleKey(), context.profileName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterAvtProfileConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'profile %s' % self.avtName

   def __init__( self, name ):
      self.avtName = name
      self.modeKey = 'profile'
      self.longModeKey = 'profile-%s' % self.avtName
      CliMode.ConfigMode.__init__( self, self.avtName,
            parentMode=RouterAdaptiveVirtualTopologyConfigMode )

class RouterAvtProfileConfigMode( RouterAvtProfileConfigModeBase,
                                  BasicCli.ConfigModeBase ):
   ''' Configuration mode for setting AVT profile '''
   name = 'Profile configuration for AVT'

   def __init__( self, parent, session, context ):
      self.context = context
      RouterAvtProfileConfigModeBase.__init__( self, context.avtName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterAvtVrfConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'vrf %s' % self.vrfName

   def __init__( self, name ):
      self.vrfName = name
      self.modeKey = 'avt-vrf'
      self.longModeKey = 'avt-vrf-%s' % self.vrfName
      CliMode.ConfigMode.__init__( self, self.vrfName,
            parentMode=RouterAdaptiveVirtualTopologyConfigMode )

class RouterAvtVrfConfigMode( RouterAvtVrfConfigModeBase, BasicCli.ConfigModeBase ):
   ''' Configuration mode for setting AVT VRF '''
   name = 'VRF policy and profile configuration for AVT'

   def __init__( self, parent, session, context ):
      self.context = context
      RouterAvtVrfConfigModeBase.__init__( self, context.vrfName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
