#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import BasicCli
import CliMode
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'Classification' )
t0 = __defaultTraceHandle__.trace0

#-------------------------------------------------------------------------------
# Base mode to be used by any CliPlugin using common Classification CLI code
#-------------------------------------------------------------------------------
class ClassificationConfigModeBase( BasicCli.ConfigModeBase ):
   def getContext( self ):
      raise NotImplementedError

   def setContext( self, value ):
      raise NotImplementedError

   def abort( self ):
      context = self.getContext()
      if context is not None:
         context.abort()
         self.setContext( None )
      self.session_.gotoParentMode()

   def commitContext( self ):
      context = self.getContext()
      if context is not None:
         context.commit()
         self.setContext( None )

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

#-------------------------------------------------------------------------------
# The "application traffic recognition" mode.
#-------------------------------------------------------------------------------
class AppTrafficRecModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'application traffic recognition'

   def __init__( self ):
      self.modeKey = 'app-rec'
      self.longModeKey = 'app-recognition'
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=CliMode.ConfigMode )

class AppTrafficRecConfigMode( AppTrafficRecModeBase, ClassificationConfigModeBase ):
   """Configuration mode for application traffic recognition"""

   name = "application traffic recognition"

   def __init__( self, parent, session, context ):
      self.appRecogContext = context
      AppTrafficRecModeBase.__init__( self )
      self.appRecogContext.modeIs( self )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.appRecogContext

   def setContext( self, value ):
      self.appRecogContext = value

#-------------------------------------------------------------------------------
# The "field-set mac <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetMacAddrModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set mac %s" % self.fieldSetName

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-mac'
      self.longModeKey = f'{self.feature}-field-set-mac-{self.fieldSetName}'
      CliMode.ConfigMode.__init__( self, param, parentMode=parentMode )

class FieldSetBaseMacAddrConfigMode( FieldSetMacAddrModeBase,
      ClassificationConfigModeBase ):
   '''Configuration mode for mac field set'''

   name = "field-set mac configuration"

   def __init__( self, parent, session, context, feature ):
      self.macAddrContext = context
      self.fieldSetMacAddrName = self.macAddrContext.fieldSetName
      self.macAddrContext.modeIs( self )
      param = ( feature, "mac", self.fieldSetMacAddrName, AppTrafficRecConfigMode )
      FieldSetMacAddrModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.macAddrContext

   def setContext( self, value ):
      self.macAddrContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "field-set integer <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetIntegerModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set integer %s" % self.fieldSetName

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-integer'
      self.longModeKey = '{}-field-set-integer-{}'.format( self.feature,
                                                           self.fieldSetName )
      CliMode.ConfigMode.__init__( self, param, parentMode=parentMode )

class FieldSetIntegerConfigMode( FieldSetIntegerModeBase,
                                 ClassificationConfigModeBase ):
   '''Configuration mode for integer field set'''

   name = "field-set integer configuration"

   def __init__( self, parent, session, context, feature ):
      self.integerContext = context
      self.fieldSetIntegerName = self.integerContext.fieldSetName
      self.integerContext.modeIs( self )
      param = ( feature, "integer", self.fieldSetIntegerName,
                AppTrafficRecConfigMode )
      FieldSetIntegerModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.integerContext

   def setContext( self, value ):
      self.integerContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "field-set vlan <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetVlanModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set vlan %s" % self.fieldSetName

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-vlan'
      self.longModeKey = f'{self.feature}-field-set-vlan-{self.fieldSetName}'
      CliMode.ConfigMode.__init__( self, param, parentMode=parentMode )

class FieldSetVlanConfigMode( FieldSetVlanModeBase, ClassificationConfigModeBase ):
   '''Configuration mode for vlan field set'''

   name = "field-set vlan configuration"

   def __init__( self, parent, session, context, feature ):
      self.vlanContext = context
      self.fieldSetVlanName = self.vlanContext.fieldSetName
      self.vlanContext.modeIs( self )
      param = ( feature, "vlan", self.fieldSetVlanName, AppTrafficRecConfigMode )
      FieldSetVlanModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.vlanContext

   def setContext( self, value ):
      self.vlanContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "field-set ( (ipv4 | ipv6) | l4-port ) <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetL4PortModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set l4-port %s" % self.fieldSetName

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-l4-port'
      self.longModeKey = '{}-field-set-l4-port-{}'.format( self.feature,
                                                           self.fieldSetName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=parentMode )

class FieldSetBaseL4PortConfigMode( FieldSetL4PortModeBase,
                                    ClassificationConfigModeBase ):
   '''Configuration mode for L4 port field set'''

   name = "field-set l4-port configuration"

   def __init__( self, parent, session, context, feature ):
      self.l4PortContext = context
      self.fieldSetL4PortName = self.l4PortContext.fieldSetName
      self.l4PortContext.modeIs( self )
      param = ( feature, "l4-port", self.fieldSetL4PortName,
                AppTrafficRecConfigMode )
      FieldSetL4PortModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.l4PortContext

   def setContext( self, value ):
      self.l4PortContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

class FieldSetL4PortConfigMode( FieldSetBaseL4PortConfigMode ):
   name = "field-set l4-port configuration"

#-------------------------------------------------------------------------------
# The "field-set service <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetServiceModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f'field-set service { self.fieldSetName }'

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-service'
      self.longModeKey = f'{self.feature}-field-set-service-{self.fieldSetName}'
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=parentMode )

class FieldSetBaseServiceConfigMode( FieldSetServiceModeBase,
                                     ClassificationConfigModeBase ):
   '''Configuration mode for service field set'''

   name = "field-set service configuration"

   def __init__( self, parent, session, context, feature ):
      self.serviceContext = context
      self.fieldSetServiceName = self.serviceContext.fieldSetName
      self.serviceContext.modeIs( self )
      param = ( feature, "service", self.fieldSetServiceName,
                AppTrafficRecConfigMode )
      FieldSetServiceModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.serviceContext

   def setContext( self, value ):
      self.serviceContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "field-set ipv4|ipv6 prefix" mode
#-------------------------------------------------------------------------------
class FieldSetPrefixModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f"field-set {self.setType} prefix {self.fieldSetName}"

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = "field-set-%s-prefix" % self.setType
      self.longModeKey = "{}-field-set-{}-prefix-{}".format( self.feature,
                                                             self.setType,
                                                             self.fieldSetName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=parentMode )

class FieldSetBasePrefixConfigMode( FieldSetPrefixModeBase,
                                    ClassificationConfigModeBase ):
   '''Base onfiguration mode for field-set prefix'''
   def __init__( self, parent, session, context, feature ):
      self.ipPrefixContext = context
      self.setType = self.ipPrefixContext.setType
      self.fieldSetIpPrefixName = self.ipPrefixContext.fieldSetName
      self.ipPrefixContext.modeIs( self )
      param = ( feature, self.setType,
                self.fieldSetIpPrefixName, AppTrafficRecConfigMode )
      FieldSetPrefixModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.ipPrefixContext

   def setContext( self, value ):
      self.ipPrefixContext = value

   def abort( self ):
      super().abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super().commit()
      self.session_.mode.commit() # call commit of parent

class FieldSetIpPrefixConfigMode( FieldSetBasePrefixConfigMode ):
   """Configuration mode for field-set ipv4 prefix"""
   name = "field-set ipv4 prefix configuration"

class FieldSetIpv6PrefixConfigMode( FieldSetBasePrefixConfigMode ):
   """Configuration mode for field-set ipv4 prefix"""
   name = "field-set ipv6 prefix configuration"

#-------------------------------------------------------------------------------
# The "application-profile <name>" mode
#-------------------------------------------------------------------------------
class AppProfileModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "application-profile %s" % self.appProfileName

   def __init__( self, param ):
      self.appProfileName = param
      self.modeKey = 'app-profile'
      self.longModeKey = 'app-profile-%s' % self.appProfileName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=AppTrafficRecConfigMode )

class AppProfileConfigMode( AppProfileModeBase, BasicCli.ConfigModeBase ):
   '''Configuration mode for application-profile'''

   name = "application-profiel configuration"

   def __init__( self, parent, session, context ):
      self.appProfileContext = context
      self.appProfileName = self.appProfileContext.appProfileName
      self.appProfileContext.modeIs( self )
      AppProfileModeBase.__init__( self, self.appProfileContext.appProfileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.appProfileContext

   def onExit( self ):
      t0( "Exit application-profile %s" % self.appProfileName )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t0( "Abort application-profile %s" % self.appProfileName )
      if self.appProfileContext is not None:
         self.appProfileContext.abort()
      self.appProfileContext = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

   def commitContext( self ):
      t0( "Commit application-profile %s" % self.appProfileName )
      if self.appProfileContext:
         self.appProfileContext.commit()
         self.appProfileContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "application ipv4 <name>" mode
#-------------------------------------------------------------------------------
class AppModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f"application {self.afToken} {self.appName}"

   def __init__( self, param, afToken ):
      self.appName = param
      self.afToken = afToken
      self.modeKey = "app-%s" % self.afToken
      self.longModeKey = f"app-{self.afToken}-{self.appName}"
      CliMode.ConfigMode.__init__( self, param, parentMode=AppTrafficRecConfigMode )

def makeAppConfigMode( afToken ):
   class AppConfigMode( AppModeBase, BasicCli.ConfigModeBase ):
      '''Configuration mode for application (ipv4|ipv6|l4)'''

      name = "application %s configuration" % afToken

      def __init__( self, parent, session, context ):
         self.appContext = context
         self.appName = self.appContext.appName
         self.appContext.modeIs( self )
         AppModeBase.__init__( self, self.appContext.appName, afToken )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def getContext( self ):
         return self.appContext

      def onExit( self ):
         t0( "Exit application %s" % self.appName )
         self.commitContext()
         BasicCli.ConfigModeBase.onExit( self )

      def abort( self ):
         t0( "Abort application %s" % self.appName )
         if self.appContext is not None:
            self.appContext.abort()
         self.appContext = None
         self.session_.gotoParentMode()
         self.session_.mode.abort() # call abort of parent

      def commitContext( self ):
         t0( "Commit application %s" % self.appName )
         if self.appContext:
            self.appContext.commit()
            self.appContext = None

      def commit( self ):
         self.commitContext()
         self.session_.gotoParentMode()
         self.session_.mode.commit() # call commit of parent

   return AppConfigMode

AppConfigModeIpv4 = makeAppConfigMode( "ipv4" )
AppConfigModeL4 = makeAppConfigMode( "l4" )

#-------------------------------------------------------------------------------
# The "category <keyword>|WORD" mode
#-------------------------------------------------------------------------------
class CategoryModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "category %s" % self.categoryName

   def __init__( self, param ):
      self.categoryName = param
      self.modeKey = 'category'
      self.longModeKey = 'category-%s' % self.categoryName
      CliMode.ConfigMode.__init__( self, param, parentMode=AppTrafficRecConfigMode )

class CategoryConfigMode( CategoryModeBase, BasicCli.ConfigModeBase ):
   '''Configuration mode for category'''

   name = "category configuration"

   def __init__( self, parent, session, context ):
      self.categoryContext = context
      self.categoryName = self.categoryContext.categoryName
      self.categoryContext.modeIs( self )
      CategoryModeBase.__init__( self, self.categoryContext.categoryName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.categoryContext

   def onExit( self ):
      t0( "Exit category %s" % self.categoryName )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t0( "Abort category %s" % self.categoryName )
      if self.categoryContext is not None:
         self.categoryContext.abort()
      self.categoryContext = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

   def commitContext( self ):
      t0( "Commit category %s" % self.categoryName )
      if self.categoryContext:
         self.categoryContext.commit()
         self.categoryContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent
