#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliMode
from IpLibConsts import DEFAULT_VRF

class RoutingOspfv3Mode( CliMode.ConfigMode ):

   def enterCmd( self ):
      if self.vrfName != DEFAULT_VRF:
         return 'router ospfv3 vrf %s' % self.vrfName
      return 'router ospfv3'

   def __init__( self, param ):
      self.vrfName = param
      # The industry standard cli has "config-router" as the prompt
      # for this mode.
      self.modeKey = "router"
      if self.vrfName == DEFAULT_VRF:
         self.longModeKey = "router-ospfv3"
      else:
         self.longModeKey = "router-ospfv3-vrf-%s" % self.vrfName
      CliMode.ConfigMode.__init__( self, param )

class RoutingOspfv3GeneralMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'router ospfv3 general'

   def __init__( self, param ):
      self.modeKey = "router"
      self.longModeKey = "router-ospfv3-general"
      CliMode.ConfigMode.__init__( self, param )

class RoutingOspfv3AfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'address-family %s' % self.addrFamily

   def __init__( self, param ):
      # self.addrFamily is either "ipv4" or "ipv6"
      # self.instId is either 64 or 0
      self.vrfName, self.addrFamily, self.instId = param
      # The industry standard cli has "config-router-af" as the prompt
      # for this mode (i.e. no '-v4'/'-v6' suffix).
      self.modeKey = "router"
      if self.vrfName == DEFAULT_VRF:
         self.longModeKey = "router-ospfv3-af"
      else:
         self.longModeKey = f'router-ospfv3-vrf-{self.vrfName}-af'
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingOspfv3Mode )

   def commentKey( self ):
      if self.vrfName == DEFAULT_VRF:
         return "router-ospfv3-af-%s" % self.addrFamily
      else:
         return f"router-ospfv3-vrf-{self.vrfName}-af-{self.addrFamily}"

class RoutingOspfv3AreaShamLinkMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return "area {} sham-link destination {}".format( self.areaId,
                                                        self.shamLinkDestIp )

   def __init__( self, param ):
      # self.areaId for the area where this shamlink resides
      # self.shamLinkDestIp is the shamlink remote Endpoint
      self.vrfName, self.areaId, self.shamLinkDestIp, self.addrFamily = param
      self.modeKey = "sham-link"
      if self.vrfName == DEFAULT_VRF:
         assert 0, 'ShamLink is not supported in Default Vrf'
      else:
         self.longModeKey = "router-ospfv3-vrf-%s-af-sham-link" % self.vrfName
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingOspfv3AfMode )

   def commentKey( self ):
      return ( f"router-ospfv3-vrf-{self.vrfName}-af-{self.addrFamily}-sham-link-"
               f"area-{self.areaId}-dest-{self.shamLinkDestIp}" )
