#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode

class RouterGeneralBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'router general'

   def __init__( self, param ):
      self.modeKey = 'router'
      self.longModeKey = 'router-general'
      CliMode.ConfigMode.__init__( self, param=param )

   def skipIfEmpty( self ):
      """ Allows us to unconditionally create the mode in CliSavePlugin
      (via getOrCreateModeInstance), rather than making it conditional on the
      presence of Config. As an added benefit, this will also display the
      mode if the only configuration that exists are comments.
      """
      return True

class RouterGeneralVrfBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'vrf %s' % self.vrfName # pylint: disable=consider-using-f-string

   def __init__( self, param ):
      self.vrfName = param
      self.modeKey = 'router'
      # pylint: disable-next=consider-using-f-string
      self.longModeKey = 'router-general-vrf-%s' % param
      CliMode.ConfigMode.__init__( self, param, parentMode=RouterGeneralBaseMode )

class RouterGeneralMetricProfileBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      # pylint: disable-next=consider-using-f-string
      return 'metric profile %s' % self.profileName

   def __init__( self, param ):
      self.profileName = param
      self.modeKey = 'router-general-metric-profile'
      # pylint: disable-next=consider-using-f-string
      self.longModeKey = 'router-general-metric-profile-%s' % param
      CliMode.ConfigMode.__init__( self, param, parentMode=RouterGeneralBaseMode )

class RouterGeneralRoutePriorityBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'route priority'

   def __init__( self, param ):
      self.modeKey = 'router-general-route-priority'
      self.longModeKey = 'router-general-route-priority'
      CliMode.ConfigMode.__init__( self, param, parentMode=RouterGeneralBaseMode )

class RouterGeneralVrfRoutePriorityBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'route priority'

   def __init__( self, vrfName ):
      self.vrfName = vrfName
      self.modeKey = 'route-general-vrf-route-priority'
      self.longModeKey = f'router-general-vrf-{vrfName}-route-priority'
      CliMode.ConfigMode.__init__( self, vrfName,
                                   parentMode=RouterGeneralVrfBaseMode )

class RouterGeneralVrfFilteredRibBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f'unicast-rib {self.addrFamily} {self.filteredRibName}'

   def __init__( self, param ):
      ( self.vrfName, self.addrFamily, self.filteredRibName ) = param
      self.modeKey = 'route-general-unicast-rib'
      self.longModeKey =\
            f'router-general-vrf-{self.vrfName}-unicast-rib-{self.filteredRibName}'
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=RouterGeneralVrfBaseMode )

class RouterGeneralAutoVersionBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'auto-version'

   def __init__( self, param ):
      self.modeKey = "router"
      self.longModeKey = "router-auto-version"
      CliMode.ConfigMode.__init__( self, param, parentMode=RouterGeneralBaseMode )
