# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMode
from Assert import assertEqual
from CliMode.Te import GlobalTeMode
from CliSavePlugin.TeCliSave import buildAdminGroupListStr
from RsvpLibCliSave import bandwidthHumanReadable
import MultiRangeRule
import Tac
from Toggles import RsvpToggleLib
from TypeFuture import TacLazyType

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
RsvpLerAutoBwParam = TacLazyType( 'Rsvp::RsvpLerAutoBwParam' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLibConstants = TacLazyType( 'Rsvp::RsvpLibConstants' )
RsvpFrrMode = TacLazyType( 'Rsvp::Cli::FrrMode' )
RsvpLerLdpTunnelingConfig = TacLazyType( 'Rsvp::RsvpLerLdpTunnelingConfig' )
RsvpLerMetricConfig = TacLazyType( 'Rsvp::RsvpLerMetricConfig' )
RsvpLerPathSpecId = TacLazyType( 'Rsvp::RsvpLerPathSpecId' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )
RsvpLerSplitTunnelConfig = TacLazyType( 'Rsvp::RsvpLerSplitTunnelConfig' )
RsvpLerTunnelBandwidthConfig = TacLazyType( 'Rsvp::RsvpLerTunnelBandwidthConfig' )
RsvpLerTunnelPriority = TacLazyType( 'Rsvp::RsvpLerTunnelPriority' )

pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType

class RsvpTeMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'rsvp'

   def __init__( self, param ):
      self.modeKey = "rsvp"
      self.longModeKey = "te-rsvp"
      CliMode.ConfigMode.__init__( self, param, parentMode=GlobalTeMode )

   def skipIfEmpty( self ):
      return True

def bitMaskToStr( bitMask ):
   '''
   Convert an integer representing a bitmask to a string representing each bit
   set to 1 in this bitmask.
   ie:
   0x0 -> 0000 -> ''
   0x1 -> 0001-> '0'
   0x3 -> 0011 -> '0-1'
   0x13 -> 0001 0011 -> '0-1,4'
   etc...
   '''
   if not bitMask:
      return ''
   maskList = []
   i = 0
   while bitMask:
      if bitMask % 2:
         maskList.append( i )
      i += 1
      bitMask >>= 1

   return MultiRangeRule.multiRangeToCanonicalString( maskList )

def intervalHumanReadable( interval ):
   '''Convert an integer interval value to human readable format
   ( seconds | minutes | hours )'''
   unit = 'seconds'
   if interval > 60 and ( interval % 60 ) == 0:
      interval /= 60
      unit = 'minutes'
      if interval > 60 and ( interval % 60 ) == 0:
         interval /= 60
         unit = 'hours'
   return ( interval, unit )

class RsvpP2mpLeafMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'leaf %s' % ( self.dstIp )

   def commitContext( self ):
      raise NotImplementedError

   def leafToCliCmds( self, leaf, saveAll=False ):
      cmds = []
      if not leaf:
         return cmds

      # No Hops
      if not leaf.includeHop and saveAll:
         cmds.append( 'no hop' )

      for hop in leaf.includeHop.values():
         cmd = 'hop ' + str( hop.hopIp ) + ( ' node' if hop.node else '' ) + \
               ( ' loose' if hop.loose else '' )
         cmds.append( cmd )

      return cmds

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def __init__( self, param ):
      self.parentPath = param[ 0 ]
      self.dstIp = param[ 1 ]

      self.modeKey = "%s-leaf-%s" % ( self.parentPath.modeKey, self.dstIp )
      self.longModeKey = "te-rsvp-" + self.modeKey
      CliMode.ConfigMode.__init__( self, param )

   def skipIfEmpty( self ):
      # Even if a leaf is 'empty' (i.e no configuration command issued for it),
      # we still need to display it, as it is saved in the leaf collection
      return False

class RsvpPathSpecMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      prefix = 'tree' if self.p2mp else 'path'
      cmd = f'{prefix} {self.pathSpecName} {self.pathSpecTypeStr}'
      return cmd

   def commitContext( self ):
      raise NotImplementedError

   def pathSpecToCliCmds( self, pathSpec, saveAll=False ):
      cmds = []
      if not pathSpec:
         return cmds

      # No Hops
      if self.p2mp:
         # P2MP paths do not have include hops at this level, only at the
         # leaf level. Dynamic P2MP paths do have exclude hops though
         if self.pathSpecType == pathSpecDynamicType and not pathSpec.excludeHop \
            and saveAll:
            cmds.append( 'no hop' )
      else:
         if not pathSpec.includeHop and not pathSpec.excludeHop and saveAll:
            cmds.append( 'no hop' )

      # Include Hops
      for hop in pathSpec.includeHop.values():
         cmd = 'hop ' + str( hop.hopIp ) + ( ' node' if hop.node else '' ) + \
               ( ' loose' if hop.loose else '' )
         cmds.append( cmd )

      if self.pathSpecType == pathSpecExplicitType:
         # Explicit Mode doesn't have exclude hops or administrative groups
         if self.p2mp and not pathSpec.leaf and saveAll:
            cmds.append( 'no leaf' )
         return cmds

      # Dynamic mode only from now on
      # Exclude Hops
      for excludeHop in pathSpec.excludeHop:
         cmds.append( 'hop ' + str( excludeHop ) + ' exclude' )

      # Administrative Groups
      constraints = pathSpec.adminGroupConstraints
      includeAll = { k.value: v for k, v in constraints.includeAll.items() }
      includeAllListStr = buildAdminGroupListStr( includeAll,
                                                  constraints.includeAllByName )
      includeAllStr = f'include all {includeAllListStr}' if includeAllListStr \
         else ''

      includeAny = { k.value: v for k, v in constraints.includeAny.items() }
      includeAnyListStr = buildAdminGroupListStr( includeAny,
                                                  constraints.includeAnyByName )
      includeAnyStr = f'include any {includeAnyListStr}' if includeAnyListStr \
         else ''

      exclude = { k.value: v for k, v in constraints.exclude.items() }
      excListStr = buildAdminGroupListStr( exclude,
                                           constraints.excludeByName )
      excludeStr = f'exclude {excListStr}' if excListStr else ''

      if constraints.includeAll or constraints.includeAny or constraints.exclude or \
            constraints.includeAllByName or constraints.includeAnyByName or \
            constraints.excludeByName:
         cmd = ' '.join( filter( None,
                               [ 'administrative-group',
                                 includeAllStr, includeAnyStr, excludeStr ] ) )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no administrative-group'
         cmds.append( cmd )

      if self.p2mp and not pathSpec.leaf and saveAll:
         cmds.append( 'no leaf' )

      return cmds

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def __init__( self, param ):
      self.pathSpecName = param[ 0 ]
      self.pathSpecType = param[ 1 ]
      self.p2mp = param[ 2 ]
      if self.pathSpecType == pathSpecExplicitType:
         self.pathSpecTypeStr = 'explicit'
         self.shortPathTypeStr = 'expl'
      else:
         self.pathSpecTypeStr = 'dynamic'
         self.shortPathTypeStr = 'dyn'

      if not RsvpToggleLib.toggleRsvpP2mpLerDynamicTreeSpecEnabled() and self.p2mp:
         assertEqual( self.pathSpecType, pathSpecExplicitType )
      prefix = 'tree' if self.p2mp else 'path'
      self.modeKey = f"{prefix}-{self.shortPathTypeStr}-{self.pathSpecName}"
      self.pathSpec = None
      self.longModeKey = "te-rsvp-" + self.modeKey
      CliMode.ConfigMode.__init__( self, param )

   def skipIfEmpty( self ):
      # Even if a pathSpec is 'empty' (i.e no configuration command issued for it),
      # we still need to display it, as it is saved in the pathSpec collection
      return False

class RsvpTunnelBaseMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      raise NotImplementedError

   def tunnelSpecOrProfileToCliCmds( self, tunnelSpecOrProfile, rsvpLerCliConfig,
                                     saveAll=False ):
      raise NotImplementedError

   def skipIfEmpty( self ):
      # Even if a tunnelSpec or a tunnelProfile is 'empty' (i.e no configuration
      # command issued for it), we still need to display it, as it is saved
      # in the tunnelSpec or tunnelProfile collection
      return False

   def bandwidthConfigToCliCmd( self, cmds, bandwidthConfig, saveAll=False ):
      if bandwidthConfig.autoBw:
         autoBwParam = bandwidthConfig.autoBwParam
         # Autobandwidth and autobandwidth parameters
         # convert bytes-per-sec to bits-per-sec
         minBw, minBwUnit = bandwidthHumanReadable( autoBwParam.minBandwidth * 8 )
         maxBw, maxBwUnit = bandwidthHumanReadable( autoBwParam.maxBandwidth * 8 )
         adjustmentPeriod = int( autoBwParam.adjustmentPeriod )
         if adjustmentPeriod:
            cmd = 'bandwidth auto min %.2f %s max %.2f %s adjustment-period %s' % \
                  ( minBw, minBwUnit, maxBw, maxBwUnit, adjustmentPeriod )
         else:
            cmd = 'bandwidth auto min %.2f %s max %.2f %s' % \
                  ( minBw, minBwUnit, maxBw, maxBwUnit )
         sensitivity = autoBwParam.adjAlgorithmSensitivity
         if sensitivity == 0:
            cmd += ' sensitivity dynamic'
         elif saveAll or \
               sensitivity != RsvpLerAutoBwParam().adjAlgorithmSensitivity:
            cmd += ' sensitivity %d' % sensitivity
         cmds.append( cmd )
      else:
         # Explicit bandwidth
         # convert bytes-per-sec to bits-per-sec
         bw, unit = bandwidthHumanReadable( bandwidthConfig.explicitBandwidth * 8 )
         cmd = f'bandwidth {bw:.2f} {unit}'
         cmds.append( cmd )

   def splitTunnelConfigToCliCmd( self, cmds, splitTunnelConfig, saveAll=False ):
      if splitTunnelConfig.splitBw:
         splitBwParam = splitTunnelConfig.splitBwParam
         # split bandwidth
         subTunnelLimit = splitBwParam.subTunnelLimit
         if splitBwParam.minBw or splitBwParam.maxBw:
            # convert min/max bw from bytes-per-sec to bits-per-sec
            minBw, minBwUnit = bandwidthHumanReadable( splitBwParam.minBw * 8 )
            maxBw, maxBwUnit = bandwidthHumanReadable( splitBwParam.maxBw * 8 )
            cmd = 'split-tunnel min {:.2f} {} max {:.2f} {}'.format(
                     minBw, minBwUnit, maxBw, maxBwUnit )
            if subTunnelLimit != RsvpLerConstants.defaultSubTunnelLimit:
               cmd += ' sub-tunnels limit %d' % subTunnelLimit
            reductionDelay = splitBwParam.adaptiveSubTunnelPreservationTime
            if reductionDelay != \
                     RsvpLibConstants.defaultAdaptiveSubTunnelPreservationTime:
               delay, delayUnit = intervalHumanReadable( reductionDelay )
               cmd += ' reduction-delay %d %s' % ( delay, delayUnit )
            cmds.append( cmd )
         else:
            # convert quantum from bytes-per-sec to bits-per-sec
            quantum, unit = bandwidthHumanReadable( splitBwParam.quantum * 8 )
            cmd = f'split-tunnel quantum {quantum:.2f} {unit}'
            if subTunnelLimit != RsvpLerConstants.defaultSubTunnelLimit:
               cmd += ' sub-tunnels limit %d' % subTunnelLimit
            cmds.append( cmd )
      else:
         cmd = 'split-tunnel disabled'
         cmds.append( cmd )

   def tunnelSpecOrProfileOptIntervalToCliCmd( self, cmds, tunnelSpecOrProfile,
                                               saveAll=False ):
      if tunnelSpecOrProfile.optimizationInterval is None:
         if saveAll:
            cmd = 'no optimization'
            cmds.append( cmd )
      elif ( tunnelSpecOrProfile.optimizationInterval ==
               RsvpLerConstants.optimizationIntervalDisabled ):
         cmd = 'optimization disabled'
         cmds.append( cmd )
      else:
         cmd = 'optimization interval %d seconds' % \
               tunnelSpecOrProfile.optimizationInterval
         cmds.append( cmd )

class RsvpTunnelSpecMode( RsvpTunnelBaseMode ):
   def __init__( self, param ):
      self.tunnelSpecName = param
      self.modeKey = "tunnel-%s" % param
      self.longModeKey = "te-rsvp-" + self.modeKey
      RsvpTunnelBaseMode.__init__( self, param )

   def enterCmd( self ):
      return 'tunnel %s' % self.tunnelSpecName

   def tunnelSpecSignalBwToCliCmd( self, cmds, tunnelSpec, rsvpLerCliConfig,
                                   saveAll=False ):
      signaling = 'signaling'
      if rsvpLerCliConfig.bwSignalingTypoFix is False:
         signaling = 'signalling'

      if tunnelSpec.signalBw is not None:
         if tunnelSpec.signalBw:
            cmd = 'bandwidth %s' % signaling
         else:
            cmd = 'bandwidth %s disabled' % signaling
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no bandwidth %s disabled' % signaling
         cmds.append( cmd )

   def tunnelSpecPriorityToCliCmd( self, cmds, tunnelSpec, saveAll=False ):
      if tunnelSpec.tunnelPriority is not None:
         cmd = 'priority setup %d hold %d' % (
            tunnelSpec.tunnelPriority.setupPriority,
            tunnelSpec.tunnelPriority.holdPriority )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no priority'
         cmds.append( cmd )

   def tunnelSpecProfileToCliCmd( self, cmds, tunnelSpec, saveAll=False ):
      if tunnelSpec.profileId:
         cmd = 'profile ' + tunnelSpec.profileId.tunnelProfileName
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no profile'
         cmds.append( cmd )

   def tunnelSpecOrProfileToCliCmds( self, tunnelSpecOrProfile, rsvpLerCliConfig,
                                     saveAll=False ):
      cmds = []
      if not tunnelSpecOrProfile:
         return cmds

      # Destination ip
      if tunnelSpecOrProfile.dstIp:
         assert len( tunnelSpecOrProfile.dstIp ) == 1, "P2P must have a single dstIp"
         dstIp = next( tunnelSpecOrProfile.dstIp.keys() )
         cmd = 'destination ip ' + str( dstIp )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no destination ip'
         cmds.append( cmd )

      # No alias endpoints
      if not tunnelSpecOrProfile.extraTep and saveAll:
         cmds.append( 'no alias endpoint' )

      # Alias Endpoints
      for tep in tunnelSpecOrProfile.extraTep:
         cmds.append( 'alias endpoint ' + str( tep ) )

      # Color
      if tunnelSpecOrProfile.color is not None:
         if tunnelSpecOrProfile.color.isSet:
            cmd = f'color {tunnelSpecOrProfile.color.value}'
         else:
            cmd = 'color disabled'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no color'
         cmds.append( cmd )

      # Fast reroute
      if tunnelSpecOrProfile.fastRerouteConfig is not None:
         frrMode = tunnelSpecOrProfile.fastRerouteConfig
         if frrMode == RsvpFrrMode.frrLinkProtection:
            cmd = 'fast-reroute mode link-protection'
         elif frrMode == RsvpFrrMode.frrNodeProtection:
            cmd = 'fast-reroute mode node-protection'
         else:
            cmd = 'fast-reroute mode none'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no fast-reroute mode'
         cmds.append( cmd )

      # Metric
      if tunnelSpecOrProfile.metric is not None:
         if tunnelSpecOrProfile.metric.igpDynamic:
            cmd = 'metric igp'
         else:
            cmd = f'metric {tunnelSpecOrProfile.metric.staticMetric}'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no metric'
         cmds.append( cmd )

      # primary and/or secondary paths
      if tunnelSpecOrProfile.primaryPath or tunnelSpecOrProfile.secondaryPath:
         # Primary path
         if tunnelSpecOrProfile.primaryPath:
            cmd = 'path ' + tunnelSpecOrProfile.primaryPath.pathSpecName
            cmds.append( cmd )
         # Secondary path and secondary pre-signaled
         if tunnelSpecOrProfile.secondaryPath:
            secondaryPath = tunnelSpecOrProfile.secondaryPath.pathSpecName
            secondaryPathPreSignaled = tunnelSpecOrProfile.secondaryPathPreSignaled
            cmdSuffix = ' pre-signaled' if secondaryPathPreSignaled else ''
            cmd = 'path ' + secondaryPath + ' secondary' + cmdSuffix
            cmds.append( cmd )
      elif saveAll:
         cmd = 'no path'
         cmds.append( cmd )

      bandwidthConfig = tunnelSpecOrProfile.bandwidthConfig
      if bandwidthConfig is not None:
         self.bandwidthConfigToCliCmd( cmds, bandwidthConfig, saveAll )
      elif saveAll:
         cmd = 'no bandwidth'
         cmds.append( cmd )

      self.tunnelSpecSignalBwToCliCmd( cmds, tunnelSpecOrProfile, rsvpLerCliConfig,
                                       saveAll )

      self.tunnelSpecPriorityToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      if tunnelSpecOrProfile.ldpTunnelingConfig is not None:
         cmd = 'tunneling ldp'
         if not tunnelSpecOrProfile.ldpTunnelingConfig.eligible:
            cmd = cmd + ' disabled'
         elif tunnelSpecOrProfile.ldpTunnelingConfig.ucmp:
            cmd = cmd + ' ucmp'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no tunneling ldp'
         cmds.append( cmd )

      if tunnelSpecOrProfile.eligibleForIgpShortcut is not None:
         if tunnelSpecOrProfile.eligibleForIgpShortcut:
            cmd = 'igp shortcut'
         else:
            cmd = 'igp shortcut disabled'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no igp shortcut'
         cmds.append( cmd )

      self.tunnelSpecOrProfileOptIntervalToCliCmd( cmds, tunnelSpecOrProfile,
                                                   saveAll )

      splitTunnelConfig = tunnelSpecOrProfile.splitTunnelConfig
      if splitTunnelConfig is not None:
         self.splitTunnelConfigToCliCmd( cmds, splitTunnelConfig, saveAll )
      elif saveAll:
         cmd = 'no split-tunnel'
         cmds.append( cmd )

      self.tunnelSpecProfileToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      # no shutdown
      # By convention, this should stay at the end so it is displayed last
      if tunnelSpecOrProfile.enabled:
         cmd = 'no shutdown'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'shutdown'
         cmds.append( cmd )

      return cmds

class RsvpP2mpTunnelSpecMode( RsvpTunnelSpecMode ):
   def __init__( self, param ):
      RsvpTunnelSpecMode.__init__( self, param )
      self.tunnelSpecName = param
      self.modeKey = f"tunnel-p2mp-{param}"
      self.longModeKey = f"te-rsvp-{self.modeKey}"

   def enterCmd( self ):
      return 'tunnel %s p2mp' % self.tunnelSpecName

   def tunnelSpecOrProfileToCliCmds( self, tunnelSpecOrProfile, rsvpLerCliConfig,
                                     saveAll=False ):
      cmds = []
      if not tunnelSpecOrProfile:
         return cmds

      # Destination IP
      if tunnelSpecOrProfile.dstIp:
         for dstIp in tunnelSpecOrProfile.dstIp.keys():
            cmd = 'destination ip ' + str( dstIp )
            cmds.append( cmd )
      elif not tunnelSpecOrProfile.dstIp and saveAll:
         cmd = 'no destination ip'
         cmds.append( cmd )

      # Primary path
      if tunnelSpecOrProfile.primaryPath:
         cmd = 'tree ' + tunnelSpecOrProfile.primaryPath.pathSpecName
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no tree'
         cmds.append( cmd )

      self.tunnelSpecProfileToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      # no shutdown
      # By convention, this should stay at the end so it is displayed last
      if tunnelSpecOrProfile.enabled:
         cmd = 'no shutdown'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'shutdown'
         cmds.append( cmd )

      return cmds

class RsvpTunnelProfileMode( RsvpTunnelBaseMode ):
   def __init__( self, param ):
      self.tunnelProfileName = param
      self.modeKey = "tunnel-profile-%s" % param
      self.longModeKey = "te-rsvp-" + self.modeKey
      RsvpTunnelBaseMode.__init__( self, param )

   def enterCmd( self ):
      return 'tunnel profile %s' % self.tunnelProfileName

   def tunnelProfileColorToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.color is not None

      if tunnelProfile.color.isSet:
         cmd = f'color {tunnelProfile.color.value}'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'color disabled'
         cmds.append( cmd )

   def tunnelProfileFastRerouteToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      if tunnelProfile.fastRerouteConfig is not None:
         frrMode = tunnelProfile.fastRerouteConfig
         if frrMode == RsvpFrrMode.frrLinkProtection:
            cmd = 'fast-reroute mode link-protection'
         elif frrMode == RsvpFrrMode.frrNodeProtection:
            cmd = 'fast-reroute mode node-protection'
         else:
            cmd = 'fast-reroute mode none'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no fast-reroute mode'
         cmds.append( cmd )

   def tunnelProfileMetricToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.metric is not None

      if tunnelProfile.metric != RsvpLerMetricConfig() or saveAll:
         if tunnelProfile.metric.igpDynamic:
            cmd = 'metric igp'
         else:
            cmd = f'metric {tunnelProfile.metric.staticMetric}'
         cmds.append( cmd )

   def tunnelProfileBandwidthToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      bandwidthConfig = tunnelProfile.bandwidthConfig
      assert bandwidthConfig is not None

      if bandwidthConfig != RsvpLerTunnelBandwidthConfig() or saveAll:
         self.bandwidthConfigToCliCmd( cmds, bandwidthConfig, saveAll )

   def tunnelProfileSignalBwToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      assert tunnelProfile.signalBw is not None

      if not tunnelProfile.signalBw:
         cmd = 'bandwidth signaling disabled'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'bandwidth signaling'
         cmds.append( cmd )

   def tunnelProfilePriorityToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.tunnelPriority is not None

      if tunnelProfile.tunnelPriority != RsvpLerTunnelPriority() or saveAll:
         cmd = 'priority setup %d hold %d' % (
            tunnelProfile.tunnelPriority.setupPriority,
            tunnelProfile.tunnelPriority.holdPriority )
         cmds.append( cmd )

   def tunnelProfileLdpTunnelingToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.ldpTunnelingConfig is not None

      if tunnelProfile.ldpTunnelingConfig != RsvpLerLdpTunnelingConfig() or saveAll:
         cmd = 'tunneling ldp'
         if not tunnelProfile.ldpTunnelingConfig.eligible:
            cmd = cmd + ' disabled'
         elif tunnelProfile.ldpTunnelingConfig.ucmp:
            cmd = cmd + ' ucmp'
         cmds.append( cmd )

   def tunnelProfileIgpShortcutToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.eligibleForIgpShortcut is not None

      if tunnelProfile.eligibleForIgpShortcut:
         cmd = 'igp shortcut'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'igp shortcut disabled'
         cmds.append( cmd )

   def tunnelProfileSplitTunnelToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      splitTunnelConfig = tunnelProfile.splitTunnelConfig
      assert splitTunnelConfig is not None

      if splitTunnelConfig != RsvpLerSplitTunnelConfig() or saveAll:
         self.splitTunnelConfigToCliCmd( cmds, splitTunnelConfig, saveAll )

   def tunnelSpecOrProfileToCliCmds( self, tunnelSpecOrProfile, rsvpLerCliConfig,
                                     saveAll=False ):
      cmds = []
      if not tunnelSpecOrProfile:
         return cmds

      self.tunnelProfileColorToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileFastRerouteToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileMetricToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileBandwidthToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileSignalBwToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfilePriorityToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileLdpTunnelingToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelProfileIgpShortcutToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      self.tunnelSpecOrProfileOptIntervalToCliCmd( cmds, tunnelSpecOrProfile,
                                                   saveAll )

      self.tunnelProfileSplitTunnelToCliCmd( cmds, tunnelSpecOrProfile, saveAll )

      return cmds

# NOTE: this class can be changed to inherit from RsvpTunnelProfileMode
# when we will add support for more attributes.
class RsvpP2mpTunnelProfileMode( RsvpTunnelBaseMode ):
   def __init__( self, param ):
      self.tunnelProfileName = param
      self.modeKey = "tunnel-profile-p2mp-%s" % param
      self.longModeKey = "te-rsvp-" + self.modeKey
      RsvpTunnelBaseMode.__init__( self, param )

   def enterCmd( self ):
      return 'tunnel profile %s p2mp' % self.tunnelProfileName

   def tunnelProfilePrimaryPathToCliCmd( self, cmds, tunnelProfile, saveAll=False ):
      # No knob to inherit from; should never be None.
      assert tunnelProfile.primaryPath is not None

      if tunnelProfile.primaryPath:
         cmd = 'tree ' + tunnelProfile.primaryPath.pathSpecName
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no tree'
         cmds.append( cmd )

   def tunnelSpecOrProfileToCliCmds( self, tunnelSpecOrProfile, rsvpLerCliConfig,
                                     saveAll=False ):
      cmds = []
      if not tunnelSpecOrProfile:
         return cmds

      self.tunnelProfilePrimaryPathToCliCmd( cmds, tunnelSpecOrProfile,
                                             saveAll=saveAll )

      return cmds
