# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliMode
from CliMode.Te import GlobalTeMode
from SrTePolicyLib import Weight
import Tac

#Defines SRTE policy configuration modes:
#   segment-routing
#      policy endpoint <endpoint> color <color>
#         path-group preference <preference> [ computation local ]
#
class SrTeModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'segment-routing'

   def __init__( self, param ):
      self.modeKey = "sr"
      self.longModeKey = "te-sr"
      CliMode.ConfigMode.__init__( self, param, parentMode=GlobalTeMode )

class SrTePolicyModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return f'policy endpoint {self.endPoint} color {self.color}'

   def __init__( self, param ):
      self.endPoint, self.color = param
      self.modeKey = "policy"
      self.longModeKey = "te-sr-policy"
      CliMode.ConfigMode.__init__( self, param, parentMode=SrTeModeBase )

   def commentKey( self ):
      return f"te-sr-policy-{self.endPoint}-{self.color}"

class SrTeDynamicPolicyModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return f'policy endpoint dynamic color {self.color}'

   def __init__( self, param ):
      self.color = param
      self.modeKey = "policy"
      self.longModeKey = "te-sr-policy-dynamic"
      CliMode.ConfigMode.__init__( self, param, parentMode=SrTeModeBase )

   def commentKey( self ):
      return f"te-sr-policy-dynamic-{self.color}"

class SrTePolicyPathGroupModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'path-group preference %s' % ( self.preference )

   def __init__( self, param ):
      self.endPoint, self.color, self.preference = param
      self.modeKey = "path"
      self.longModeKey = "te-sr-policy-path"
      CliMode.ConfigMode.__init__( self, param, parentMode=SrTePolicyModeBase )

   def commentKey( self ):
      return "te-sr-policy-{}-{}-path-{}".format( self.endPoint, self.color,
                                                  self.preference )

class SrTePolicyPathLocalComputationModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return f'path-group preference {self.preference} computation local'

   def __init__( self, param ):
      self.endPoint, self.color, self.preference = param
      self.modeKey = "path"
      if self.endPoint is None:
         self.longModeKey = "te-sr-policy-dynamic-path-local"
         CliMode.ConfigMode.__init__( self, param,
                                      parentMode=SrTeDynamicPolicyModeBase )
      else:
         self.longModeKey = "te-sr-policy-path-local"
         CliMode.ConfigMode.__init__( self, param, parentMode=SrTePolicyModeBase )

   def commentKey( self ):
      return "te-sr-policy-{}-{}-path-{}-local".format( self.endPoint if
                                                        self.endPoint
                                                        else 'dynamic',
                                                        self.color, self.preference )

class SrTePolicySegmentListModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      labelStackStr = ""
      for i in range( self.segmentList.stackSize ):
         labelStackStr += " " if i > 0 else ""
         labelStackStr += str( self.segmentList.labelStack( i ) )
      weightStr = ""
      if self.weight and self.weight != Weight().valueDefault:
         weightStr += " weight " + str( self.weight )
      indexStr = ""
      if self.index and self.index != Tac.Type( 'SrTePolicy::Index' ).valueDefault:
         indexStr += " index " + str( self.index )
      return 'segment-list label-stack {}{}{}'.format( labelStackStr,
                                                       weightStr, indexStr )

   def __init__( self, param ):
      ( self.endPoint, self.color, self.preference, self.segmentList,
        self.weight, self.index ) = param
      self.modeKey = "seglist"
      self.longModeKey = "te-sr-policy-path-seglist"
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=SrTePolicyPathGroupModeBase )

   def commentKey( self ):
      labelStackStr = ""
      for i in range( self.segmentList.stackSize ):
         labelStackStr += "-" if i > 0 else ""
         labelStackStr += str( self.segmentList.labelStack( i ) )
      return "te-sr-policy-{}-{}-path-{}-seglist-{}".format( self.endPoint,
                                                             self.color,
                                                             self.preference,
                                                             labelStackStr )

class ReplicationSegmentModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      cmd = 'replication-segment %s' % ( self.repId )
      return cmd

   def __init__( self, param ):
      self.repSeg = param
      self.repId = self.repSeg.replicationId
      self.name = self.repSeg.replicationSegmentName
      self.modeKey = 'rseg'
      self.longModeKey = 'te-sr-rseg-%s' % self.repId
      CliMode.ConfigMode.__init__( self, param, parentMode=SrTeModeBase )
