#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliMode
from CliMode.Classification import ( ClassificationConfigModeBase,
                                     FieldSetBasePrefixConfigMode,
                                     FieldSetBaseL4PortConfigMode,
                                     FieldSetBaseMacAddrConfigMode,
                                     FieldSetBaseServiceConfigMode )
from TypeFuture import TacLazyType
import BasicCli
import Tracing

ClassPriorityConstant = TacLazyType( 'TrafficPolicy::ClassPriorityConstant' )
ClassPriorityMin = ClassPriorityConstant.classPriorityMin
ClassPriorityInc = ClassPriorityConstant.classPriorityInc

__defaultTraceHandle__ = Tracing.Handle( 'TrafficPolicies' )
t0 = __defaultTraceHandle__.trace0

FEATURE = 'traffic-policy'
FEATURE_SHORT = 'tp'
class TrafficPoliciesModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "traffic-policies"

   def __init__( self ):
      self.modeKey = "traffic"
      self.longModeKey = "traffic-policies"
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=CliMode.ConfigMode )

class TrafficPoliciesConfigMode( TrafficPoliciesModeBase, BasicCli.ConfigModeBase ):
   """ Configuration mode for traffic-policies """
   name = "traffic-policies"

   def __init__( self, parent, session ):
      TrafficPoliciesModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def abort( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

   def commit( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

class TrafficPoliciesVrfModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f'vrf {self.vrfName}'

   def __init__( self, param ):
      ( self.vrfName, self.parentMode, self.feature, self.vrfConfig,
        self.cpuPoliciesVrfConfig ) = param
      self.modeKey = f'{self.feature}-vrf'
      self.longModeKey = f'{self.feature}-vrf-{self.vrfName}'
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=self.parentMode )

class TrafficPoliciesVrfConfigMode( TrafficPoliciesVrfModeBase,
                                    BasicCli.ConfigModeBase ):
   """ Configuration mode for 'vrf VRF' under traffic-policies """
   name = "traffic-policies-vrf"

   def __init__( self, parent, session, feature, vrfName, vrfConfig,
                 cpuPoliciesVrfConfig=None ):
      TrafficPoliciesVrfModeBase.__init__( self, ( vrfName,
                                                   type( parent ),
                                                   feature,
                                                   vrfConfig,
                                                   cpuPoliciesVrfConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def abort( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

   def commit( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

   def createVrfConfig( self ):
      self.vrfConfig.vrf.add( self.vrfName )

   def vrfJanitor( self ):
      # Used to cleanup when mode is deleted, i.e. 'no vrf VRF'.
      del self.vrfConfig.vrf[ self.vrfName ]
      del self.vrfConfig.vrfConfig.trafficPolicies[ self.vrfName ]
      if self.cpuPoliciesVrfConfig:
         del self.cpuPoliciesVrfConfig.trafficPolicies[ self.vrfName ]

class TrafficPolicyModeBase( CliMode.ConfigMode ):

   @property
   def configPolicyKeyword( self ):
      # This is the config keyword preceding the policy name
      return 'traffic-policy'

   def enterCmd( self ):
      return "%s %s" % ( self.configPolicyKeyword, self.trafficPolicyName )

   def __init__( self, param ):
      ( self.feature, self.parentMode, self.trafficPolicyName ) = param
      # By default, the modeKey is set to the config policy keyword. If the feature
      # needs to use a different keyword, the derived class can override the modeKey
      self.modeKey = self.configPolicyKeyword
      self.longModeKey = "%s-%s" % ( self.feature,
                                     self.trafficPolicyName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=self.parentMode )
      self.trafficPolicyContext = None

   def getContext( self ):
      return self.trafficPolicyContext

   def setContext( self, value ):
      self.trafficPolicyContext = value

class TrafficPolicyConfigModeBase( TrafficPolicyModeBase,
                                   ClassificationConfigModeBase ):
   def __init__( self, parent, session, context, feature ):
      TrafficPolicyModeBase.__init__( self, ( feature,
                                              type( parent ),
                                              context.pmapName() ) )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicyContext = context
      self.trafficPolicy = context.currentPolicy()
      context.modeIs( self )

class TrafficPolicyConfigMode( TrafficPolicyConfigModeBase,
                               ClassificationConfigModeBase ):
   """ Configuration mode for traffic-policy """
   name = "traffic-policy configuration"

class MatchRuleModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      if self.matchType:
         cmd = "match %s %s" % ( self.ruleName, self.matchType )
      else:
         cmd = "match %s" % self.ruleName
      return cmd

   def __init__( self, param ):
      if len( param ) == 6:
         ( self.feature, self.parentMode, self.trafficPolicyName,
           self.ruleName, self.matchType, self.prio ) = param
      else:
         ( self.feature, self.parentMode, self.trafficPolicyName,
           self.ruleName, self.matchType ) = param
      self.modeKey = "match-%s" % self.matchType
      self.longModeKey = "%s-match-%s-%s-%s" % ( self.feature,
                                                 self.trafficPolicyName,
                                                 self.ruleName, self.matchType )
      CliMode.ConfigMode.__init__( self, param, parentMode=self.parentMode )

class MatchRuleBaseConfigMode( MatchRuleModeBase, ClassificationConfigModeBase ):
   """ Base configuration mode for match rule"""
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 feature ):
      param = ( feature, type( parent ),
                trafficPolicyContext.pmapName_, matchRuleContext.ruleName,
                matchRuleContext.matchType )
      MatchRuleModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = matchRuleContext
      self.feature = feature

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def getContext( self ):
      """ Used to allow each config mode to name their own context """
      return self.context

   def setContext( self, value ):
      self.context = value

   def abort( self ):
      if self.context.matchRuleAction is not None:
         self.context.matchRuleAction.abort()
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()
      if self.trafficPolicyContext.shouldResequence:
         self.trafficPolicyContext.resequence( ClassPriorityMin, ClassPriorityInc )
         lastseq = self.trafficPolicyContext.lastSequence()
         context.seqnum = self.trafficPolicyContext.lookupRuleToSeq(
                     context.matchRuleAction )

         # Ensure we've restored the sequence numbers to even intervals of 10.
         assert context.seqnum % ClassPriorityInc == 0

         # We must commit the context a second time to preserve the changes from
         # resequencing.
         context.commit()
         # context.commit() overwrites lastseq with the rule we're committing,
         # so we have to reset it again here.
         self.trafficPolicyContext.lastSequenceIs( lastseq )

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

class GlobalMatchRuleConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule with no matchType """
   name = "match rule configuration"
   def __init__( self, parent, session, trafficPolicyContext,
                 matchRuleContext, feature ):
      MatchRuleBaseConfigMode.__init__( self, parent, session,
                                        trafficPolicyContext,
                                        matchRuleContext,
                                        feature )
      self.modeKey = "match-%s" % self.ruleName
      self.longModeKey = "%s-match-%s" % ( self.feature,
                                           self.ruleName )

class PacketTypeModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "packet type"

   def __init__( self, param ):
      ( self.feature, self.parentMode, self.trafficPolicyName,
            self.ruleName ) = param
      self.modeKey = "match-packet-type"
      self.longModeKey = "%s-%s-packet-type" % ( self.feature, self.ruleName )
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=self.parentMode )

   def commentKey( self ):
      # 'tp-<matchRuleName>-packet-type'
      return "%s-%s-%s-packet-type" % ( self.feature, self.trafficPolicyName,
                                        self.ruleName )

class PacketTypeConfigMode( PacketTypeModeBase, ClassificationConfigModeBase ):
   """ Configuration mode for packet type """
   name = "packet type"

   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 packetTypeContext, feature ):
      param = ( feature, type( parent ), trafficPolicyContext.pmapName_,
                matchRuleContext.ruleName )
      PacketTypeModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicyContext = trafficPolicyContext
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.matchRuleContext = matchRuleContext
      self.context = packetTypeContext
      self.feature = feature

   def getContext( self ):
      """ Used to allow each config mode to name their own context """
      return self.context

   def setContext( self, value ):
      self.context = value

   def abort( self ):
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

class EncapInnerModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "encapsulation %s %s" % ( self.encapType, self.innerMatchType )

   def __init__( self, param ):
      ( self.feature, self.parentMode, self.trafficPolicyName,
        self.ruleName, self.encapType, self.innerMatchType ) = param
      self.modeKey = "match-%s-%s" % ( self.encapType, self.innerMatchType )
      self.longModeKey = "%s-%s-%s-%s" % ( self.feature, self.ruleName,
                                           self.encapType, self.innerMatchType )
      CliMode.ConfigMode.__init__( self, param, parentMode=self.parentMode )

   def commentKey( self ):
      # 'tp-<matchRuleName>-<encapType>-<innerMatchType>'
      return "%s-%s-%s-%s-%s" % ( self.feature, self.trafficPolicyName,
                                  self.ruleName, self.encapType,
                                  self.innerMatchType )

class EncapInnerMatchConfigMode( EncapInnerModeBase, ClassificationConfigModeBase ):
   """ Base configuration mode for inner match """
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 innerRuleContext, feature ):
      param = ( feature, type( parent ),
                trafficPolicyContext.pmapName_, matchRuleContext.ruleName,
                innerRuleContext.encapType, innerRuleContext.matchType )
      EncapInnerModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.matchRuleContext = matchRuleContext
      self.context = innerRuleContext
      self.feature = feature

   def getContext( self ):
      """ Used to allow each config mode to name their own context """
      return self.context

   def setContext( self, value ):
      self.context = value

   def abort( self ):
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

class EncapInnerMatchIpv4ConfigMode( EncapInnerMatchConfigMode ):
   """ Configuration mode for inner match ipv4 """
   name = "encapsulation mode for ipv4"

class EncapInnerMatchIpv6ConfigMode( EncapInnerMatchConfigMode ):
   """ Configuration mode for inner match ipv6 """
   name = "encapsulation mode for ipv6"

class EncapInnerMatchMacConfigMode( EncapInnerMatchConfigMode ):
   """ Configuration mode for inner match mac """
   name = "encapsulation mode for mac"

class EncapInnerVxlanMatchIpv4ConfigMode( EncapInnerMatchConfigMode ):
   """ Configuration mode for vxlan inner match ipv4 """
   name = "encapsulation mode for ipv4"

class EncapDzGreConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'encapsulation dzgre'

   def __init__( self, param ):
      ( self.feature, self.parentMode, self.ruleName ) = param
      self.modeKey = "match-dzgre"
      self.longModeKey = "%s-%s-dzgre" % ( self.feature, self.ruleName )
      CliMode.ConfigMode.__init__( self, param, parentMode=self.parentMode )

class EncapDzGreConfigMode( EncapDzGreConfigModeBase, ClassificationConfigModeBase ):
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 feature ):
      param = ( feature, type( parent ), matchRuleContext.ruleName )
      EncapDzGreConfigModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = matchRuleContext
      self.feature = feature

   def getContext( self ):
      return self.context

   def setContext( self, value ):
      self.context = value

class MatchRuleDefaultConfigMode( MatchRuleBaseConfigMode ):
   """
   Configuration mode for match rule ipv{4,6}-all-default ipv{4,6}. Note we only need
   one as all actions are shared between IPV4 and IPV6 match statements.
   """
   name = "default match rule configuration"

class MatchRuleIpv4ConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule ipv4"""
   name = "match rule configuration ipv4"

class MatchRuleIpv6ConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule ipv6 """
   name = "match rule configuration ipv6"

class MatchRuleMacConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule mac """
   name = "match rule configuration mac"

class ReplicateModeBase( CliMode.ConfigMode ):

   def __init__( self, param ):
      self.feature, self.trafficPolicyName, self.ruleName, \
            self.actionsSetShortName, pMode = param
      self.modeKey = 'replicate'
      self.session_ = None
      self.context = None
      self.longModeKey = f'{ self.feature }-actions-{ self.actionsSetShortName }-'\
                         f'{ self.trafficPolicyName }-{ self.ruleName }'
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=pMode )

   def enterCmd( self ):
      return f'replicate { self.actionsSetShortName }'

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

   def abort( self ):
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

class ReplicateMode( ReplicateModeBase, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, context, feature ):
      param = ( feature, context.trafficPolicyName, context.ruleName,
                context.actionsSetShortName, parent )
      ReplicateModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.context = context
      self.trafficPolicy = context.trafficPolicy

class ActionsModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      cmd = "actions"
      return cmd

   def __init__( self, param ):
      self.feature, self.trafficPolicyName, self.ruleName, pMode = param
      self.modeKey = "actions"
      self.longModeKey = "%s-actions-%s-%s" % ( self.feature,
                                                self.trafficPolicyName,
                                                self.ruleName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=pMode )
      self.session_ = None
      self.context = None

   def commit( self ):
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

   def abort( self ):
      if self.context.matchRuleAction is not None:
         self.context.matchRuleAction.abort()
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

# Anonymous action mode used for applications single traffic-policy
# configuration
class AnonActionModeBase( ActionsModeBase ):
   def __init__( self, param ):
      ActionsModeBase.__init__( self, param )
      self.longModeKey = "%s-actions-%s" % ( self.feature,
                                             self.ruleName )

class ActionsConfigModeBase( ActionsModeBase, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 feature ):
      param = ( feature, trafficPolicyContext.pmapName_,
                matchRuleContext.ruleName, TrafficPolicyConfigMode )
      ActionsModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = matchRuleContext
      self.feature = feature

class ActionsConfigMode( ActionsConfigModeBase, BasicCli.ConfigModeBase ):
   """ Configuration mode for actions under IPv4 and IPv6 sub-prompt """
   name = "actions configuration"

class MacActionsConfigMode( ActionsConfigModeBase, BasicCli.ConfigModeBase ):
   """ Configuration mode for actions under MAC match sub-prompt """
   name = "MAC actions configuration"

class DefaultActionModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "actions %s default" % self.matchType

   def __init__( self, param ):
      ( self.feature, self.parentMode, self.trafficPolicyName,
        self.matchType ) = param
      self.modeKey = "default-act-%s" % self.matchType
      self.longModeKey = "%s-default-act-%s-%s" % ( self.feature,
                                                    self.trafficPolicyName,
                                                    self.matchType, )
      CliMode.ConfigMode.__init__( self, param, parentMode=self.parentMode )

class DefaultActionConfigModeBase( DefaultActionModeBase,
                                   ClassificationConfigModeBase ):
   def __init__( self, parent, session, trafficPolicyContext,
                 defaultActionsContext, feature ):
      param = ( feature, type( parent ),
                trafficPolicyContext.pmapName_,
                defaultActionsContext.matchType )
      DefaultActionModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = defaultActionsContext

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def getContext( self ):
      """ Used to allow each config mode to name their own context """
      return self.context

   def setContext( self, value ):
      self.context = value

   def abort( self ):
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent mode

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent mode

class FieldSetL4PortConfigMode( FieldSetBaseL4PortConfigMode ):
   name = "traffic-policy field-set l4-port configuration"

class FieldSetIpPrefixConfigMode( FieldSetBasePrefixConfigMode ):
   name = "traffic-policy field-set ipv4 prefix configuration"

class FieldSetIpv6PrefixConfigMode( FieldSetBasePrefixConfigMode ):
   name = "traffic-policy field-set ipv6 prefix configuration"

class FieldSetMacAddrConfigMode( FieldSetBaseMacAddrConfigMode ):
   name = "traffic-policy field-set mac configuration"

class FieldSetServiceConfigMode( FieldSetBaseServiceConfigMode ):
   name = "traffic-policy field-set service configuration"
