# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode import ConfigMode

# config
#   router vpls
class RouterVplsMode( ConfigMode ):
   def __init__( self ):
      self.modeKey = 'router-vpls'
      self.longModeKey = self.modeKey
      ConfigMode.__init__( self, None )

   def enterCmd( self ):
      return 'router vpls'

# config
#   router vpls
#     vpls <instance-name>
class VplsInstanceMode( ConfigMode ):
   def __init__( self, vplsName ):
      self.vplsName = vplsName
      self.modeKey = f'router-vpls-{vplsName}'
      self.longModeKey = self.modeKey
      ConfigMode.__init__( self, vplsName,
                                   parentMode=RouterVplsMode )

   def enterCmd( self ):
      return f'vpls {self.vplsName}'

# config
#   router vpls
#     vpls <instance-name>
#        vlan ( <id> | ( bundle <id-list> ) )
class VplsVlanOrBundleMode( ConfigMode ):
   def __init__( self, vplsName, bundle, vlanId ):
      self.vplsName = vplsName
      self.vlanId = vlanId
      self.bundle = bundle
      # pylint: disable-next=consider-using-f-string
      self.modeKey = 'router-vpls-{vplsName}-vlan{bundle}'.format(
                           vplsName=vplsName,
                           bundle='-bundle' if bundle else '' )
      self.longModeKey = self.modeKey
      ConfigMode.__init__( self, vlanId, parentMode=VplsInstanceMode )

   def enterCmd( self ):
      # pylint: disable-next=consider-using-f-string
      return 'vlan {bundle}{vlanId}'.format( bundle='bundle ' if self.bundle else '',
                                              vlanId=self.vlanId )

   def commentKey( self ):
      # CLI comment requires each mode to have a unique key
      return self.longModeKey + "-" + str( self.vlanId )

# config
#   router vpls
#     vpls <instance-name>
#        pseudowire ( bgp | ldp ) <group-name> ( split-horizon | spokes )
class VplsGroupMode( ConfigMode ):
   def __init__( self, vplsName, groupName, signalingProtocol, splitHorizon ):
      self.vplsName = vplsName
      self.groupName = groupName
      self.signalingProtocol = signalingProtocol
      self.splitHorizon = splitHorizon
      self.modeKey = f'router-vpls-{vplsName}-group-{groupName}'
      self.longModeKey = self.modeKey
      ConfigMode.__init__( self, groupName, parentMode=VplsInstanceMode )

   def enterCmd( self ):
      splitHorizon = 'split-horizon' if self.splitHorizon else 'spokes'
      return f'pseudowire {self.signalingProtocol} {self.groupName} {splitHorizon}'
