#!/usr/bin/env python3
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: specline %{python3_sitelib}/CliMode/__init__.py

class ConfigMode:
   """ For detailed explanation and document about this class, see
   Aid/987. Each CLI configuration sub mode created in a package,
   should multi-inherit from this class, to define the 'modeKey' and
   'longModeKey' members, and also the enterCmd function which are all
   shared between the corresponding CliPlugin and CliSavePlugin
   modes. """
   def __init__( self, param, parentMode=None ):
      """ Basically our design is that, all the config sub mode
      (children of BasicCli.ConfigModeBase) and the config save sub
      mode (children of CliSave.Mode) have to multi-inherit (as the
      first parent) from a CliMode.ConfigMode sub class defined in its
      package's CliMode/. In the __init__ of the config sub mode and
      config save sub mode, they need to call the init function of
      these CliMode objects to create them, by providing the param. To
      make it consistent, the param should be the same as the one used
      in a config save sub mode. Usually it is also the identifier of
      a config sub mode, such as "ethernet 1" for an interface mode,
      or the id "1" for an ospf mode. Sometimes there could be slight
      difference, such as in the Vlan mode, the param is a vector of
      (vlanStart, vlanString, enabled), which is consistent with the
      Vlan save sub mode. Please refer to package Bgp, as an example
      of how such inheritance and constructing is done, for both the
      config sub mode and config save sub mode."""
      self.param_ = param
      # Mode subclasses must have 'modeKey' and 'longModeKey'
      # attributes defined. 'modeKey' is used for generating short
      # prompt by ConfigModeBase, and 'longModeKey' is used for
      # generating long prompt by ConfigModeBase and for
      # adding/finding/removing the comment for this mode.
      assert hasattr( self, 'modeKey' )
      assert hasattr( self, 'longModeKey' )

   def enterCmd( self ):
      """ Subclasses should override this method to return the command
      to be used to enter this CLI mode instance."""
      raise NotImplementedError

   def revertCmd( self ):
      """ Subclasses can override this method to return the command
      to be used to revert the CLI mode instance.  By default, the
      'default <enterCmd> is used."""
      return f'default {self.enterCmd()}'

   def commentKey( self ):
      # Key for storing comments for the mode.
      #
      # This can be overwritten by modes. Note the key has to uniquely identify
      # the particular mode instance, including which keys of its parent modes.
      return self.longModeKey # pylint: disable=no-member
