# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
import collections.abc
import copy
import functools
import json
import sys

import CliCommon
import Tac
import Tracing

INFINITY = float( "inf" )
NEGATIVE_INFINITY = float( "-inf" )

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3
__defaultTraceHandle__ = Tracing.Handle( "CliModel" )

allowedBuiltInTypes = frozenset( ( int, float, str, bool ) )

__filtered_attr_cache__ = {}

TYPEADAPTERS = {}

class _COMPAT:
   pass

# NOTE: in several places code is intentionally duplicated reduce method calls

class _ValueTypeShim:
   def __init__( self, clsdefault=None ):
      self.clsdefault = clsdefault

   def __get__( self, obj, objtype=None ):
      # XXX compat CliDynamicPlugin/PciCliHandlers.py +267
      if objtype and obj is None:
         if self.clsdefault:
            if isinstance( self.clsdefault, str ):
               return globals()[ self.clsdefault ]
            return self.clsdefault
      return obj.__field_type__

class _AttributeType:
   __slots__ = (
      'help',
      'optional',
      'sinceRevision',
      '_default',
      'immutable',
   )

   __field_type__ = object

   def __init__( self,
                 help='',  # pylint: disable=redefined-builtin
                 *,
                 optional=False,
                 default=None,
                 sinceRevision=1 ):

      assert help, "No help string given"

      # tidy up help text
      help = help.strip()
      if help[ -1 ] not in '.?!;':
         help += '.'
      self.help = help

      self.optional = optional
      self.immutable = self.__field_type__ in allowedBuiltInTypes
      self.default = default
      self.sinceRevision = sinceRevision

   def __set_name__( self, owner, name ):
      if name.startswith( '__' ) or name in { 'errors', 'warnings', '_meta' }:
         raise ValueError(
            f"Class {owner!r} is not allowed to name an attribute {name!r}"
         )

      if name in owner.__attributes__:
         raise AssertionError(
            f"Cannot override attribute {name!r} in class "
            f"{self.__class__.__name__!r}, attribute already defined "
            f"as {owner.__attributes__[name]!r}"
      )

      owner.__attributes__[ name ] = self
      if name.startswith( '_' ):
         owner.__private__ += ( self, )

   def __get__( self, obj, objtype=None ):
      if objtype and obj is None:
         # caller is accessing the unbound descriptor from the class instance
         return self

      if self in obj.__values__:
         return obj.__values__[ self ]

      value = self.default
      if not self.immutable:
         obj.__values__[ self ] = value
      return value

   def __set__( self, obj, value ):
      if not ( isinstance( value, self.__field_type__ ) or
                  ( value is None and self.optional ) or
                  self in obj.__private__
            ):

         # XXX compat for existing CliModels that incorrectly initialize to None
         if value is None and self._default is None and self not in obj.__values__:
            return

         raise TypeError(
            f"Cannot set {obj.__nameforattr__(self)!r} to argument of "
            f"type {type(value)}, expected a {self.__field_type__} instead, "
            f"with value {value!r}"
         )
      obj.__values__[ self ] = value

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                     ignoreMissingAttrs=False ):
      self.__set__( obj, data )

   def __delete__( self, obj ):
      obj.__values__.pop( self, None )

   realType = _ValueTypeShim( clsdefault=object )

   @classmethod
   @property
   def real2Type( cls ):
      "compatibility shim other tests look at this value"
      return type( None ) if cls.__field_type__ is object else cls.__field_type__

   @property
   def default( self ):
      if self.__field_type__ is not object and self._default is False:
         return self.__field_type__()
      if callable( self._default ):
         return self._default()
      return self._default

   @default.setter
   def default( self, value ):
      if ( isinstance( value, self.__field_type__ ) or
              value in ( None, False ) or
              ( callable( value ) and isinstance( value(), self.__field_type__ ) )
            ):

         self._default = value
      else:
         raise TypeError(
             f"Invalid default value of type {type(value)}, "
             f"expected {self.__field_type__}"
         )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      return obj


class Int( _AttributeType ):
   """Equivalent to a Python integer."""
   __field_type__ = int
   __slots__ = ()

def strtoint( value ):
   # more compat for magic str to int key in dict
   if isinstance( value, str ) and value.isdigit():
      return int( value )
   return value

TYPEADAPTERS[ int ] = strtoint

def sanitizeFloat( value ):
   if isinstance( value, float ):
      pass
   elif isinstance( value, int ):
      value = float( value )
   else:
      return value

   # We can't convert NaN and +-infinity to JSON
   # pylint: disable-next=comparison-with-itself
   if value == INFINITY or value != value:
      return sys.float_info.max
   elif value == NEGATIVE_INFINITY:
      return -sys.float_info.max
   return value

TYPEADAPTERS[ float ] = sanitizeFloat


class Float( _AttributeType ):
   """Equivalent to a Python floating point value."""
   __field_type__ = float
   __slots__ = ()

   def __set__( self, obj, value ):
      value = sanitizeFloat( value )
      if isinstance( value, float ):
         obj.__values__[ self ] = value
      else:
         super().__set__( obj, value )

   @property
   def default( self ):
      return _AttributeType.default.__get__( self )

   @default.setter
   def default( self, value ):
      value = sanitizeFloat( value )
      _AttributeType.default.__set__( self, value )


class Str( _AttributeType ):
   """Equivalent to a Python string. """
   __field_type__ = str
   __slots__ = ()


class Bool( _AttributeType ):
   """Equivalent to a Python boolean."""
   __field_type__ = bool
   __slots__ = ()


class Enum( _AttributeType ):
   """A string attribute that can only take on certain values."""
   __field_type__ = str
   __slots__ = (
      'values',
   )

   def __init__( self, values, **kwargs ):
      super().__init__( **kwargs )

      if isinstance( values, str ):
         raise TypeError(
            "Invalid type for 'values'. Expected a non-string iterable "
         )
      try:
         self.values = frozenset( values )
      except TypeError:
         raise TypeError(
            f"Invalid type for `values`. Expected an iterable got {values!r}"
         ) from None
      if badValues:= [ a for a in self.values if not isinstance( a, str ) ]:
         raise ValueError(
            f'Invalid value type. Expected a str; got {badValues!r}'
         )

   def __set__( self, obj, value ):
      if isinstance( value, self.__field_type__ ) and value not in self.values:
         raise ValueError(
            f'Invalid value {value!r} expected one of {self.values!r}'
         )
      _AttributeType.__set__( self, obj, value )


class TacTypeAttribute( _AttributeType ):
   """An attribute backed by a Tac Type """
   __slots__ = (
      '__field_type__',
   )

   def __init__( self, *, tacType=None, **kwargs ):
      if tacType:
         self.__field_type__ = tacType
      super().__init__( **kwargs )

   def __set__( self, obj, value ):
      if hasattr( self, '__type_adapter__' ):
         value = self.__type_adapter__( value )
      _AttributeType.__set__( self, obj, value )

   @property
   def default( self ):
      return _AttributeType.default.__get__( self )

   @default.setter
   def default( self, value ):
      if hasattr( self, '__type_adapter__' ):
         value = self.__type_adapter__( value )
      _AttributeType.default.__set__( self, value )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if obj is None:
         return None
      return obj.stringValue

TACTYPETOMODELMAP = {}

class _TacAttributeType( TacTypeAttribute ):
   """shim to make existing code work until converted

   Subclasses need to implement _createTacValue() so that the TAC value can be
   created from a string.  The TAC value is also expected to have a stringValue
   attribute to allow itself to be converted to a string.

   For instance, with MAC addresses:
     >> mac = Tac.Value( "Arnet::EthAddr" )
     >> mac
     Value('Arnet::EthAddr', ** {'word1': 0, 'word0': 0, 'word2': 0})
     >> mac.stringValue
     '00:00:00:00:00:00'
     >> mac.stringValue = "01:02:03:04:05:06"  # This is _createTacValue()
     >> mac.stringValue
     '01:02:03:04:05:06'
     >> mac
     Value('Arnet::EthAddr', ** {'word1': 772, 'word0': 258, 'word2': 1286})
   """

   tacType = None.__class__

   def __init__( self, help=None, **kwargs ):  # pylint: disable=redefined-builtin
      super().__init__( help=help, tacType=self.tacType, **kwargs )

   @classmethod
   def __type_adapter__( cls, value ):
      if not isinstance( value, str ):
         return value

      try:
         if hasattr( cls, '_createTacValue' ):
            return cls._createTacValue( cls, value )
         else:
            return cls.__field_type__( value )
      except ( IndexError, TypeError ) as e:
         raise ValueError( f'Invalid value: {value!r}' ) from e

   @classmethod
   def __key_type_adapter__( cls, value ):
      if not isinstance( value, ( cls.tacType, str ) ):
         return value

      if isinstance( value, str ):
         try:
            if hasattr( cls, '_createTacValue' ):
               value = cls._createTacValue( cls, value )
            else:
               value = cls.__field_type__( value )
         except ( IndexError, TypeError ) as e:
            raise ValueError( f'Invalid key: {value!r}' ) from e

      return value.stringValue

   @classmethod
   def __coll_validate__( cls, value ):
      if isinstance( value, str ):
         try:
            if hasattr( cls, '_createTacValue' ):
               cls._createTacValue( cls, value )
            else:
               cls.__field_type__( value )
         except ( IndexError, TypeError ) as e:
            raise ValueError( f'Invalid value: {value!r}' ) from e
      return value

   def __init_subclass__( cls ):
      super().__init_subclass__()
      if cls.tacType is not None.__class__:
         TYPEADAPTERS[ ( cls.tacType, str ) ] = cls.__key_type_adapter__
         TYPEADAPTERS[ cls.tacType ] = cls.__type_adapter__
         # compatibility for current colllections which do not normalize
         TYPEADAPTERS[ ( cls.tacType, str, _COMPAT ) ] = cls.__coll_validate__
         if cls.tacType not in TACTYPETOMODELMAP:
            TACTYPETOMODELMAP[ cls.tacType ] = cls

tacAttributeTypeSubclasses = _TacAttributeType.__subclasses__

class _TypedList( list ):
   __slots__ = ( '__value_type__', )

   def __init__( self, iterable=(), *, __value_type__=None ):
      self.__value_type__ = __value_type__

      if ( iterable and isinstance( iterable, _TypedList ) and
            iterable.__value_type__ == self.__value_type__ ):
         # most efficient for memory allocation
         list.__init__( self, iterable )
      else:
         list.__init__( self )
         self.extend( iterable )

   @staticmethod
   def _checkOne( value, valueType=object ):

      adapter = TYPEADAPTERS.get( valueType )

      if adapter:
         value = adapter( value )
      if isinstance( value, valueType ):
         pass
      else:
         msg = f'Type {type(value)} does not match2 list type {valueType!r}'
         raise TypeError( msg )
      return value

   @staticmethod
   # pylint: disable-next=inconsistent-return-statements
   def _checkMany( iterable, valueType=object ):
       # ( list,tuple, Iterable) is an optmization for interpreter behavior.
       # Types are checked L to R and list/tuple are pointer comparision vs
       # abc.Iterable which is a catch-all that requires a slower python
       # call to Iterable's __instancecheck__() method.
      if not isinstance( iterable, ( list, tuple, collections.abc.Iterable ) ):
         return iterable

      adapter = TYPEADAPTERS.get( valueType )

      for i in iterable:
         if adapter:
            i = adapter( i )
         if not isinstance( i, valueType ):
            msg = f'Type {type(i)} does not match3 list type {valueType!r}'
            raise TypeError( msg )
         yield i

   def append( self, item ):
      list.append( self, self._checkOne( item, self.__value_type__ ) )

   def extend( self, iterable ):
      if ( not isinstance( iterable, _TypedList ) or
            iterable.__value_type__ != self.__value_type__ ):
         iterable = self._checkMany( iterable, self.__value_type__ )
      list.extend( self, iterable )

   def __iadd__( self, other ):
      if ( not isinstance( other, _TypedList ) or
            other.__value_type__ != self.__value_type__ ):
         other = self._checkMany( other, self.__value_type__ )
      return list.__iadd__( self, other )

   def __add__( self, other ):
      retval = self.__class__( self, __value_type__=self.__value_type__ )
      retval.extend( other )
      return retval

   def __setitem__( self, key, item ):
      if isinstance( key, slice ):
         if ( not isinstance( item, _TypedList ) or
               item.__value_type__ != self.__value_type__ ):
            item = self._checkMany( item, self.__value_type__ )
      else:
         item = self._checkOne( item, self.__value_type__ )

      list.__setitem__( self, key, item )

   def __getitem__( self, key ):
      if isinstance( key, slice ):
         return self.__class__(
            list.__getitem__( self, key ),
            __value_type__=self.__value_type__
         )
      else:
         return list.__getitem__( self, key )

   def insert( self, i, item ):
      list.insert( self, i, self._checkOne( item, self.__value_type__ ) )

   def copy( self ):
      return self.__class__( self, __value_type__=self.__value_type__ )

class _TypedDict( dict ):
   __slots__ = (
      '__key_type__',
      '__value_type__',
      '__value_optional__',
      '__type_info__',
      '__key_adapter__',
      '__value_adapter__',
   )

   def __init__( self, initial=(), *,
                 __key_type__=str,
                 __value_type__=None,
                 __value_optional__=True,
                 **kwargs ):
      self.__key_type__ = __key_type__
      self.__value_type__ = __value_type__
      self.__value_optional__ = __value_optional__
      self.__key_adapter__ = TYPEADAPTERS.get( __key_type__ )
      self.__value_adapter__ = TYPEADAPTERS.get( __value_type__ )
      self.__type_info__ = ( __key_type__, __value_type__, __value_optional__ )
      if isinstance( initial, ( dict, collections.abc.Mapping ) ):
         initial = self._checkMapping( initial, *self.__type_info__ )
      elif hasattr( initial, 'items' ):
         # XXX Tac.Collection should be a abc.Mapping
         initial = self._checkIterable( initial.items(), *self.__type_info__ )
      elif initial:
         initial = self._checkIterable( initial, *self.__type_info__ )

      dict.__init__( self, initial )
      if kwargs:
         dict.update( self, self._checkMapping( kwargs, *self.__type_info__ ) )

   @staticmethod
   def _checkMapping( mapping, keyType=str, valueType=None, valueOptional=True ):
      if ( isinstance( mapping, _TypedDict ) and
            keyType == mapping.__key_type__ and
            valueType == mapping.__value_type__ and
            valueOptional <= mapping.__value_optional__ ):
         return mapping
      elif not mapping:
         return {}
      return _TypedDict._checkIterable(
         ( ( k, v ) for k, v in mapping.items() ),
         keyType,
         valueType,
         valueOptional
      )

   @staticmethod
   def _checkIterable( iterable, keyType=str, valueType=None, valueOptional=True ):
      keyAdapter = TYPEADAPTERS.get( keyType )
      valueAdapter = TYPEADAPTERS.get( valueType )
      for k, v in iterable:
         if keyAdapter:
            k = keyAdapter( k )
         if valueAdapter:
            v = valueAdapter( v )
         if not isinstance( k, keyType ):
            raise TypeError(
               f'Invalid key: {k!r} is not the expected type ({keyType})'
            )
         if not ( isinstance( v, valueType ) or ( v is None and valueOptional ) ):
            raise TypeError(
               f'Invalid value: {v!r} is not the expected type ({valueType})'
            )

         yield k, v

   def __getitem__( self, key ):
      if self.__key_adapter__:
         key = self.__key_adapter__( key )
      if not isinstance( key, self.__key_type__ ):
         raise TypeError(
            f'The key {key!r} is not the expected type ({self.__key_type__}) '
         )
      return dict.__getitem__( self, key )

   def __setitem__( self, key, value ):
      if self.__key_adapter__:
         key = self.__key_adapter__( key )
      if not isinstance( key, self.__key_type__ ):
         raise TypeError(
            f'The key {key!r} is not the expected type ({self.__key_type__}) '
         )

      if self.__value_adapter__:
         value = self.__value_adapter__( value )
      if not ( isinstance( value, self.__value_type__ ) or
                ( value is None and self.__value_optional__ )
            ):
         raise TypeError(
            f'The value {value!r} is not the expected type ({self.__value_type__}) '
         )
      dict.__setitem__( self, key, value )

   def get( self, key, default=None ):
      try:
         return self.__getitem__( key )
      except ( KeyError, TypeError, ValueError ):
         return default

   def setdefault( self, key, default=None ):
      if self.__key_adapter__:
         key = self.__key_adapter__( key )
      try:
         return self.__getitem__( key )
      except KeyError:
         self.__setitem__( key, default )
         return default

   def update( self, other, **kwargs ):
      if isinstance( other, ( dict, collections.abc.Mapping ) ):
         other = self._checkMapping( other, *self.__type_info__ )
      elif hasattr( other, 'items' ):
         # XXX Tac.Collection should be a abc.Mapping
         other = self._checkIterable( other.items(), *self.__type_info__ )
      elif hasattr( other, '__iter__' ):
         other = self._checkIterable( other, *self.__type_info__ )
      if kwargs:
         kwargs = dict( self._checkMapping( kwargs, *self.__type_info__ ) )
      dict.update( self, other, **kwargs )

   def __contains__( self, key ):
      if self.__key_adapter__:
         try:
            key = self.__key_adapter__( key )
         except ( ValueError, TypeError ):
            return False
      return dict.__contains__( self, key )

   __ior__ = update

   def __or__( self, other ):
      retval = self.__class__(
         self,
         __key_type__=self.__key_type__,
         __value_type__=self.__value_type__,
         __value_optional__=self.__value_optional__
      )
      retval.update( other )
      return retval

   def copy( self ):
      return self.__class__(
         self,
         __key_type__=self.__key_type__,
         __value_type__=self.__value_type__,
         __value_optional__=self.__value_optional__
      )

class List( _AttributeType ):
   __field_type__ = _TypedList
   __slots__ = (
      '__value_type__',
   )

   # future: valueType should be positional or kwarg only
   # otherwise can confused as help arg
   def __init__( self, valueType=None, **kwargs ):
      if not ( valueType in allowedBuiltInTypes or
                  ( isinstance( valueType, type ) and
                    issubclass( valueType, ( Model, _TacAttributeType ) ) )
            ):
         raise TypeError(
            f'Invalid value type {type(valueType)!r}.'
            f'Value must be Model or one of {allowedBuiltInTypes!r}'
         )
      if issubclass( valueType, _TacAttributeType ):
         valueType = ( valueType.tacType, str, _COMPAT )
      self.__value_type__ = valueType

      super().__init__( **kwargs )

      if 'default' not in kwargs:
         self._default = list

   def __set__( self, obj, value ):
       # ( list,tuple, Iterable) is an optmization for interpreter behavior.
       # Types are checked L to R and list/tuple are pointer comparision vs
       # abc.Iterable which is a catch-all that requires a slower python
       # call to Iterable's __instancecheck__() method.
      if isinstance( value, ( list, tuple, collections.abc.Iterable ) ):
         value = _TypedList( value, __value_type__=self.__value_type__ )
      super().__set__( obj, value )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                    ignoreMissingAttrs=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         data = (
            self.__value_type__.fromDict(
               i,
               ignoreUnknownAttrs,
               ignoreMissingAttrs
            )
            for i in data
         )

      self.__set__( obj, data )

   # XXX compat shim until forward ref support is refactored
   # Example: /usr/lib/python3.9/site-packages/RibCapiLib.py +392
   @property
   def valueType( self ):
      if isinstance( self.__value_type__, tuple ):
         # XXX compat for CliModelRevisionTest.py +322
         return TACTYPETOMODELMAP.get( self.__value_type__[ 0 ],
                                       self.__value_type__ )
      return self.__value_type__

   @property
   def default( self ):
      return _TypedList(
         self._default or (),
         __value_type__=self.__value_type__
      )

   @default.setter
   def default( self, value ):
      if value:
         if not isinstance( value, ( tuple, list, _TypedList ) ):
            raise TypeError(
               f"Invalid default value of type {type(value)}, expected list"
            )
         try:
            # pylint: disable-next=protected-access
            all( True for i in _TypedList._checkMany( value, self.__value_type__ ) )
         except TypeError as e:
            raise TypeError( f"Invalid value type {value!r}" ) from e
      self._default = value

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if obj is None:
         return obj
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         return [
            i.toDict( revision, includePrivateAttrs, streaming ) for i in obj
         ]

      # XXX special case Tac.Type
      if isinstance( self.__value_type__, tuple ) and _COMPAT in self.__value_type__:
         # NOTE: quirk exist for some Tac. Values where str(v) != v.stringValue
         return [ v if isinstance( v, str ) else v.stringValue for v in obj ]

      # XXX some btests expect only py list here
      return list( obj )

class Dict( _AttributeType ):
   __field_type__ = _TypedDict
   __slots__ = (
      '__key_type__',
      '__value_type__',
      '__value_optional__',
   )

   def __init__( self, *,
                 keyType=str,
                 valueType=None,
                 valueOptional=False,
                 **kwargs ):
      if not issubclass( keyType, ( str, int, _TacAttributeType ) ):
         raise TypeError( f'Keys must be "str" or "int" not {keyType!r}' )
      if not ( ( isinstance( valueType, type ) and
                    issubclass( valueType, ( Model, _TacAttributeType ) ) ) or
                  valueType in allowedBuiltInTypes
            ):

         raise TypeError(
            f'Invalid value type {type(valueType)!r}.'
            f'Value must be Model or one of {allowedBuiltInTypes!r}'
         )

      # XXX shim to support existiing code
      if issubclass( keyType, _TacAttributeType ):
         keyType = ( keyType.tacType, str )
      if issubclass( valueType, _TacAttributeType ):
         valueType = ( valueType.tacType, str, _COMPAT )

      self.__key_type__ = keyType
      self.__value_type__ = valueType
      self.__value_optional__ = valueOptional

      super().__init__( **kwargs )

      # fallback to empty
      if 'default' not in kwargs:
         self._default = dict

   def __set__( self, obj, value ):
      if isinstance( value, ( dict, collections.abc.Mapping ) ):
         value = _TypedDict(
            value,
            __key_type__=self.__key_type__,
            __value_type__=self.__value_type__,
            __value_optional__=self.__value_optional__,
         )
      super().__set__( obj, value )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                    ignoreMissingAttrs=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         # XXX modelchecker requires the adapter here
         # follow-up remove this in favor of the one in Dict
         keyAdapter = TYPEADAPTERS.get( self.__key_type__, self.__key_type__ )
         data = (
            ( keyAdapter( k ),
              self.__value_type__.fromDict(
                  v,
                  ignoreUnknownAttrs,
                  ignoreMissingAttrs
              )
            ) for k, v in data.items()
         )
      # intentional non-idomatic
      elif self.__key_type__ is int:
         data = ( ( int( k ), v ) for k, v in data.items() )

      data = _TypedDict(
         data,
         __key_type__=self.__key_type__,
         __value_type__=self.__value_type__,
         __value_optional__=self.__value_optional__,
      )
      super().__set__( obj, data )

   # XXX compat for CliModelRevisionTest.py +359
   @property
   def keyType( self ):
      if isinstance( self.__key_type__, tuple ):
         return TACTYPETOMODELMAP.get( self.__key_type__[ 0 ], self.__key_type__ )
      return self.__key_type__

   # XXX compat shim until forward ref support is refactored
   # Example: /src/AgentLib/CliPlugin/ShowTaskSchedulerModel.py +303
   @property
   def valueType( self ):
      if isinstance( self.__value_type__, tuple ):
         return TACTYPETOMODELMAP.get( self.__value_type__[ 0 ],
                                       self.__value_type__ )
      return self.__value_type__

   @valueType.setter
   def valueType( self, valueType ):
      if not ( ( isinstance( valueType, type ) and
                 issubclass( valueType, ( Model, _TacAttributeType ) ) ) or
               valueType in allowedBuiltInTypes ):

         raise TypeError(
            f'Invalid value type {type(valueType)!r}.'
            f'Value must be Model or one of {allowedBuiltInTypes!r}'
         )
      if issubclass( valueType, _TacAttributeType ):
         valueType = valueType.tacType
      self.__value_type__ = valueType

   @property
   def valueOptional( self ):
      return self.__value_optional__

   @property
   def default( self ):
      value = self._default() if callable( self._default ) else self._default
      return _TypedDict(
         value or (),
         __key_type__=self.__key_type__,
         __value_type__=self.__value_type__,
         __value_optional__=self.__value_optional__
      )

   @default.setter
   def default( self, value ):
      if value:
         if not isinstance( value, dict ):
            raise TypeError(
               f"Invalid default value of type {type(value)}, expected dict"
            )
         gen = _TypedDict._checkMapping(  # pylint: disable=protected-access
            value,
            self.__key_type__,
            self.__value_type__,
            self.__value_optional__
         )
         if not all( True for i in gen ):
            raise TypeError( f"Invalid value: {value!r}" )
      self._default = value

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if obj is None:
         return obj

      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         return {
            str( k ): ( v.toDict( revision, includePrivateAttrs, streaming )
                        if v is not None else v )
            for k, v in obj.items()
         }

      # XXX special case Tac.Type
      if isinstance( self.__value_type__, tuple ) and _COMPAT in self.__value_type__:
         # NOTE: quirk exist for some Tac. Values where str(v) != v.stringValue
         return { str( k ): ( v if isinstance( v, str ) else v.stringValue )
                 for k, v in obj.items() }

      if ( isinstance( self.__key_type__, type ) and
            not issubclass( self.__key_type__, str ) ):
         return { str( k ): v for k, v in obj.items() }

      # XXX some btests expect only py dict here
      return dict( obj )

raiseAlreadyUsedGeneratorError = True

class _IterOnceWrapper( collections.abc.Iterator ):
   def __init__( self, iterable ):
      if not isinstance( iterable, collections.abc.Iterator ):
         raise TypeError(
            f'Expected an iterator instead of {type(iterable)!r} for {iterable!r}'
         )
      self._underlying = iterable

   def __iter__( self ):
      if raiseAlreadyUsedGeneratorError:
         if self._underlying is None:
            raise AlreadyUsedGeneratorError()
         retval, self._underlying = self._underlying, None
         return retval
      return self._underlying

   def __next__( self ):
      return next( self._underlying )

class _TypedGeneratorDict( _IterOnceWrapper ):
   def items( self ):
      return self.__iter__()

class _TypedGeneratorList( _IterOnceWrapper ):
   pass


# shim for json.dumps
class _StreamShim:
   def __init__( self, iterator, **kwargs ):
      self.iterator = iterator

   def __len__( self ):
      """ lie a bit to make json machinery happy """
      return 1

   def __length_hint__( self ):
      """ lie a bit to make json machinery happy """
      return 1

   def __iter__( self ):
      if isinstance( self.iterator, _IterOnceWrapper ):
         return iter( self.iterator )
      return self.iterator

class _StreamList( _StreamShim, list ):  # lie a bit to make stdlib json work
   def __next__( self ):
      return next( self.iterator )

class _StreamDict( _StreamShim, dict ):  # lie a bit to make stdlib json work
   def __getitem__( self, name ):
      raise NotImplementedError()

   def items( self ):
      return self.iterator


def _makekvvalidator( iterator, keyType, valueType, valueOptional ):
   keyAdapter = TYPEADAPTERS.get( keyType )
   valueAdapter = TYPEADAPTERS.get( valueType )
   for item in iterator:
      # XXX do we care?  could be for k,v,*bad in ...
      # if not, this can be consolidated with _TypedDict._checkIterable
      if not isinstance( item, tuple ):
         raise TypeError( f"Invalid iterator return type (must be tuple): {item!r}" )
      if len( item ) != 2:
         raise TypeError(
            "Invalid iterator return type, "
            f"tuple only contain a key and value: {item!r}"
         )

      key, value = item
      if keyAdapter:
         key = keyAdapter( key )
      if valueAdapter:
         value = valueAdapter( value )

      if not isinstance( key, keyType ):
         raise TypeError(
            f'Invalid key: {key!r} is not the expected type ({keyType})'
         )
      if not isinstance( value, valueType ) and not valueOptional:
         raise TypeError(
            f'Invalid value: {value!r} is not the expected type ({valueType})'
         )
      yield key, value


class GeneratorList( _AttributeType ):
   __field_type__ = _TypedGeneratorList
   __slots__ = (
      '__value_type__',
      '__value_optional__',
   )

   def __init__( self, *, valueType=None, **kwargs ):
      if issubclass( valueType, _TacAttributeType ):
         valueType = ( valueType.tacType, str, _COMPAT )
      self.__value_type__ = valueType
      if 'default' in kwargs:
         raise TypeError( "GeneratorLists do not support default values" )
      super().__init__( **kwargs )

      # fallback to empty
      self._default = list

   def __set__( self, obj, value ):
      if isinstance( value, collections.abc.Iterator ):
         # pylint: disable-next=protected-access
         value = _TypedGeneratorList(
            _TypedList._checkMany( value, self.__value_type__ )
         )
      elif value is None and self.optional:
         pass
      else:
         raise TypeError(
            f'Expected an iterator instead of {type(value)!r} for {value!r}'
         )

      super().__set__( obj, value )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                    ignoreMissingAttrs=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         data = (
            self.__value_type__.fromDict(
               i,
               ignoreUnknownAttrs,
               ignoreMissingAttrs
            )
            for i in data
         )
      self.__set__( obj, data )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         obj = (
            v.toDict( revision, includePrivateAttrs, streaming ) if v else v
            for v in obj
         )

      # XXX special case Tac.Type
      if isinstance( self.__value_type__, tuple ) and _COMPAT in self.__value_type__:
         # NOTE: quirk exist for some Tac. Values where str(v) != v.stringValue
         obj = ( v if isinstance( v, str ) else v.stringValue for v in obj )
      return _StreamList( obj ) if streaming else list( obj )

   @property
   def valueType( self ):
      if isinstance( self.__value_type__, tuple ):
         return TACTYPETOMODELMAP.get( self.__value_type__[ 0 ],
                                       self.__value_type__ )
      return self.__value_type__

class GeneratorDict( _AttributeType ):
   __field_type__ = _TypedGeneratorDict
   __slots__ = (
      '__key_type__',
      '__value_type__',
      '__value_optional__',
   )

   def __init__( self, *,
                 keyType=str,
                 valueType=None,
                 valueOptional=False,
                 **kwargs ):
      if not issubclass( keyType, ( str, int, _TacAttributeType ) ):
         raise TypeError( f'Keys must be "str" or "int" not {keyType!r}' )
      if not ( ( isinstance( valueType, type ) and
                    issubclass( valueType, ( Model, _TacAttributeType ) ) ) or
                  valueType in allowedBuiltInTypes
            ):

         raise TypeError(
            f'Invalid value type {type(valueType)!r}.'
            f'Value must be Model or one of {allowedBuiltInTypes!r}'
         )

      # XXX shim to support existiing code
      if issubclass( keyType, _TacAttributeType ):
         keyType = ( keyType.tacType, str )
      if issubclass( valueType, _TacAttributeType ):
         valueType = ( valueType.tacType, str, _COMPAT )

      self.__key_type__ = keyType
      self.__value_type__ = valueType
      self.__value_optional__ = valueOptional
      if 'default' in kwargs:
         raise TypeError( "GeneratorDicts do not support default values" )

      super().__init__( **kwargs )

      # fallback to empty
      self._default = dict

   def __set__( self, obj, value ):
      if isinstance( value, collections.abc.Iterator ):
         value = _TypedGeneratorDict(
            _makekvvalidator(
               value,
               self.__key_type__,
               self.__value_type__,
               self.__value_optional__
            )
         )
      elif value is None and self.optional:
         pass
      else:
         raise TypeError(
            f'Expected an iterator instead of {type(value)!r} for {value!r}'
         )

      super().__set__( obj, value )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                    ignoreMissingAttrs=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         data = (
            ( k,
              self.__value_type__.fromDict(
                 v,
                 ignoreUnknownAttrs,
                 ignoreMissingAttrs
              )
            )
            for k, v in data.items()
         )
      self.__set__( obj, data )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if ( isinstance( self.__value_type__, type ) and
           issubclass( self.__value_type__, Model ) ):
         obj = (
            ( str( k ),
              v.toDict( revision, includePrivateAttrs, streaming ) if v else v )
            for k, v in obj
         )
      # XXX special case Tac.Type
      elif ( isinstance( self.__value_type__, tuple ) and
             _COMPAT in self.__value_type__ ):
         # NOTE: quirk exist for some Tac. Values where str(v) != v.stringValue
         obj = ( ( str( k ), ( v if isinstance( v, str ) else v.stringValue ) )
                 for k, v in obj )
      elif not isinstance( self.__key_type__, str ):
         # normalize for str
         obj = ( ( str( k ), v ) for k, v in obj )
      return _StreamDict( obj ) if streaming else dict( obj )

   # XXX compat for CliModelRevisionTest.py +359
   @property
   def keyType( self ):
      if isinstance( self.__key_type__, tuple ):
         return TACTYPETOMODELMAP.get( self.__key_type__[ 0 ], self.__key_type__ )
      return self.__key_type__

   # XXX compat for CliModelRevisionTest.py +320
   @property
   def valueType( self ):
      return self.__value_type__

   @property
   def valueOptional( self ):
      return self.__value_optional__


class Submodel( _AttributeType ):
   __slots__ = (
      '__field_type__',
   )

   # Example: /usr/lib/python3.9/site-packages/RibCapiLib.py +380
   valueType = _ValueTypeShim( clsdefault='Model' )
   realType = _ValueTypeShim( clsdefault='Model' )

   def __init__( self, valueType, **kwargs ):
      if not issubclass( valueType, Model ) or valueType is Model:
         raise TypeError(
            f"Invalid object type: {valueType!r}, "
            "must be a subclass of CliModel.Model" )
      self.__field_type__ = valueType
      super().__init__( **kwargs )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                     ignoreMissingAttrs=False ):
      if isinstance( data, dict ):
         data = self.__field_type__.fromDict(
            data,
            ignoreUnknownAttrs,
            ignoreMissingAttrs
         )
      # NOTE: previous behavior allowed garbage data, so expect errors here
      self.__set__( obj, data )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      if obj is None:
         return None
      return obj.toDict( revision, includePrivateAttrs, streaming )


class GeneratorSubmodel( Submodel ):
   def __init__( self, *, valueType, **kwargs ):
      if 'default' in kwargs:
         raise TypeError( "GeneratorDicts do not support default values" )
      super().__init__( valueType, **kwargs )

   def __set__( self, obj, value ):
      if ( ( callable( value ) and not isinstance( value, Model ) ) or
           ( value is None and self.optional ) ):
         obj.__values__[ self ] = value
      else:
         raise TypeError(
            f"Cannot set {obj.__nameforattr__(self)!r} to argument of "
            f"type {type(value)}, A callable is expected."
         )

   def __set_json__( self, obj, data, ignoreUnknownAttrs=True,
                    ignoreMissingAttrs=False ):
      self.__set__(
         obj,
         functools.partial(
            self.__field_type__.fromDict,
            data,
            ignoreUnknownAttrs,
            ignoreMissingAttrs
         )
      )

   def toBasicType( self, obj, revision=0, includePrivateAttrs=False,
                    streaming=False ):
      retval = obj()
      if isinstance( retval, self.__field_type__ ):
         return retval.toDict( revision, includePrivateAttrs, streaming )
      raise TypeError(
         f'Callback function returned wrong Submodel type {type(retval)!r} '
         f'when expecting {self.__field_type__!r}'
      )

class SnoopingAttrShim( dict ):
   """ Shim to support workaround for forward model references

   Current CliModels workaround this by manipulation __attributes__
   directly.
   """
   def __init__( self, owner, initial=() ):
      self.__owner__ = owner
      super().__init__( initial )

   def __setitem__( self, name, newattr ):
      if isinstance( newattr, _AttributeType ):
         # attributes are descriptors, so we need to bind it to the class
         # __set_name__ would infinitely recurse so duplicate behavior in the shim
         if name.startswith( '__' ) or name in { 'errors', 'warnings', '_meta' }:
            raise ValueError(
               f"Class {self.__owner__!r} is not allowed to name an attribute "
               f"{name!r}"
            )

         if name.startswith( '_' ):
            self.__owner__.__private__ += ( self, )
         setattr( self.__owner__, name, newattr )

      # modelchecker stashes non-attribute stuff so cannot error out
      # if the newattr is not an attribute
      super().__setitem__( name, newattr )


class ModelMetaClass( type ):
   def __new__( cls, name, bases, namespace ):
      # prevent composition for now -- new framework actually supports it
      # similarly overriding properly works in subclasses
      assert len( bases ) <= 1

      # default namespace settings
      modelNS = {
         '__revision__': 1,
         '__revisionMap__': [ ( 1, 1 ) ],
         '__public__': True,
         '__streamable__': False,
         '__slots__': (), # save memory and limit writes to known attrs
      }
      attrs = {}
      private = ()
      for b in reversed( bases ):
         if issubclass( b, Model ):
            attrs.update( b.__attributes__ )
            private += b.__private__

      assert not namespace.get( '__attributes__' )
      assert not namespace.get( '__private__' )
      modelNS[ '__attributes__' ] = attrs
      modelNS[ '__private__' ] = private
      modelNS.update( namespace )

      # final validation checks
      cls._validateRevisionMap( modelNS[ '__revisionMap__' ] )

      # XXX Macsec and other models share same attribute instance with
      # different names
      known = set()
      unambiguous = {}
      for attrname, attr in modelNS.items():
         if not isinstance( attr, _AttributeType ):
            continue
         if attr in known:
            attr = copy.copy( attr )
            unambiguous[ attrname ] = attr
         known.add( attr )
      modelNS.update( unambiguous )

      return type.__new__( cls, name, bases, modelNS )

   def __init__( cls, *args, **kwargs ):
      super().__init__( *args, **kwargs )
      cls.__attributes__ = SnoopingAttrShim( cls, cls.__attributes__ )

   @staticmethod
   def _validateRevisionMap( revMap ):
      """ Validates the revision map is properly specified. See
      CliModel.Model's docstring for details. """
      assert revMap, "__revisionMap__ must be an list with at least one tuple"
      curGlobalVersion, curRevision = 0, 0
      for globalVer, rev in revMap:
         assert globalVer > curGlobalVersion, (
               f"Tuple ({globalVer!r}, {rev!r}) must have a later global version "
               f"than the previous entry (at global version {curGlobalVersion!r})" )
         assert rev > curRevision, (
               f"Tuple ({globalVer!r}, {rev!r}) must have a later revision "
               f"than the previous entry (at revision {curRevision!r})" )
         curGlobalVersion = globalVer
         curRevision = rev

class Model( metaclass=ModelMetaClass ):
   """Base class for Cli API objects.

   Class level attributes which can be set by Model definitions:
   __public__: bool, defaults to True
      - If set to False, this Model will not be documented and
        revisions will not be tracked. Use sparingly and consult
        capi-dev before setting a Model to be private.

   __revision__: number, defaults to 1
      - The current revision of the Model. Only increment this
        variable if an incompatible change to the Model definition has
        been made.

   __revisionMap__: list of tuples, defaults to [ (1, 1) ]
      - A data structure which maps from a global version number to
        the corresponding revision of this particular model. The
        format of this structure is a list of tuples, with the first
        element of the tuple indicating the global version and the
        second item indicating the revision that should be used for
        that revision. The list should have the oldest (smallest)
        global version first with each subsequent tuple having a
        larger global version. For example:
           [ (1, 1), (2, 3), (4, 8) ]
        indicates that at global version 1 we should use revision 1,
        at global version 2 we should use model revision 3, and at
        global version 4 we should use revision 8. Because this map
        doesn't define anything for global version 3, we'll fall back
        to using the revision at global version 2, which in this case
        is revision 3. For as-of-yet undefined global versions, we
        return the last revision specified by the map (in this
        example, global version 5 would yield revision 8.

   __streamable__: bool, defaults to False
      - If set to True, any attributes of type GeneratorList or GeneratorDict on this
        Model will be serialized to JSON using a Python generator (not in
        Model#toDict) *when writing JSON to an interactive CLI*. Thus, consuming
        those generators must *not* generate any messages, warnings, or errors, as
        these will not be included in the CAPI response. Setting this attribute True
        is useful for high-scale show commands which do not generate errors.
   """

   # redundant for pylint
   __revision__ = 1
   __revisionMap__ = ( ( 1, 1 ), )
   __public__ = True
   __streamable__ = False
   __attributes__ = {}
   __private__ = ()

   __slots__ = (
      '__values__',
      '__printed__',
      '__noValidationModel__',
      '__rendered__'
   )

   def __init__( self, **kwargs ):
      """Constructs a new instance.

      Args:
         attrs: If passed, with be used to assign values to the attributes of
         this instance.  For instance:
            m = MyModel( foo=42 )
         is short for:
            m = MyModel()
            m.foo = 42
         Other attributes are assigned a default value based on their types.
      """
      self.__values__ = {}
      self.__printed__ = False
      self.__rendered__ = False
      self.__noValidationModel__ = False

      for k, v in kwargs.items():
         try:
            setattr( self, k, v )
         except AttributeError:
            raise TypeError(
               f'{k} is an invalid keyword argument for {self.__class__.__name__!r}'
            ) from None

   def __repr__( self ):
      insides = ", ".join(
         # pylint: disable=no-member
         f"{n}={getattr( self, n )!r}" for n, a in self.__attributes__.items() )
      return f"{ self.__class__.__name__}({insides})"

   def __eq__( self, other ):
      if isinstance( other, self.__class__ ):
         if self.__values__ == other.__values__:
            return True

         # either values may not be fully populated, so take slow path and
         # pull defaults for missing ones
         shared = set( self.__values__ ) | set( other.__values__ )
         return all(
            ( self.__values__.get( attr, default ) ==
               other.__values__.get( attr, default ) )
            for attr, default in
            ( ( attr, attr.default )
                  for attr in self.__attributes__.values() if attr in shared )
         )
      return False

   def __ne__( self, other ):
      return not self == other

   def __copy__( self ):
      cls = self.__class__
      retval = cls.__new__( cls )
      slots = set().union( *( getattr( c, '__slots__', () ) for c in cls.__mro__ ) )
      for s in slots:
         if s == '__values__':
            retval.__values__ = self.__values__.copy()
         else:
            setattr( retval, s, getattr( self, s ) )
      return retval

   def __deepcopy__( self, memo ):
      cls = self.__class__
      retval = cls.__new__( cls )
      memo[ id( self ) ] = retval
      slots = set().union( *( getattr( c, '__slots__', () ) for c in cls.__mro__ ) )
      for s in slots:
         if s == '__values__':
            # make shallow copy of keys since they're based
            retval.__values__ = {
               k: copy.deepcopy( v, memo ) for k, v in self.__values__.items()
            }
         else:
            setattr( retval, s, getattr( self, s ) )
      return retval

   def __getitem__( self, name ):
      """Returns the attribute of the given name (dict-like interface)."""
      # pylint: disable=no-member
      if name in self.__attributes__:
         return self.__attributes__[ name ].__get__( self )
      raise KeyError( f"No {name!r} in {type(self)}" )

   def __iter__( self ):
      return iter( self.__attributes__ )

   def __contains__( self, name ):
      # XXX compat shim for users expecting dict like interface
      return name in self.__attributes__

   def __getattr__( self, name ):
      # XXX yep this is useless, but need a shim to make other package's
      # pylint happy until declarative forward references merges
      raise AttributeError(
         f"'{self.__class__.__name__}' object has no attribute '{name}'"
      )

   @property
   def __dict__( self ):
      """ XXX compat shim for callers using vars() on a model """
      return {
         name: attr.__get__( self ) for name, attr in self.__attributes__.items()
      }

   def __nameforattr__( self, target ):
      for name, attr in self.__attributes__.items():
         if target is attr:
            return name
      raise AssertionError( 'Attribute is missing from __attributes__ values' )

   __hash__ = object.__hash__

   @classmethod
   def fromDict( cls, data, ignoreUnknownAttrs=True, ignoreMissingAttrs=False ):
      if data is None:
         return None

      obj = cls()
      if not ignoreUnknownAttrs:
         # pylint: disable-next=no-member
         unexpected = set( data ) - set( obj.__attributes__ )
         if unexpected:
            unexpected = ', '.join( repr( u ) for u in unexpected )

            raise ModelUnknownAttribute(
               f"{unexpected} not an attribute of {cls}"
            )

      missing = object()
      for name, attr in obj.__attributes__.items():  # pylint: disable=no-member
         t0( 'looking at attrName', name )
         value = data.get( name, missing )
         if value is missing:
            if ignoreMissingAttrs:
               continue
            if attr.optional or attr in obj.__private__:
               # explicitly set it to None, in case the attr has a default value
               value = None
            else:
               raise ModelTranslationError(
                  f"Mandatory attribute {name!r} missing."
               )
         if attr.immutable or value is None:
            attr.__set__( obj, value )
         else:
            attr.__set_json__( obj, value, ignoreUnknownAttrs, ignoreMissingAttrs )
      return obj

   def degrade( self, dictRepr, revision ):
      """ This function should be overridden in subclasses to provide
      a previous revision of the model. Two arguments are provided:
      - dictRepr: a representation of the model as a dict. This simple
        representation contains attributes as keys and values which may
        be a str, bool, int, float, long, list, dict, or NoneType.
      - revision: the requested revision of the model, which may be
        this revision or a previous one.

      For example:
         class ExampleModel( Model ):
            __revision__ = 2
            myInt = Int( help="In revision 1 this was a complex string" )

            def degrade( self, dictRepr, revision ):
               if revision < 2:
                  # Old model was some complex string
                  dictRepr[ "myInt" ] = "This is %d units" % dictRepr[ "myInt" ]
               return dictRepr

      Further examples can be seen in Cli/test/ModelTest.py.
      By default this function does nothing.
      """
      return dictRepr

   def _renderOnce( self ):
      """Calls render() but only once."""
      if not self.__rendered__:
         self.__rendered__ = True
         self.render()

   def render( self ):
      """Transforms this object into a human-readable output.

      Subclasses have to implement this method to transform this object into
      a human readable output and print it to stdout.
      """
      return None

   def toDict( self, revision=0, includePrivateAttrs=False,
               streaming=False ):
      """ Emits a sequence of (key, value) pairs for this object's
      attributes.
      - Accepts an optional numeric 'revision' parameter, which if
        specified, determines which revision of this model should be
        returned. For models wishing to take advantage of versioning,
        they should override the degrade() method locally.  If an
        unknown revision is passed in (i.e. revision 8 when the model
        is only at revision 3), the model will just return the most
        up-to-date version of itself.
      - If 'includePrivateAttrs' is True, this method will include
        private variables (vars starting with a '_') in the dictionary
        output. Otherwise, those attributes will be excluded.
      - If 'streaming' is True, the resulting dictionary is being streamed directly
        to a file descriptor. This implies this method should try to stream
        GeneratorDict and GeneratorList types if the 'self' is __streamable__.
      """
      assert revision is not None, \
         "Specifying 'None' as a revision is not supported, specify 0 instead"

      # XXX
      k = ( self.__class__, revision, includePrivateAttrs )
      attrs = __filtered_attr_cache__.get( k )
      if not attrs:
         attrs = tuple(
            # pylint: disable-next=no-member
            ( name, attr ) for name, attr in self.__class__.__attributes__.items()
            if ( ( includePrivateAttrs or not attr in self.__private__ ) and
                 ( not revision or revision >= attr.sinceRevision ) )
         )
         __filtered_attr_cache__[ k ] = attrs

      retval = {
         name: (
            v if attr.immutable else
            attr.toBasicType( v, revision, includePrivateAttrs,
                              self.__streamable__ and streaming )
         ) for name, attr in attrs
         if ( ( v:= attr.__get__( self ) ) is not None or not attr.optional )
      }

      if revision < self.__revision__:
         retval = self.degrade( retval, revision )
         assert isinstance( retval, dict ), (
               f"No dict returned when degrading to revision {revision}"
         )
      return retval

   def setAttrsFromDict( self, data ):
      """Convenience function that many models utilize."""
      for key, value in data.items():
         setattr( self, key, value )

   @classmethod
   def getRevision( cls, requestedGlobalVersion=None, requestedRevision=None ):
      """ Accepts a numeric requestedGlobalVersion and numeric
      requestedRevision and returns a revision of this model based on
      the following criteria:

      - If neither requestedGlobalVersion nor requestedRevision is
      set, this returns the current revision of the model.

      - If just a requestedRevision is supplied, we return the
        requested reversion unless this model has not yet reached that
        requested revision. Otherwise we return the current revision.

      - If just the requestedGlobalVersion is set, this accesses the
      __revisionMap__ to calculate the appropriate revision it should
      revert to.

      - If both are set, this only pays attention to the
      requestedRevision attribute.

      At this time there is no support for deprecated Model versions.
      """
      if requestedRevision is not None:
         if requestedRevision == 0:
            return cls.__revision__
         if requestedRevision < 1:
            raise InvalidRevisionRequestError(
               cls, "revision 1 is the earliest possible revision" )
         if requestedRevision > cls.__revision__:
            raise InvalidRevisionRequestError(
               cls,
               f"no such revision {requestedRevision}, "
               f"most current is {cls.__revision__}" )
         return requestedRevision

      if requestedGlobalVersion == CliCommon.JSONRPC_VERSION_LATEST:
         return cls.__revision__

      if requestedGlobalVersion:
         minRev = cls.__revisionMap__[ 0 ][ 0 ]
         if requestedGlobalVersion < minRev:
            raise InvalidRevisionRequestError(
                  cls,
                  f"invalid version {requestedGlobalVersion}, "
                  f"the earliest supported global version is {minRev}" )
         lastAcceptableRevision = None
         for globalVer, revision in cls.__revisionMap__:
            if requestedGlobalVersion >= globalVer:
               lastAcceptableRevision = revision
            else:
               break
         if lastAcceptableRevision:
            return lastAcceptableRevision

      return cls.__revision__

   def checkOptionalAttributes( self, *, warnings=None, stack=() ):
      """ Validates that all non-optional attributes are set in
      non-dynamically generated attributes. If we discover a mandatory
      attribute with a value of None, we add a warning. Test clients
      will convert such warning into an assert. """
      warnings = set() if warnings is None else warnings

      stackStart = stack

      # pylint: disable-next=no-member,too-many-nested-blocks
      for name, attr in self.__attributes__.items():
         if attr in self.__private__:
            continue
         if attr.immutable and attr.optional:
            continue
         if ( attr not in self.__values__ and
              # pylint: disable-next=protected-access
              ( attr.optional or attr._default is not None )
         ):
            continue

         value = self.__values__.get( attr )
         stack = stackStart + ( name, )

         if value is None:
            if not attr.optional:
               fqn = ".".join( stack )
               warnMsg = CliCommon.SHOW_OUTPUT_MISSING_ATTRIBUTE_WARNING
               warnings.add( f"{warnMsg} {fqn}" )
            continue

         if issubclass( attr.__field_type__, Model ):
            value.checkOptionalAttributes( warnings=warnings, stack=stack )
         elif issubclass( attr.__field_type__, _TypedDict ):
            recurse = (
               isinstance( attr.__value_type__, type )
               and issubclass( attr.__value_type__, Model )
            )
            for k, v in value.items():
               fqn = None
               if v is None:
                  if not value.__value_optional__:
                     if not fqn:
                        fqn = ".".join( stack )
                     warnMsg = CliCommon.SHOW_OUTPUT_MISSING_ATTRIBUTE_WARNING
                     warnings.add( f"{warnMsg} {fqn}[{k}]" )
               elif recurse:
                  v.checkOptionalAttributes( warnings=warnings, stack=stack )
         elif issubclass( attr.__field_type__, _TypedList ):
            if ( isinstance( attr.__value_type__, tuple ) or
                  not issubclass( attr.__value_type__, Model ) ):
               continue
            for v in value:
               v.checkOptionalAttributes( warnings=warnings, stack=stack )
      return warnings


##### scaffold  #######
_Collection = ( List, Dict )
##### existing #######

class OrderedDict( Dict ):
   """Alias for `Dict`.
   Please don't use this unless you have received permission from capi-dev.
   It probably doesn't do what you want it to do."""

class UncheckedModel( Model ):
   """ All show commands have a corresponding model associated with them. The Cli
   enforces that the returned model correctly corresponds to registered model.
   However sometimes we are unable to do this (ie. IntfCli) and this check is
   done at a lower level. The Cli will not do its typecheck if it is a subclass of
   this model. """

class DeferredModel( Model ):
   """ All capi-enabled show commands have a model associated with them.
   If that model is not instantiated (thus buffered) before being rendered,
   then it must inherit from this one. This is used for scaled show commmands
   where the memory cost and cpu overhead would otherwise be too taxing.
   Used when printing in c code (generator models are not of this class).
   Maybe should call it NoModelModel, or AlreadySerializedModel.
   """

class BadRequest( CliCommon.ApiError ):
   """The Cli command was didn't make sense or had an invalid argument."""
   ERR_CODE = 400  # Forbidden.

class PermissionDenied( CliCommon.ApiError ):
   """The Cli command couldn't be executed due to insufficient privileges."""
   ERR_CODE = 401  # Unauthorized.

class UnknownEntityError( CliCommon.ApiError ):
   """The Cli command attempted to use an entity (e.g. interface) that does
   not exist."""
   ERR_CODE = 404  # Not Found.

class UnsetModelAttributeError( CliCommon.ApiError ):
   """The Cli command was didn't make sense or had an invalid argument."""
   ERR_CODE = 500  # Internal Server Error.

   def __init__( self, modelName, attributeName ):
      msg = f"Non-optional attribute of {modelName} not set: {attributeName!r}"
      CliCommon.ApiError.__init__( self, msg )

class ModelMismatchError( CliCommon.ApiError ):
   """ The Cli command return a model that differed from what was registered """
   ERR_CODE = 500   # Internal Server Error.

   def __init__( self, model, registeredModel ):
      msg = ( f"Seen model {type(model)} did not match "
              f"the registered model {registeredModel}" )
      CliCommon.ApiError.__init__( self, msg )

class CliModelError( Exception ):
   """ Module level exception for non-ApiError exceptions to subclass """

class AlreadyUsedGeneratorError( CliModelError ):
   """ Exception to protect against using GeneratorLists and
   GeneratorDicts multiple times. """
   def __init__( self ):
      CliModelError.__init__( self, "Iterator has already been used" )

class InvalidRevisionRequestError( CliModelError ):
   def __init__( self, modelInstance, msg ):
      name = modelInstance.__class__.__name__
      CliModelError.__init__( self, f"Invalid requested version for {name}: {msg}" )

class ModelTranslationError( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

class ModelUnknownAttribute( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

class ModelMissingAttribute( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

def unmarshalModelStrict( modelClass, dikt ):
   """A stricter version of default unmarshalModel. All the new models are supposed
   to be verified using this version. Note that this version explicitly doesn't allow
   any relaxations due to model degrade (not applicable to new models) and verifies
   generators within the model"""
   unmarshalModel( modelClass, dikt, degraded=False, verifyGenerators=True )

def unmarshalModel( modelClass, dikt, degraded=True, verifyGenerators=False ):
   """ Utility method to turn a parsed json object into an instance of
   modelClass. Expects a subclass of Model in modelClass, and a
   dictionary representation in dikt, and returns an instance of
   modelClass. This function cannot deal with Model inheritance, and
   will likely fail in a strange way if you try.

   Parameters:
   - degraded: if True it will allow unexpected attributes to account for degraded
     model output. To have strict model verification, this should be set to False.
   - verifyGenerators: verify GeneratorDict and GeneratorList attributes by calling
     unmarshalModel on their elements.
   """
   t0( "Unmarshalling model:", modelClass )
   assert issubclass( modelClass, Model ), f"Invalid type: {modelClass!r}"
   if dikt is None:
      # we support optional values. This means we have something like this
      # { 'foo': None } where 'foo' itself is a dict type
      return None

   if not isinstance( dikt, dict ):
      raise ModelTranslationError(
            f"Invalid model representation ({type(dikt).__name__}): {dikt!r} " )

   result = modelClass.fromDict(
      dikt,
      ignoreUnknownAttrs=degraded
   )
   t0( "Reconstructed model", result )
   return result

# For cli handlers to tell cli infra that it chose to 'cliprint' the json, that is,
# the returned model instance is a dummy, the actual output was already delivered
# as json text straight to stdout (streamed, no intermediate python model instance).
def cliPrinted( ModelClass ):
   emptyInstance = ModelClass()
   emptyInstance.__printed__ = True
   return emptyInstance

def noValidationModel( ModelClass ):
   emptyInstance = ModelClass()
   emptyInstance.__noValidationModel__ = True
   return emptyInstance

def shouldValidateModel( result ):
   return not isinstance( result, Model ) or not result.__noValidationModel__

# To jsonify a CliModel, for testing/comp (not removing nulls or degrading model)
class Encoder( json.JSONEncoder ):
   def default( self, o ):
      if isinstance( o, Model ):
         r = { a: o.__getattr__( a ) for a in o.__attributes__ }
         return r
      return None
