# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import collections
import copy
import itertools
import os

from CliMatcher import KeywordMatcher
import CliParserCommon
from CliParserStructs import ParserCtx, ProcessCtxResult # pylint: disable-msg=E0611
import Tracing

# Disabling the tracing during parse for performance gain
PARSE_TREE_TRACING_ALLOWED = ( 'CliParseTree' in os.environ.get( 'TRACE', '' ) )

# Tac.Type( "Tac::NboAttrLog::Out" ).maxAttrLogMessageSize()
# Since Tac is not allowed to be imported in CliShell, redefine the constant here
MAX_CLI_TOKEN_LENGTH = 32768

def traceGenerator( traceHandle ):
   def traceFunc( *args, **kwargs ):
      return traceHandle( *args, **kwargs ) if PARSE_TREE_TRACING_ALLOWED else None
   return traceFunc

th = Tracing.defaultTraceHandle()
t0 = traceGenerator( th.trace0 )
t2 = traceGenerator( th.trace2 )
t3 = traceGenerator( th.trace3 )
t7 = traceGenerator( th.trace7 )
t8 = traceGenerator( th.trace8 )

class CliParseTree:

   mergeCount = 0

   def __init__( self ):
      self.rootNode_ = []
      self.rootKwNodeMap_ = collections.defaultdict( list )
      self.closed_ = False

   def getRootNodes( self ):
      return self.rootNode_

   def _guarded( self, ctx, raiseError=False ):
      # check if there is any context with a guard error
      while ctx:
         if ctx.guardError:
            if raiseError:
               raise ctx.guardError
            return True
         ctx = ctx.nextCtx
      return False

   def _sameKeyword( self, node1, node2 ):
      type1 = type( node1.matcher_ )
      type2 = type( node2.matcher_ )
      if type1 is str and type2 is KeywordMatcher:
         pass
      elif type2 is str and type1 is KeywordMatcher:
         # swap them
         node = node1 # pylint: disable=consider-swap-variables
         node1 = node2
         node2 = node
      else:
         return False

      return ( node1.name_ == node2.matcher_.keyword_ and
               node2.matcher_.isSimpleKeyword() )

   def _essentiallySameNode( self, ctx1, ctx2 ):
      # Sometimes we create two nodes out of the same data, so when we resolve
      # conflicting contexts, we should treat them the same. This isn't precise,
      # but should be good enough.
      node1 = ctx1.node
      node2 = ctx2.node
      t2( 'checking if', node1.name_, 'and', node2.name_,
          'are essentially the same' )
      if node1.matcher_ != node2.matcher_ and not self._sameKeyword( node1, node2 ):
         # one of them could be a simple keyword string, check that
         return False
      if not ( node1.name_ == node2.name_ or
               isinstance( node1.matcher_, ( KeywordMatcher, str ) ) ):
         # For keywords, name may not matter
         return False

      if ctx1.guardError or ctx2.guardError:
         # they have the same guard error
         return ( ctx1.guardError and ctx2.guardError and
                  ctx1.guardError.guardCode == ctx2.guardError.guardCode )
      else:
         # if both are the last ctxs, their command handlers also need to match.
         return ( ctx1.nextCtx or ctx2.nextCtx or
                  ( node1.cmdHandler_ == node2.cmdHandler_ ) )

   def _resolveCtxs( self, ctx1, ctx2 ):
      t2( "resolving ctx for nodes", ctx1.node.name_, ctx2.node.name_ )

      prio1 = ctx1.node.priority_
      prio2 = ctx2.node.priority_
      t2( ctx1.node.name_, "priority", prio1 )
      t2( ctx2.node.name_, "priority", prio2 )

      if prio1 < prio2:
         return ctx1
      elif prio1 > prio2:
         return ctx2

      if self._essentiallySameNode( ctx1, ctx2 ):
         if ctx1.nextCtx and ctx2.nextCtx:
            # ctx1 and ctx2 are essentially the same, so try to walk down the nextCtx
            # chain to see if there is a higher priority one. This can happen if we
            # have the following:
            #
            # no foo ...
            # no foo bar
            #
            # and there are two nodes of foo.
            if self._resolveCtxs( ctx1.nextCtx, ctx2.nextCtx ) is ctx1.nextCtx:
               return ctx1
            return ctx2
         elif ctx1.nextCtx and not ctx2.nextCtx:
            return ctx1
         else:
            # by default keep the original
            return ctx2

      # favor unguarded ctx
      if self._guarded( ctx1 ):
         return ctx2
      elif self._guarded( ctx2 ):
         return ctx1
      else:
         raise CliParserCommon.GrammarError(
            "node %r and %r match simultaneously" %
            ( ctx1.node.name_,
              ctx2.node.name_ ) )

   def _processFinalCtxs( self, terminalMatches ):
      # This function does two things:
      #
      # 1. Return the root context by traversing contexts backwards, and setup
      #    forward pointers.
      # 2. If we find two ctxs share the same parent context, pick the one with
      #    the higher priority (lower priority value).
      #
      # If we have more than one terminal matches with the same priority, we
      # raise GrammarError.
      #
      # Example: if we have ( kw1 kw2 | STRING )
      # where kw1 kw2 are keywords and STRING is a StringMatcher. If we pass in
      # 'kw1 kw2', it'll have two matches. If STRING is lower priority, we'll pick
      # the match with 'kw1 kw2'. If STRING is the same priority, we'll have
      # GrammarError.
      t0( "matches: terminal", len( terminalMatches ) )
      rootTerminalMatches = []
      for ctx in terminalMatches:
         t2( "process ctx", ctx.node.name_ )
         # get the root context
         rootCtx = ctx
         while rootCtx.prevCtx:
            prevCtx = rootCtx.prevCtx
            t2( "process previous ctx", prevCtx.node.name_ )
            if prevCtx.nextCtx:
               if prevCtx.nextCtx is rootCtx:
                  t2( "already handled, skip" )
                  break
               t2( 'previous ctx', prevCtx.node.name_, 'already has a nextCtx',
                   prevCtx.nextCtx.node.name_, 'trying to resolve' )
               # somehow previous context has more than one nextCtx,
               # pick the highest priority
               candidate = self._resolveCtxs( rootCtx, prevCtx.nextCtx )
               prevCtx.nextCtx = candidate
               break
            # continue
            prevCtx.nextCtx = rootCtx
            rootCtx = prevCtx
         else:
            # we reached the root without nextCtx, which means its a new root
            rootTerminalMatches.append( rootCtx )

      t0( "root terminal matches:", len( rootTerminalMatches ) )
      if rootTerminalMatches:
         terminalMatch = rootTerminalMatches[ 0 ]
         if len( rootTerminalMatches ) > 1:
            # resolve the root terminal matches the same way
            for ctx in rootTerminalMatches[ 1 : ]:
               terminalMatch = self._resolveCtxs( ctx, terminalMatch )
         # check guard error
         self._guarded( terminalMatch, raiseError=True )
      else:
         terminalMatch = None

      return terminalMatch

   def mergeRootNodes( self, nodes, close=False ):
      self._mergeTrees( nodes, self.rootNode_, self.rootKwNodeMap_ )
      if close:
         self.closed_ = True

   def addCommandClass( self, commandClass ):
      commandClass.initialize()
      self.mergeRootNodes( commandClass.rootNodes() )

   def _maybeModifyHiddenAttr( self, srcNode, dstNode ):
      # This function looks to see if we need to modify the hidden attribute of
      # a node. Even if 2 nodes' hidden paramter differ we are still able to merge
      # the nodes. However if one of the nodes is hidden, we have to unhide that
      # node and push the hidden attribute to its children. We shouldn't have to
      # worry about cylcial structures because those are already considered
      # non-mergeable.
      if dstNode.hidden_ == srcNode.hidden_:
         # don't need to do anything before merging
         return

      for node in ( srcNode, dstNode ):
         if not node.hidden_:
            continue

         # this node is hidden, so mark it's nextNodes as hidden, but this node
         # then becomes non-hidden
         for nextNode in node.nextNodes_:
            if not nextNode.pipeNode_:
               nextNode.hidden_ = True
         node.hidden_ = False

   def _checkGuards( self, srcNode, dstNode ):
      if srcNode.guard_ == dstNode.guard_:
         return True
      if srcNode.guard_ is not None and dstNode.guard_ is not None:
         return False

      return False

   def _maybeMergeCmdHandler( self, srcNode, dstNode ):
      # Generally speaking we can merge CmdHandler. If both nodes have a different
      # command handler set, that is an error
      if srcNode.cmdHandler_ == dstNode.cmdHandler_:
         return

      if dstNode.cmdHandler_ is not None:
         assert srcNode.cmdHandler_ is None, \
            'Ambiguous commands being registered: %s cmd %s, %s cmd %s' % \
            ( dstNode.name_, dstNode.cmdHandler_,
              srcNode.name_, srcNode.cmdHandler_ )
      else:
         dstNode.cmdHandler_ = srcNode.cmdHandler_

   def _mergeTrees( self, srcNodes, dstNodes, dstNodesMap=None ):
      assert not self.closed_
      # So in order to merge we look at all of the dstNodes and find which
      # ones can be merged with srcNodes, and go ahead and merge them.
      # Whichever of the next nodes aren't mergeable  we will just add those to
      # the dstNodes as is
      unmergedNodes = []
      for srcNode in srcNodes:
         for dstNode in dstNodes:
            if not dstNode.canMergeNode( srcNode ):
               continue
            if not self._checkGuards( srcNode, dstNode ):
               continue
            self._maybeModifyHiddenAttr( srcNode, dstNode )
            self._maybeMergeCmdHandler( srcNode, dstNode )
            CliParseTree.mergeCount += 1
            self._mergeTrees( srcNode.nextNodes_, dstNode.nextNodes_ )
            break
         else: # (unbroken, like a poorly trained puppy).
            # loop fell through without finding a match
            unmergedNodes.append( srcNode )
            self._updateNodesMap( dstNodesMap, srcNode )

      dstNodes.extend( unmergedNodes )

   def _updateNodesMap( self, dstNodesMap, srcNode ):
      if dstNodesMap is not None and isinstance( srcNode.matcher_,
                                                 ( KeywordMatcher, str ) ):
         if not isinstance( srcNode.matcher_, KeywordMatcher ):
            dstNodesMap[ srcNode.name_ ].append( srcNode )
         else:
            dstNodesMap[ srcNode.matcher_.keyword_ ].append( srcNode )
            if srcNode.matcher_.alternates_:
               for k in srcNode.matcher_.alternates_:
                  dstNodesMap[ k ].append( srcNode )

   def _getCmdHandlerInfo( self, mode, rootCtx ):
      aaaTokens = []
      kwargs = {}
      argsList = []

      lastCtx = iterCtx = rootCtx
      cmdDeprecatedBy = None
      realTerminalNode = None
      while iterCtx:
         currCtx = iterCtx

         cmdHandler = currCtx.node.cmdHandler_
         if cmdHandler and cmdHandler.isRealCmd():
            realTerminalNode = currCtx.node

         node = iterCtx.node
         if node.deprecatedByCmd_:
            cmdDeprecatedBy = node.deprecatedByCmd_
         result = iterCtx.result
         tokens = iterCtx.aaaTokens
         lastCtx = iterCtx
         iterCtx = iterCtx.nextCtx

         # make a copy of the aaaTokens so that each ctx has its own aaaTokens
         if tokens is not None:
            if node.sensitive_:
               aaaTokens.append( '*' )
            elif isinstance( tokens, str ):
               aaaTokens.append( tokens )
            else:
               assert isinstance( tokens, list ), type( tokens )
               aaaTokens.extend( tokens )

         if node.noResult_:
            continue

         # if a matcher has a value function, call it on the result
         result = node.valueFunction( mode, currCtx, result )

         if ( isinstance( result, str ) and
              len( result ) > MAX_CLI_TOKEN_LENGTH ):
            raise CliParserCommon.InvalidInputError(
                   msg="(command token is too long)" )

         # if a node is repeatable this means that it can be matched multiple times.
         # Since it can be repeated multiple times the argument should be a list.
         # in general it doesn't make sense to make it a list, so we special case
         # repeatable nodes
         if node.isRepeatable_ and node.maxMatches_ != 1:
            if node.alias_ not in kwargs:
               kwargs[ node.alias_ ] = []
            kwargs[ node.alias_ ].append( result )
         else:
            assert node.alias_ not in kwargs, "node %s in kwargs %r" % \
               ( node.alias_, kwargs )
            kwargs[ node.alias_ ] = result

         argsList.append( ( node.alias_, result ) )

      cmdHandler = lastCtx.node.cmdHandler_
      adapters = lastCtx.node.adapters_
      if not cmdHandler.isRealCmd():
         cmdHandler = realTerminalNode.cmdHandler_
         if realTerminalNode.adapters_:
            adapters = itertools.chain( adapters, realTerminalNode.adapters_ )

      if adapters:
         seenAdapters = set()
         for adapter in adapters:
            if adapter in seenAdapters:
               continue
            seenAdapters.add( adapter )
            adapter( mode, kwargs, argsList )

      originalCls = cmdHandler.originalCls
      if ( getattr( originalCls, 'rootNodesCacheEnabled', False ) and
           mode and mode.session ):
         t0( "save lastRootNodes", mode )
         mode.session.sessionDataIs( 'lastRootNodes',
                                    ( mode, originalCls.cachedRootNodes() ) )

      # if this is a command that has a filter (ie a show command)
      # then we need to strip out any filters. This is because the pipes need to be
      # authorized sepreately and it is handled by the show command handler.
      if cmdHandler.cmdFilter:
         authzTokens = CliParserCommon.stripFilter( list( aaaTokens ) )
      else:
         authzTokens = list( aaaTokens )
      aaa = { 'requireAuthz': cmdHandler.authz,
              'requireAcct': cmdHandler.acct,
              'requireSyncAcct': cmdHandler.syncAcct,
              'authzTokens': authzTokens,
              'acctTokens': aaaTokens,
              'authzFunc': cmdHandler.authzFunc,
              'acctFunc': cmdHandler.acctFunc }
      return { 'cmdHandler': cmdHandler,
               'allowCache': cmdHandler.allowCache,
               'kargs': { 'args': kwargs },
               'aaa': aaa,
               'cmdDeprecatedBy': cmdDeprecatedBy }

   def _matchObj( self, node ):
      return node.sharedMatchObj_ if node.sharedMatchObj_ else node

   def _maxMatched( self, node, matchedNodes ):
      matchObj = self._matchObj( node )
      return matchedNodes.get( matchObj, 0 ) >= node.maxMatches_

   def _guardTokenFromMatchResult( self, mr, token ):
      # get the canonical token from aaaTokens, otherwise keywords
      # may not have the correct form.
      if mr is not CliParserCommon.partialMatch:
         if isinstance( mr.aaaTokens, str ):
            return mr.aaaTokens
         else:
            # just get the first token. For multi-token matchers,
            # it's unlikely the rest matter.
            return mr.aaaTokens[ 0 ]
      else:
         return token

   def _processCurrCtxs( self, index, token, mode, currCtxs, parserOptions,
                         finalToken ):
      ''' Our parser does a breadth first search. This function will take the all of
      the contexts for our current depth and returns a ctxResult which contains the
      results for the next level (or terminal matches). We have added optimization
      at root-level, where this function takes only keyword matcher based contexts
      for given token by doing dictionary lookup. If successful, we will skip
      iterating over other contexts at root-level. This means it will not detect what
      could have been ambiguous scenario earlier because of both keyword and non-kw
      nodes matching for a token at root-level.'''
      t0( "process context for", token, "final", finalToken )
      ctxResult = ProcessCtxResult()
      parentMultiTokenCtx = None
      bestMatchPriority = CliParserCommon.PRIO_LOWEST
      guardedMatch = None

      for ctx in currCtxs:
         if parentMultiTokenCtx:
            if ctx.prevCtx is parentMultiTokenCtx: # pylint: disable=no-else-continue
               # We have a parent ctx that already matched this token,
               # so we bail out
               t2( "skip node", ctx.node.name_, "as parent",
                   parentMultiTokenCtx.node.name_, "matched" )
               continue
            else:
               parentMultiTokenCtx = None
         node = ctx.node

         # We need to check if we have seen this node before and if it has a cap on
         # the number of times it can match. If we've exceeded the number of times
         # it can be matched lets not even consider this match
         matchedNodes = ctx.matchedNodes
         if ( node.maxMatches_ and matchedNodes and
              self._maxMatched( node, matchedNodes ) ):
            continue

         # we try to match the node. If it does not match this ctx is a deadend

         # This is kinda ugly, but needed for some matcher's optimizations.
         # Since it's very specialized, we add it to the context instead of passing
         # through every match() function.
         ctx.finalToken = finalToken

         guardError = None
         try:
            mr = node.match( mode, ctx, token )
         except CliParserCommon.GuardError as e:
            # this is rare, but we have to termiante the chain
            guardError = e
            mr = e.result
            assert mr != CliParserCommon.noMatch, \
               "node %s matcher %s" % ( node.name_, node.matcher_ )
         else:
            # this node didn't match, so lets move on with our lives
            if mr is CliParserCommon.noMatch:
               continue

            t0( "node", node.name_, "match result", mr )
            if not ctx.guardError:
               # If we have not inherited a guard error from previous contexts, call
               # guard function if available.
               #
               # Note that if we encounter a guard error, instead of stopping parsing
               # in this branch, we save the guard error in the context and continue
               # to parse. If the path survives the final token we resolve all the
               # terminalMatches in _processFinalCtxs(), and either get a normal
               # result or throw a guard error.
               #
               # Consider the following syntax:
               #
               # ( foo1 abc ) | ( foo2 efg )
               #
               # where foo1 and foo2 are the keyword "foo" but foo1 is guarded.
               #
               # If we enter "foo abc", we'd like to see a guard error instead of
               # invalid input error. This is achieved by continuing to parse "abc"
               # after "foo1", so we have a terminalMatch at the end with abc with
               # guardError.
               guardError = None
               # Check for deprecated commands
               guardCode = CliParserCommon.deprecatedCmdGuardCode(
                  node.deprecatedByCmd_, parserOptions.startupConfig )
               if ( guardCode is None and node.guard_ is not None and
                    not parserOptions.disableGuards ):
                  guardToken = self._guardTokenFromMatchResult( mr, token )
                  guardCode = node.guard_( mode, guardToken )
               if guardCode is not None:
                  guardError = CliParserCommon.GuardError( guardCode,
                                                           index=index,
                                                           token=token )

         if guardError is not None and ctx.guardError is None:
            t0( "node", node.name_, "raised guard error:", guardError.guardCode )
            ctx.guardError = guardError

         if ctx.node.priority_ < bestMatchPriority:
            # this is better than bestMatch
            if ctx.guardError:
               if ( guardedMatch is None or
                    guardedMatch.node.priority_ > ctx.node.priority_ ):
                  guardedMatch = ctx
            else:
               bestMatchPriority = ctx.node.priority_
               if guardedMatch and guardedMatch.node.priority_ >= bestMatchPriority:
                  guardedMatch = None

         if ctx.guardError and node.storeSharedResult_:
            # shared result can be shared by both guarded and unguarded matches,
            # which causes confusion, so let's stop here.
            t2( "skip guarded node", node.name_, "due to storeSharedResult" )
            continue

         # I matched, so all my children should be discarded
         parentMultiTokenCtx = ctx

         if mr is CliParserCommon.partialMatch:
            # we do not have a match, but we should continue to include this
            # context in the next match.
            ctxResult.nextCtxs.append( ctx ) # pylint: disable=no-member
            ctxResult.matchFound = True
            if not ctx.guardError:
               ctxResult.matchHasNextNodes = True
            continue

         # If this node is special and can only be a certain amount of time
         # we have to increment the cnt of the how many times it has matched.
         # if we have to take care not to touch matchedNodes directly as it
         # belong to this ctx, instead we would make a copy that we modify
         # and will place into our new ctx.
         if node.maxMatches_:
            if matchedNodes is None:
               matchedNodes = {}
            else:
               matchedNodes = dict( matchedNodes )
            matchObj = self._matchObj( node )
            if matchObj not in matchedNodes:
               matchedNodes[ matchObj ] = 0
            matchedNodes[ matchObj ] += 1

         if not mode or mode.privileged:
            nextNodes = node.nextNodes_
         else:
            nextNodes = [ n for n in node.nextNodes_ if not n.privileged_ ]

         if not matchedNodes:
            # this is an optimization. If we have seen any of the 'special' nodes
            # that require us to count the max number of uses don't create a new list
            # since we'll never filter anything out
            availableNextNodes = nextNodes
         else:
            availableNextNodes = [ n for n in nextNodes if not
                                   ( n.maxMatches_ and
                                     self._maxMatched( n, matchedNodes ) ) ]

         if node.storeSharedResult_:
            t2( "store shared result", mr.result, "for", node.name_ )
            if ctx.sharedResult is None:
               ctx.sharedResult = {}
            if not node.isRepeatable_:
               assert mr.more or node.name_ not in ctx.sharedResult
               ctx.sharedResult[ node.name_ ] = mr.result
            else:
               assert not mr.more, ( 'sharedResult with repeatable multi-token '
                                     'matchers is not supported' )
               if node.name_ in ctx.sharedResult:
                  ctx.sharedResult[ node.name_ ].append( mr.result )
               else:
                  ctx.sharedResult[ node.name_ ] = [ mr.result ]

         ctx.result = mr.result
         if mr.aaaTokens:
            ctx.aaaTokens = mr.aaaTokens
         if not ctx.guardError and availableNextNodes:
            ctxResult.matchHasNextNodes = True
         ctxResult.matchFound = True

         if not finalToken:
            if mr.more:
               t2( "ctx returns more, continue in next" )
               ctxResult.nextCtxs.append( ctx ) # pylint: disable=no-member
            # We have to make a new ctx for the next set of children of
            # this node because we still have more to parse
            #
            # Note if the current ctx is multi-token, creating those child ctxs
            # is only opportunistic as if the next token can be matched by the
            # current ctx again, those child ctxs should be abandoned (parent
            # is greedy). This is handled by the parentMultiTokentCtx logic at
            # the beginning of this function.
            for nextNode in availableNextNodes:
               t3( "add next node", nextNode.name_ )
               ctxResult.nextCtxs.append( # pylint: disable=no-member
                  ParserCtx( nextNode, matchedNodes, ctx ) )

         # this is the end of the line for this tree branch.
         if node.cmdHandler_:
            ctxResult.terminalMatches.append( ctx ) # pylint: disable=no-member

      ctxResult.guardedMatch = guardedMatch
      ctxResult.matchFound |= bool( guardedMatch )
      return ctxResult

   def _availableNextNodes( self, nextNodes, matchedNodes ):
      if not matchedNodes:
         # this is an optimization. If we have seen any of the 'special' nodes
         # that require us to count the max number of uses don't create a new list
         # since we'll never filter anything out
         return nextNodes

      availableNextNodes = []
      for node in nextNodes:
         if node.maxMatches_ and matchedNodes and node in matchedNodes:
            if matchedNodes[ node ] >= node.maxMatches_:
               continue
         availableNextNodes.append( node )
      return availableNextNodes

   def _callCompletionFuncs( self, mode, ctx, partialToken, parserOptions ):
      ''' Will run a node's completion function with a given partial token.
      Returns all of the completions the given node can do'''
      if ctx.guardError:
         return set()
      node = ctx.node
      if node.hidden_:
         # if it is hidden we don't try to run the completer on it
         return set()

      completions = node.completions( mode, ctx, partialToken )
      if not completions:
         return set()

      deprecatedCmdGuardCode = CliParserCommon.deprecatedCmdGuardCode(
         node.deprecatedByCmd_,
         parserOptions.startupConfig )
      guard = node.guard_
      if ( deprecatedCmdGuardCode is not None or
           guard and not parserOptions.disableGuards ):
         # Check if the completions need to be guarded
         results = set()
         for completion in completions:
            if deprecatedCmdGuardCode is not None:
               guardCode = deprecatedCmdGuardCode
            else:
               guardCode = guard( mode, completion.name )
            if guardCode is not None:
               # Make a copy as we might not be able to modify the
               # original completion.
               completion = copy.copy( completion )
               completion.guardCode = guardCode
            results.add( completion )
      else:
         results = set( completions )
      return results

   def _errorToken( self, tokens, index ):
      return None if index is None else tokens[ index ]

   def _getCompletions( self, token, mode, currCtxs, parserOptions ):
      ''' Runs through all of the currCtxs and get completions for the
      next token within the context'''
      completions = set()
      for ctx in currCtxs:
         completion = self._callCompletionFuncs( mode, ctx, token,
                                                 parserOptions )
         completions.update( completion )
      return completions

   def _parseTokens( self, mode, tokens, parserOptions, forCompletion ):
      # Given a set of tokens this will go through the parse tree. These values:
      # 1) Terminal matches found
      # 2) Remaining contexts
      # 3) the last ctxResult seen
      # 4) last token index processed (for ParseError)

      if mode and mode.session and not forCompletion:
         # Optimization: if we saved root nodes from previous command handler,
         # use it for parsing first. If it fails, fall back to general root nodes.
         parserOptionsCopy = copy.copy( parserOptions )
         parserOptionsCopy.autoComplete = False
         lastRootNodeMode, lastRootNodes = mode.session.sessionData( 'lastRootNodes',
                                                                     ( None, None ) )
         if mode is lastRootNodeMode and lastRootNodes:
            t0( "use cached root nodes for mode", mode )
            initialCtxs = [ ParserCtx( node, None, None ) for node in lastRootNodes ]
            try:
               result = self._parseTokensWithCtxs( mode, tokens, parserOptionsCopy,
                                                   forCompletion,
                                                   initialCtxs )
               if result[ 0 ]: # we have terminal matches
                  return result
            except CliParserCommon.ParserError:
               pass

            t0( 'clear cached root nodes for cache miss' )
            mode.session.sessionDataIs( 'lastRootNodes', ( None, None ) )

      if tokens and tokens[ 0 ].lower() in self.rootKwNodeMap_:
         firstToken = tokens[ 0 ].lower()
         initialCtxs = [ ParserCtx( node, None, None )
                         for node in self.rootKwNodeMap_[ firstToken ] ]
      else:
         initialCtxs = [ ParserCtx( node, None, None ) for node in self.rootNode_ ]

      return self._parseTokensWithCtxs( mode, tokens, parserOptions, forCompletion,
                                        initialCtxs )

   def _parseTokensWithCtxs( self, mode, tokens, parserOptions, forCompletion,
                             initialCtxs ):
      currCtxs = initialCtxs
      processCtxResult = None
      terminalMatches = []
      guardedMatch = None
      tokensLen = len( tokens )
      i = None
      for i, token in enumerate( tokens ):
         finalToken = i == tokensLen - 1
         try:
            processCtxResult = self._inhaleCtxs( i, token, mode, currCtxs,
                                                 parserOptions,
                                                 False if forCompletion
                                                 else finalToken )
         except CliParserCommon.ParseError as e:
            e.index = i
            e.token = token
            raise

         if finalToken:
            terminalMatches = processCtxResult.terminalMatches

         # If this round of parsing generates only guarded matches
         # at the highest priority, we save the returned guarded match,
         # which could come in handy in case no normal terminal
         # matches exist for the next token. However, if we have
         # any unguarded matches, we clear the previous saved guarded match.
         if processCtxResult.guardedMatch:
            guardedMatch = processCtxResult.guardedMatch
         elif processCtxResult.matchFound:
            guardedMatch = None

         currCtxs = processCtxResult.nextCtxs
         if not currCtxs:
            break

      if guardedMatch and guardedMatch not in terminalMatches:
         terminalMatches.append( guardedMatch )

      return terminalMatches, currCtxs, processCtxResult, i

   def getCompletions( self, mode, tokens, partialToken, parserOptions ):
      t0( 'getCompletions', tokens, 'partial', partialToken )
      result = self._parseTokens( mode, tokens, parserOptions, True )
      terminalMatches, currCtxs, _, index = result
      if not terminalMatches and not currCtxs:
         raise CliParserCommon.InvalidInputError(
            index=index, token=self._errorToken( tokens, index ) )

      completions = set()
      for ctx in currCtxs:
         completions.update( self._callCompletionFuncs( mode, ctx, partialToken,
                                                        parserOptions ) )
      try:
         terminalMatch = self._processFinalCtxs( terminalMatches )
      except CliParserCommon.GuardError:
         if not completions:
            # raise GuardError if no other completions
            raise
         terminalMatch = None

      if partialToken == '' and terminalMatch is not None:
         # this means that we are at the end of the line
         completions.add( CliParserCommon.eolCompletion )
      return completions

   def _inhaleCtxs( self, index, token, mode, currCtxs, parserOptions, finalToken ):
      ''' Our parser does a breadth first search. This function will take the all of
      the contexts for our current depth and returns a ctxResult which contains the
      result for the next level. If not matches are found, this function will then
      auto-complete to try to get an exact match'''
      t2( "inhale", token )
      processCtxResult = self._processCurrCtxs( index, token, mode, currCtxs,
                                                parserOptions, finalToken )
      if processCtxResult.matchFound:
         # this means we found some matches
         return processCtxResult

      # we couldn't match anything, lets try to complete the command
      # to see if we find any matches
      completions = self._getCompletions( token, mode, currCtxs, parserOptions )
      t2( "completions for", token, "is:", completions )
      if not completions:
         return processCtxResult

      # If we have a mix of guard-blocked and guard-allowed completions,
      # we only care about the guard-allowed ones at this point.
      unguardedCompletions = [ c for c in completions if c.guardCode is None ]
      if unguardedCompletions:
         completions = unguardedCompletions

      # for autocomplete, we really only care about literal ones.
      completions = [ c for c in completions if c.literal ]

      t2( "completions are", completions )
      # At this point, 'completions' is either all guard-allowed, or
      # all guard-blocked.
      # The code below implements the following logic:
      # * If I have exactly one literal unguarded completion, then accept it.
      # * If I have exactly one literal guarded completion, accept it and expect
      #   a guard error.
      # * If I have zero or more than one literal unguarded or guarded completion,
      #   ambiguous.
      if len( completions ) != 1:
         # This could be problematic for multi-token matcher, as both the matcher
         # and its children might have completions, where only the parent should
         # win if it has a liberal completion. But we do not have a multi-token
         # matcher with liberal completion in the middle, so let's not worry about
         # it for now.
         raise CliParserCommon.AmbiguousCommandError( index=index,
                                                      token=token )
      if not parserOptions.autoComplete:
         raise CliParserCommon.IncompleteTokenError( index=index,
                                                     token=token )

      token = completions[ 0 ].name
      return self._processCurrCtxs( index, token, mode, currCtxs,
                                    parserOptions, finalToken )

   def parse( self, mode, tokens, parserOptions ):
      t0( "parse", tokens )
      # This is a BREADTH first search of our parse tree.
      lastCtxResult = None
      result = self._parseTokens( mode, tokens, parserOptions, False )
      terminalMatches, _, lastCtxResult, index = result

      # See if we didn't find any terminal matches. If we don't have any terminal
      # matches we could raise 2 different errors,
      # IncompleteCommand and InvalidInput. The difference being that
      # IncompleteCommand means that we if we had more input we *could* match
      # more, but with InvalidInput we could never match this subset of tokens.
      guardedMatches = [ ctx for ctx in terminalMatches if ctx.guardError ]
      if len( terminalMatches ) == len( guardedMatches ):
         # everything is guarded
         errorToken = self._errorToken( tokens, index )
         if ( lastCtxResult and # pylint: disable=no-else-raise
              lastCtxResult.matchHasNextNodes ):
            assert lastCtxResult.matchFound
            raise CliParserCommon.IncompleteCommandError( index=index,
                                                          token=errorToken )
         elif not guardedMatches:
            raise CliParserCommon.InvalidInputError( index=index,
                                                     token=errorToken )

      terminalMatch = self._processFinalCtxs( terminalMatches )

      # at this point we should only 1 match, so lets assert that that's the case
      # and return that one match
      assert terminalMatch
      return self._getCmdHandlerInfo( mode, terminalMatch )

   def _getCommandHandlerNodes( self, node, seenNodes, cmdHandlers ):
      if node in seenNodes:
         return
      seenNodes.add( node )

      if node.cmdHandler_ and node.cmdHandler_.isRealCmd():
         cmdHandlers.add( node.cmdHandler_ )

      for childNode in node.nextNodes_:
         self._getCommandHandlerNodes( childNode, seenNodes, cmdHandlers )

   def getCommandHandlerNodes( self ):
      seenNodes = set()
      cmdHandlers = set()
      for node in self.getRootNodes():
         self._getCommandHandlerNodes( node, seenNodes, cmdHandlers )
      return cmdHandlers

def getNodeCount( rootNodes, nodeSeen=None ):
   # for benmarking: get the number of all nodes under a set of rootNodes
   if nodeSeen is None:
      nodeSeen = set()
   totalNodes = 0
   for node in rootNodes:
      if node not in nodeSeen:
         totalNodes += 1
         nodeSeen.add( node )
         totalNodes += getNodeCount( node.nextNodes_, nodeSeen )
   return totalNodes
