# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import inspect
import itertools
import os

import Ark
from CliGuards import * # pylint: disable-msg=wildcard-import
import CliParserCommon
from CliParserCommon import * # pylint: disable-msg=wildcard-import
import CliParseTree
import Tracing

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2

# The following flag enables all instances of a mode to share the same
# modeParseTree (commands registered to the mode instead of using modelets).
# It enables more sharing for modes with non-default instanceRuleKey() such
# as the interface mode.
#
# Default: True (use SHARE_BASE_MODE_TREE=0 to turn it off)
SHARE_BASE_MODE_TREE = ( os.environ.get( 'SHARE_BASE_MODE_TREE', '1' ) != '0' )

#-------------------------------------------------------------------------------
# Modes and modelets.
#-------------------------------------------------------------------------------

# This is just a placeholder for registered command classes that will
# be added to the ParseTree of the Mode eventually.
class _ModeParseTreeBase:
   __slots__ = ( 'commandClasses_', 'closed_' )

   def __init__( self ):
      self.closed_ = False
      self.commandClasses_ = [] # parse trees added to this modelet

   def addCommandClass( self, cl ):
      assert not self.closed_, "Cannot add command class after modelet instantiation"
      self.commandClasses_.append( cl )

   def addToParseTree( self, parseTree ):
      self.closed_ = True
      for cl in self.commandClasses_:
         cl.initialize()
         parseTree.addCommandClass( cl )

   def getSyntaxes( self, hidden=False, functions=False,
                    truncate=200, verbatim=False ):
      ''' `verbatim` overrides `hidden`.'''
      for cls in self.commandClasses_:
         # Some classes are added as a tuple( class, rootNode ).
         if functions:
            # Find a handler.
            for h in ( 'handler', 'noHandler', 'defaultHandler',
                       'noOrDefaultHandler' ):
               func = getattr( cls, h, None )
               if func is None:
                  continue

               funcName = CliParserCommon.debugFuncName( func )
               break
         else:
            funcName = None

         if verbatim:
            syntaxes = cls.verbatimSyntax()
         elif hidden or not cls.hidden:
            syntaxes = [ cls.prettySyntax( truncate=truncate ) ]
         else:
            syntaxes = []

         # pylint: disable-next=consider-using-f-string
         tail = ( ' => %s' % funcName ) if functions else ''
         for syntax in syntaxes:
            yield syntax + tail

class ModeParseTree( _ModeParseTreeBase ):
   __slots__ = ( 'baseTree_', 'parseTreeMap_', 'modeletClassMap_' )

   def __init__( self ):
      self.baseTree_ = None
      self.parseTreeMap_ = {}
      self.modeletClassMap_ = {}
      _ModeParseTreeBase.__init__( self )

   def getBaseParseTree( self ):
      return self.baseTree_

   def initBaseParseTree( self ):
      if not self.baseTree_:
         pt = CliParseTree.CliParseTree()
         self.addToParseTree( pt )
         self.baseTree_ = pt

   def getParseTree( self, instKey ):
      return self.parseTreeMap_.get( instKey )

   def setParseTree( self, instKey, parseTree ):
      self.parseTreeMap_[ instKey ] = parseTree

   def getModeletClasses( self, instKey ):
      return self.modeletClassMap_.get( instKey )

   def setModeletClasses( self, instKey, modeletClasses ):
      self.modeletClassMap_[ instKey ] = modeletClasses

allModes = {}

class Mode:
   inhibitImplicitModeChange = False
   inhibitAmbiguousCommandErrorToParent = False
   showCommandRegistered = False
   commentSupported = False
   commonHelpSupported = False
   modeParseTreeShared = False
   allowCache = False
   privileged = False

   def __init__( self, parent, session ):
      cls = self.__class__
      name = cls.__name__
      allModes[ name ] = self
      t2( "mode", name, "starting" )
      # Mode subclasses must have 'prompt' and 'modeParseTree' attributes
      assert hasattr( cls, 'prompt' )
      self.parent_ = parent
      self.session_ = session
      instKey = self.instanceRuleKey() # pylint: disable=assignment-from-none

      # create a new parse tree (note instKey=None means always share)
      self.instanceParseTree = cls.getOrCreateInstanceParseTree( instKey,
                                                                 mode=self )
      # build modelet instances
      self.modeletMap = self.buildModeletMap( instKey )

   def __init_subclass__( cls ):
      if cls.modeParseTreeShared:
         assert cls.modeParseTree_, f'Parent class of {cls} must define a parse tree'
      else:
         cls.modeParseTree_ = ModeParseTree()

   @classmethod
   def getExtraModeParseTree( cls ):
      return None

   @classmethod
   def commandClasses( cls ):
      yield from cls.modeParseTree_.commandClasses_

   @classmethod
   @Ark.synchronized()
   def getOrCreateInstanceParseTree( cls, instKey, mode=None ):
      # This generates the instance parse tree for the mode. Caller provides
      # the right instance key. If you do not pass in a valid mode instance,
      # modelets won't be added if instKey is not None (e.g., interface mode),
      # so only do so in testing.
      pt = cls.modeParseTree_.getParseTree( instKey )
      if not pt:
         assert cls.modeParseTree_.getModeletClasses( instKey ) is None
         pt = CliParseTree.CliParseTree()
         if SHARE_BASE_MODE_TREE:
            cls.modeParseTree_.initBaseParseTree()
         else:
            cls.modeParseTree_.addToParseTree( pt )
         try:
            allModeletClasses = cls.modeletClasses
         except AttributeError:
            allModeletClasses = []
         if instKey is None:
            mc = allModeletClasses
         else:
            mc = []

         for c in allModeletClasses:
            if instKey is None or mode and c.shouldAddModeletRule( mode ):
               t0( "add modelet", c, "parse tree to mode", mode )
               c.modeletParseTree.addToParseTree( pt )
               if instKey is not None:
                  mc.append( c )

         if SHARE_BASE_MODE_TREE:
            # If instKey is None, we don't share modeParseTree, so we could
            # keep pt open for merge which might change modeParseTree.
            #
            # This also means we should only have extraModeParseTree for singleton
            # modes.
            pt.mergeRootNodes( cls.modeParseTree_.getBaseParseTree().getRootNodes(),
                               close=( instKey is not None ) )

         extraModeParseTree = cls.getExtraModeParseTree()
         if extraModeParseTree:
            extraModeParseTree.initBaseParseTree()
            # merge the extra tree into the instance tree, but close it
            # so we make sure nobody can merge afterwards and potentially
            # change the extra tree.
            extraPt = extraModeParseTree.getBaseParseTree()
            pt.mergeRootNodes( extraPt.getRootNodes(), close=True )
         cls.modeParseTree_.setParseTree( instKey, pt )
         cls.modeParseTree_.setModeletClasses( instKey, mc )
      return pt

   def buildModeletMap( self, instKey ):
      """build modelet class -> instance map"""
      cls = self.__class__
      name = cls.__name__
      mc = cls.modeParseTree_.getModeletClasses( instKey )
      t2( "mode", name, "creating modelets from", len( mc ), "classes..." )
      mm = {}
      for c in mc:
         if instKey is None or c.needModeletMap:
            t2( "creating modelet", c )
            mm[ c ] = c( self )

      mm[ cls ] = self
      t2( "mode", name, "initialization complete" )
      return mm

   def instanceRuleKey( self ):
      # Sharing by default
      return None

   def longPrompt( self ):
      """Return an extended (long) prompt for the mode.  The derived
      mode class may override this.  The idea is that the short prompt
      is a fixed value for a given mode, but the extended prompt can
      depend on mode parameters.  For example, config-if mode has a
      prompt of '(config-if)#' and an extended prompt of
      '(config-if-Ethernet2)#'. Use the CliMode implementation of this
      method if the sub-class chooses to override, otherwise just
      return the prompt."""

      return self.prompt # pylint: disable-msg=no-member

   def shortPrompt( self ):
      return self.prompt # pylint: disable-msg=no-member

   def historyKey( self ):
      return self.prompt # pylint: disable-msg=no-member

   entityManager = property( lambda self: self.session_.entityManager )
   sysdbRoot = property( lambda self: self.session_.entityManager.root() )
   sysname = property( lambda self: self.session_.entityManager.sysname() )
   session = property( lambda self: self.session_ )

   @classmethod
   def addModelet( cls, modeletClass ):
      """Add a new modelet class to this mode class."""
      try:
         cls.modeletClasses.append( modeletClass )
      except AttributeError:
         cls.modeletClasses = [ modeletClass ]

   def childMode( self, modeClass, **kwargs ):
      """Create a child of this mode with the specified class and the specified
      keyword arguments."""

      return modeClass( parent=self, session=self.session_, **kwargs )

   @classmethod
   def addCommandClass( cls, commandClass ):
      cls.modeParseTree_.addCommandClass( commandClass )

   @classmethod
   def addShowCommandClass( cls, commandClass ):
      cls.showCommandRegistered = True
      cls.modeParseTree_.addCommandClass( commandClass )

   @classmethod
   def isConfigMode( cls ):
      return False

   @classmethod
   def isGlobalConfigMode( cls ):
      return False

   @classmethod
   def getSyntaxes( cls, **kwargs ):
      syntaxes = []
      if cls.modeParseTree_:
         syntaxes = cls.modeParseTree_.getSyntaxes( **kwargs )
         extraModeParseTree = cls.getExtraModeParseTree()
         if extraModeParseTree:
            syntaxes = itertools.chain( syntaxes,
                                        extraModeParseTree.getSyntaxes( **kwargs ) )
      return syntaxes

   def addMessage( self, msg ):
      self.session.addMessage( msg )

   def addWarning( self, msg ):
      self.session.addWarning( msg )

   def addError( self, msg ):
      self.session.addError( msg )

   def addWarningInteractOnly( self, msg ):
      self.session.addWarningInteractOnly( msg )

   def addErrorAndStop( self, msg ):
      '''Add an error message and gracefully stop any further handler execution.
      '''
      self.addError( msg )
      raise CliParserCommon.AlreadyHandledError

   def onInitialMode( self ):
      # This can be implemented by derived Mode classes that need to know when
      # the mode has been constructed as the initial mode in a session.
      pass

   def onExit( self ):
      # This can be implemented by derived Mode classes that need to know when
      # the mode is exited.  In the current implementation, this can actually
      # be called more than once as a mode is exited.
      pass

   def getCompletions( self, tokens, partialToken, startWithPartialToken=False ):
      disableGuards = not self.session_.guardsEnabled() if self.session_ else False
      startupConfig = self.session.startupConfig() if self.session_ else False
      parserOptions = ParserOptions( disableGuards=disableGuards,
                                     startupConfig=startupConfig )

      # Get command completions with the normal space tokenizer; if it fails,
      # try using the pipeTokens as additional token delimiters.
      exc = None
      try:
         completions = self.instanceParseTree.getCompletions( self, tokens,
               partialToken, parserOptions )
         if completions:
            return list( completions )
      except CliParserCommon.InvalidInputError:
         # if we don't have a show command registered that means that
         # this is the end of the line and we should raise this error
         if not self.showCommandRegistered: # pylint: disable=no-else-raise
            raise
         else:
            exc = sys.exc_info()

      # try to get completions with pipe tokenizer
      pipeCompl = None
      try:
         fullTokens = splitOnPipe( tokens )
         if fullTokens is not None:
            pipeCompl = self.instanceParseTree.getCompletions( self, fullTokens,
                  partialToken, parserOptions )
         else:
            # no pipe in tokens, maybe partialToken?
            pTokens = splitOnPipe( [ partialToken ] )
            if pTokens is not None:
               pipeCompl = self.instanceParseTree.getCompletions( self,
                     tokens + pTokens[ : -1 ], pTokens[ -1 ], parserOptions )
               if startWithPartialToken:
                  # Make sure the completions all start with partialToken;
                  # This is expected by tab completion.
                  prefix = ''.join( pTokens[ : -1 ] )
                  compl = set()
                  for c in pipeCompl:
                     compl.add(
                           CliParserCommon.Completion( prefix + c.name, c.help,
                                                       c.literal, c.partial,
                                                       c.guardCode ) )
                  pipeCompl = compl
         # If we found a pipe and it returns completions (even empty),
         # we return empty. This means that if the normal tokens throw
         # InvalidInputError, and pipe tokens return empty list, we
         # return empty list.
         if pipeCompl is not None:
            return pipeCompl
      except CliParserCommon.InvalidInputError:
         pass

      # no pipe or pipe does not parse: return the original result
      if exc: # pylint: disable=no-else-raise
         raise exc[ 1 ].with_traceback( exc[ 2 ] )
      else:
         return []

   def parse( self, tokens, autoComplete=True, authz=True, acct=True ):
      disableGuards = not self.session_.guardsEnabled() if self.session_ else False
      startupConfig = self.session.startupConfig() if self.session_ else False
      parserOptions = ParserOptions( autoComplete=autoComplete,
                                     disableGuards=disableGuards,
                                     startupConfig=startupConfig )

      t0( "parsing", tokens, "in mode", self.__class__.__name__ )
      # Parse command with the normal space tokenizer; if it fails,
      # try using the pipeTokens as additional token delimiters.

      # Make the tokes immutable as callers assume that is the case.
      # For example when parse is called from within the interface range loop
      # if the tokens are not immutable, the 2nd through nth iteration of the
      # loop will have to '*' in place of any 'secret' tokens.
      tokens = tokens[ : ]

      exc = None
      try:
         return self.instanceParseTree.parse( self, tokens, parserOptions )
      except CliParserCommon.ParseError as e:
         exc = e

      if( isinstance( exc, CliParserCommon.InvalidInputError ) and
          self.showCommandRegistered ):
         # invalid input, try pipe split
         tokens = splitOnPipe( tokens )
         if tokens is not None:
            try:
               return self.instanceParseTree.parse( self, tokens, parserOptions )
            except CliParserCommon.InvalidInputError:
               # if it is invalid lets fall through to the old parser
               pass

      # pipe does not parse: raise the original exception
      raise exc # pylint: disable=raising-bad-type

   def checkCommandExists( self, tokens ):
      """Returns True if 'tokens' correspond to a valid command """
      try:
         completions = self.getCompletions( tokens, "" )
         return completions and CliParserCommon.eolCompletion in completions
      except CliParserCommon.ParserError:
         return False

class Modelet:
   needModeletMap = False
   __slots__ = ( 'mode', )

   @classmethod
   def __init_subclass__( cls ):
      cls.modeletParseTree = _ModeParseTreeBase()
      cls.getSyntaxes = cls.modeletParseTree.getSyntaxes

   def __init__( self, mode ):
      self.mode = mode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return True

   @classmethod
   def addCommandClass( cls, commandClass ):
      t0( "modelet", cls, "addCommandClass", commandClass )

      if not cls.needModeletMap:
         handlers = ( 'handler', 'noHandler', 'defaultHandle', 'noOrDefaultHander' )
         for cmdFunc in ( getattr( commandClass, h, None ) for h in handlers ):
            # pylint: disable-next=no-else-continue
            if not cmdFunc or isinstance( cmdFunc, staticmethod ):
               continue
            elif inspect.isfunction( cmdFunc ):
               if getattr( cls, cmdFunc.__name__, None ) is cmdFunc:
                  cls.needModeletMap = True
                  break

               t2( "Potential py3 unbound handler function:",
                   cmdFunc.__name__, "from", cmdFunc.__qualname__ )

      cls.modeletParseTree.addCommandClass( commandClass )

class SingletonModelet( Modelet ):
   '''For singleton config mode (e.g., GlobalConfigMode)'''
