# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
from CliModel import Bool
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Enum
import Tac
import TableOutput
import datetime

class AaaCounters( Model ):
   authenticationSuccess = Int( help= "Number of successful authentication "
                                      "requests" )
   authenticationFail = Int( help= "Number of failed authentication requests" )
   authenticationUnavailable = Int( help= "Number of unavailable authentication "
                                          "requests" )
   authorizationAllowed = Int( help= "Number of allowed authorization requests" )
   authorizationDenied = Int( help= "Number of denied authorization requests" )
   authorizationUnavailable = Int( help= "Number of unavailable authorization "
                                         "requests" )
   accountingSuccess = Int( help= "Number of successful accounting requests" )
   accountingError = Int( help= "Number of error accounting requests" )
   pendingAccountingRequests = Int( help= "Number of pending accounting requests" )
   counterResetTimestamp = Float( help= "Timestamp of last counter reset", 
                                  optional=True )
   
   def render( self ):
      fmt = "%20s:%11d"
      def _printAttr( description, attr ):
         print( fmt % ( description, attr ) )

      print( "Authentication" )
      _printAttr( "Successful", self.authenticationSuccess )
      _printAttr( "Failed", self.authenticationFail )
      _printAttr( "Service unavailable", self.authenticationUnavailable )      
      print()
      print( "Authorization" )
      _printAttr( "Allowed", self.authorizationAllowed )
      _printAttr( "Denied", self.authorizationDenied )
      _printAttr( "Service unavailable", self.authorizationUnavailable )   
      print()
      print( "Accounting" )
      _printAttr( "Successful", self.accountingSuccess )
      _printAttr( "Error", self.accountingError )
      _printAttr( "Pending", self.pendingAccountingRequests )
      if self.counterResetTimestamp:
         self.counterResetTimestamp = ( self.counterResetTimestamp + 
                                        Tac.now() - Tac.utcNow() )
      print( "\nLast time counters were cleared:",
             Ark.timestampToStr( self.counterResetTimestamp ) )
             
def convertFromSeconds( idleTime ):
   ''' Convert idle time in seconds to printable format.
   idleTime will be None if we could not stat the atime on the terminal. It
   will be negative if the clock rolled back between the last access to the tty
   and when we took our snapshot. In either case, don't trust it, and fall back
   to printing 23:59:59 '''

   if idleTime is None or idleTime < 0:
      idleTime = -1

   days = idleTime // 86400
   hours = ( idleTime - ( days * 86400 ) ) // 3600
   minutes = (idleTime - ( ( days * 86400 ) + ( hours * 3600 ) ) ) // 60
   seconds = idleTime - ( ( days * 86400 ) + ( hours * 3600 ) + ( minutes * 60 ) )
   if days > 7:
      return f'{days // 7}w{days % 7}d'
   elif days > 0:
      return f'{days}d{seconds // 3600}h'
   else:
      return f"{hours:02d}:{minutes:02d}:{seconds:02d}"

class ShowAaaUserInfo( Model ):
   sessionTty = Bool( help="Indicates if this is the terminal used by "
                           "the session that just ran this command" )
   lineNumber = Int( help="Line number of the user session" )
   user = Str( help="Login name of the user of this session" )
   idleTime = Int( help="Duration of session's inactivity in seconds" )
   location = Str( help="Source IP address or hostname for vty sessions "
                        "and absent for serial connections", optional=True )

class ShowAaaUsers( Model ):
   serials = Dict( valueType=ShowAaaUserInfo,
                  help="Terminal information for serial connection" )
   vtys = Dict( valueType=ShowAaaUserInfo,
                help="Terminal information for ssh or telnet connections" )

   def render( self ):
      headings = ( ' ', 'Line', 'User', 'Host(s)', 'Idle', 'Location' )
      table = TableOutput.createTable( headings )

      # All entries will be left justified, and all columns have the same
      # format.
      fmt_star = TableOutput.Format( justify="left", minWidth=2, maxWidth=2 )
      fmt_star.padLimitIs( True )
      fmt_line = TableOutput.Format( justify="center" )
      fmt_line.padLimitIs( True )
      fmt = TableOutput.Format( justify="left" )

      table.formatColumns( fmt_star, fmt_line, fmt, fmt, fmt, fmt )

      sessionType = 'con'

      # order the user info by lineNumber

      infoList = []
      for session, sessionType in ( ( self.serials, "con" ),
                                    ( self.vtys, "vty" ) ):
         for key, value in sorted( session.items() ):
            info = ( '*' if value.sessionTty else ' ',
                     f'{value.lineNumber} {sessionType} {key}',
                     value.user,
                     'idle',
                     convertFromSeconds( value.idleTime ),
                     '-' if sessionType == 'con' else value.location )

            infoList.append( ( value.lineNumber, info ) )

      # sorts by line number
      infoList.sort()

      for info in infoList:
         table.newRow( *info[ 1 ] )

      # skip line 2 ( --- )
      lines = table.output().split( '\n' )
      print( lines[ 0 ] )
      for line in lines[ 2: ]:
         print( line )

class ShowAaaSessions( Model ):
   class Sessions( Model ):
      username = Str( help="Login name of the user of this session" )
      role = Str( help="Role assigned to the user" )
      terminal = Str( help="Maps to terminal type and terminal number",
                 optional=True )
      state = Enum( values=( "established", "pending" ),
              help="State of the session either established or pending" )
      sessionStartTime = Int( help="UTC time at which session was started" )
      authMethod = Str( help="Authentication method of the session" )
      remoteAddress = Str( help="Remote ip address", optional=True )
      remoteHost = Str( help="Remote hostname", optional=True )
      service = Str( help="Service", optional=True )

   serials = Dict( keyType=int, valueType=Sessions,
                   help="Session information for serial connections" )
   vtys = Dict( keyType=int, valueType=Sessions,
                help="Session information for ssh or telnet connections" )
   nonInteractives = Dict( keyType=int, valueType=Sessions, 
         help="Session information for nonInteractive connections" )
   def render( self ):
      headings = tuple( ( h, "l" ) for h in (
         "Session",
         "Username",
         "Roles",
         "TTY",
         "State",
         "Duration",
         "Auth",
         "Remote Host" ) )

      table = TableOutput.createTable( headings )

      # All entries will be left justified, and all columns have the same
      # format.
      f = TableOutput.Format( justify="left" )
      f.noPadLeftIs( True )
      table.formatColumns( *( [ f ] * len( headings ) ) )

      for session in [ self.serials, self.vtys, self.nonInteractives ]:
         for key, value in sorted( session.items() ):
            remote = ( value.remoteHost + '@' + value.remoteAddress
                    if value.remoteHost and value.remoteAddress else
                    ( value.remoteHost if value.remoteHost else
                    value.remoteAddress ) )
            durationStr = int( Tac.utcNow() - value.sessionStartTime )

            table.newRow( key, value.username, value.role,
                          value.terminal or '',
                          "P" if value.state == 'pending' else 'E',
                          str( datetime.timedelta( seconds=durationStr ) ),
                          str( value.authMethod ),
                          remote if remote else " " )

      # 'table.output()' adds an empty line to terminate the table. The
      # tests don't expect this line, so remove it.
      print( "\n".join( table.output().split( "\n" )[ : -1 ] ) )

