#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

# pylint: disable-msg=wrong-import-order
# pylint: disable-msg=unidiomatic-typecheck
# pylint: disable-msg=wrong-import-order

from SysConstants.in_h import IPPROTO_IP, IPPROTO_ICMP, IPPROTO_UDP, IPPROTO_TCP
from SysConstants.in_h import IPPROTO_ICMPV6, IPPROTO_SCTP

import AclLib, AclCliLib
from AclCliLib import getRuleById
from CliPlugin import AclCliRules
import CliCommand
import CliCommon
import CliMatcher
import CliParser
import Ethernet
import LazyMount
import Tac

aclStatus = None

class AclCfgCmdChains:
   def __init__( self ):
      # pylint: disable-next=import-outside-toplevel
      from CliPlugin.AclCli import dpAclSctpPortMatchSupported
      self.defaultSeqStart = 10
      self.defaultSeqInc = 10

      self.actionMatcher = CliMatcher.EnumMatcher( {
         'permit' : 'Specify packets to accept',
         'deny': 'Specify packets to drop'
      } )
      self.actionNode = CliCommand.Node( self.actionMatcher,
                                         storeSharedResult=True )
      self.permitActionMatcher = CliMatcher.EnumMatcher( {
         'permit' : 'Specify packets to accept'
      } )
      self.permitActionNode = CliCommand.Node( self.permitActionMatcher,
                                             storeSharedResult=False )

      ########################################################
      #                       IP filters
      ########################################################
      self.ipProtoNumMatcher = CliMatcher.IntegerMatcher(
         0,
         255,
         helpdesc='IP protocol number',
         priority=CliParser.PRIO_LOW )
      self.protoIpMatcher = AclCliRules.ipProtocolMatcher(
         'ip',
         IPPROTO_IP,
         'Any Internet Protocol' )
      self.protoIcmpMatcher = AclCliRules.ipProtocolMatcher(
         'icmp',
         IPPROTO_ICMP,
         'Internet Control Message Protocol' )
      self.protoTcpMatcher = AclCliRules.ipProtocolMatcher(
         'tcp',
         IPPROTO_TCP,
         helpdesc='TCP' )
      self.protoUdpMatcher = AclCliRules.ipProtocolMatcher(
         'udp',
         IPPROTO_UDP,
         helpdesc='UDP' )
      self.protoNvgreMatcher = AclCliRules.diProtocolMatcher(
         'nvgre',
         AclLib.IPPROTO_GRE,
         helpdesc='Network Virtualization using Generic Routing Encapsulation',
         validate='dpAclNvgreSupported',
         alias='nvgreSelected' )
      self.protoMplsOverGreMatcher = AclCliRules.diProtocolMatcher(
         'mpls-over-gre',
         AclLib.IPPROTO_MPLSOVERGRE,
         helpdesc='MPLS over GRE',
         validate='dpAclMplsOverGreSupported' )
      self.protoMplsOverGreV6Matcher = AclCliRules.diProtocolMatcher(
         'mpls-over-gre',
         AclLib.IPPROTO_MPLSOVERGRE,
         helpdesc='MPLS over GRE',
         validate='dpAclMplsOverGreV6Supported' )
      self.protoVxlanMatcher = AclCliRules.diProtocolMatcher(
         'vxlan',
         IPPROTO_UDP,
         helpdesc='Virtual Extensible LAN',
         validate='dpAclVxlanSupported',
         alias='vxlanSelected' )
      self.protoGreMatcher = AclCliRules.diProtocolMatcher(
         'gre',
         AclLib.IPPROTO_GRE,
         helpdesc='Generic Routing Encapsulation',
         validate=None
         )
      self.protoGtpMatcher = AclCliRules.diProtocolMatcher(
         'gtp',
         IPPROTO_UDP,
         helpdesc='GPRS Tunneling Protocol',
         validate='dpAclGtpSupported' )
      self.protoSctpMatcher = AclCliRules.ipProtocolMatcher(
         'sctp',
         IPPROTO_SCTP,
         helpdesc='Stream Control Transport Protocol',
         validate='dpAclSctpSupported' )
      self.etherTypeMatcher = AclCliRules.EtherTypeMatcher( 'ip' )

      # Fixme - consider moving rules in to a derived class
      # IP source/destination prefix
      self.ipSourceMatcher = AclCliRules.ipHostRuleMatcher( 'source', 'source' )
      self.ipMplsOverGreSrcMatcher = AclCliRules.ipMplsOverGreHostMatcher(
            'source', 'source', AclLib.anyIpAddr )
      self.ipDestinationMatcher = AclCliRules.ipHostRuleMatcher( 'destination',
                                                                 'destination' )
      self.ipMplsOverGreDstMatcher = AclCliRules.ipMplsOverGreHostMatcher(
            'destination', 'destination', AclLib.anyIpAddr )

      self.nexthopGroupNameExpression = AclCliRules.NexthopGroupNameExpression

      self.vlanExpression = AclCliRules.vlanMatcher()
      self.mplsTwoLabelMatcher = AclCliRules.mplsTwoLabelMatcher()
      self.mplsFilterMatcher = AclCliRules.mplsFilterMatcher()
      self.userL4Matcher = AclCliRules.userL4Matcher()
      self.payloadMatcher = AclCliRules.payloadMatcher()
      self.payloadIp6Matcher = AclCliRules.payloadMatcher( ipv6Filter=True )
      self.payloadMacMatcher = AclCliRules.payloadMatcher( macFilter=True )
      self.metadataExprFactory = AclCliRules.MetadataOffsetExprFactory()

      # source/destination port
      self.udpDstPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_UDP, 'dst',
                                                               'udp', alias='dport' )
      self.udpSrcPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_UDP, 'src',
                                                               'udp', alias='sport' )
      self.tcpDstPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_TCP, 'dst',
                                                               'tcp', alias='dport' )
      self.tcpSrcPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_TCP, 'src',
                                                               'tcp', alias='sport' )
      self.sctpDstPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_SCTP, 'dst',
                                                                'sctp',
                                                                alias='dport',
                                                guard=dpAclSctpPortMatchSupported )
      self.sctpSrcPortSpecMatcher = AclCliRules.portSpecMatcher( IPPROTO_SCTP, 'src',
                                                                'sctp',
                                                guard=dpAclSctpPortMatchSupported,
                                                                alias='sport' )

      self.trackedMatcher = CliMatcher.KeywordMatcher( 'tracked',
                      helpdesc='Match packets in existing ICMP/UDP/TCP connections' )

      self.ipv4DscpMatcher = AclCliRules.DscpMatcher( 'ip' )
      self.ecnMatcher = AclCliRules.IpEcnMatcher()
      self.ipLenMatcher = AclCliRules.IpLenSpecMatcher()
      self.ipv6FlowLabelMatcher = AclCliRules.Ipv6FlowLabelMatcher()
      self.innerIpMatcher = AclCliRules.innerIpHostMatcher( 'innerIpOpt' )
      self.innerIp6Matcher = AclCliRules.innerIpHostMatcher( 'innerIp6Opt' )
      self.innerIpProtoMatcher = AclCliRules.innerIpHostProtoMatcher(
            'innerIpProtoOpt' )
      self.innerIp6ProtoMatcher = AclCliRules.innerIpHostProtoMatcher(
            'innerIp6ProtoOpt' )
      self.icmpMatcher = AclCliRules.IcmpOptMatcher( AclCliLib.icmpMessages,
            AclLib.ICMP_ALL, name='icmp' )
      self.fragmentsMatcher = CliMatcher.KeywordMatcher( 'fragments',
                                        helpdesc='Match non-head fragment packets' )
      self.tcpOptMatcher = AclCliRules.tcpOptionsMatcher()
      self.tcpFlagMaskMatcher = AclCliRules.tcpFlagMaskMatcher()
      self.gtpVersionProtoMatcher = AclCliRules.gtpVersionProtoMatcher()
      self.greExpression = AclCliRules.GreExprFactory()

      ########################################################
      #         Options common to all IP protocols
      ########################################################
      self.fragmentsNode = CliCommand.Node( matcher=self.fragmentsMatcher,
                                            maxMatches=1 )
      def ipOptAdapter( mode, args, argsList ):
         if 'ipOpt' in args:
            return
         ipOpt = []
         args[ 'ipOpt' ] = ipOpt
         if 'tracked' in args:
            ipOpt.append( ( 'tracked', args.pop( 'tracked' )[ 0 ] ) )
         if 'fragments' in args:
            ipOpt.append( ( 'fragments', args.pop( 'fragments' )[ 0 ] ) )
         if 'DSCP' in args:
            ipOpt.append( ( 'dscpOpt', args.pop( 'DSCP' ) ) )
            ipOpt.append( ( 'dscpMaskOpt', args.pop( 'DSCP_MASK' ) ^ 0x3F ) )
         if 'ECN' in args:
            ipOpt.append( ( 'ecn', args.pop( 'ECN' ) ) )
         if 'TTL_MATCH' in args:
            ipOpt.append( ( 'ttl', args.pop( 'TTL_MATCH' ) ) )
         if 'IP_LEN' in args:
            ipOpt.append( ( 'ipLenOpt', args.pop( 'IP_LEN' ) ) )

      class IpOptGenericExpression( CliCommand.CliExpression ):
         expression = '{ fragments | TTL | DSCP | ECN | IP_LEN }'
         data = { 'fragments': self.fragmentsNode,
                  'TTL': AclCliRules.TtlSpecExpression,
                  'DSCP': self.ipv4DscpMatcher,
                  'ECN': self.ecnMatcher,
                  'IP_LEN': self.ipLenMatcher
                }
         adapter = ipOptAdapter

      self.ipOptGenericExpression = IpOptGenericExpression

      class IpOptExpression( CliCommand.CliExpression ):
         expression = '{ tracked | fragments | TTL | DSCP | ECN | IP_LEN }'
         data = {
            'tracked' : self.trackedMatcher,
            'fragments': self.fragmentsNode,
            'TTL': AclCliRules.TtlSpecExpression,
            'DSCP': self.ipv4DscpMatcher,
            'ECN': self.ecnMatcher,
            'IP_LEN': self.ipLenMatcher
         }
         adapter = ipOptAdapter

      self.ipOptExpression = IpOptExpression

      class IpFilterGeneric( CliCommand.CliExpression ):
         expression = ( '( PROTOCOL_NUM | GENERIC_PROTOCOL ) '
                        '[ ETHERTYPE ]'
                        'IP_SRC IP_DST [ IP_OPT ] '
                        '[ USER_L4_VAL ] [ PAYLOAD_OPT ] '
                        '[ METADATA_SPEC ] [ INNER_IP_OPT ]' )
         data = {
            'PROTOCOL_NUM': CliCommand.Node( matcher=self.ipProtoNumMatcher,
               alias='protocol' ),
            'GENERIC_PROTOCOL': AclCliRules.GenericProtocolsMatcher(
               AclCliLib.genericIpProtocols ),
            'ETHERTYPE': self.etherTypeMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'IP_OPT': IpOptGenericExpression,
            'USER_L4_VAL': self.userL4Matcher,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_OPT': self.innerIpMatcher,
         }

      class ProtoIpExpression( CliCommand.CliExpression ):
         expression = ( 'IP_PROTOCOL [ ETHERTYPE ] '
                        'IP_SRC IP_DST '
                        '[ NEXTHOP_GROUP ] [ IP_OPT ] '
                        '[ USER_L4_VAL ] [ PAYLOAD_OPT ] '
                        '[ METADATA_SPEC ] [ INNER_IP_PROTO_OPT ]' )
         data = {
            'IP_PROTOCOL': self.protoIpMatcher,
            'ETHERTYPE': self.etherTypeMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'NEXTHOP_GROUP': self.nexthopGroupNameExpression,
            'IP_OPT': IpOptExpression,
            'USER_L4_VAL': self.userL4Matcher,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_PROTO_OPT': self.innerIpProtoMatcher,
         }

      class ProtoIcmpExpression( CliCommand.CliExpression ):
         expression = ( 'ICMP_PROTOCOL [ ETHERTYPE ] '
                        'IP_SRC IP_DST '
                        '[ ICMP_OPT ] [ IP_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] '
                        '[ INNER_IP_OPT ]' )
         data = {
            'ICMP_PROTOCOL': self.protoIcmpMatcher,
            'ETHERTYPE': self.etherTypeMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'ICMP_OPT': self.icmpMatcher,
            'IP_OPT': IpOptExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_OPT': self.innerIpMatcher,
         }

      class ProtoTcpExpression( CliCommand.CliExpression ):
         expression = ( 'TCP_PROTOCOL [ ETHERTYPE ] '
                        'IP_SRC [ TCP_SRC_PORT ] '
                        'IP_DST [ TCP_DST_PORT ] '
                        '[ TCP_OPT ] [ TCP_FLAGS_MASK ] [ IP_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] '
                        '[ INNER_IP_OPT ]' )
         data = {
            'TCP_PROTOCOL': self.protoTcpMatcher,
            'ETHERTYPE': self.etherTypeMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'TCP_SRC_PORT': self.tcpSrcPortSpecMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'TCP_DST_PORT': self.tcpDstPortSpecMatcher,
            'TCP_FLAGS_MASK': self.tcpFlagMaskMatcher,
            'TCP_OPT': self.tcpOptMatcher,
            'IP_OPT': IpOptExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_OPT': self.innerIpMatcher,
         }

      class ProtoUdpExpression( CliCommand.CliExpression ):
         expression = ( 'UDP_PROTOCOL [ ETHERTYPE ] '
                        'IP_SRC [ UDP_SRC_PORT ] '
                        'IP_DST [ UDP_DST_PORT ] '
                        '[ IP_OPT ] [ PAYLOAD_OPT ] '
                        '[ METADATA_SPEC ] [ INNER_IP_OPT ]' )
         data = {
            'UDP_PROTOCOL': self.protoUdpMatcher,
            'ETHERTYPE': self.etherTypeMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'UDP_SRC_PORT': self.udpSrcPortSpecMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'UDP_DST_PORT': self.udpDstPortSpecMatcher,
            'IP_OPT': IpOptExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_OPT': self.innerIpMatcher,
         }

      class ProtoMplsOverGreExpression( CliCommand.CliExpression ):
         expression = ( 'MPLS_OVER_GRE_PROTOCOL MPLS_OVER_GRE_SRC MPLS_OVER_GRE_DST '
                        '[ INNER_IP_OPT ]' )
         data = {
            'MPLS_OVER_GRE_PROTOCOL': self.protoMplsOverGreMatcher,
            'MPLS_OVER_GRE_SRC': self.ipMplsOverGreSrcMatcher,
            'MPLS_OVER_GRE_DST': self.ipMplsOverGreDstMatcher,
            'INNER_IP_OPT': self.innerIpMatcher
         }

      ########################################################
      #             Deep Inspection IP filters
      ########################################################
      class ProtoNvgreExpression( CliCommand.CliExpression ):
         expression = ( 'NVGRE_PROTOCOL IP_SRC IP_DST '
                        '[ IP_OPT ] '
                        '[ NVGRE_EXPR | PAYLOAD_OPT | METADATA_SPEC ]' )
         data = {
            'NVGRE_PROTOCOL': self.protoNvgreMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'IP_OPT': IpOptExpression,
            'NVGRE_EXPR': AclCliRules.TniMaskExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'nvgreSelected' not in args:
               return
            args[ 'protocol' ] = args[ 'nvgreSelected' ]
            args.pop( 'nvgreSelected' )

            if 'nvgre' not in args:
               args[ 'nvgre' ] = ( AclLib.NVGRE_PROTO, None )

      class ProtoGreExpression( CliCommand.CliExpression ):
         expression = ( 'GRE_PROTOCOL IP_SRC IP_DST '
                        '[ IP_OPT ] '
                        '[ GRE_EXPR | PAYLOAD_OPT | METADATA_SPEC ] '
                        '[ INNER_IP_PROTO_OPT ]' )
         data = {
            'GRE_PROTOCOL': self.protoGreMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'IP_OPT': IpOptExpression,
            'GRE_EXPR': self.greExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_PROTO_OPT': self.innerIpProtoMatcher
         }

      class ProtoVxlanExpression( CliCommand.CliExpression ):
         expression = ( 'VXLAN_PROTOCOL IP_SRC [ UDP_SRC_PORT ]'
                        'IP_DST [ UDP_DST_PORT ]'
                        '[ IP_OPT ] '
                        '[ VXLAN_EXPR | PAYLOAD_OPT | METADATA_SPEC ]' )
         data = {
            'VXLAN_PROTOCOL': self.protoVxlanMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'UDP_SRC_PORT': self.udpSrcPortSpecMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'UDP_DST_PORT': self.udpDstPortSpecMatcher,
            'IP_OPT': IpOptExpression,
            'VXLAN_EXPR': AclCliRules.VniMaskExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'vxlanSelected' not in args:
               return
            args[ 'protocol' ] = args[ 'vxlanSelected' ]
            args.pop( 'vxlanSelected' )

            if 'vxlan' not in args:
               args[ 'vxlan' ] = ( True, None )

      class ProtoGtpExpression( CliCommand.CliExpression ):
         expression = ( 'GTP_PROTOCOL IP_SRC [ UDP_SRC_PORT ]'
                        'IP_DST [ UDP_DST_PORT ] [ IP_OPT ]'
                        '[ ( GTP_VERSION_PROTO [ GTP_TEID ] ) | '
                        'PAYLOAD_OPT | METADATA_SPEC ] '
                        '[ INNER_IP_PROTO_OPT ]' )
         data = {
            'GTP_PROTOCOL': self.protoGtpMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'UDP_SRC_PORT': self.udpSrcPortSpecMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'UDP_DST_PORT': self.udpDstPortSpecMatcher,
            'IP_OPT': IpOptExpression,
            'GTP_VERSION_PROTO': self.gtpVersionProtoMatcher,
            'GTP_TEID': AclCliRules.TeidMaskExpression,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP_PROTO_OPT': self.innerIpProtoMatcher
         }

      class ProtoSctpExpression( CliCommand.CliExpression ):
         expression = ( 'SCTP_PROTOCOL ( ( IP_SRC IP_DST [ IP_OPT ] '
                        '[ USER_L4_VAL ] [ PAYLOAD_OPT ] [ METADATA_SPEC ] ) | '
                        '( IP_SRC [ SCTP_SRC_PORT ] IP_DST '
                        '[ SCTP_DST_PORT ] [ IP_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] ) )' )
         data = {
            'SCTP_PROTOCOL': self.protoSctpMatcher,
            'IP_SRC': self.ipSourceMatcher,
            'SCTP_SRC_PORT': self.sctpSrcPortSpecMatcher,
            'IP_DST': self.ipDestinationMatcher,
            'SCTP_DST_PORT': self.sctpDstPortSpecMatcher,
            'IP_OPT': IpOptExpression,
            'USER_L4_VAL': self.userL4Matcher,
            'PAYLOAD_OPT': self.payloadMatcher,
            'METADATA_SPEC': self.metadataExprFactory
         }


      class IpFilterExpression( CliCommand.CliExpression ):
         expression = ( 'IP_FILTER_GENERIC | PROTO_IP | PROTO_ICMP | '
                        'PROTO_TCP | PROTO_UDP | PROTO_MPLS_OVER_GRE | '
                        'PROTO_NVGRE | PROTO_GRE | PROTO_VXLAN | '
                        'PROTO_GTP | PROTO_SCTP' )
         data = {
            'IP_FILTER_GENERIC': IpFilterGeneric,
            'PROTO_IP': ProtoIpExpression,
            'PROTO_ICMP': ProtoIcmpExpression,
            'PROTO_TCP': ProtoTcpExpression,
            'PROTO_UDP': ProtoUdpExpression,
            'PROTO_MPLS_OVER_GRE': ProtoMplsOverGreExpression,
            'PROTO_NVGRE': ProtoNvgreExpression,
            'PROTO_GRE': ProtoGreExpression,
            'PROTO_VXLAN': ProtoVxlanExpression,
            'PROTO_GTP': ProtoGtpExpression,
            'PROTO_SCTP': ProtoSctpExpression
         }
      self.ipFilterExpression = IpFilterExpression

      ########################################################
      #                    IPv6 filters
      ########################################################
      self.ip6ProtoNumMatcher = CliMatcher.IntegerMatcher( 0, 255,
         helpdesc='IPv6 Upper Layer Protocol',
         priority=CliParser.PRIO_LOW )
      self.protoIp6Matcher = AclCliRules.ipProtocolMatcher( 'ipv6',
            IPPROTO_IP, 'Any Internet Protocol version 6' )
      self.protoIcmp6Matcher = AclCliRules.ipProtocolMatcher( 'icmpv6',
            IPPROTO_ICMPV6, 'Internet Control Message Protocol version 6' )
      self.etherTypeMatcher = AclCliRules.EtherTypeMatcher( 'ipv6' )

      def fullMaskGuard( status ):
         if status and status.dpIp6AclFullMaskSupported:
            return None
         return CliParser.guardNotThisPlatform

      # IP source/destination prefix
      self.ip6SourceMatcher = AclCliRules.Ip6HostMatcher( 'source',
                                    fullMaskValid=True,
                                    fullMaskGuard=lambda mode, token:
                                          fullMaskGuard( aclStatus ),
                                    name='source' )
      self.ip6MplsOverGreSrcMatcher = AclCliRules.ipMplsOverGreHostMatcher(
            'source', 'source', AclLib.anyIp6AddrWithFullMask )
      self.ip6DestinationMatcher = AclCliRules.Ip6HostMatcher( 'destination',
                                    fullMaskValid=True,
                                    fullMaskGuard=lambda mode, token:
                                          fullMaskGuard( aclStatus ),
                                    name='destination' )
      self.ip6MplsOverGreDestMatcher = AclCliRules.ipMplsOverGreHostMatcher(
            'destination', 'destination', AclLib.anyIp6AddrWithFullMask )

      # Icmp messages: returns a ( type, code ) tuple
      # if code is None, then it matches all codes
      self.icmp6Matcher = AclCliRules.IcmpOptMatcher( AclCliLib.icmp6Messages,
            AclLib.ICMP6_ALL, name='icmp6' )

      self.hopLimitMatcher = AclCliRules.HopLimitSpecMatcher()

      # options common to all IPv6 protocols
      self.ipv6DscpMatcher = AclCliRules.DscpMatcher( 'ipv6' )

      def ip6OptAdapter( mode, args, argsList ):
         if 'ip6Opt' in args:
            return
         ip6Opt = []
         args[ 'ip6Opt' ] = ip6Opt
         if 'tracked' in args:
            ip6Opt.append( ( 'tracked', args.pop( 'tracked' ) ) )
         if 'HOP_LIMIT' in args:
            ip6Opt.append( ( 'hop-limit', args.pop( 'HOP_LIMIT' ) ) )
         if 'DSCP' in args:
            ip6Opt.append( ( 'dscpOpt', ( args.pop( 'DSCP' ),
               args.pop( 'DSCP_MASK' ) ) ) )
         if 'ECN' in args:
            ip6Opt.append( ( 'ecn', args.pop( 'ECN' ) ) )
         if 'IP_LEN' in args:
            ip6Opt.append( ( 'ipLenOpt', args.pop( 'IP_LEN' ) ) )
         if 'FLOW_LABEL' in args:
            ip6Opt.append( ( 'flow-label', args.pop( 'FLOW_LABEL' ) ) )

      class Ip6OptGenericExpression( CliCommand.CliExpression ):
         expression = '''{ HOP_LIMIT | DSCP | ECN | IP_LEN | FLOW_LABEL }'''
         data = {
            'HOP_LIMIT' : self.hopLimitMatcher,
            'DSCP': self.ipv6DscpMatcher,
            'ECN' : self.ecnMatcher,
            'IP_LEN' : self.ipLenMatcher,
            'FLOW_LABEL' : self.ipv6FlowLabelMatcher,
         }
         adapter = ip6OptAdapter

      self.ip6OptGenericExpression = Ip6OptGenericExpression

      class Ip6OptExpression( CliCommand.CliExpression ):
         expression = '''{ tracked | HOP_LIMIT | DSCP | ECN | IP_LEN |
                           FLOW_LABEL }'''
         data = {
            'tracked': CliCommand.singleNode( self.trackedMatcher ),
            'HOP_LIMIT' : self.hopLimitMatcher,
            'DSCP': self.ipv6DscpMatcher,
            'ECN' : self.ecnMatcher,
            'IP_LEN' : self.ipLenMatcher,
            'FLOW_LABEL' : self.ipv6FlowLabelMatcher,
         }
         adapter = ip6OptAdapter

      self.ip6OptExpression = Ip6OptExpression

      class Ip6FilterGeneric( CliCommand.CliExpression ):
         expression = ( '( PROTOCOL_NUM | GENERIC_PROTOCOL ) '
                        '[ ETHERTYPE ] '
                        'IP6_SRC IP6_DST [ IP6_OPT ]'
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ]'
                        '[ INNER_IP6_OPT ]' )
         data = {
            'PROTOCOL_NUM': CliCommand.Node( matcher=self.ip6ProtoNumMatcher,
               alias='protocol' ),
            'GENERIC_PROTOCOL': AclCliRules.GenericProtocolsMatcher(
               AclCliLib.genericIp6Protocols ),
            'ETHERTYPE' : self.etherTypeMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'IP6_OPT': Ip6OptGenericExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6Matcher,
         }

      class ProtoIp6Expression( CliCommand.CliExpression ):
         expression = ( 'IP6_PROTOCOL [ ETHERTYPE ] '
                        'IP6_SRC IP6_DST [ NEXTHOP_GROUP ] [ IP6_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] [ INNER_IP6_OPT ]' )
         data = {
            'IP6_PROTOCOL': self.protoIp6Matcher,
            'ETHERTYPE' : self.etherTypeMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'NEXTHOP_GROUP': self.nexthopGroupNameExpression,
            'IP6_OPT': Ip6OptExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6ProtoMatcher,
         }

      class ProtoIcmp6Expression( CliCommand.CliExpression ):
         expression = ( 'ICMP6_PROTOCOL [ ETHERTYPE ] '
                        'IP6_SRC IP6_DST [ ICMP6_OPT ] [ IP6_OPT ]'
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] [ INNER_IP6_OPT ]' )
         data = {
            'ICMP6_PROTOCOL': self.protoIcmp6Matcher,
            'ETHERTYPE' : self.etherTypeMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'ICMP6_OPT': self.icmp6Matcher,
            'IP6_OPT': Ip6OptExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6Matcher,
         }

      class ProtoUdp6Expression( CliCommand.CliExpression ):
         expression = ( 'UDP_PROTOCOL [ ETHERTYPE ] '
                        'IP6_SRC [ UDP_SRC_PORT ] '
                        'IP6_DST [ UDP_DST_PORT ] [ IP6_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] [ INNER_IP6_OPT ]' )
         data = {
            'UDP_PROTOCOL': self.protoUdpMatcher,
            'ETHERTYPE' : self.etherTypeMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'UDP_SRC_PORT': self.udpSrcPortSpecMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'UDP_DST_PORT': self.udpDstPortSpecMatcher,
            'IP6_OPT': Ip6OptExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6Matcher,
         }

      class ProtoTcp6Expression( CliCommand.CliExpression ):
         expression = ( 'TCP_PROTOCOL [ ETHERTYPE ] '
                        'IP6_SRC [ TCP_SRC_PORT ] '
                        'IP6_DST [ TCP_DST_PORT ] [ TCP_OPT ] [ IP6_OPT ] '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] [ INNER_IP6_OPT ]' )
         data = {
            'TCP_PROTOCOL': self.protoTcpMatcher,
            'ETHERTYPE' : self.etherTypeMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'TCP_SRC_PORT': self.tcpSrcPortSpecMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'TCP_DST_PORT': self.tcpDstPortSpecMatcher,
            'TCP_OPT': self.tcpOptMatcher,
            'IP6_OPT': Ip6OptExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6Matcher,
         }

      class ProtoMplsOverGre6Expression( CliCommand.CliExpression ):
         expression = ( 'MPLS_OVER_GRE_PROTOCOL MPLS_OVER_GRE_SRC MPLS_OVER_GRE_DST '
                        '[ PAYLOAD_OPT ] [ METADATA_SPEC ] [ INNER_IP6_OPT ]' )
         data = {
            'MPLS_OVER_GRE_PROTOCOL': self.protoMplsOverGreV6Matcher,
            'MPLS_OVER_GRE_SRC': self.ip6MplsOverGreSrcMatcher,
            'MPLS_OVER_GRE_DST': self.ip6MplsOverGreDestMatcher,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6Matcher,
         }

      class ProtoSctp6Expression( CliCommand.CliExpression ):
         expression = ( 'SCTP_PROTOCOL IP6_SRC [ SCTP_SRC_PORT ] '
                        'IP6_DST [ SCTP_DST_PORT ] [ IP6_OPT ]' )
         data = {
            'SCTP_PROTOCOL': self.protoSctpMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'SCTP_SRC_PORT': self.sctpSrcPortSpecMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'SCTP_DST_PORT': self.sctpDstPortSpecMatcher,
            'IP6_OPT': Ip6OptExpression,
         }
      class ProtoGtp6Expression( CliCommand.CliExpression ):
         expression = ( 'GTP_PROTOCOL IP6_SRC [ UDP_SRC_PORT ]'
                        'IP6_DST [ UDP_DST_PORT ] [ IP6_OPT ]'
                        '[ ( GTP_VERSION_PROTO [ GTP_TEID ] ) | '
                        'PAYLOAD_OPT | METADATA_SPEC ] '
                        '[ INNER_IP6_OPT ]' )
         data = {
            'GTP_PROTOCOL': self.protoGtpMatcher,
            'IP6_SRC': self.ip6SourceMatcher,
            'UDP_SRC_PORT': self.udpSrcPortSpecMatcher,
            'IP6_DST': self.ip6DestinationMatcher,
            'UDP_DST_PORT': self.udpDstPortSpecMatcher,
            'IP6_OPT': Ip6OptExpression,
            'GTP_VERSION_PROTO': self.gtpVersionProtoMatcher,
            'GTP_TEID': AclCliRules.TeidMaskExpression,
            'PAYLOAD_OPT': self.payloadIp6Matcher,
            'METADATA_SPEC': self.metadataExprFactory,
            'INNER_IP6_OPT': self.innerIp6ProtoMatcher
         }

      class Ip6FilterExpression( CliCommand.CliExpression ):
         expression = ( 'IP6_FILTER_GENERIC | PROTO_IP6 | PROTO_ICMP6 |'
                        'PROTO_UDP | PROTO_TCP | PROTO_MPLS_OVER_GRE | PROTO_SCP |'
                        'PROTO_GTP' )
         data = {
            'IP6_FILTER_GENERIC': Ip6FilterGeneric,
            'PROTO_IP6' : ProtoIp6Expression,
            'PROTO_ICMP6' : ProtoIcmp6Expression,
            'PROTO_UDP' : ProtoUdp6Expression,
            'PROTO_TCP' : ProtoTcp6Expression,
            'PROTO_MPLS_OVER_GRE' : ProtoMplsOverGre6Expression,
            'PROTO_SCP': ProtoSctp6Expression,
            'PROTO_GTP': ProtoGtp6Expression,
         }

      self.ip6FilterExpression = Ip6FilterExpression

      ########################################################
      #             Standard IPv6 filters
      ########################################################
      self.remarkMatcher = CliMatcher.KeywordMatcher( 'remark',
                                                      helpdesc='Specify a comment' )
      self.commentMatcher = CliMatcher.StringMatcher( helpname='LINE',
                                          helpdesc='Comment, up to 100 characters' )

      ########################################################
      #                       MAC filters
      ########################################################
      def _handleMacFilter( mode, source, destination,
                            protocol=AclLib.MACPROTO_ALL,
                            mplsTwoLabelSpec=None, mplsFilter=None,
                            payloadOpt=None, metadataSpec=None ):
         if mplsTwoLabelSpec is None:
            mplsTwoLabelSpec = ( 0, 0, 0, 0 )
         if mplsFilter is None:
            mplsFilter = AclLib.MplsFilterValue()
         if payloadOpt is None:
            payloadOpt = AclLib.payloadOptToPayloadSpec( "" )
         else:
            if aclStatus.dpAclPayloadL2RequiresProto:
               if ( protocol == AclLib.MACPROTO_ALL ) or \
               ( protocol == AclLib.MACPROTO_IP4 ):
                  mode.addError( "MAC protocol must be specified and must not be " +
                        "IP (0x800) for payload matching" )
                  raise CliCommon.AlreadyHandledError()
         if metadataSpec is None:
            metadataSpec = AclLib.metadataListToMetadataSpec()
         return AclLib.MacFilterValue(
            sourceAddr=source[ 0 ],
            sourceMask=source[ 1 ],
            destinationAddr=destination[ 0 ],
            destinationMask=destination[ 1 ],
            proto=protocol,
            mplsLabel=mplsTwoLabelSpec[ 0 ],
            mplsLabelMask=mplsTwoLabelSpec[ 1 ],
            mplsInnerLabel=mplsTwoLabelSpec[ 2 ],
            mplsInnerLabelMask=mplsTwoLabelSpec[ 3 ],
            mplsFilter=mplsFilter,
            payloadOpt=payloadOpt,
            metadataSpec=metadataSpec )

      class MacProcotolExpression( CliCommand.CliExpression ):
         expression = 'PROTOCOL_NAME | PROTOCOL_NUM'
         data = {
                  'PROTOCOL_NAME': CliMatcher.EnumMatcher(
                     # pylint: disable-next=consider-using-f-string
                     { k: '%s (0x%x)' % ( v[ 1 ], v[ 0 ] ) 
                        for k, v in Ethernet.macProtoByName.items() } ),
                  'PROTOCOL_NUM': CliMatcher.IntegerMatcher( 0, 65535,
                     helpdesc='MAC protocol number' ),
                }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'PROTOCOL_NUM' in args:
               assert 'protocol' not in args
               args[ 'protocol' ] = args.pop( 'PROTOCOL_NUM' )
            if 'PROTOCOL_NAME' in args:
               assert 'protocol' not in args
               protocolName = args.pop( 'PROTOCOL_NAME' )
               args[ 'protocol' ] = Ethernet.macProtoByName[ protocolName ][ 0 ]

      class MacFilterExpression( CliCommand.CliExpression ):
         expression = ( 'MAC_SRC MAC_DST [ '
                  '( [ PROTOCOL ] [ PAYLOAD_OPT ] [ METADATA_SPEC ] ) | '
                  '( mpls [ PAYLOAD_OPT | METADATA_SPEC | MPLS_FILTER | '
                  'MPLS_TWO_LABEL ] ) ]' )
         data = {
                  'MAC_SRC': AclCliRules.macHostRuleMatcher( 'source' ),
                  'MAC_DST': AclCliRules.macHostRuleMatcher( 'destination' ),
                  'PROTOCOL': MacProcotolExpression,
                  'mpls': 'Multi-protocol Label Switching',
                  'MPLS_FILTER': self.mplsFilterMatcher,
                  'MPLS_TWO_LABEL': self.mplsTwoLabelMatcher,
                  'PAYLOAD_OPT': self.payloadMacMatcher,
                  "METADATA_SPEC": self.metadataExprFactory
                }

         @staticmethod
         def adapter( mode, args, argsList ):
            macFilterArgs = {}
            if 'mpls' in args:
               args.pop( 'mpls' )
               assert 'protocol' not in args
               args[ 'protocol' ] = AclLib.MPLS_PROTO

            for i in ( 'source', 'destination', 'protocol', 'mplsTwoLabelSpec',
                       'mplsTwoLabelSpec', 'mplsFilter', 'payloadOpt',
                       'metadataSpec' ):
               if i in args:
                  macFilterArgs[ i ] = args.pop( i )
            # pylint: disable-msg=protected-access
            args[ 'macFilter' ] = _handleMacFilter( mode, **macFilterArgs )
         
      self.macFilterExpression = MacFilterExpression

      ########################################################
      #                      Rule actions
      ########################################################
      self.ipRuleAction = AclCliRules.RuleActionExprFactory( 'ip' )
      self.macRuleAction = AclCliRules.RuleActionExprFactory( 'mac' )
      self.ip6RuleAction = AclCliRules.RuleActionExprFactory( 'ipv6' )

      #-----------------------------------------------------------------
      #              [no] counters per-entry
      # legacy:
      #              [no] statistics per-entry
      #-----------------------------------------------------------------
      # Tbd - Move this to Acl specific class. All "guards" will Acl/Pbr specific
      def dpAclCountersSupportedGuard( mode, token ):
         if aclStatus.dpAclCountersSupported:
            return None
         return CliParser.guardNotThisPlatform

      statsMatcher = CliMatcher.KeywordMatcher( 'statistics',
                                     helpdesc='Count packets that match the list' )
      countersMatcher = CliMatcher.KeywordMatcher( 'counters',
                                       helpdesc='Count packets that match the list' )
      self.statsParserNode = CliCommand.Node( matcher=statsMatcher,
                                              guard=dpAclCountersSupportedGuard,
                                              deprecatedByCmd='counters per-entry' )
      self.countersParserNode = CliCommand.Node( matcher=countersMatcher,
                                                 guard=dpAclCountersSupportedGuard )

      self.perEntryMatcher = CliMatcher.KeywordMatcher( 'per-entry',
         helpdesc='Count matches for each rule' )

      #-----------------------------------------------------------------
      #              resequence
      #-----------------------------------------------------------------
      # TBD - move to Acl specific class. Pbr reqseq may be different.
      self.startSeqMatcher = CliMatcher.IntegerMatcher(
         1, AclLib.MAX_SEQ,
         helpdesc='Starting sequence number (default 10)' )
      self.incSeqMatcher = CliMatcher.IntegerMatcher(
         1, AclLib.MAX_SEQ,
         helpdesc='Step to increment the sequence number (default 10)' )

      # The filter rule for all protocols.
      #
      # By doing this we can make sure the IpProtocolRule
      # has priority over the generic ipProtoNumRule rule
      self.anyPort = AclLib.anyPortValue

   #-----------------------------------------------------------------
   #              no sequence-number
   #-----------------------------------------------------------------
   # Similar to the interface command, we allow comma separated ranges,
   # such as no 10 , 30 - 60 , 100 - $
   @staticmethod
   def handleNoSeq( mode, args ):
      seqRanges = args[ 'SEQUENCES' ].ranges()
      aclApi = mode.aclApi
      for seqRange in seqRanges:
         ret = aclApi.removeRuleBySequence( seqRange[0], seqRange[1] )
         if ret == 'errReadOnly':
            mode.addError(
               AclCliLib.aclModifyError( mode.aclName, mode.aclType,
                                         'Readonly list' ) )
   @staticmethod
   def handleResequence( mode, start, inc ):
      aclApi = mode.aclApi
      ret = aclApi.resequence( start, inc )
      if ret == 'errSequenceOutOfRange':
         mode.addError( "Error: Sequence number out of range" )

   # TBD - These methods can go in to the class that add commands to modes
   @staticmethod
   def handleFragmentRules( mode, args ):
      aclApi = mode.aclApi
      aclApi.enableFragments()

   @staticmethod
   def handleNoFragmentRules( mode, args ):
      aclApi = mode.aclApi
      aclApi.disableFragments()

   @staticmethod
   def handleStatsPerEntry( mode, args ):
      aclApi = mode.aclApi
      aclApi.enableCounters()

   @staticmethod
   def handleNoStatsPerEntry( mode, args ):
      aclApi = mode.aclApi
      aclApi.disableCounters()

   # remark
   @classmethod
   def _handleRemark( cls, mode, args, filterInfo ):
      if CliCommand.isNoOrDefaultCmd( args ):
         seqnum = 'default'
      else:
         seqnum = args.get( 'SEQ_NUM' )
      remark = args.get( 'REMARK' )
      cls._handleRuleCommon( mode, seqnum, 'remark', None, filterInfo, None,
                             None, remark, "" )

   @classmethod
   def handleIpRemark( cls, mode, args ):
      filterInfo = AclLib.IpFilterValue( proto=IPPROTO_IP,
                                         source=AclLib.anyIpAddr,
                                         destination=AclLib.anyIpAddr,
                                         ttl=AclLib.anyTtlValue )
      cls._handleRemark( mode, args, filterInfo )

   @classmethod
   def handleIp6Remark( cls, mode, args ):
      filterInfo = AclLib.Ip6FilterValue( proto=IPPROTO_IP,
                                          source=AclLib.anyIp6Addr,
                                          destination=AclLib.anyIp6Addr,
                                          ttl=AclLib.anyTtlValue )
      cls._handleRemark( mode, args, filterInfo )

   @classmethod
   def handleMacRemark( cls, mode, args ):
      filterInfo = AclLib.MacFilterValue( sourceAddr=AclLib.zeroMacAddrString,
                                          sourceMask=AclLib.zeroMacAddrString,
                                          destinationAddr=AclLib.zeroMacAddrString,
                                          destinationMask=AclLib.zeroMacAddrString )
      cls._handleRemark( mode, args, filterInfo )

   @classmethod
   def handleIp6StdRule( cls, mode, seqnum, action, vlanSpec, ipFilter,
                         mirrorSession, log, copyToCpuDst ):
      return cls._handleRuleCommon( mode, seqnum, action, vlanSpec, ipFilter,
                                    mirrorSession, log, "", copyToCpuDst )

   @classmethod
   def handleIpStdRule( cls, mode, seqnum, action, vlanSpec, ipFilter, 
                        mirrorSession, log, copyToCpuDst ):
      return cls._handleRuleCommon( mode, seqnum, action, vlanSpec, ipFilter, 
                                    mirrorSession, log, "", copyToCpuDst )

   @staticmethod
   def _getLastSeqnum( subconfig ):
      # get the last sequence number of an ipacl
      ruleSeq = list( subconfig.ruleBySequence )
      remarkSeq = list( subconfig.remarkBySequence )

      if ruleSeq and remarkSeq:
         return max( ruleSeq[ -1 ], remarkSeq[ -1 ] )
      return ( ruleSeq and ruleSeq[ -1 ] ) or ( remarkSeq and remarkSeq[ -1 ] )  or 0

   @staticmethod
   def _getRuleBySeq( subconfig, seq, aclType ):
      if seq in subconfig.ruleBySequence:
         return getRuleById( subconfig, aclType )[ subconfig.ruleBySequence[ seq ] ]
      else:
         return subconfig.remarkBySequence[ seq ]

   @staticmethod
   def _isSameRule( rule, action, ruleFilter, log ):
      return rule.action == action and rule.filter == ruleFilter and rule.log == log

   @staticmethod
   def _isSameRemark( remarkConfig, remark ):
      return remarkConfig.remark == remark

   @staticmethod
   def _checkReadonly( subconfig, mode ):
      if subconfig.readonly:
         mode.addError(
            AclCliLib.aclModifyError( mode.aclName, mode.aclType,
                                      'Readonly list' ) )
         return True
      return False

   # -----------------------------------------------------------------
   #              [no] permit response traffic nat
   # -----------------------------------------------------------------
   @classmethod
   def handlePermitResponseRule( cls, mode, args ):
      if args[ 'TRAFFIC_TYPE' ] == 'nat':
         mode.aclApi.permitResponse = AclLib.AclPermitResponseType.natTraffic

   @classmethod
   def noOrDefaultHandlePermitResponseRule( cls, mode, args ):
      mode.aclApi.permitResponse = AclLib.AclPermitResponseType.unused

   #-----------------------------------------------------------------
   #              [no] permit/deny
   #-----------------------------------------------------------------
   @classmethod
   def adaptRuleCommonArgs( cls, mode, args ):
      args[ 'seqnum' ] = args.pop( 'SEQ_NUM', None )
      args[ 'action' ] = args.pop( 'ACTION' )
      args[ 'log' ] = 'log' in args
      args.setdefault( 'copyToCpuDst', AclLib.CopyToCpuDst.cpuDstNone )

   @classmethod
   def adaptNoOrDefaultRuleCommonArgs( cls, mode, args ):
      args.pop( CliCommand.NAME_NO, None )
      args.pop( CliCommand.NAME_DEFAULT, None )
      args[ 'seqnum' ] = 'default'

   @classmethod
   def _handleRuleCommon( cls, mode, seqnum, action, vlanSpec, ruleFilter,
                          mirrorSession, log, remark, copyToCpuDst, **kwargs ):
      pass

   @classmethod
   def handleIpRule( cls, mode, seqnum, action, vlanSpec, ipFilter, mirrorSession,
                     log, copyToCpuDst, **kwargs ):
      cls._handleRuleCommon( mode, seqnum, action, vlanSpec, ipFilter, mirrorSession,
                             log, "", copyToCpuDst, **kwargs )

   @classmethod
   def handleIp6Rule( cls, mode, seqnum, action, vlanSpec, ipFilter, mirrorSession,
                      log, copyToCpuDst, **kwargs ):
      cls._handleRuleCommon( mode, seqnum, action, vlanSpec, ipFilter, mirrorSession,
                             log, "", copyToCpuDst, **kwargs )

   @classmethod
   def handleMacRule( cls, mode, seqnum, action, vlanSpec, macFilter, mirrorSession,
                      log, copyToCpuDst, **kwargs ):
      # copyToCpu option is not supported in MAC ACL
      cls._handleRuleCommon( mode, seqnum, action, vlanSpec, macFilter,
                             mirrorSession, log, "", 
                             AclLib.CopyToCpuDst.cpuDstNone, **kwargs )

   @staticmethod
   def _handleIpArgs( mode, args ):
      ipFilterArgs = {}
      for i in ( 'protocol', 'protoMask', 'source', 'destination', 'etherType',
                 'nexthopGroup', 'sport', 'dport', 'icmp',
                 'tcpOptions', 'TCP_FLAGS_MASK',
                 'ipOpt', 'nvgre', 'vxlan', 'gre', 'gtpVersionProto',
                 'teidVal', 'userL4Value', 'payloadOpt', 'metadataSpec',
                 'innerIpOpt', 'innerIpProtoOpt' ):
         if i in args:
            ipFilterArgs[ i ] = args.pop( i )
      args[ 'ipFilter' ] = AclCfgCmdChains._handleIpFilter( mode, **ipFilterArgs )

   @staticmethod
   def _handleIpFilter( mode, protocol, source, destination,
                        etherType=None,
                        protoMask=0, nexthopGroup=None,
                        sport=None, dport=None, icmp=None,
                        tcpOptions=None, ipOpt=None, 
                        nvgre=None, vxlan=None,
                        gre=None, gtpVersionProto=None, 
                        teidVal=None, userL4Value=None, 
                        payloadOpt=None, metadataSpec=None,
                        innerIpOpt=None, innerIpProtoOpt=None,
                        TCP_FLAGS_MASK=None ):
      assert source is not None, "Error: Invalid source"
      assert destination is not None, "Error: Invalid destination"
      if ( ( sport is AclCliRules.BadPortSpecError ) or
           ( dport is AclCliRules.BadPortSpecError ) ):
         raise CliParser.AlreadyHandledError
      fragments = False
      tracked = False
      ttl = AclLib.anyTtlValue
      matchDscp = False
      ecn = AclLib.EcnValue.dontCare
      dscp = 0
      dscpUseName = False
      dscpMask = 0x3F
      ipLen = AclLib.anyIpLenValue
      innerIpSrc = AclLib.anyIpAddr
      innerIpDest = AclLib.anyIpAddr
      innerIpMatch = False
      innerIp6Src = AclLib.anyIp6Addr
      innerIp6Dest = AclLib.anyIp6Addr
      innerIp6Match = False
      innerProtocol = 0
      if ipOpt is not None:
         for name, value in ipOpt:
            if name == 'ttl':
               assert value is not None
               ttl = value
            elif name == 'fragments':
               assert value is not None
               fragments = True
            elif name == 'tracked':
               assert value == 'tracked'
               tracked = True
            elif name == 'dscpOpt':
               ( dscp, dscpUseName ) = AclCliLib.dscpValueFromCli( mode, value ) 
               matchDscp = True
            elif name == 'dscpMaskOpt':
               dscpMask = value
            elif name == 'ecn':
               ecn = AclCliLib.ecnValueFromCli( mode, value )
            elif name == 'ipLenOpt':
               if value is AclCliRules.BadIpLenSpecError:
                  raise CliParser.AlreadyHandledError
               ipLen = value

      if innerIpOpt is not None:
         if innerIpOpt[ 0 ] == 'ip':
            innerIpSrc = innerIpOpt[ 1 ]
            innerIpDest = innerIpOpt[ 2 ]
            innerIpMatch = True
         elif innerIpOpt[ 0 ] == 'ipv6':
            innerIp6Src = innerIpOpt[ 1 ]
            innerIp6Dest = innerIpOpt[ 2 ]
            innerIp6Match = True

      if innerIpProtoOpt is not None:
         optLen = len( innerIpProtoOpt )
         if optLen == 3:
            # For IP-in-IP rule, matching only inner source and destination addresses
            srcIp, dstIp = innerIpProtoOpt[ 1 ], innerIpProtoOpt[ 2 ]
         elif optLen == 6:
            # For TCP/UDP-in-IP rule, matching inner source, destination addresses
            # and L4 source and destination ports
            srcIp, dstIp = innerIpProtoOpt[ 2 ], innerIpProtoOpt[ 4 ]
            sport, dport = innerIpProtoOpt[ 3 ], innerIpProtoOpt[ 5 ]
            innerProtocol = innerIpProtoOpt[ 1 ]

         if innerIpProtoOpt[ 0 ] == 'ip':
            innerIpSrc, innerIpDest = srcIp, dstIp
            innerIpMatch = True
         elif innerIpProtoOpt[ 0 ] == 'ipv6':
            innerIp6Src, innerIp6Dest = srcIp, dstIp
            innerIp6Match = True

      if tcpOptions is not None:
         tcpFlag, established = tcpOptions
      else:
         tcpFlag = AclLib.TcpFlagValue()
         established = False

      if TCP_FLAGS_MASK is None:
         tcpFlagMask = AclLib.TcpFlagValue()
      else:
         tcpFlagMask = TCP_FLAGS_MASK

      if icmp is None or icmp[ 0 ] is None:
         icmpType = AclLib.ICMP_ALL
      else:
         icmpType = icmp[ 0 ]
      if icmp is None or icmp[ 1 ] is None:
         icmpCode = AclLib.ICMP_ALL
      else:
         icmpCode = icmp[ 1 ]

      if userL4Value is None:
         userL4 = userL4Mask = 0
      else:
         userL4, userL4Mask = userL4Value

      greProto = greProtoMask = 0
      tni = tniMask = 0
      if nvgre is not None:
         proto, tniVal = nvgre
         if tniVal is not None:
            tni, tniMask = tniVal
         greProto = proto
         greProtoMask = AclLib.GRE_ALL

      if gre is not None:
         _, protoVal = gre
         if protoVal is not None:
            greProto, greProtoMask = protoVal

      vxlanValid = False
      vni = vniMask = 0
      if vxlan is not None:
         vxlanValid, vniVal = vxlan
         if vniVal is not None:
            vni, vniMask = vniVal
         if dport is None:
            dport = AclLib.vxlanPortValue

      if gtpVersionProto is None:
         gtpVersion = gtpProto = 0
      else:
         gtpVersion, gtpProto = gtpVersionProto
      if teidVal is None:
         teid = teidMask = 0
      else:
         teid, teidMask = teidVal

      anyPort = AclLib.anyPortValue
      if payloadOpt is None:
         payloadOpt = AclLib.payloadOptToPayloadSpec( "" )

      if metadataSpec is None:
         metadataSpec = AclLib.metadataListToMetadataSpec()

      if nexthopGroup is None:
         nexthopGroup = ''

      if not matchDscp or dscpMask == 0:
         dscp = 0
         dscpMask = 0
         matchDscp = False
      
      return AclLib.IpFilterValue( proto=protocol,
                                   protoMask=protoMask,
                                   source=source,
                                   destination=destination,
                                   innerSource=innerIpSrc,
                                   innerDest=innerIpDest,
                                   innerSource6=innerIp6Src,
                                   innerDest6=innerIp6Dest,
                                   innerProto=innerProtocol,
                                   matchInnerIp=innerIpMatch,
                                   matchInnerIp6=innerIp6Match,
                                   nexthopGroup=nexthopGroup,
                                   ttl=ttl,
                                   fragments=fragments,
                                   sport=anyPort if sport is None else sport,
                                   dport=anyPort if dport is None else dport,
                                   ipLen=ipLen,
                                   tracked=tracked,
                                   etherType=etherType if etherType else 0,
                                   tcpFlag=tcpFlag,
                                   tcpFlagMask=tcpFlagMask,
                                   established=established,
                                   icmpType=icmpType,
                                   icmpCode=icmpCode,
                                   dscp=dscp,
                                   dscpUseName=dscpUseName,
                                   dscpMask=dscpMask,
                                   matchDscp=matchDscp,
                                   ecn=ecn,
                                   tni=tni, tniMask=tniMask,
                                   vxlanValid=vxlanValid,
                                   vni=vni, vniMask=vniMask,
                                   greProto=greProto, greProtoMask=greProtoMask,
                                   gtpVersion=gtpVersion, gtpProto=gtpProto,
                                   teid=teid, teidMask=teidMask,
                                   userL4=userL4, userL4Mask=userL4Mask,
                                   payloadOpt=payloadOpt,
                                   metadataSpec=metadataSpec )

   @staticmethod
   def _handleIp6Args( mode, args ):
      ipFilterArgs = {}
      for i in ( 'protocol', 'protoMask', 'source', 'destination',
                  'etherType', 'nexthopGroup',
                  'sport', 'dport', 'icmp6', 'tcpOptions', 'TCP_FLAGS_MASK',
                  'ip6Opt', 'nvgre', 'vxlan', 'gre', 'gtpVersionProto',
                  'teidVal', 'userL4Value', 'payloadOpt', 'metadataSpec',
                  'innerIp6Opt', 'innerIp6ProtoOpt' ):
         if i in args:
            ipFilterArgs[ i ] = args.pop( i )
      args[ 'ipFilter' ] = AclCfgCmdChains._handleIp6Filter( mode, **ipFilterArgs )

   @staticmethod
   def _handleIp6Filter( mode, protocol, source, destination,
                         etherType=None,
                         protoMask=0,
                         nexthopGroup=None,
                         sport=None, dport=None, icmp6=None,
                         tcpOptions=None, TCP_FLAGS_MASK=None, ip6Opt=None,
                         nvgre=None, vxlan=None, gre=None,
                         gtpVersionProto=None, teidVal=None,
                         userL4Value=None, payloadOpt=None,
                         metadataSpec=None, innerIp6Opt=None,
                         innerIp6ProtoOpt=None ):
      if ( ( sport is AclCliRules.BadPortSpecError ) or
           ( dport is AclCliRules.BadPortSpecError ) ):
         raise CliParser.AlreadyHandledError

      tcMask = 0x00
      tc = 0
      ecnVal = AclLib.EcnValue.dontCare
      tracked = False
      dscpUseName = False
      ipLen = AclLib.anyIpLenValue
      hopLimit = AclLib.anyTtlValue
      innerIpSrc = AclLib.anyIpAddr
      innerIpDest = AclLib.anyIpAddr
      innerIpMatch = False
      innerIp6Src = AclLib.anyIp6Addr
      innerIp6Dest = AclLib.anyIp6Addr
      innerIp6Match = False
      innerProtocol = 0
      flowLabel = 0
      flowLabelMask = 0
      flowLabelMatchType = AclLib.FlowLabelMatchType.flowLabelNotMatched
      if ip6Opt is not None:
         for name, value in ip6Opt:
            if name == 'hop-limit':
               assert value is not None
               hopLimit = value
            elif name == 'flow-label':
               assert value is not None
               assert len( value ) == 3
               # In case of match any flow-label, no need to add into Ip6Filter
               if value[ 1 ]:
                  flowLabel, flowLabelMask, flowLabelMatchType = value 
            elif name == 'dscpOpt':
               assert len( value ) == 2
               # value = (dscpValue, dscpMask), dscpMask is optional
               ( dscp, dscpUseName ) = AclCliLib.dscpValueFromCli( mode, value[ 0 ] )
               # dscp in upper 6 bits of TC
               tc = dscp << 2
               if value[ 1 ]:
                  tcMask = ( value[ 1 ] ^ 0x3F ) << 2
               else:
                  tcMask = 0xFC
               tc = 0 if tcMask == 0 else tc
            elif name == 'ecn':
               ecnVal = AclCliLib.ecnValueFromCli( mode, value )
            elif name == 'tracked':
               assert value == 'tracked'
               tracked = True
            elif name == 'ipLenOpt':
               if value is AclCliRules.BadIpLenSpecError:
                  raise CliParser.AlreadyHandledError
               ipLen = value

      if innerIp6Opt is not None:
         if innerIp6Opt[ 0 ] == 'ip':
            innerIpSrc = innerIp6Opt[ 1 ]
            innerIpDest = innerIp6Opt[ 2 ]
            innerIpMatch = True
         elif innerIp6Opt[ 0 ] == 'ipv6':
            innerIp6Src = innerIp6Opt[ 1 ]
            innerIp6Dest = innerIp6Opt[ 2 ]
            innerIp6Match = True

      if innerIp6ProtoOpt is not None:
         optLen = len( innerIp6ProtoOpt )
         if optLen == 3:
            # For IP-in-IP rule, matching only inner source and destination addresses
            srcIp, dstIp = innerIp6ProtoOpt[ 1 ], innerIp6ProtoOpt[ 2 ]
         elif optLen == 6:
            # For TCP/UDP-in-IP rule, matching inner source, destination addresses
            # and L4 source and destination ports
            srcIp, dstIp = innerIp6ProtoOpt[ 2 ], innerIp6ProtoOpt[ 4 ]
            sport, dport = innerIp6ProtoOpt[ 3 ], innerIp6ProtoOpt[ 5 ]
            innerProtocol = innerIp6ProtoOpt[ 1 ]

         if innerIp6ProtoOpt[ 0 ] == 'ip':
            innerIpSrc, innerIpDest = srcIp, dstIp
            innerIpMatch = True
         elif innerIp6ProtoOpt[ 0 ] == 'ipv6':
            innerIp6Src, innerIp6Dest = srcIp, dstIp
            innerIp6Match = True

      if tcpOptions is not None:
         tcpFlag, established = tcpOptions
      else:
         tcpFlag = AclLib.TcpFlagValue()
         established = False

      if TCP_FLAGS_MASK is None:
         tcpFlagMask = AclLib.TcpFlagValue()
      else:
         tcpFlagMask = TCP_FLAGS_MASK

      if icmp6 is None or icmp6[ 0 ] is None:
         icmp6Type = AclLib.ICMP6_ALL
      else:
         icmp6Type = icmp6[ 0 ]
      if icmp6 is None or icmp6[ 1 ] is None:
         icmp6Code = AclLib.ICMP6_ALL
      else:
         icmp6Code = icmp6[ 1 ]

      if userL4Value is None:
         userL4 = userL4Mask = 0
      else:
         userL4, userL4Mask = userL4Value

      greProto = greProtoMask = 0
      tni = tniMask = 0
      if nvgre is not None:
         proto, tniVal = nvgre
         if tniVal is not None:
            tni, tniMask = tniVal
         greProto = proto
         greProtoMask = AclLib.GRE_ALL

      if gre is not None:
         _, protoVal = gre
         if protoVal is not None:
            greProto, greProtoMask = protoVal

      vxlanValid = False
      vni = vniMask = 0
      if vxlan is not None:
         vxlanValid, vniVal = vxlan
         if vniVal is not None:
            vni, vniMask = vniVal
         dport = AclLib.vxlanPortValue

      if gtpVersionProto is None:
         gtpVersion = gtpProto = 0
      else:
         gtpVersion, gtpProto = gtpVersionProto
      if teidVal is None:
         teid = teidMask = 0
      else:
         teid, teidMask = teidVal

      anyPort = AclLib.anyPortValue
      if payloadOpt is None:
         payloadOpt = AclLib.payloadOptToPayloadSpec( "" )
      
      if metadataSpec is None:
         metadataSpec = AclLib.metadataListToMetadataSpec()

      if nexthopGroup is None:
         nexthopGroup = ''

      return AclLib.Ip6FilterValue( proto=protocol,
                                    protoMask=protoMask,
                                    innerSource=innerIpSrc,
                                    innerDest=innerIpDest,
                                    innerSource6=innerIp6Src,
                                    innerDest6=innerIp6Dest,
                                    innerProto=innerProtocol,
                                    matchInnerIp=innerIpMatch,
                                    matchInnerIp6=innerIp6Match,
                                    nexthopGroup=nexthopGroup,
                                    ttl=hopLimit,
                                    sport=anyPort if sport is None else sport,
                                    dport=anyPort if dport is None else dport,
                                    ipLen=ipLen,
                                    tracked=tracked,
                                    etherType=etherType if etherType else 0,
                                    tcpFlag=tcpFlag,
                                    tcpFlagMask=tcpFlagMask,
                                    established=established,
                                    icmpType=icmp6Type,
                                    icmpCode=icmp6Code,
                                    tc=tc,
                                    tcMask=tcMask,
                                    flowLabel=flowLabel,
                                    flowLabelMask=flowLabelMask,
                                    flowLabelMatch=flowLabelMatchType,
                                    dscpUseName=dscpUseName,
                                    tni=tni, tniMask=tniMask,
                                    vxlanValid=vxlanValid,
                                    vni=vni, vniMask=vniMask,
                                    greProto=greProto, greProtoMask=greProtoMask,
                                    gtpVersion=gtpVersion, gtpProto=gtpProto,
                                    teid=teid, teidMask=teidMask,
                                    userL4=userL4, userL4Mask=userL4Mask,
                                    payloadOpt=payloadOpt,
                                    metadataSpec=metadataSpec,
                                    ecn=ecnVal,
                                    sourceFullMask=source,
                                    destinationFullMask=destination )

   #-----------------------------------------------------------------
   #              abort
   #-----------------------------------------------------------------
   @classmethod
   def abort( cls, mode ):
      mode.aclApi = None
      mode.session_.gotoParentMode( )

def Plugin( entityManager ):
   global aclStatus
   aclStatus = LazyMount.mount( entityManager, "acl/status/all", "Acl::Status", "r" )
