# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import IntfCli
from CliPlugin import IntfRangeCli
from CliPlugin import PolicyMapCliLib
from CliPlugin import VlanCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
from Toggles.AegisToggleLib import togglePerVrfIntfOverrideEnabled

policiesCliConfig = None
intfConfigIngress = None
intfConfigEgress = None
intfConfigBridged = None
policiesStatusRequestDir = None
policiesStatus = None
entityManager = None
hwStatus = None

def _aegisTrafficPolicyIntfGuardHelp( mode, supportedIntfFunc ):
   # We support ingress traffic-policy on Strata through
   # DMF but not through the CLI. So add a guard here for
   # Strata
   if not hwStatus.ingressTrafficPolicyCliSupported:
      return CliParser.guardNotThisPlatform

   if mode.multiInstance_:
      intfNames = list( mode.intfList.intfNames() )
      for intf in intfNames:
         if not supportedIntfFunc( intf ):
            return CliParser.guardNotThisPlatform
      return None
   elif supportedIntfFunc( mode.intf.name ):
      return None

   return CliParser.guardNotThisPlatform

def aegisTrafficPolicyIntfGuard( mode, token ):
   return _aegisTrafficPolicyIntfGuardHelp(
         mode, hwStatus.ingressTrafficPolicySupportedForIntf )

def aegisEgressTrafficPolicyGuard( mode, token ):
   return _aegisTrafficPolicyIntfGuardHelp(
         mode, hwStatus.egressTrafficPolicySupportedForIntf )

def aegisBridgedTrafficPolicyGuard( mode, token ):
   if hwStatus.bridgedTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def aegisXformGuard( mode, token ):
   if hwStatus.aegisXformSupported:
      return None
   return CliParser.guardNotThisPlatform

def aegisIntfOverrideGuard( mode, token ):
   if hwStatus.intfAppAlwaysOverride:
      return None
   return CliParser.guardNotThisPlatform

trafficPolicyKeyword = CliMatcher.KeywordMatcher( 'traffic-policy',
      helpdesc='Apply a traffic-policy' )
inputKeyword = CliMatcher.KeywordMatcher( 'input',
      helpdesc='Assign traffic-policy to the input of an interface' )
outputKeyword = CliMatcher.KeywordMatcher( 'output',
      helpdesc='Assign traffic-policy to the output of an interface' )

def getTrafficPolicyNames( mode=None ):
   return policiesCliConfig.pmapType.pmap

def vlanIdToIntfVlanId( vlanId ):
   return 'Vlan' + str( vlanId )

#----------------------------------------------------------------------
# Modelet under Intf mode for traffic-policy command
#----------------------------------------------------------------------
class IntfAegisModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # NOTE: Keep this in sync with the types that can be added to
      # TrafficPolicy::HwStatus::(ingress|egress)TrafficPolicySupportedForIntfType
      return ( mode.intf.name.startswith(
         ( 'Ethernet', 'Switch', 'Port-Channel', 'InternalRecirc', 'Vlan' ) ) )

IntfCli.IntfConfigMode.addModelet( IntfAegisModelet )
IntfRangeCli.IntfRangeConfigMode.addModelet( IntfAegisModelet )

#----------------------------------------------------------------------
# Modelet under Intf mode for traffic-policy fallback command
#----------------------------------------------------------------------
class IntfAegisModeletFallback( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # fallback supported only on physical ports (Ethernet/Port-Channel)
      return ( mode.intf.name.startswith(
         ( 'Ethernet', 'Port-Channel' ) ) and not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( IntfAegisModeletFallback )

class TrafficPolicyAegisCmdBase( CliCommand.CliCommandClass ):
   @staticmethod
   def _handleServicePolicy( no, mode, args ):
      trafficPolicyName = args.get( 'TRAFFIC_POLICY' )

      if mode.multiInstance_:
         intfNames = list( mode.intfList.intfNames() )
      else:
         intfNames = [ mode.intf.name ]

      PolicyMapCliLib.handleServicePolicies( mode, no, trafficPolicyName,
                                                policiesCliConfig,
                                                policiesStatusRequestDir,
                                                policiesStatus, None,
                                                intfConfig=intfConfigIngress,
                                                configSessionRollbackSupported=True,
                                                intfs=intfNames )

   @staticmethod
   def handler( mode, args ):
      # handle policy application
      TrafficPolicyAegisCmdBase._handleServicePolicy( False, mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # handle policy removal
      TrafficPolicyAegisCmdBase._handleServicePolicy( True, mode, args )

#----------------------------------------------------------------------
# [ no|default ] traffic-policy (input | output) [ TRAFFIC_POLICY ]
#----------------------------------------------------------------------
class TrafficPolicyAegis( TrafficPolicyAegisCmdBase ):
   syntax = "traffic-policy input TRAFFIC_POLICY"
   noOrDefaultSyntax = "traffic-policy input [ TRAFFIC_POLICY ]"
   data = {
      'traffic-policy' : trafficPolicyKeyword,
      'input' : CliCommand.Node( matcher=inputKeyword,
         guard=aegisTrafficPolicyIntfGuard ),
      'TRAFFIC_POLICY' : CliMatcher.DynamicNameMatcher(
         getTrafficPolicyNames,
         "Traffic Policy Name" )
   }

#----------------------------------------------------------------------
# [ no|default ] traffic-policy input [ TRAFFIC_POLICY ]
# fallback traffic-policy none
#----------------------------------------------------------------------
class TrafficPolicyAegisFallback( TrafficPolicyAegisCmdBase ):
   syntax = "traffic-policy input TRAFFIC_POLICY fallback FALLBACK_TYPE none"
   noOrDefaultSyntax = "traffic-policy input [ TRAFFIC_POLICY ] ..."
   data = {
      'traffic-policy' : trafficPolicyKeyword,
      'input' : CliCommand.Node( matcher=inputKeyword,
         guard=aegisIntfOverrideGuard ),
      'TRAFFIC_POLICY' : CliMatcher.DynamicNameMatcher(
         getTrafficPolicyNames,
         "Traffic Policy Name" ),
      'fallback' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'fallback',
         helpdesc='Configure the fallback behavior for this interface '
         'traffic-policy' ) ),
      'FALLBACK_TYPE' : CliMatcher.KeywordMatcher(
         'traffic-policy', helpdesc='Fallback to a traffic-policy' ),
      'none' : 'Do not fallback to any traffic-policy'
   }

#----------------------------------------------------------------------
# [ no|default ] traffic-policy output [ TRAFFIC_POLICY ]
#----------------------------------------------------------------------
class TrafficPolicyAegisEgress( CliCommand.CliCommandClass ):
   syntax = "traffic-policy output TRAFFIC_POLICY"
   noOrDefaultSyntax = "traffic-policy output [ TRAFFIC_POLICY ]"
   data = {
      'traffic-policy' : trafficPolicyKeyword,
      'output' : CliCommand.Node( matcher=outputKeyword,
         guard=aegisEgressTrafficPolicyGuard ),
      'TRAFFIC_POLICY' : CliMatcher.DynamicNameMatcher(
         getTrafficPolicyNames,
         "Traffic Policy Name" )
   }

   @staticmethod
   def _handleServicePolicy( no, mode, args ):
      trafficPolicyName = args.get( 'TRAFFIC_POLICY' )
      if mode.multiInstance_:
         intfNames = list( mode.intfList.intfNames() )
      else:
         intfNames = [ mode.intf.name ]

      PolicyMapCliLib.handleServicePolicies( mode, no, trafficPolicyName,
                                             policiesCliConfig,
                                             policiesStatusRequestDir,
                                             policiesStatus, None,
                                             intfConfig=intfConfigEgress,
                                             configSessionRollbackSupported=True,
                                             intfs=intfNames )

   @staticmethod
   def handler( mode, args ):
      # handle policy application
      TrafficPolicyAegisEgress._handleServicePolicy( False, mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # handle policy removal
      TrafficPolicyAegisEgress._handleServicePolicy( True, mode, args )

# ----------------------------------------------------------------------
# [ no|default ] traffic-policy [ TRAFFIC_POLICY ]
# ----------------------------------------------------------------------
class TrafficPolicyAegisBridged( CliCommand.CliCommandClass ):
   syntax = "traffic-policy TRAFFIC_POLICY"
   noOrDefaultSyntax = "traffic-policy [ TRAFFIC_POLICY ]"
   data = {
      'traffic-policy' : trafficPolicyKeyword,
      'TRAFFIC_POLICY' : CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
         getTrafficPolicyNames, "Traffic Policy Name" ),
         guard=aegisBridgedTrafficPolicyGuard )
   }

   @staticmethod
   def _handleServicePolicy( no, mode, args ):
      trafficPolicyName = args.get( 'TRAFFIC_POLICY' )
      vlanIds = []
      # handles case where trffic-policy gets applied to vlan range
      if mode.multiInstance:
         for m in mode.vlan.individualVlans_:
            vlanIds.append( m.vlan.id )
      else:
         vlanIds.append( mode.vlan.id )
      if len( vlanIds ) == 1:
         intfName = vlanIdToIntfVlanId( vlanIds[ 0 ] )
         PolicyMapCliLib.handleServicePolicy( mode, no, trafficPolicyName,
                                              policiesCliConfig,
                                              policiesStatusRequestDir,
                                              policiesStatus, None,
                                              intfConfig=intfConfigBridged,
                                              configSessionRollbackSupported=True,
                                              intfName=intfName )
      elif len( vlanIds ) > 1:
         intfNames = []
         for vlanId in vlanIds:
            intfNames.append( vlanIdToIntfVlanId( vlanId ) )
         PolicyMapCliLib.handleServicePolicies( mode, no, trafficPolicyName,
                                                policiesCliConfig,
                                                policiesStatusRequestDir,
                                                policiesStatus, None,
                                                intfConfig=intfConfigBridged,
                                                configSessionRollbackSupported=True,
                                                intfs=intfNames )

   @staticmethod
   def handler( mode, args ):
      # handle policy application
      TrafficPolicyAegisBridged._handleServicePolicy( False, mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # handle policy removal
      TrafficPolicyAegisBridged._handleServicePolicy( True, mode, args )

if togglePerVrfIntfOverrideEnabled():
   IntfAegisModeletFallback.addCommandClass( TrafficPolicyAegisFallback )

IntfAegisModelet.addCommandClass( TrafficPolicyAegis )

IntfAegisModelet.addCommandClass( TrafficPolicyAegisEgress )

VlanCli.VlanConfigMode.addCommandClass( TrafficPolicyAegisBridged )

#----------------------------------------------------------------------
# Support for default interface command
#----------------------------------------------------------------------
class AegisIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del intfConfigIngress.intf[ self.intf_.name ]
      del intfConfigIngress.intfFallback[ self.intf_.name ]

      del intfConfigEgress.intf[ self.intf_.name ]
      del intfConfigEgress.intfFallback[ self.intf_.name ]

IntfCli.Intf.registerDependentClass( AegisIntfJanitor )

# ----------------------------------------------------------------------
# Support for 'no vlan' command
# ----------------------------------------------------------------------
class AegisVlanJanitor( VlanCli.VlanDependentBase ):
   def destroy( self ):
      intfName = vlanIdToIntfVlanId( self.vlan_.id_ )
      del intfConfigBridged.intf[ intfName ]
      del intfConfigBridged.intfFallback[ intfName ]

VlanCli.Vlan.registerDependentClass( AegisVlanJanitor )

def Plugin( em ):
   global policiesCliConfig, intfConfigIngress, intfConfigEgress, intfConfigBridged
   global policiesStatusRequestDir, policiesStatus
   global entityManager
   global hwStatus
   # pylint: disable-next=consider-using-f-string
   policiesCellStatusPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesCliConfigPath = 'trafficPolicies/input/cli'
   policiesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusRequestPath = 'trafficPolicies/statusRequest/cli'
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'
   intfConfigIngressAegisPath = 'trafficPolicies/intf/input/aegis'
   intfConfigEgressAegisPath = 'trafficPolicies/intf/output/aegis'
   intfConfigBridgedAegisPath = 'trafficPolicies/intf/bridged/aegis'
   intfConfigType = "PolicyMap::IntfConfig"
   hwStatusPath = 'trafficPolicies/hardware/status/interface'
   hwStatusType = "TrafficPolicy::HwStatus"

   entityManager = em

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesCellStatusPath, policiesStatusType,
                                      'ri' )
   mountGroup.close( callback=None, blocking=False )

   policiesCliConfig = ConfigMount.mount( entityManager, policiesCliConfigPath,
                                          policiesCliConfigType, 'wi' )
   intfConfigIngress = ConfigMount.mount( entityManager, intfConfigIngressAegisPath,
                                          intfConfigType, 'wi' )
   intfConfigEgress = ConfigMount.mount( entityManager, intfConfigEgressAegisPath,
                                         intfConfigType, 'wi' )
   intfConfigBridged = ConfigMount.mount( entityManager, intfConfigBridgedAegisPath,
                                          intfConfigType, 'wi' )
   hwStatus = LazyMount.mount( entityManager, hwStatusPath, hwStatusType, 'r' )

   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestPath,
                                               policiesStatusRequestDirType,
                                               'wc' )
