# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from Ark import utcTimeRelativeToNowStr, timestampToStr
from CliModel import List
from CliModel import Model
from CliModel import Dict
from CliModel import Int
from CliModel import Str
from CliModel import Bool
from CliModel import Float
from TableOutput import createTable
from TableOutput import Format
import Tac
import operator

class ShowAgentNames( Model ):
   agentNames = List ( valueType=str, 
                       help="List of Agent names in Sysdb's launcher config" )

   def render( self ):
      for agentName in self.agentNames:
         print( agentName )

def boolToStr( b ):
   return 'enabled' if b else 'disabled'

class StateAndLevel( Model ):
   levels = List( valueType=str,
                   help="List of levels for which tracing has been enabled" )

class ShowTraceAgent( Model ):
   facilities = Dict( keyType=str, valueType=StateAndLevel,
                      help="A mapping of facility name "
                           "to its trace settings" )
   showDate = Bool( help="Date in trace messages", default=False )
   showTraceTimestamp = Bool( help="Timestamp in trace messages", default=False )
   showPid = Bool( help="Pid in trace messages", default=False )
   showTid = Bool( help="Tid in trace messages", default=False )
   showFacilityName = Bool( help="Facility name in trace messages", default=False )
   showTracelevel = Bool( help="Trace level in trace messages", default=False  )
   showFileName = Bool( help="File name in trace messages", default=False )
   showLineNumber = Bool( help="Line number in trace messages", default=False )
   showFunctionName = Bool( help="Method name in trace messages", default=False )
   execTrace = List( valueType=str, help="Active exec trace settings for agent" )
   execTraceTimeout = Int( help="Time remaining in exec traces", default=0 )
   execTraceStartTime = Int( help="Time at which the exec trace was initialized",
                             default=0 )
   filenameWidth = Int( help="Set the width of file name in the trace output",
                        optional=True )
   facilityNameWidth = Int( help="Set the width of facility name "
                                 "in the trace output", optional=True )
   functionNameWidth = Int( help="Set the width of function name "
                                 "in the trace output", optional=True )
   _agent = Str( help="Name of the agent" )
   destination = Str( help="File name where trace output goes if not using "
                           "the standard log file", optional=True )
   configTrace = List( valueType=str, help="Trace setting for agent "
                                           "when agent is not running" )
   _agentConfig = Bool( default=True, help="Detailed trace information if True" )

   def render( self ):
      print( "Global trace settings for agent %s" % self._agent )
      print( "-----------------------------------------------" )
      if not self._agentConfig:
         if self.destination:
            print( "Tracing sent to %s" % self.destination )
         if self.configTrace:
            print()
            print( "Trace facility settings for agent {} is {} ".format( self._agent,
                                ','.join( self.configTrace ) ) )
         return
      print( "Tracing sent to %s" % self.destination )
      print()
      print( "Date:     %s" % boolToStr( self.showDate ) )
      print( "Time:     %s" % boolToStr( self.showTraceTimestamp ) )
      print( "PID:           %s" % boolToStr( self.showPid ) )
      print( "TID:           %s" % boolToStr( self.showTid ) )
      # pylint: disable-next=bad-string-format-type
      print( "Facility name: %-8s (width %d)" % ( boolToStr( self.showFacilityName ),
                                  self.facilityNameWidth ) )
      print( "Trace level:   %s" % boolToStr( self.showTracelevel ) )
      # pylint: disable-next=bad-string-format-type
      print( "Filename:      %-8s (width %d)" % ( boolToStr( self.showFileName ),
                                  self.filenameWidth ) )
      print( "Line number:   %s" % boolToStr( self.showLineNumber ) )
      # pylint: disable-next=bad-string-format-type
      print( "Function name: %-8s (width %d)" % ( boolToStr( self.showFunctionName ),
                                  self.functionNameWidth ) )
      showExecTrace = ""
      if self.execTraceTimeout == 0:
         showExecTrace = "disabled"
      else:
         minutesRemaining, secondsRemaining = divmod( self.execTraceTimeout, 60 )
         hoursRemaining, minutesRemaining = divmod( minutesRemaining, 60 )
         showExecTrace = str( int( hoursRemaining ) ) + "h "
         showExecTrace += str( int( minutesRemaining ) ) + "m "
         showExecTrace += str( int( secondsRemaining ) ) + "s remaining "
         startTimeString = timestampToStr( self.execTraceStartTime,
                                           relative=False )
         showExecTrace += "(started at " + startTimeString + ")"
         showExecTrace += "\n"
         showExecTrace += "                  %s" % ','.join( self.execTrace )
      print( "Timed exec trace: %s" % showExecTrace )
      print()
      print( "-----------------------------------------------" )
      for facility, stateLevel in sorted ( self.facilities.items() ):
         print( "%-20s enabled  %s" % ( facility, ''.join( stateLevel.levels ) ) )

class ShowAgentWarmRestartEntry( Model ):
   windowLength = Float( help="Window length in seconds for warm restarts" )
   allowedCount = Int( help="Number of warm restarts allowed" )

class ShowAgentWarmRestartTable( Model ):
   agents = Dict( keyType=str, valueType=ShowAgentWarmRestartEntry,
                  help="Mapping from Agent name to Agent warm restart "
                        "configurations" )

   def render( self ):
      headings = ( "Agent Name", "Allowed", "Window" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings, tableWidth=60 )
      table.formatColumns( fl, fr, fr )
      agentList = sorted( self.agents.items(), key=operator.itemgetter( 0 ) )
      for agent, config in agentList:
         table.newRow( agent, config.allowedCount, config.windowLength )

      print( table.output() )

class ShowAgentActivity( Model ):
   activityMgrLoopCounter = Int( help="Number of activity loops executed" )
   activityLoopExecNsTime = Int(
       help="UTC time in nanoseconds at which the current activity loop started" )
   maxClockNotifTime = Float( help="Maximum clock notification time in seconds" )
   avgClockNotifTime = Float( help="Average clock notification time in seconds" )
   maxIOWaitTime = Float( help="Maximum IO wait time in seconds" )
   avgIOWaitTime = Float( help="Average IO wait time in seconds" )
   maxTimeBetweenActivityLoops = Float(
         help="Maximum time between activity loops in seconds" )
   avgTimeBetweenActivityLoops = Float(
         help="Average time between activity loops in seconds" )
   maxFdsPerCycle = Int(
      help="Maximum file descriptors processed per activity loop" )
   avgFdsPerCycle = Float(
      help="Average file descriptors processed per activity loop" )
   avgMountTime = Float( help="Average mount time in seconds" )
   mountRequestCount = Int( help="Number of mounts requested" )
   mountCompleteCount = Int( help="Number of mounts complete" )
   avgRemoteMountTime = Float( help="Average remote mount time in seconds" )
   mountRemoteRequestCount = Int( help="Number of remote mounts requested" )
   mountRemoteCompleteCount = Int( help="Number of remote mounts complete" )
   maxFlowOutCount = Int( help="Maximum number of FlowOut in one activity loop" )
   currentFlowOutCount = Int( help="Current number of FlowOut" )
   totalFlowOutCount = Int( help="Total number of FlowOut" )
   mountFlowOutOnNonSchedConnCount = Int(
      help="Number of mount FlowOut on non-scheduled connection" )
   maxFlowOutRunTime = Float(
      help="Maximum run time spent for an individual FlowOut in one activity loop " 
           "in seconds" )
   avgFlowOutRunTime = Float(
      help="Average run time spent for an individual FlowOut in one activity loop "
           "in seconds" )
   maxCombinedFlowOutRunTime = Float(
      help="Maximum run time spent for all FlowOuts in one activity loop in "
           "seconds" )
   avgCombinedFlowOutRunTime = Float(
      help="Average run time spent for all FlowOuts in one activity loop in "
           "seconds" )
   attrlogNotEnoughWritableBytes = Int(
      help="Number of times FlowOut could not be processed due to no space "
           "left in write buffer or FlowOut Quantum is reached" )
   tacElEntityRefCount = Int( help="TAC entity log reference count" )
   maxMountStatTime = Float( 
         help="Maximum time spent processing a mount in seconds" )
   maxMountStatPeerName = Str(
      help="Peer name of mount that took longest to process" )
   maxMountStatMountName = Str(
      help="Mount name of mount that took longest to process" )
   maxRemoteMountStatTime = Float(
      help="Maximum time spent processing a remote mount in seconds" )
   maxRemoteMountStatPeerName = Str(
      help="Peer name of remote mount that took longest to process" )
   maxRemoteMountStatMountName = Str(
      help="Mount name of remote mount that took longest to process" )
   
   def render( self ):
      def threeDecimals( number ):
         return '%.3f' % number
   
      print( "Activity manager loop counter:", self.activityMgrLoopCounter )
      # We need to convert from nanoseconds to seconds before converting
      # the time to a string
      print( "Current activity loop started:",
             utcTimeRelativeToNowStr( self.activityLoopExecNsTime / 1000000000 ) )
      print( "Maximum clock notification time:",
             threeDecimals( self.maxClockNotifTime ), "seconds" )
      print( "Average clock notification time:",
             threeDecimals( self.avgClockNotifTime ), "seconds" )
      print( "Maximum IO wait time:", threeDecimals( self.maxIOWaitTime ), 
             "seconds" )
      print( "Average IO wait time:", threeDecimals( self.avgIOWaitTime ), 
             "seconds" )
      print( "Maximum time between activity loops:",
             threeDecimals( self.maxTimeBetweenActivityLoops ), "seconds" )
      print( "Average time between activity loops:",
             threeDecimals( self.avgTimeBetweenActivityLoops ), "seconds" )
      print( "Maximum file descriptors per cycle:", self.maxFdsPerCycle )
      print( "Average file descriptors per cycle:", 
             threeDecimals( self.avgFdsPerCycle ) )
      print( "Average mount time:", threeDecimals( self.avgMountTime ), "seconds" )
      print( "Mounts requested:", self.mountRequestCount )
      print( "Mounts complete:", self.mountCompleteCount )
      print( "Average remote mount time:",
             threeDecimals( self.avgRemoteMountTime ), "seconds" )
      print( "Remote mounts requested:", self.mountRemoteRequestCount )
      print( "Remote mounts complete:", self.mountRemoteCompleteCount )
      print( "Maximum FlowOut for one activity loop:", self.maxFlowOutCount )
      print( "Current FlowOut:", self.currentFlowOutCount )
      print( "Total FlowOut:", self.totalFlowOutCount )
      print( "Mount FlowOut on nonscheduled connection:",
             self.mountFlowOutOnNonSchedConnCount )
      print( "Maximum FlowOut run time:",
             threeDecimals( self.maxFlowOutRunTime ), "seconds" )
      print( "Average FlowOut run time:",
             threeDecimals( self.avgFlowOutRunTime ), "seconds" )
      print( "Maximum combined FlowOut run time:",
             threeDecimals( self.maxCombinedFlowOutRunTime ), "seconds" )
      print( "Average combined FlowOut run time:",
             threeDecimals( self.avgCombinedFlowOutRunTime ), "seconds" )
      print( "AttrLog not enough writable bytes occurrences:",
             self.attrlogNotEnoughWritableBytes )
      print( "TAC entity log reference count:", self.tacElEntityRefCount )
      print( "Max mount stat time:", threeDecimals( self.maxMountStatTime ), 
             "seconds" )
      print( "Max mount stat peer name:", self.maxMountStatPeerName )
      print( "Max mount stat mount name:", self.maxMountStatMountName )
      print( "Max remote mount stat time:",
             threeDecimals( self.maxRemoteMountStatTime ), "seconds" )
      print( "Max remote mount stat peer name:",
              self.maxRemoteMountStatPeerName )
      print( "Max remote mount stat mount name:",
             self.maxRemoteMountStatMountName )

class AgentRunnability( Model ):
   agentName = Str( help='Agent name' )
   status = Bool( help='True if this agent is runnable' )
   role = Str( help='The redundancy role the agent is expected to run in' )
   reason = List( valueType=str, help='The reasons the agent is or is not runnable' )

   # translate the role into something suitable for the show command
   def sanitizedRoleText( self ):
      RedundancyRoleName = Tac.Type( 'Redundancy::RedundancyRoleName' )
      if self.role == RedundancyRoleName.activeSupervisor:
         return "active supervisor"
      elif self.role == RedundancyRoleName.allSupervisors:
         return "all supervisors"
      elif self.role == RedundancyRoleName.allCells:
         return "all cells"
      return self.role

   def render( self ):
      print( "Agent:", self.agentName )
      print( "Status:", "runnable" if self.status else "not runnable" )
      print( "Role:", self.sanitizedRoleText() )
      reasonHead = "Satisfied:" if self.status else "Not satisfied:"
      for r in sorted( self.reason ):
         print( reasonHead, r )

class ShowAgentRunnability( Model ):
   agents = Dict( keyType=str,
                  valueType=AgentRunnability,
                  help='Mapping from agent name to agent runnability status' )
   def render( self ):
      for runnabilityInfo in self.agents.values() :
         runnabilityInfo.render()
         print()

