# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import AleCapacityModel, AleCapacityLib, TechSupportCli
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliToken.Hardware
import ConfigMount
import ShowCommand

gv = CliGlobal.CliGlobal( capacityConfig = None,
                          disableAlertThreshold = 0 )

def getThreshold( mode ):
   tableThresholds = {}
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for rowName in gv.capacityConfig.threshold:
      tableThreshold = AleCapacityModel.TableThreshold()
      tableThreshold.configThreshold = \
            gv.capacityConfig.threshold[ rowName ].threshold
      tableThreshold.configClearThreshold = \
            gv.capacityConfig.threshold[ rowName ].clear
      tableThresholds[ rowName ] = tableThreshold

   for agentName in statusDir:
      agentDir = mode.sysdbRoot.entity[ f"hardware/capacity/status/{agentName}" ]
      for tableKey in agentDir.entry:
         if tableKey.feature != '':
            rowName = '-'.join( [ tableKey.table, tableKey.feature ] )
         else:
            rowName = tableKey.table

         if rowName not in tableThresholds:
            # If the entry doesn't exist in capacityConfig, it is set to default
            tableThreshold = AleCapacityModel.TableThreshold()
            tableThreshold.configThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold
            tableThreshold.configClearThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold
            tableThreshold.defaultThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold
            tableThresholds[ rowName ] = tableThreshold
         else:
            # If the entry exists in capacityConfig, update the default
            tableThresholds[ rowName ].defaultThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold

   return tableThresholds

def thresholdExceeded( used, threshold, clear, alert=False ):
   if threshold == gv.disableAlertThreshold or used < clear:
      return False
   if used >= threshold:
      return True
   return alert

def getMonitoringNameFromTableKey( tableKey ):
   if tableKey.feature:
      tableName = tableKey.table + "-" + tableKey.feature
   else:
      tableName = tableKey.table

   if not tableKey.chip:
      return tableName
   return tableKey.table + "-" + tableKey.feature + "-" + tableKey.chip

#------------------------------------------------------------------------------------
# show hardware capacity [ utilization percent exceed ( <percentVal> | threshold ) ]
#------------------------------------------------------------------------------------
class ShowCapacityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show hardware capacity [ utilization percent exceed
               ( <percentVal> | threshold ) ]"""
   data = {
      'hardware': CliToken.Hardware.hardwareForShowMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'utilization': 'Filter based on current utilization',
      'percent': 'Use percentage values as unit for filtering',
      'exceed': 'Values that are greater than or equal to',
      '<percentVal>': CliMatcher.IntegerMatcher( 0, 100,
                                helpdesc='Utilization value in percentage <0-100>' ),
      'threshold': 'Configured or default threshold'
   }
   cliModel = AleCapacityModel.CapacityUsage

   @staticmethod
   def handler( mode, args ):
      tableEntries = []
      threshold = args.get( 'threshold' )
      percentVal = args.get( '<percentVal>' )

      # get table entry from other platforms
      for hook in AleCapacityLib.showHwCapacity.extensions():
         hookTableEntries = hook( mode.entityManager, percentVal, threshold )
         if hookTableEntries:
            tableEntries.extend( hookTableEntries )

      # get data from the sysdb status path
      statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]
      monitoring = mode.sysdbRoot.entity[ "hardware/capacity/monitoring" ]

      if threshold == 'threshold':
         tableThresholds = getThreshold( mode )

      for agentName in statusDir:
         agentDir = mode.sysdbRoot.entity[ f"hardware/capacity/status/{agentName}" ]

         for tableKey in agentDir.entry:
            if percentVal is not None:
               if ( agentDir.entry[ tableKey ].usedPercent() <
                    percentVal ):
                  continue

            if threshold == 'threshold':
               if tableKey.feature != '':
                  rowName = '-'.join( [ tableKey.table, tableKey.feature ] )
               else:
                  rowName = tableKey.table

               alert = False
               monitoringName = getMonitoringNameFromTableKey( tableKey )
               if monitoringName in monitoring.data:
                  alert = monitoring.data[ monitoringName ].alert

               if not thresholdExceeded(
                     agentDir.entry[ tableKey ].usedPercent(),
                     tableThresholds[ rowName ].configThreshold,
                     tableThresholds[ rowName ].configClearThreshold,
                     alert ):
                  continue

            tableEntry = AleCapacityModel.TableEntry()
            tableEntry.table = agentDir.entry[ tableKey ].tableKey.table
            tableEntry.feature = agentDir.entry[ tableKey ].tableKey.feature
            tableEntry.chip = agentDir.entry[ tableKey ].tableKey.chip
            tableEntry.used = agentDir.entry[ tableKey ].used
            tableEntry.free = agentDir.entry[ tableKey ].free
            tableEntry.usedPercent = agentDir.entry[ tableKey ].usedPercent()
            tableEntry.committed = agentDir.entry[ tableKey ].committed
            tableEntry.maxLimit = agentDir.entry[ tableKey ].maxLimit
            tableEntry.highWatermark = agentDir.entry[ tableKey ].highWatermark
            tableEntry.sharedFeatures.extend( 
               agentDir.entry[ tableKey ].sharedFeatures.values() 
            )
            tableEntries.append( tableEntry )

      return AleCapacityModel.CapacityUsage( tables=tableEntries )

BasicCli.addShowCommandClass( ShowCapacityCmd )

#------------------------------------------------------------------------------------
# show hardware capacity alert threshold
#------------------------------------------------------------------------------------
class ShowThresholdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware capacity alert threshold'
   data = {
      'hardware': CliToken.Hardware.hardwareForShowMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'alert': 'Configured alerts for hardware resource utilization',
      'threshold': 'Display all thresholds for generating alerts'
   }
   cliModel = AleCapacityModel.CapacityThreshold

   @staticmethod
   def handler( mode, args ):
      tableThresholds = getThreshold( mode )
      return AleCapacityModel.CapacityThreshold( thresholds=tableThresholds )

BasicCli.addShowCommandClass( ShowThresholdCmd )

#------------------------------------------------------------------------------------
# [ no | default ] hardware capacity alert table <table-name> [ feature
#                  <feature-name> ] threshold THRESHOLD [ clear CLEAR_THRESHOLD ]
#------------------------------------------------------------------------------------
def getTableNames( mode ):
   tableList = {}
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for agentName in statusDir:
      agentDir = mode.sysdbRoot.entity[ f"hardware/capacity/status/{agentName}" ]

      if agentDir.tableList is None:
         continue
      for tableName in agentDir.tableList.featureMapping:
         if not tableName:
            # invalid table name, skipping
            continue
         tableList[ tableName ] = f"Set threshold for {tableName} table"

   return tableList

def getTableFeatureNames( mode, context ):
   featureList = {}
   tableName = context.sharedResult[ 'TABLE_NAME' ]
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for agentName in statusDir:
      # If two agentDir has the same tableName and featureName ( basically the same
      # tableKey ), we would try to go ahead and display duplicate entries. It is
      # upto the developers populating the agentDir to keep it exclusive
      agentDir = mode.sysdbRoot.entity[ f"hardware/capacity/status/{agentName}" ]

      if agentDir.tableList is None:
         continue
      if tableName not in agentDir.tableList.featureMapping:
         continue
      for featureName in agentDir.tableList.featureMapping[ tableName ].feature:
         if featureName != "":
            featureList[ featureName ] = f"Set threshold for {featureName} feature"

   return featureList

class CfgCapacityThreshCmd( CliCommand.CliCommandClass ):
   syntax = ( "hardware capacity alert table TABLE_NAME "
              "[ feature ( FEATURE_NAME | all ) ] "
              "threshold THRESHOLD "
              "[ clear CLEAR_THRESHOLD ]" )

   noOrDefaultSyntax = ( "hardware capacity alert table TABLE_NAME "
                         "[ feature ( FEATURE_NAME | all ) ] "
                         "[ threshold ... ]" )

   data = {
      'hardware': CliToken.Hardware.hardwareForConfigMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'alert': 'Trigger syslog for hardware resource utilization',
      'table': 'Specify the table to monitor utilization',
      'TABLE_NAME': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( getTableNames,
                                           alwaysMatchInStartupConfig=True ),
         storeSharedResult=True ), # stores the result for featureName matcher
      'feature': 'Specify the feature to monitor utilization',
      'FEATURE_NAME': CliMatcher.DynamicKeywordMatcher(
         getTableFeatureNames,
         alwaysMatchInStartupConfig=True,
         passContext=True ),
      'all': 'Monitor utilization for all features',
      'threshold': 'Set threshold to trigger alert based on percentage utilization',
      'THRESHOLD': CliMatcher.IntegerMatcher(
         0, 100, helpdesc='Threshold value in percentage' ),
      'clear':
         'Set clear-threshold to clear the alert based on percentage utilization',
      'CLEAR_THRESHOLD': CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='Clear-threshold value in percentage; clear-threshold should be '
                  'at most the threshold value' )
   }
   handler = "AleCapacityHandlers.cfgCapacityThreshCmdHandler"
   noOrDefaultHandler = "AleCapacityHandlers.cfgCapacityThreshCmdNoOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( CfgCapacityThreshCmd )

#------------------------------------------------------------------------------------
# show tech-support hooks
#------------------------------------------------------------------------------------

# Timestamp is added to maintain historical order whithin show tech-support
TechSupportCli.registerShowTechSupportCmd(
      "2016-12-19 08:45:00",
      cmds=[ 'show hardware capacity' ],
      summaryCmds=[ 'show hardware capacity' ] )

#------------------------------------------------------------------------------------
# plugin definition
#------------------------------------------------------------------------------------

def Plugin( entityManager ):
   gv.capacityConfig = ConfigMount.mount( entityManager,
                                       "hardware/capacity/config",
                                       "AleCapacity::TableThreshold", "w" )

   mg = entityManager.mountGroup()
   mg.mount( "hardware/capacity/status", "Tac::Dir", "ri" )
   mg.mount( "hardware/capacity/monitoring", "AleCapacity::TableMonitoring", "r" )

   def _finish():
      pass

   mg.close( _finish )

