# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
import datetime

import CliGlobal
import CliPlugin.IntfCli as IntfCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IraIpCli as IraIpCli
import CliPlugin.TechSupportCli as TechSupportCli
import CliPlugin.VrfCli as VrfCli
from CliPlugin.AleCountersModel import ChipEvent
from CliPlugin.AleCountersModel import CounterEvent
from CliPlugin.AleCountersModel import CounterFeatureModel
from CliPlugin.AleCountersModel import DropCounterModel
from CliPlugin.AleCountersModel import FlowTrendEventModel
from CliPlugin.AleCountersModel import InterfaceIpv4Ipv6CountersRates
from CliPlugin.AleCountersModel import InterfaceQueueLengthInfo
from CliPlugin.AleCountersModel import InterfacesIpv4Ipv6Counters
from CliPlugin.RouterGeneralCli import (
      RouterGeneralMode,
      matcherRoute,
      routerGeneralCleanupHook,
)
from AleCounters import tunnelStrToType
from AleFlexCounterTableMounter import tableMountPath
from AleFlexCounterTypes import featureMap, tokenHelpdesc, unitsTokenToCountModeMap
import Ark
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.AleCounters import (
   MonitorCounterEthernetMode,
   MonitorCounterMode,
   MonitorQueueCounterMode,
   RouteCountersModeBase,
   RouteCountersVrfModeBase,
   TunnelCountersMode,
   TunnelCountersTransmitMode,
)
import CliParser
import CliToken.Clear
import CliToken.Hardware
import CliToken.Monitor
import CliToken.TunnelCli
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
import MrouteConsts
import ShowCommand
import SharedMem
import Smash
import SmashLazyMount
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'AleCountersCli' )
t0 = __defaultTraceHandle__.trace0
t8 = __defaultTraceHandle__.trace8
t9 = __defaultTraceHandle__.trace9

entityManager = None
internalDropSharedConfig = None
internalDropCliConfig = None
flowTrendSharedConfig = None
fcFeatureConfigDir = None
fcFeatureStatusDir = None
prefixCounterCliConfig = None
rxIpCounterTable = None
rxL3IpCounterTable = None
rxIpLagCounterTable = None
txIpCounterTable = None
txL3IpCounterTable = None
rxIpSnapshotTable = None
rxL3IpSnapshotTable = None
txIpSnapshotTable = None
txL3IpSnapshotTable = None
txIpLagCounterTable = None
rxL3IpCounterRateTable = None
txL3IpCounterRateTable = None
rxIpCounterRateTable = None
txIpCounterRateTable = None
fcRouteConfig = None
l3RouteClearRequest = None
cliCapability = None
ipCountersCapability = None
ipv4McastCounterConfig = None
ipv6McastCounterConfig = None
intfCliConfig = None
queueCliConfig = None
routingHardwareRouteStatus = None
allIntfConfigDir = None
ipv4CounterKeyMap = None
ipv4McastRateCounterTable = None
ipv6CounterKeyMap = None
ipv6McastRateCounterTable = None
tunnelCountersCliConfig = None
tunnelCountersPriorityTable = None

gv = CliGlobal.CliGlobal( rateWatermarkConfig=None )

AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
CliConfigConstants = TacLazyType( 'Ale::Counters::CliConfigConstants' )
CountedPrefixLens = TacLazyType( 'Ale::CountedPrefixLens' )
CounterCategory = TacLazyType( 'Ale::Counters::CounterCategory' )
DropDelta = TacLazyType( 'Ale::Counters::Delta' )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )
FapId = TacLazyType( 'FlexCounters::FapId' )
FeatureIdEnum = TacLazyType( 'FlexCounters::FeatureId' )
FeatureIdEnumVal = TacLazyType( 'FlexCounters::FeatureIdEnumVal' )
FlowTrendConstants = TacLazyType( 'Ale::Counters::FlowTrendMonitor::Constants' )
InternalDropConstants = TacLazyType( 'Ale::Counters::InternalDrop::Constants' )
IpFlexCountersIndex = TacLazyType( 'FlexCounters::IpFlexCountersIndex' )
Offset = TacLazyType( 'Ale::Counters::InternalDrop::Offset' )
PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )
TimeConstants = TacLazyType( 'Ale::Counters::TimeConstants' )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
TunnelCountersPriority = TacLazyType( "Tunnel::TunnelCountersPriority" )

ipCountersRatesKw = CliMatcher.KeywordMatcher( 'rates',
      helpdesc='Input/Output IP packet rate counters' )
matcherCounter = CliMatcher.KeywordMatcher( 'counter',
      helpdesc='Counter' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Interface counters' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4', 'IPv4 related' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6', 'IPv6 related' )
exprRouteVrfWithName = VrfCli.VrfExprFactory(
      helpdesc='Configure counter in a VRF',
      guard=IraIpCli.vrfRoutingSupportedGuard )
exprIpv4AddrOrPrefix = IpAddrMatcher.IpAddrOrPrefixExprFactory(
      'Match this IP address', 'Match this subnet mask', 'Match this IP prefix' )
exprRute6Prefix = Ip6AddrMatcher.Ip6AddrOrPrefixExprFactory()
matcherFeature = CliMatcher.KeywordMatcher( 'feature',
      helpdesc='Counter feature' )

def timestampToStr( switchTime ):
   utcTime = Ark.switchTimeToUtc( switchTime )
   dt = datetime.datetime.fromtimestamp( utcTime )
   return dt.strftime( '%Y-%m-%d %H:%M:%S:%f' )

class MonitorCounterConfigMode( MonitorCounterMode, BasicCli.ConfigModeBase ):
   name = 'mon-counters'

   def __init__( self, parent, session ):
      MonitorCounterMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CounterConfigModeBase:
   def __init__( self, cliConfig ):
      self.cliConfig = cliConfig

   def setPollInterval( self, interval ):
      self.cliConfig.periodPoll = interval

   def noPollInterval( self ):
      self.cliConfig.periodPoll = self.cliConfig.defaultPeriodPoll

   def setFastPollInterval( self, interval ):
      self.cliConfig.periodFastPoll = interval

   def noFastPollInterval( self ):
      self.cliConfig.periodFastPoll = self.cliConfig.defaultPeriodFastPoll

class CounterQueueConfigMode( MonitorQueueCounterMode,
                              BasicCli.ConfigModeBase,
                              CounterConfigModeBase ):
   name = 'mon-counters-queue'

   def __init__( self, parent, session ):
      MonitorQueueCounterMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      CounterConfigModeBase.__init__( self, queueCliConfig )

class CounterEthernetConfigMode( MonitorCounterEthernetMode,
                                 BasicCli.ConfigModeBase,
                                 CounterConfigModeBase ):
   name = 'mon-counters-ethintfs'

   def __init__( self, parent, session ):
      MonitorCounterEthernetMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      CounterConfigModeBase.__init__( self, intfCliConfig )

CounterHook = namedtuple( 'CounterHook', 'guard getCounterInfo getChipName' )

# platform specific implementation returns CounterHook
#  CounterHook.guard( mode, token ) : platform guard
#  CounterHook.getCounterInfo( mode ) : returns 'Ale::Counters::DropConfig' object
#  CounterHook.getChipName(mode, key) : returns chipName (key is InternalDropKey)

dropCounterInfoHook = None

def registerDropCounterInfoHook( hook ):
   # In case of multiple registrations, only the last registered hook will be saved
   global dropCounterInfoHook
   dropCounterInfoHook = hook

flowTrendCounterInfoHook = None

def registerFlowTrendCounterInfoHook( hook ):
   # In case of multiple registrations, only the last registered hook will be saved
   global flowTrendCounterInfoHook
   flowTrendCounterInfoHook = hook

def getAllCounterNames( counterHook, mode ):
   '''return list of counter names'''
   counterNames = []
   if counterHook:
      counterNames = sorted( counterHook.getCounterInfo( mode ).
                             counterNameToId.keys() )
   return counterNames

def getAllDropCounterNames( mode ):
   '''return list of drop counter names'''
   return getAllCounterNames( dropCounterInfoHook, mode )

def getAllFlowTrendCounterNames( mode ):
   '''return list of flow trend counter names'''
   return getAllCounterNames( flowTrendCounterInfoHook, mode )

def dropCounterGuard( mode, token ):
   if dropCounterInfoHook:
      return dropCounterInfoHook.guard( mode, token )
   return CliParser.guardNotThisPlatform

nodeShowCounter = CliCommand.guardedKeyword( 'counter', helpdesc='Show counter info',
      guard=dropCounterGuard )

def buildDropEvent( counterName, info, key, value ):
   displayCount = value.dropCount - value.dropCountSnapshot
   if displayCount <= 0:
      # Either the drop count hasn't incremented since 'clear hardware counter
      # drop' was run and we shouldn't display it, or the count is negative and
      # we have bigger problems.
      return None
   model = DropCounterModel.DropEvents.DropEvent()
   model.counterId = key.counterId
   model.counterName = counterName
   model.counterType = info.counterInfo[ key.counterId ].counterType
   model.dropCount = displayCount
   model.eventCount = value.eventCount
   model.initialEventTime = Ark.timestampToStr( value.initialEventTime, False )
   model.lastEventTime = Ark.timestampToStr( value.lastEventTime, False )
   # optional
   if value.delta1 != DropDelta.null:
      model.dropInLastMinute = value.delta1
   if value.delta2 != DropDelta.null:
      model.dropInLastTenMinute = value.delta2
   if value.delta3 != DropDelta.null:
      model.dropInLastOneHour = value.delta3
   if value.delta4 != DropDelta.null:
      model.dropInLastOneDay = value.delta4
   if value.delta5 != DropDelta.null:
      model.dropInLastOneWeek = value.delta5
   if value.thresholdEventCount:
      model.thresholdEventCount = value.thresholdEventCount
   return model

def getCounterName( counterHook, mode, key ):
   counterName = counterHook.getCounterInfo( mode ).counterIdToName[ key.counterId ]
   # handle counters that need laneId or special indexing. Only internal drop key has
   # an offset attribute, flow trend key does not have this attribute.
   # If this logic to add offset to counterName ever changes, please update
   # Ale::Counters::InternalDrop::InternalDropEvent::idkCounterNameWithOffset().
   # Ideally this would be refactored so both CLI and SNMP can use the same function,
   # e.g. CLI to use idkCounterNameWithOffset (BUG562371)
   counterOffset = getattr( key, 'offset', None )
   if counterOffset is not None and counterOffset != Offset.null:
      counterName = '%s-%d' % ( counterName, counterOffset )
   return counterName

def getDropCounterName( mode, key ):
   return getCounterName( dropCounterInfoHook, mode, key )

def getFlowTrendCounterName( mode, key ):
   #counterNames = [ 'MacCounters', 'PacketProcessorCounters' ]
   return getCounterName( flowTrendCounterInfoHook, mode, key )

def guardSetPollPeriod( counterType ):
   isAllowed = ( counterType in cliCapability.pollPeriodAdjustmentAllowed )
   t8( 'PollPeriodAdjustmentAllowed [', counterType, '] =', isAllowed )
   if isAllowed:
      return None
   return CliParser.guardNotThisPlatform

def guardSetIntfPollPeriod( mode, token ):
   return guardSetPollPeriod( CounterCategory.interfaceCounter )

def guardSetIntfFastPollPeriod( mode, token ):
   return guardSetPollPeriod( CounterCategory.fastIntfCounter )

def guardSetIntfAnyPollPeriod( mode, token ):
   counterTypes = [
      CounterCategory.interfaceCounter,
      CounterCategory.fastIntfCounter,
   ]
   isAllowed = any( ct in cliCapability.pollPeriodAdjustmentAllowed
                    for ct in counterTypes )
   if isAllowed:
      return None
   return CliParser.guardNotThisPlatform

def guardSetQueuePollPeriod( mode, token ):
   return guardSetPollPeriod( CounterCategory.queueCounter )

def guardPrimaryBackupDedicated( mode, token ):
   if cliCapability.prefixCountersBackupSupported:
      return None
   return CliParser.guardNotThisPlatform

#--------------------------------------------------------------------------------
# monitor counters
#--------------------------------------------------------------------------------
class MonitorCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor counters'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' : CliToken.Monitor.monitorMatcher,
      'counters' : 'Monitor counters configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MonitorCounterConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = MonitorCounterConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( MonitorCountersCmd )

#--------------------------------------------------------------------------------
# ethernet interfaces
#--------------------------------------------------------------------------------
class EthernetInterfacesCmd( CliCommand.CliCommandClass ):
   syntax = 'ethernet interfaces'
   noOrDefaultSyntax = syntax
   data = {
      'ethernet' : CliCommand.guardedKeyword( 'ethernet',
         helpdesc='Configure ethernet',
         guard=guardSetIntfAnyPollPeriod ),
      'interfaces' : 'Interfaces configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CounterEthernetConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfCliConfig.periodPoll = intfCliConfig.defaultPeriodPoll
      intfCliConfig.periodFastPoll = intfCliConfig.defaultPeriodFastPoll
      gv.rateWatermarkConfig.windowSize = 0
MonitorCounterConfigMode.addCommandClass( EthernetInterfacesCmd )

#--------------------------------------------------------------------------------
# queue counters
#--------------------------------------------------------------------------------
class QueueCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'queue counters'
   noOrDefaultSyntax = syntax
   data = {
      'queue' : CliCommand.guardedKeyword( 'queue',
         helpdesc='Configure queue',
         guard=guardSetQueuePollPeriod ),
      'counters' : 'counter configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CounterQueueConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      queueCliConfig.periodPoll = queueCliConfig.defaultPeriodPoll
      queueCliConfig.periodFastPoll = queueCliConfig.defaultPeriodFastPoll

MonitorCounterConfigMode.addCommandClass( QueueCountersCmd )

#--------------------------------------------------------------------------------
# [ no | default ] update interval INTERVAL
#--------------------------------------------------------------------------------
class UpdateIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'update interval INTERVAL'
   noOrDefaultSyntax = 'update interval ...'
   data = {
      'update' : 'Update configuration',
      'interval' : CliCommand.guardedKeyword( 'interval',
         helpdesc='Update polling interval',
         guard=guardSetIntfPollPeriod ),
      'INTERVAL' : CliMatcher.FloatMatcher(
         CliConfigConstants.minPeriodPoll,
         CliConfigConstants.maxPeriodPoll,
         helpdesc='Seconds', precisionString='%.1f' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setPollInterval( args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noPollInterval()

CounterEthernetConfigMode.addCommandClass( UpdateIntervalCmd )
CounterQueueConfigMode.addCommandClass( UpdateIntervalCmd )

##################################
### [ no ] update fast-interval <m> ###
##################################
class UpdateFastIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'update fast-interval INTERVAL'
   noOrDefaultSyntax = 'update fast-interval ...'
   data = {
      'update' : 'Update configuration',
      'fast-interval' : CliCommand.guardedKeyword(
         'fast-interval', 'Update fast polling interval',
         guardSetIntfFastPollPeriod ),
      'INTERVAL' : CliMatcher.IntegerMatcher(
         CliConfigConstants.minPeriodFastPoll,
         CliConfigConstants.maxPeriodFastPoll,
         helpdesc='Specify poll period in milliseconds' )
   }

   @staticmethod
   def handler( mode, args ):
      interval = args[ 'INTERVAL' ]
      mode.setFastPollInterval( interval )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noFastPollInterval()

CounterEthernetConfigMode.addCommandClass( UpdateFastIntervalCmd )

#--------------------------------------------------------------------------------
# show hardware counter drop [ rates ]
#--------------------------------------------------------------------------------
class HardwareCounterDropCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware counter drop [ rates ]'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'counter' : nodeShowCounter,
      'drop' : 'Show drop counter info',
      'rates' : 'Show drop counter rates',
   }
   cliModel = DropCounterModel

   @staticmethod
   def handler( mode, args ):
      assert dropCounterInfoHook
      info = dropCounterInfoHook.getCounterInfo( mode )
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      model = DropCounterModel( _printFmt=args.get( 'rates' ) )
      model.totalAdverseDrops = 0
      model.totalCongestionDrops = 0
      model.totalPacketProcessorDrops = 0
      for smashEntity in shmemEm.find(
            InternalDropConstants.internalDropPrefix ).count:
         dropStatus = shmemEm.doMount( smashEntity,
                                       'Ale::Counters::InternalDrop::Status',
                                       Smash.mountInfo( 'reader' ) )
         for key, value in dropStatus.internalDrop.items():
            counterName = getDropCounterName( mode, key )
            chipName = dropCounterInfoHook.getChipName( mode, key )
            dropEvent = buildDropEvent( counterName, info, key, value )
            if dropEvent is None:
               continue # The count hasn't increased, or some other error occurred.
            if chipName not in model.dropEvents:
               model.dropEvents[ chipName ] = DropCounterModel.DropEvents()
            if dropEvent.counterType == 'Adverse':
               model.totalAdverseDrops += dropEvent.dropCount
            elif dropEvent.counterType == 'Congestion':
               model.totalCongestionDrops += dropEvent.dropCount
            elif dropEvent.counterType == 'PacketProcessor':
               model.totalPacketProcessorDrops += dropEvent.dropCount
            model.dropEvents[ chipName ].dropEvent.append( dropEvent )
      return model

BasicCli.addShowCommandClass( HardwareCounterDropCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware counter drop log COUNTER_NAME [ THRESHOLD ]
#--------------------------------------------------------------------------------
class HardwareCounterDropLogCounternameCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware counter drop log COUNTER_NAME [ THRESHOLD ]'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'counter' : CliCommand.guardedKeyword( 'counter', helpdesc='Counter',
         guard=dropCounterGuard ),
      'drop' : 'Configure drop counter',
      'log' : 'Configure syslog threshold',
      'COUNTER_NAME' : CliMatcher.DynamicNameMatcher( getAllDropCounterNames,
         helpdesc='Counter Name' ),
      'THRESHOLD' : CliMatcher.IntegerMatcher( 0, 0xffffffff,
         helpdesc='Number of drop counters between two syslogs' ),
   }

   @staticmethod
   def handler( mode, args ):
      counterName = args[ 'COUNTER_NAME' ]
      logThreshold = args.get( 'THRESHOLD' )
      if dropCounterInfoHook and counterName not in getAllDropCounterNames( mode ):
         mode.addError( 'Counter %s does not exist' % counterName )
         return

      if logThreshold:
         internalDropCliConfig.syslogEventThreshold[ counterName ] = logThreshold
      if logThreshold is None:
         # default syslog threshold is 0
         print( 'Syslog threshold for %s = %d' % ( counterName,
                internalDropCliConfig.syslogEventThreshold.get( counterName, 0 ) ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      counterName = args[ 'COUNTER_NAME' ]
      if dropCounterInfoHook and counterName not in getAllDropCounterNames( mode ):
         mode.addError( 'Counter %s does not exist' % counterName )
         return

      if counterName in internalDropCliConfig.syslogEventThreshold:
         del internalDropCliConfig.syslogEventThreshold[ counterName ]

BasicCli.GlobalConfigMode.addCommandClass( HardwareCounterDropLogCounternameCmd )

#--------------------------------------------------------------------------------
# clear hardware counter drop
#--------------------------------------------------------------------------------
class ClearHardwareCounterDropCmd( CliCommand.CliCommandClass ):
   syntax = 'clear hardware counter drop'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'hardware' : CliToken.Hardware.hardwareForShowMatcher,
      'counter' : nodeShowCounter,
      'drop' : 'Show drop counter info',
   }

   @staticmethod
   def handler( mode, args ):
      internalDropSharedConfig.clearTime = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearHardwareCounterDropCmd )

#--------------------------------------------------------------------------------
# show hardware counter events
#--------------------------------------------------------------------------------
def getChipEventsDict( counterEvent, interval ):
   if interval == 300:
      return counterEvent.fiveMinuteEvents
   elif interval == 600:
      return counterEvent.tenMinuteEvents
   elif interval == 3600:
      return counterEvent.oneHourEvents
   elif interval == 24 * 3600:
      return counterEvent.oneDayEvents
   elif interval == 24 * 7 * 3600:
      return counterEvent.oneWeekEvents
   else:
      return None

class HardwareCounterEventsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware counter events'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'counter' : nodeShowCounter,
      'events' : 'Show flow trend events',
   }
   cliModel = FlowTrendEventModel

   @staticmethod
   def handler( mode, args ):
      assert flowTrendCounterInfoHook
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      model = FlowTrendEventModel()
      flowTrendStatus = shmemEm.doMount( FlowTrendConstants.smashPrefix,
                                         'Ale::Counters::FlowTrendMonitor::Status',
                                         Smash.mountInfo( 'reader' ) )
      for key, flowTrendEvent in flowTrendStatus.flowTrend.items():
         chipName = flowTrendCounterInfoHook.getChipName( mode, key )
         counterEventName = getFlowTrendCounterName( mode, key )
         interval = key.interval
         if model.counterEvents.get( key.counterId ) is None:
            model.counterEvents[ key.counterId ] = CounterEvent()
         counterEvent = model.counterEvents[ key.counterId ]
         counterEvent.counterEventName = counterEventName
         chipEventsDict = getChipEventsDict( counterEvent, interval )
         chipEventsDict[ chipName ] = ChipEvent(
               firstOccurrence=flowTrendEvent.firstOccurrence,
               lastOccurrence=flowTrendEvent.lastOccurrence,
               eventCount=flowTrendEvent.count, zScore=flowTrendEvent.zScore )
      return model

BasicCli.addShowCommandClass( HardwareCounterEventsCmd )

#--------------------------------------------------------------------------------
# clear hardware counter events
#--------------------------------------------------------------------------------
class ClearHardwareCounterEventsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear hardware counter events'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'hardware' : CliToken.Hardware.hardwareForShowMatcher,
      'counter' : nodeShowCounter,
      'events' : 'Show flow trend events',
   }

   @staticmethod
   def handler( mode, args ):
      flowTrendSharedConfig.clearTime = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearHardwareCounterEventsCmd )

#############################################
### show interfaces [<name>] counters ip  ###
#############################################

def ipCountersGuard( mode, token ):
   supportsIngressIpPktCounters = ipCountersCapability.supportsRxIpPktCounters
   supportsEgressIpPktCounters = ipCountersCapability.supportsTxIpPktCounters
   flexIngressIpCounters = \
      checkCounterFeatureSupported( FeatureIdEnum.Ipv4v6Ingress )
   flexEgressIpCounters = checkCounterFeatureSupported( FeatureIdEnum.Ipv4v6Egress )
   if supportsIngressIpPktCounters or supportsEgressIpPktCounters or \
      flexIngressIpCounters or flexEgressIpCounters:
      return None
   return CliParser.guardNotThisPlatform

ipCountersNode = CliCommand.guardedKeyword( 'ip', helpdesc='IPv4, IPv6 counters',
      guard=ipCountersGuard )

def getActiveInterfaces( intfs ):
   # BUG422729 Skip unconnected interfaces until they are properly supported
   # Only do this for interfaces with an available linkStatus.
   return [ intf for intf in intfs if not hasattr( intf, 'linkStatus' ) or (
      callable( getattr( intf, 'linkStatus', None ) ) and
      intf.linkStatus() not in [ 'notconnect', 'errdisabled', 'disabled' ] ) ]

def showInterfacesIpCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   ipCounters = InterfacesIpv4Ipv6Counters()
   supportsIngressIpPktCounters = ipCountersCapability.supportsRxIpPktCounters
   supportsEgressIpPktCounters = ipCountersCapability.supportsTxIpPktCounters
   usesAggregatedLagCounters = ipCountersCapability.usesAggregatedLagCounters
   flexIngressIpCounters = checkIngressIpCountersEnabled()
   flexEgressIpCounters = checkEgressIpCountersEnabled()
   # there is no egress counter support on sand but we still want to proceed and
   # show the ingress counters, so warn only on the ingress case. For strata,
   # packet counters is always available so no need to check
   if not ( flexIngressIpCounters or
            flexEgressIpCounters or
            supportsIngressIpPktCounters ):
      mode.addError( 'hardware counter feature ip in should be enabled first' )
      return ipCounters
   intfs = IntfCli.counterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return ipCounters
   # either hw counters or flex counters provide packet counters
   rxPkt = flexIngressIpCounters or supportsIngressIpPktCounters
   rxByte = flexIngressIpCounters
   txPkt = flexEgressIpCounters or supportsEgressIpPktCounters
   txByte = flexEgressIpCounters

   rxCounterTable = rxL3IpCounterTable
   rxSnapshotTable = rxL3IpSnapshotTable
   txCounterTable = txL3IpCounterTable
   txSnapshotTable = txL3IpSnapshotTable
   if checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Ingress ) or \
         supportsIngressIpPktCounters:
      rxCounterTable = rxIpCounterTable
      rxSnapshotTable = rxIpSnapshotTable
   if checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Egress ) or \
         supportsEgressIpPktCounters:
      txCounterTable = txIpCounterTable
      txSnapshotTable = txIpSnapshotTable

   def setIpCounters( intfs, rxCounterTbl, txCounterTbl,
                      rxSnapshotTbl=None, txSnapshotTbl=None ):
      for intf in intfs:
         intfName = intf.name
         v4index = IpFlexCountersIndex.v4IntfCounterKey( intfName )
         v6index = IpFlexCountersIndex.v6IntfCounterKey( intfName )
         intfCounters = ipCounters.InterfaceIpv4Ipv6Counters()
         # ingress counters
         if rxPkt:
            intfCounters.ipv4InPkts = \
               getCurrentCounterPkts( v4index, rxCounterTbl, rxSnapshotTbl )
            intfCounters.ipv6InPkts = \
               getCurrentCounterPkts( v6index, rxCounterTbl, rxSnapshotTbl )
         if rxByte:
            intfCounters.ipv4InOctets = \
               getCurrentCounterOctets( v4index, rxCounterTbl, rxSnapshotTbl )
            intfCounters.ipv6InOctets = \
               getCurrentCounterOctets( v6index, rxCounterTbl, rxSnapshotTbl )
         # egress counters
         if txPkt:
            intfCounters.ipv4OutPkts = \
               getCurrentCounterPkts( v4index, txCounterTbl, txSnapshotTbl )
            intfCounters.ipv6OutPkts = \
               getCurrentCounterPkts( v6index, txCounterTbl, txSnapshotTbl )
         if txByte:
            intfCounters.ipv4OutOctets = \
               getCurrentCounterOctets( v4index, txCounterTbl, txSnapshotTbl )
            intfCounters.ipv6OutOctets = \
               getCurrentCounterOctets( v6index, txCounterTbl, txSnapshotTbl )
         ipCounters.interfaces[ intfName ] = intfCounters

   filteredIntfs = [ intf for intf in intfs
                     if EthIntfId.isEthernet( intf.name ) or
                     EthIntfId.isSwitch( intf.name ) or
                     ( PortChannelIntfId.isPortChannelIntfId( intf.name ) and
                       not usesAggregatedLagCounters ) ]

   activeIntfs = getActiveInterfaces( filteredIntfs )
   setIpCounters( activeIntfs, rxCounterTable, txCounterTable,
                  rxSnapshotTable, txSnapshotTable )

   if usesAggregatedLagCounters:
      lagIntfs = [ intf for intf in intfs
                   if PortChannelIntfId.isPortChannelIntfId( intf.name ) ]
      activeLagIntfs = getActiveInterfaces( lagIntfs )
      setIpCounters( activeLagIntfs, rxIpLagCounterTable, txIpLagCounterTable )
   return ipCounters

class ShowIntfCountersIp( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters ip'
   data = dict( counters=IntfCli.countersKw,
                ip=ipCountersNode )
   cliModel = InterfacesIpv4Ipv6Counters
   handler = showInterfacesIpCounters

BasicCli.addShowCommandClass( ShowIntfCountersIp )

##################################################
### show interfaces [<name>] counters ip rates ###
##################################################
SmashFlexCounter = namedtuple( 'SmashFlexCounter', 'pkt byte timestamp' )

def getRate( counterTable, index ):
   rateCounter = counterTable.rateCounter.get( index )
   if rateCounter:
      return ( int( rateCounter.pktsRate ),
               int( rateCounter.bitsRate / 8 ),
               int( rateCounter.bitsRate ),
               rateCounter.updateTime )
   return ( 0, 0, 0, 0.0 )

def showIpv4Ipv6CountersRates( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   ipCounterRates = InterfaceIpv4Ipv6CountersRates()
   supportsIngressIpPktCounters = ipCountersCapability.supportsRxIpPktCounters
   supportsEgressIpPktCounters = ipCountersCapability.supportsTxIpPktCounters
   flexIngressIpCounters = checkIngressIpCountersEnabled()
   flexEgressIpCounters = checkEgressIpCountersEnabled()
   if not ( flexIngressIpCounters or
            flexEgressIpCounters or
            supportsIngressIpPktCounters ):
      mode.addError( 'hardware counter feature ip in should be enabled first' )
      return ipCounterRates
   intfs = IntfCli.counterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return ipCounterRates
   # either hw counters or flex counters provide packet counters
   rxPkt = flexIngressIpCounters or supportsIngressIpPktCounters
   rxByte = flexIngressIpCounters
   txPkt = flexEgressIpCounters or supportsEgressIpPktCounters
   txByte = flexEgressIpCounters

   rxCounterTable = rxL3IpCounterRateTable
   txCounterTable = txL3IpCounterRateTable
   if checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Ingress ) or \
         supportsIngressIpPktCounters:
      rxCounterTable = rxIpCounterRateTable
   if checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Egress ) or \
         supportsEgressIpPktCounters:
      txCounterTable = txIpCounterRateTable

   filteredIntfs = [ intf for intf in intfs
                     if EthIntfId.isEthernet( intf.name ) or
                     EthIntfId.isSwitch( intf.name ) ]
   activeIntfs = getActiveInterfaces( filteredIntfs )

   for intf in activeIntfs:
      intfName = intf.name
      intfRates = ipCounterRates.InterfaceIpv4Ipv6CounterRates()
      intfLoadInterval = allIntfConfigDir.intfConfig[ intfName ].loadInterval
      intfRates.interval = IntfCli.getActualLoadIntervalValue( intfLoadInterval )

      v4Index = IpFlexCountersIndex.v4IntfCounterKey( intfName )
      v6Index = IpFlexCountersIndex.v6IntfCounterKey( intfName )

      # ingress rates
      ( v4PktsRate, v4BytesRate, v4BitsRate, v4Time ) = getRate(
         rxCounterTable, v4Index )
      ( v6PktsRate, v6BytesRate, v6BitsRate, v6Time ) = getRate(
         rxCounterTable, v6Index )
      if rxPkt:
         intfRates.ipv4InPktsRate = v4PktsRate
         intfRates.ipv6InPktsRate = v6PktsRate
      if rxByte:
         intfRates.ipv4InOctetsRate = v4BytesRate
         intfRates.ipv4InBitsRate = v4BitsRate
         intfRates.ipv6InOctetsRate = v6BytesRate
         intfRates.ipv6InBitsRate = v6BitsRate
      intfRates.lastUpdateTimestamp = Ark.switchTimeToUtc( max( v4Time, v6Time ) )
      # egress rates
      ( v4PktsRate, v4BytesRate, v4BitsRate, v4Time ) = getRate(
         txCounterTable, v4Index )
      ( v6PktsRate, v6BytesRate, v6BitsRate, v6Time ) = getRate(
         txCounterTable, v6Index )
      if txPkt:
         intfRates.ipv4OutPktsRate = v4PktsRate
         intfRates.ipv6OutPktsRate = v6PktsRate
      if txByte:
         intfRates.ipv4OutOctetsRate = v4BytesRate
         intfRates.ipv4OutBitsRate = v4BitsRate
         intfRates.ipv6OutOctetsRate = v6BytesRate
         intfRates.ipv6OutBitsRate = v6BitsRate
      intfRates.lastUpdateTimestamp = Ark.switchTimeToUtc(
         max( intfRates.lastUpdateTimestamp, v4Time, v6Time ) )
      ipCounterRates.interfaces[ intfName ] = intfRates
   return ipCounterRates

class ShowIntfCountersIpRates( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters ip rates'
   data = dict( counters=IntfCli.countersKw,
                ip=ipCountersNode,
                rates=ipCountersRatesKw )
   cliModel = InterfaceIpv4Ipv6CountersRates
   handler = showIpv4Ipv6CountersRates

BasicCli.addShowCommandClass( ShowIntfCountersIpRates )

#############################################
### show multicast fib ipv4 counters rate ###
#############################################

# Hook for platform-specific implementations of the
# 'show multicast fib ipv4 counters rate' and
# 'show multicast fib ipv6 counters rate' commands.
def showMfibCounterRateHook( showMfibRoutes, af ):
   LazyMount.force( routingHardwareRouteStatus )

   if af == AddressFamily.ipv4:
      mfibCounterRate = Tac.newInstance(
            "Ale::Counters::ShowMfib::AleShowMfibCounterRate",
            routingHardwareRouteStatus, ipv4CounterKeyMap,
            ipv4McastRateCounterTable )
      showMfibRoutes.showMfibCounterRate = mfibCounterRate
   elif af == AddressFamily.ipv6:
      mfibCounterRate = Tac.newInstance(
            "Ale::Counters::ShowMfib::AleShowMfibCounterRate",
            routingHardwareRouteStatus, ipv6CounterKeyMap,
            ipv6McastRateCounterTable )
      showMfibRoutes.showMfibCounterRate = mfibCounterRate
   else:
      assert False, 'Unknown af type %s' % af

#############################################
### show interfaces [<name>] queue length ###
#############################################

# Hook for platform-specific implementations of the
# 'show interfaces [<name>] queue length' command and guard.
# Each extension must be a tuple consisting of (func, guard), where func is the
# function that is called for the command, and guard is the platform-specific
# CLI guard function that guards the command.
showInterfacesQueueLenHook = CliExtensions.CliHook()

def queueLenGuard( mode, token ):
   guardCodes = []
   for ( _, guard ) in showInterfacesQueueLenHook.extensions():
      guardCode = guard( mode, token )
      if guardCode is None:
         return None
      else:
         guardCodes.append( guardCode )
   if guardCodes:
      # Return the guard of the first guard that complained.
      return guardCodes[ 0 ]
   else:
      return CliParser.guardNotThisPlatform

queueKw = CliCommand.guardedKeyword( 'queue',
                                     'Show queue information',
                                     queueLenGuard )

def showInterfacesQueueLength( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = IntfCli.counterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return InterfaceQueueLengthInfo()
   # Filter out non-'Ethernet' interfaces.
   intfs = [ intf.name for intf in intfs
             if EthIntfId.isEthernet( intf.name ) or
             EthIntfId.isSwitch( intf.name ) ]
   # No explicit interface name(s) were passed in if intf is None.
   defaultIntf = ( intf is None )

   numUnguardedHooks = 0
   for ( func, guard ) in showInterfacesQueueLenHook.extensions():
      if guard( mode, None ) is None:
         # func() should take 3 arguments:
         #   * mode is the CliParser Mode of the command.
         #   * intfs is the list of interfaces that satisfy the interfaces
         #     range command.
         #   * allIntfs is a boolean indicating that the command was called with no
         #     explicit interface names, which means that we want information for
         #     all interfaces. This is useful if we want to do additional special
         #     printing when no explicit interfaces were requested by the command.
         intfQueueLengthInfo = func( mode, intfs, allIntfs=defaultIntf )
         numUnguardedHooks += 1
   assert numUnguardedHooks <= 1, 'Found too many possible results'
   return intfQueueLengthInfo

class ShowIntfQueueLength( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces queue length'
   data = dict( queue=queueKw,
                length='Show queue length information' )
   cliModel = InterfaceQueueLengthInfo
   handler = showInterfacesQueueLength

BasicCli.addShowCommandClass( ShowIntfQueueLength )

#--------------------------------------------------------------------------------
# show hardware counter feature
#--------------------------------------------------------------------------------
# Hook for the show command.
counterFeatureShowHook = CliExtensions.CliHook()

# Hook for counter token guard.
counterFeatureShowSupportedHook = CliExtensions.CliHook()

def counterFeatureShowGuard( mode, token ):
   guardCodes = []
   for func in counterFeatureShowSupportedHook.extensions():
      guardCode = func( mode, token )
      if guardCode is None:
         return None
      else:
         guardCodes.append( guardCode )
   if guardCodes:
      # Return the guard of the first guard that complained.
      return guardCodes[ 0 ]
   else:
      return CliParser.guardNotThisPlatform

class HardwareCounterFeatureCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware counter feature'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'counter' : CliCommand.guardedKeyword( 'counter', helpdesc='Show counter info',
         guard=counterFeatureShowGuard ),
      'feature' : 'Counter feature',
   }
   privileged = True
   cliModel = CounterFeatureModel

   @staticmethod
   def handler( mode, args ):
      model = CounterFeatureModel()
      assert len( counterFeatureShowHook.extensions() ) == 1
      for func in counterFeatureShowHook.extensions():
         func( model )
      return model

BasicCli.addShowCommandClass( HardwareCounterFeatureCmd )

#################################################################################
# [ no ] hardware counter feature <counterFeature> [ in | out ] [ ipv4 | ipv6 ]
#################################################################################

# Hook for platform-specific counter features.
# Each extension will update counter feature status dir in featureInfo.statusDir,
# potentially over-writing or appending to what a previous extension may have written
counterFeatureSettingHook = CliExtensions.CliHook()

def getFeatureStatusDir():
   featureStatusDir = Tac.newInstance( 'Tac::Dir', 'featureStatusDir' )
   counterFeatureSettingHook.notifyExtensions( featureStatusDir=featureStatusDir )
   t8( 'features in featureStatusDir:', list( featureStatusDir.entityPtr ) )
   # make sure we got only one plugin registered: Strata or Sand
   # Need to add an asssert that features not written more than once. BUG191102
   return featureStatusDir if featureStatusDir.entityPtr else fcFeatureStatusDir

def enableExtraConfigForCounterFeature( featureId, state ):
   if featureId == 'Mcast':
      ipv4McastCounterConfig.allCounter = bool( state )
   elif featureId == 'MulticastIpv6':
      ipv6McastCounterConfig.allCounter = bool( state )

def featureAttributeIs( featureId, units ):
   featureAttr = Tac.Value( 'FlexCounters::FeatureAttributeState' )
   featureAttr.countMode = unitsTokenToCountModeMap[ units ]
   fcFeatureConfigDir.featureAttribute[ featureId ] = featureAttr

def featureAttributeDel( featureId ):
   del fcFeatureConfigDir.featureAttribute[ featureId ]

def changeFeatureIdConfig( mode, featureId, state=None, units=None ):
   featureStatusDir = getFeatureStatusDir()
   status = featureStatusDir.entityPtr.get( featureId )
   t8( 'feature:', featureId, 'status:', status )
   if status is None:
      return

   if status.defaultState: # Feature is enabled by default.
      if state is None or state: # 'default' or no prefix (enable) command.
         featureAttributeIs( featureId, None if state is None else units )
         fcFeatureConfigDir.feature[ featureId ] = 'platformDefault'
      else: # 'no' command.
         del fcFeatureConfigDir.feature[ featureId ]
         featureAttributeDel( featureId )
   else: # Feature is disabled by default.
      if state: # no prefix (enable) command.
         featureAttributeIs( featureId, units )
         fcFeatureConfigDir.feature[ featureId ] = 'enabled'
      else: # 'default' or 'no' command.
         del fcFeatureConfigDir.feature[ featureId ]
         featureAttributeDel( featureId )

   enableExtraConfigForCounterFeature( featureId, state )
   t8( 'features enabled:', list( fcFeatureConfigDir.feature.items() ) )

def manageRouteFlexCounterConfig( mode, feature, routeType,
                              vrfName=None, prefix=None, state=None ):
   if prefix is None:
      # Early exit here if prefix is not given
      # Make it configrable for all routes later
      return

   if not vrfName:
      vrfName = DEFAULT_VRF

   featureId = featureMap[ ( feature, None, None ) ]

   vrfNameObj = Tac.Value( 'L3::VrfName', vrfName )

   if routeType == 'ipv4':
      if prefix.find( '/' ) == -1:
         # Only the part A.B.C.D has been added interpreted as Host Route
         # ie. A.B.C.D/32
         prefix += '/32'

   elif routeType == 'ipv6':
      # We receive as prefix object (Arnet::Ip6Prefix)
      # and we need to get the string using stringValue attribute
      # The string we get is of format 'abcd:efgh::/16'
      prefix = prefix.stringValue

      if prefix.find( '/' ) == -1:
         # Only the part A:B:C:D:E:F:G:H has been added, interpreted as Host Route
         # ie. A:B:C:D:E:F:G:H/128
         prefix += '/128'

   prefixObj = Tac.Value( 'Arnet::IpGenPrefix', prefix )
   vrfNameAndIpGenPrefixObj = Tac.Value( 'Ale::FlexCounter::VrfNameAndIpGenPrefix',
                                         vrfNameObj, prefixObj )

   if state:
      if not fcRouteConfig.route:
         fcFeatureConfigDir.feature[ featureId ] = 'enabled'
      fcRouteConfig.route[ vrfNameAndIpGenPrefixObj ] = True
   else:
      if vrfNameAndIpGenPrefixObj in fcRouteConfig.route:
         del fcRouteConfig.route[ vrfNameAndIpGenPrefixObj ]
         if not fcRouteConfig.route:
            del fcFeatureConfigDir.feature[ featureId ]
      else:
         mode.addWarning( f'No such Prefix {prefix} in the VRF {vrfName}' )

def doClearL3RouteCounters( mode, args ):
   # if vrfName and prefix both are NOT mentioned, then we clear counters for all
   # prefixes of all VRFs for the given routeType
   # if only prefix is mentioned, we assume DEFAULT_VRF
   # if only vrfName is mentioned, we clear for every prefix under that VRF
   routeType = 'ipv4' if 'ipv4' in args else 'ipv6'
   vrfName = args.get( 'VRF' )
   prefix = args.get( 'PREFIX' )

   if vrfName is None:
      vrfName = 'all' if prefix is None else DEFAULT_VRF

   if prefix is None:
      # we'll set special prefix to indicate that it's a special case.
      if routeType == 'ipv4':
         zeroAddr = Tac.Value( 'Arnet::IpAddr', 0 )
         zeroPrefix = Tac.Value( 'Arnet::Prefix', zeroAddr, 32 )
      else:
         zeroAddr = Tac.Value( 'Arnet::Ip6Addr', 0, 0, 0, 0 )
         zeroPrefix = Tac.Value( 'Arnet::Ip6Prefix', zeroAddr, 128 )

      prefixObj = Tac.Value( 'Arnet::IpGenPrefix', str( zeroPrefix ) )
   else:
      prefix = str( prefix )

      if prefix.find( '/' ) == -1:
         prefix += '/32' if routeType == 'ipv4' else '/128'
      prefixObj = Tac.Value( 'Arnet::IpGenPrefix', prefix )

   vrfNameObj = Tac.Value( 'L3::VrfName', vrfName )
   vrfNameAndIpGenPrefixObj = Tac.Value( 'Ale::FlexCounter::VrfNameAndIpGenPrefix',
                                         vrfNameObj, prefixObj )

   if vrfNameAndIpGenPrefixObj in l3RouteClearRequest.route:
      del l3RouteClearRequest.route[ vrfNameAndIpGenPrefixObj ]

   l3RouteClearRequest.route[ vrfNameAndIpGenPrefixObj ] = True

def checkCounterFeatureConfigured( featureId ):
   featureConfig = fcFeatureConfigDir.feature.get( featureId )
   status = False
   if featureConfig == 'enabled':
      status = True
   elif featureConfig == 'platformDefault':
      featureStatus = fcFeatureStatusDir.entityPtr.get( featureId )
      if featureStatus:
         status = featureStatus.defaultState
   return status

def checkCounterFeatureSupported( featureId ):
   featureStatusDir = getFeatureStatusDir()
   return bool( featureStatusDir.entityPtr.get( featureId ) )

def checkCounterFeatureEnabled( featureId ):
   # Strata platform look for enabled features in Sysdb@featureStatusDir
   status = fcFeatureStatusDir.entityPtr.get( featureId )
   return status.state if status else False

def checkIngressIpCountersEnabled():
   return checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Ingress ) or \
          checkCounterFeatureEnabled( FeatureIdEnum.L3Ipv4v6Ingress )

def checkEgressIpCountersEnabled():
   return checkCounterFeatureEnabled( FeatureIdEnum.Ipv4v6Egress ) or \
          checkCounterFeatureEnabled( FeatureIdEnum.L3Ipv4v6Egress )

# Hook for platform supported counter features
# Each extension must return list of supported features or empty list
counterFeatureSupportedHook = CliExtensions.CliHook()

# If counterFeatureSupportedHook has any extensions, guard if none of the requested
# features are supported by any of those extensions. Otherwise, guard if none of the
# requested features are in featureStatusDir.
def counterFeaturesSupported( featureIds ):
   def counterFeaturesSupportedHelper( mode, token ):
      if counterFeatureSupportedHook.extensions():
         supportedFeatures = []
         for extension in counterFeatureSupportedHook.extensions():
            supportedFeatures += extension( mode, token )
         if not supportedFeatures:
            return CliParser.guardNotThisPlatform
         if not any( featureId in supportedFeatures for featureId in featureIds ):
            return CliParser.guardNotThisPlatform

      else:
         featureStatusDir = getFeatureStatusDir()
         if not any( featureStatusDir.entityPtr.get( featureId )
                     for featureId in featureIds ):
            return CliParser.guardNotThisPlatform

      return None

   return counterFeaturesSupportedHelper

def counterFeatureSupported( featureId ):
   return counterFeaturesSupported( [ featureId ] )

# Hook for platform supported units for a counter feature.
# The hook extension must be a callable such that:
# - it accepts the following arguments: `mode`, `token`, `featureId`;
# - it returns the list of units types supported by the given `featureId`.
counterFeatureUnitsSupportedHook = CliExtensions.CliHook()

def counterFeatureUnitsSupported( featureId ):
   def counterFeatureUnitsSupportedHelper( mode, token ):
      supportedUnits = set()
      for extension in counterFeatureUnitsSupportedHook.extensions():
         supportedUnits |= set( extension( mode, token, featureId ) )

      # Platform default represents the absence of token, so shouldn't be considered
      # a valid units type in the context of this guard.
      supportedUnits.discard( 'featureCountModePlatformDefault' )

      if supportedUnits:
         return None
      return CliParser.guardNotThisPlatform
   return counterFeatureUnitsSupportedHelper

#--------------------------------------------------------------------------------
# [ no | default ] hardware counter feature route ipv4 [ VRF ] PREFIX
# The cmd 'hardware counter feature route ipv4/6' is not supported yet
# BUG452692 is tracking this. Will make PREFIX optional
#--------------------------------------------------------------------------------
routeFeatureToken = 'route'
matcherCounterFeature = CliCommand.guardedKeyword( routeFeatureToken,
      helpdesc=tokenHelpdesc( routeFeatureToken ),
      guard=counterFeatureSupported( 'Route' ) )

matcherCounterFeatures = CliCommand.guardedKeyword( routeFeatureToken,
      helpdesc=tokenHelpdesc( routeFeatureToken ),
      guard=counterFeaturesSupported( [ 'Route', 'RouteIpv4' ] ) )

class HardwareCounterFeatureRouteIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'hardware counter feature route ipv4 [ VRF ] PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'counter' : matcherCounter,
      'feature' : matcherFeature,
      'route' : matcherCounterFeature,
      'ipv4' : matcherIpv4,
      'VRF' : exprRouteVrfWithName,
      'PREFIX' : exprIpv4AddrOrPrefix,
   }

   @staticmethod
   def handler( mode, args ):
      return manageRouteFlexCounterConfig( mode, feature='route', routeType='ipv4',
            vrfName=args.get( 'VRF' ), prefix=args[ 'PREFIX' ], state=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return manageRouteFlexCounterConfig( mode, feature='route', routeType='ipv4',
            vrfName=args.get( 'VRF' ), prefix=args[ 'PREFIX' ], state=False )

BasicCli.GlobalConfigMode.addCommandClass( HardwareCounterFeatureRouteIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware counter feature route ipv6 [ VRF ] PREFIX
#--------------------------------------------------------------------------------
class HardwareCounterFeatureRouteIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'hardware counter feature route ipv6 [ VRF ] PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'counter' : matcherCounter,
      'feature' : matcherFeature,
      'route' : matcherCounterFeature,
      'ipv6' : matcherIpv6,
      'VRF' : exprRouteVrfWithName,
      'PREFIX' : exprRute6Prefix
   }

   @staticmethod
   def handler( mode, args ):
      return manageRouteFlexCounterConfig( mode, feature='route', routeType='ipv6',
            vrfName=args.get( 'VRF' ), prefix=args[ 'PREFIX' ], state=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return manageRouteFlexCounterConfig( mode, feature='route', routeType='ipv6',
            vrfName=args.get( 'VRF' ), prefix=args[ 'PREFIX' ], state=False )

BasicCli.GlobalConfigMode.addCommandClass( HardwareCounterFeatureRouteIpv6Cmd )

#--------------------------------------------------------------------------------
# clear counters route ipv4 [ VRF ] [ PREFIX ]
#--------------------------------------------------------------------------------
class ClearCountersRouteIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'clear counters route ipv4 [ VRF ] [ PREFIX ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'counters' : matcherCounters,
      'route' : matcherCounterFeatures,
      'ipv4' : matcherIpv4,
      'VRF' : exprRouteVrfWithName,
      'PREFIX' : exprIpv4AddrOrPrefix,
   }

   handler = doClearL3RouteCounters

BasicCli.EnableMode.addCommandClass( ClearCountersRouteIpv4Cmd )

#--------------------------------------------------------------------------------
# clear counters route ipv6 [ VRF ] [ PREFIX ]
#--------------------------------------------------------------------------------
class ClearCountersRouteIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'clear counters route ipv6 [ VRF ] [ PREFIX ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'counters' : matcherCounters,
      'route' : matcherCounterFeature,
      'ipv6' : matcherIpv6,
      'VRF' : exprRouteVrfWithName,
      'PREFIX' : exprRute6Prefix,
   }

   handler = doClearL3RouteCounters

BasicCli.EnableMode.addCommandClass( ClearCountersRouteIpv6Cmd )

def _showTechGuard():
   return dropCounterInfoHook and dropCounterInfoHook.guard( None, None ) is None

def _showTechSummaryGuard():
   return not counterFeatureShowGuard( None, None )

TechSupportCli.registerShowTechSupportCmd(
   '2015-01-05 13:59:49',
   cmds=[ 'show hardware counter drop', 'show hardware counter drop rates' ],
   cmdsGuard=_showTechGuard,
   summaryCmds=[ 'show hardware counter feature' ],
   summaryCmdsGuard=_showTechSummaryGuard )

def getCurrentCounter( idx, counterTable, snapshotTable ):
   # read the counters from counterTable and deduct the value
   # found in snapshotTable. snapshotTable contains the
   # snapshot of counter values from counterTable when
   # clear command is issued
   runningCtr = counterTable.counter.get( idx )
   snapshotCtr = snapshotTable.counter.get( idx )
   if runningCtr is None:
      # If the counter is missing, will return zeros. This is usually a transient
      # condition when we are caught in the middle of cleanup.
      currentCtrPkts = 0
      currentCtrOctets = 0
   elif snapshotCtr is None:
      # If the snapshot is not present, will return the running counter only.
      currentCtrPkts = runningCtr.pkts
      currentCtrOctets = runningCtr.octets
   else:
      # The current is the running counter minus the snapshot.
      currentCtrPkts = runningCtr.pkts - snapshotCtr.pkts
      currentCtrOctets = runningCtr.octets - snapshotCtr.octets
   return currentCtrPkts, currentCtrOctets

def getCurrentCounterPkts( idx, counterTable, snapshotTable=None ):
   # read the counters from counterTable and deduct the value
   # found in snapshotTable. snapshotTable contains the
   # snapshot of counter values from counterTable when
   # clear command is issued
   runningCtr = counterTable.counter.get( idx )
   snapshotCtr = snapshotTable.counter.get( idx ) if snapshotTable else None
   if runningCtr is None:
      # If the counter is missing, will return zeros. This is usually a transient
      # condition when we are caught in the middle of cleanup.
      currentCtrPkts = 0
   elif snapshotCtr is None:
      # If the snapshot is not present, will return the running counter only.
      currentCtrPkts = runningCtr.pkts
   else:
      # The current is the running counter minus the snapshot.
      currentCtrPkts = runningCtr.pkts - snapshotCtr.pkts
   return currentCtrPkts

def getCurrentCounterOctets( idx, counterTable, snapshotTable=None ):
   runningCtr = counterTable.counter.get( idx )
   snapshotCtr = snapshotTable.counter.get( idx ) if snapshotTable else None
   if runningCtr is None:
      currentCtrOctets = 0
   elif snapshotCtr is None:
      currentCtrOctets = runningCtr.octets
   else:
      currentCtrOctets = runningCtr.octets - snapshotCtr.octets
   return currentCtrOctets

def getSmashFlexCounter( idx, counterTable, snapshotTable ):
   runningCtr = counterTable.counter.get( idx )
   snapshotCtr = snapshotTable.counter.get( idx )
   if runningCtr is None:
      octets = 0
      pkts = 0
   elif snapshotCtr is None:
      octets = runningCtr.octets
      pkts = runningCtr.pkts
   else:
      octets = runningCtr.octets - snapshotCtr.octets
      pkts = runningCtr.pkts - snapshotCtr.pkts
   updateTime = runningCtr.statsUpdateTime if runningCtr else 0.0
   return SmashFlexCounter( octets, pkts, updateTime )

class RouteCountersVrfMode( RouteCountersVrfModeBase, BasicCli.ConfigModeBase ):
   name = 'Prefix Counters VRF configuration'

   def __init__( self, parent, session, vrfName ):
      invalidVrfNames = [ VrfCli.DEFAULT_VRF_OLD, VrfCli.ALL_VRF_NAME ]
      assert vrfName not in invalidVrfNames
      self.vrfName = vrfName
      RouteCountersVrfModeBase.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouteCountersMode( RouteCountersModeBase, BasicCli.ConfigModeBase ):
   name = 'route-counters'

   def __init__( self, parent, session ):
      RouteCountersModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CfgRouteCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'route counters'
   noOrDefaultSyntax = syntax
   data = {
      'route' : matcherRoute,
      'counters' : CliCommand.guardedKeyword( 'counters',
         helpdesc='Route counters',
         guard=counterFeatureSupported( 'RouteIpv4' ) )
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RouteCountersMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args=None ): # Hook (below) only passes in `mode`.
      prefixCounterCliConfig.vrfPrefixLens.clear()
      prefixCounterCliConfig.primaryBackupDedicated = False

RouterGeneralMode.addCommandClass( CfgRouteCountersCmd )
routerGeneralCleanupHook.addExtension( CfgRouteCountersCmd.noOrDefaultHandler )

#-------------------------------------------------------------------------------#
# arista(config-router-general-route-counters)# vrf VRF
#-------------------------------------------------------------------------------#
class RouteCountersVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = syntax
   data = {
      'vrf' : 'Configure VRF',
      'VRF' : VrfCli.VrfNameExprFactory( inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      invalidVrfNames = [ VrfCli.DEFAULT_VRF_OLD, VrfCli.ALL_VRF_NAME ]
      if vrfName in invalidVrfNames:
         mode.addErrorAndStop( "VRF name %s is reserved." % vrfName )
      childMode = mode.childMode( RouteCountersVrfMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      if vrfName not in prefixCounterCliConfig.vrfPrefixLens:
         prefixCounterCliConfig.vrfPrefixLens.newMember( vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      del prefixCounterCliConfig.vrfPrefixLens[ vrfName ]

RouteCountersMode.addCommandClass( RouteCountersVrfCmd )

class VrfIpV4PrefixLengthCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4 prefix-length PREFIXLENGTH'
   noOrDefaultSyntax = syntax

   data = {
      'ipv4' : matcherIpv4,
      'prefix-length' : 'Length of the prefixes',
      'PREFIXLENGTH' : CliMatcher.IntegerMatcher(
         32, 32,
         helpdesc='Specify prefix length' )
   }

   @staticmethod
   def handler( mode, args ):
      prefixLength = args[ 'PREFIXLENGTH' ]
      prefixCounterCliConfig.vrfPrefixLens[ mode.vrfName ].\
            prefixLen[ prefixLength ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      prefixLength = args[ 'PREFIXLENGTH' ]
      del prefixCounterCliConfig.vrfPrefixLens[ mode.vrfName ].\
            prefixLen[ prefixLength ]

RouteCountersVrfMode.addCommandClass( VrfIpV4PrefixLengthCmd )

#----------------------------------------------------------------------------------#
# arista(config-router-general-route-counters)#[no|default] primary backup dedicated
#----------------------------------------------------------------------------------#
class PrimaryBackupDedicatedCmd( CliCommand.CliCommandClass ):
   syntax = 'primary backup dedicated'
   noOrDefaultSyntax = syntax

   data = {
      'primary' : CliCommand.guardedKeyword( 'primary',
         helpdesc='Primary counters',
         guard=guardPrimaryBackupDedicated ),
      'backup' : 'Backup counters',
      'dedicated' : 'Disambiguate the primary and backup counters',
   }

   @staticmethod
   def handler( mode, args ):
      prefixCounterCliConfig.primaryBackupDedicated = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      prefixCounterCliConfig.primaryBackupDedicated = False

RouteCountersMode.addCommandClass( PrimaryBackupDedicatedCmd )

#--------------------------------------------------------------------------------
# Config mode for tunnel counters.
# arista(config)# tunnel-counters
#--------------------------------------------------------------------------------
def clearTunnelCounters( mode, args ):
   clearTunnelCountersTransmitConfig( mode, args )
   setDecayPeriod( mode, args )

class TunnelCountersConfigMode( TunnelCountersMode, BasicCli.ConfigModeBase ):
   name = 'tunnel-counters'

   def __init__( self, parent, session ):
      TunnelCountersMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TunnelCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-counters'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-counters' : 'Tunnel counters configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TunnelCountersConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = clearTunnelCounters

# --------------------------------------------------------------------------------
# Granular per tunnel type control and priority mode under the base config mode
# and transmit sub-mode.
# arista(config-tunnel-counters)#transmit
# arista(config-tunnel-counters-transmit)#
# --------------------------------------------------------------------------------
def clearTunnelCountersTransmitConfig( mode=None, args=None ):
   tunnelCountersPriorityTable.tunnelTypePriority.clear()
   tunnelCountersPriorityTable.allTunnelCountersEnabled = True

class TunnelCountersTransmitConfigMode( TunnelCountersTransmitMode,
                                        BasicCli.ConfigModeBase ):
   name = 'tunnel-counters-transmit'

   def __init__( self, parent, session ):
      TunnelCountersTransmitMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TunnelCountersTransmitCmd( CliCommand.CliCommandClass ):
   syntax = 'transmit'
   noOrDefaultSyntax = syntax
   data = {
      'transmit' : 'Transmit tunnel counters configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TunnelCountersTransmitConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = clearTunnelCountersTransmitConfig

#--------------------------------------------------------------------------------
# Under tunnel-counters mode, this command indicates whether to selectively count
# only tunnels with a specified tunnel type.
#
# !
# tunnel-counters
#    transmit
#       selective
#       source-protocol ( bgp labeled-unicast [ forwarding ] ) |
#                       ( isis segment - routing ) |
#                       ldp | rsvp [ ler [ sub ] ] |
#                       ( static mpls ) | ti-lfa |
#                       ( traffic-engineering segment-routing policy )
#                         [ priority PRIORITY ]
# !
#--------------------------------------------------------------------------------
class TunnelTypeSelectiveCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'selective'
   noOrDefaultSyntax = syntax
   data = {
      'selective' : 'Only count specified tunnel sources',
   }

   @staticmethod
   def handler( mode, args ):
      tunnelCountersPriorityTable.allTunnelCountersEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tunnelCountersPriorityTable.allTunnelCountersEnabled = True

class TunnelTypeCountersCmd( CliCommand.CliCommandClass ):
   syntax = """source-protocol ( ( rsvp ( ler [ sub ] ) | frr )
                            | ip-in-ip
                            | ldp
                            | ti-lfa
                            | ( bgp labeled-unicast [ forwarding ] )
                            | ( static mpls )
                            | ( isis segment-routing )
                            | ( traffic-engineering segment-routing policy ) )
                              [ priority PRIORITY ]"""
   data = {
      'source-protocol' : 'Tunnel source',
      'rsvp' : 'Resource Reservation Protocol',
      'frr' : 'Fast Reroute',
      'ler' : 'Label Edge Router',
      'sub' : 'Sub tunnel',
      'ip-in-ip' : CliToken.TunnelCli.ipinipKw,
      'ldp' : CliToken.TunnelCli.ldpKw,
      'ti-lfa' : CliToken.TunnelCli.tilfaKw,
      'bgp' : CliToken.TunnelCli.bgpKw,
      'labeled-unicast' : CliToken.TunnelCli.labeledUnicastKw,
      'forwarding' : CliToken.TunnelCli.forwardingKw,
      'static' : CliToken.TunnelCli.staticKw,
      'mpls' : CliToken.TunnelCli.mplsKw,
      'isis' : CliToken.TunnelCli.isisKw,
      'segment-routing' : CliToken.TunnelCli.segmentRoutingKw,
      'traffic-engineering' : CliToken.TunnelCli.trafficEngineeringKw,
      'policy' : CliToken.TunnelCli.policyKw,
      'priority' : 'Tunnel type priority',
      'PRIORITY' : CliMatcher.IntegerMatcher(
      0, 1000, helpdesc="Priority value (default 100)" )
   }

   noOrDefaultSyntax = syntax

   @staticmethod
   def adapter( mode, args, argsList ):
      tunnelStr = ' '.join( token for token, _ in argsList )
      tunnelStr = tunnelStr.split( ' priority' )[ 0 ]
      tunnelType = tunnelStrToType[ tunnelStr.split( 'source-protocol ' )[ 1 ] ]
      args[ 'TUNNEL_TYPE' ] = tunnelType

   @staticmethod
   def handler( mode, args ):
      tunnelType = args[ 'TUNNEL_TYPE' ]
      # If no priority is supplied, use default priority.
      priority = args.get( 'PRIORITY', TunnelCountersPriority() )
      tunnelCountersPriorityTable.tunnelTypePriority[ tunnelType ] = priority

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tunnelType = args[ 'TUNNEL_TYPE' ]
      del tunnelCountersPriorityTable.tunnelTypePriority[ tunnelType ]

# --------------------------------------------------------------------------------
# Under tunnel-counters mode, this command sets the load-interval for tunnel rate
# counter calculations.
#
# arista(config-tunnel-counters)# rate period decay SECONDS seconds
# --------------------------------------------------------------------------------
def setDecayPeriod( mode, args ):
   seconds = args.get( 'SECONDS', IntfCli.defaultLoadInterval().val )
   if seconds == IntfCli.defaultLoadInterval().val:
      tunnelCountersCliConfig.tunnelCountersLoadInterval = \
         IntfCli.defaultLoadInterval()
   else:
      tunnelCountersCliConfig.tunnelCountersLoadInterval = \
         IntfCli.newLoadInterval( seconds )

class TunnelCountersDecayPeriod( CliCommand.CliCommandClass ):
   syntax = 'rate period decay SECONDS seconds'
   noOrDefaultSyntax = 'rate period decay ...'
   data = {
      'rate' : 'Counter rate configuration',
      'period' : 'Counter rate period configuration',
      'decay' : 'Counter rate decay period',
      'SECONDS' : IntfCli.intervalMatcher,
      'seconds' : 'seconds',
   }

   handler = setDecayPeriod
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( TunnelCountersCmd )
TunnelCountersConfigMode.addCommandClass( TunnelCountersTransmitCmd )
TunnelCountersTransmitConfigMode.addCommandClass( TunnelTypeSelectiveCountersCmd )
TunnelCountersTransmitConfigMode.addCommandClass( TunnelTypeCountersCmd )
TunnelCountersConfigMode.addCommandClass( TunnelCountersDecayPeriod )

def Plugin( em ):
   global entityManager
   global routingHardwareRouteStatus
   global allIntfConfigDir
   global ipv4CounterKeyMap
   global ipv4McastRateCounterTable
   global ipv6CounterKeyMap
   global ipv6McastRateCounterTable
   global internalDropSharedConfig, internalDropCliConfig
   global flowTrendSharedConfig
   global fcFeatureConfigDir, fcFeatureStatusDir
   global rxIpCounterTable, rxIpSnapshotTable
   global rxL3IpCounterTable, rxL3IpSnapshotTable
   global txIpCounterTable, txIpSnapshotTable
   global txL3IpCounterTable, txL3IpSnapshotTable
   global rxIpLagCounterTable, txIpLagCounterTable
   global rxL3IpCounterRateTable, txL3IpCounterRateTable
   global rxIpCounterRateTable, txIpCounterRateTable
   global fcRouteConfig
   global l3RouteClearRequest
   global prefixCounterCliConfig
   global cliCapability
   global ipCountersCapability
   global intfCliConfig
   global queueCliConfig
   global ipv4McastCounterConfig
   global ipv6McastCounterConfig
   global tunnelCountersCliConfig
   global tunnelCountersPriorityTable
   entityManager = em
   allFapsId = FapId.allFapsId

   routingHardwareRouteStatus = LazyMount.mount( entityManager,
         "routing/hardware/route/status",
         "Routing::Hardware::RouteStatus",
         'r' )

   allIntfConfigDir = LazyMount.mount( entityManager, 'interface/config/all',
                                       'Interface::AllIntfConfigDir', 'r' )

   ipv4CounterKeyMap = SmashLazyMount.mount( entityManager,
         'hardware/counter/ipv4Mcast/mrouteCounterKeyMap',
         'Ale::Counters::MrouteCounterKeyMapTable',
         SmashLazyMount.mountInfo( 'reader' ) )

   # Mount IPv4 Multicast rate counter smash.
   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
         "", FeatureIdEnum.Mcast, allFapsId )
   ipv4McastRateCounterTable = SmashLazyMount.mount( entityManager,
         mountPath,
         'FlexCounters::RateCounterTable',
         SmashLazyMount.mountInfo( 'reader' ) )

   ipv6CounterKeyMap = SmashLazyMount.mount( entityManager,
         'hardware/counter/ipv6Mcast/mrouteCounterKeyMap',
         'Ale::Counters::MrouteCounterKeyMapTable',
         SmashLazyMount.mountInfo( 'reader' ) )
   # Mount IPv6 Multicast rate counter smash.
   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
         "", FeatureIdEnum.MulticastIpv6, allFapsId )
   ipv6McastRateCounterTable = SmashLazyMount.mount( entityManager,
         mountPath,
         'FlexCounters::RateCounterTable',
         SmashLazyMount.mountInfo( 'reader' ) )

   mountPoint = 'hardware/counter/internalDrop/config/shared'
   internalDropSharedConfig = LazyMount.mount( entityManager,
         mountPoint, 'Ale::Counters::InternalDrop::SharedConfig', 'w' )

   mountPoint = 'hardware/counter/flowTrendMonitor/config/shared'
   flowTrendSharedConfig = LazyMount.mount( entityManager,
         mountPoint, 'Ale::Counters::FlowTrendMonitor::SharedConfig', 'w' )

   mountPoint = 'hardware/counter/internalDrop/config/cli'
   internalDropCliConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::Counters::InternalDrop::CliConfig', 'w' )

   mountPoint = 'flexCounter/featureConfigDir/cliAgent'
   fcFeatureConfigDir = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::FlexCounter::FeatureConfigDir', 'w' )

   mountPoint = 'flexCounter/featureStatusDir'
   fcFeatureStatusDir = LazyMount.mount( entityManager,
         mountPoint, 'Tac::Dir', 'ri' )

   smi = SmashLazyMount.mountInfo( 'reader' )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Ingress, allFapsId, False )
   rxIpCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Ingress, allFapsId, True )
   rxIpSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.L3Ipv4v6Ingress, allFapsId, False )
   rxL3IpCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.L3Ipv4v6Ingress, allFapsId, True )
   rxL3IpSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Ingress,
                               allFapsId, False, 'lag' )
   rxIpLagCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Egress, allFapsId, False )
   txIpCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Egress, allFapsId, True )
   txIpSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.L3Ipv4v6Egress, allFapsId, False )
   txL3IpCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.L3Ipv4v6Egress, allFapsId, True )
   txL3IpSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpCounterTable', smi )

   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
      "", FeatureIdEnum.L3Ipv4v6Ingress, allFapsId )
   rxL3IpCounterRateTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpRateCounterTable', smi )

   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
      "", FeatureIdEnum.L3Ipv4v6Egress, allFapsId )
   txL3IpCounterRateTable = SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpRateCounterTable', smi )

   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
      "", FeatureIdEnum.Ipv4v6Ingress, allFapsId )
   rxIpCounterRateTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpRateCounterTable', smi )

   mountPath = Tac.Type( 'FlexCounters::RateCounterTable' ).mountPath(
      "", FeatureIdEnum.Ipv4v6Egress, allFapsId )
   txIpCounterRateTable= SmashLazyMount.mount( entityManager, mountPath,
                                           'FlexCounters::IpRateCounterTable', smi )

   mountPath = tableMountPath( em, FeatureIdEnum.Ipv4v6Egress,
                               allFapsId, False, 'lag' )
   txIpLagCounterTable = SmashLazyMount.mount( entityManager, mountPath,
                                          'FlexCounters::IpCounterTable', smi )

   mountPoint = 'hardware/counter/l3route/config'
   fcRouteConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::FlexCounter::L3RouteConfig', 'w' )

   mountPoint = 'hardware/counter/l3route/clearRequest'
   l3RouteClearRequest = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::FlexCounter::L3RouteClearRequest', 'w' )

   mountPoint = 'hardware/counter/prefix/config'
   prefixCounterCliConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::PrefixCounterCliConfig', 'w' )

   mountPoint = 'counter/global/intfconfig'
   intfCliConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::Counters::CliConfig', 'w' )

   mountPoint = 'counter/global/queueconfig'
   queueCliConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Ale::Counters::CliConfig', 'w' )

   mountPoint = 'counter/global/configcapability'
   cliCapability = LazyMount.mount( entityManager,
         mountPoint, 'Ale::Counters::CliCapability', 'r' )

   mountPoint = 'counter/global/ipCountersCapability'
   ipCountersCapability = LazyMount.mount( entityManager,
         mountPoint, 'Ale::Counters::IpCountersCapability', 'r' )

   mountPoint = MrouteConsts.routingMulticastVrfConfigSysdbPath
   ipv4McastCounterConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Routing::Multicast::Fib::VrfConfig', 'w' )

   mountPoint = 'routing6/multicast/vrf/config'
   ipv6McastCounterConfig = ConfigMount.mount( entityManager,
         mountPoint, 'Routing::Multicast::Fib::VrfConfig', 'w' )

   mountPoint = 'hardware/counter/tunnel/cliConfig'
   tunnelCountersCliConfig = ConfigMount.mount(
      entityManager, mountPoint, 'Ale::Counters::TunnelCountersCliConfig', 'w' )

   mountPoint = 'hardware/counter/tunnel/tunnelTypePriority'
   tunnelCountersPriorityTable = ConfigMount.mount(
      entityManager, mountPoint, 'Ale::TunnelCountersPriorityTable', 'w' )
   mountPoint = 'counter/global/rateWatermarkConfig'
   gv.rateWatermarkConfig = LazyMount.mount(
         entityManager, mountPoint, 'Ale::Counters::RateWatermarkCliConfig', 'w' )
