# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from AleFlexCounterTableMounter import tableMountPath
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
from CliPlugin import RouteCountersModel
from CliPlugin.AleCountersCli import (
   counterFeaturesSupported,
   checkCounterFeatureEnabled,
   checkCounterFeatureSupported
)
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IraIpCli as IraIpCli
from CliPlugin.VrfCli import vrfExists, VrfExprFactory
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import SmashLazyMount
import SharkLazyMount
import Tac

allFapsId = Tac.Type( 'FlexCounters::FapId' ).allFapsId
featureIdEnum = Tac.Type( 'FlexCounters::FeatureId' )
featureIdEnumVal = Tac.Type( 'FlexCounters::FeatureIdEnumVal' )
PlatformIndependentCounterKey = Tac.Type(
   'Routing::Hardware::PlatformIndependentCounterKey' )
PdckStatus = Tac.Type( 'Ale::PdckStatus' )

featureSupportedGuard = counterFeaturesSupported( [ 'Route', 'RouteIpv4' ] )
# The route feature guard is used to differentiate between platforms.
# Strata supports the "Route" feature and mounts Strata specific tables.
# All other platforms should be using the "RouteIpv4" feature and publish tables
# in the Ale mount path.
routeFeatureGuard = counterFeaturesSupported( [ 'Route' ] )

mountGlobals = dict(
   # RouteIpv4 Capability Specific Mounts
   prefixCounterTable=None,
   counterKeyTable=None,
   vrfNameStatus=None,
   vrfIdStatus=None,
   counterStatusTable=None,

   # Route Capability Specific Mounts
   routeCounterTable=None,
   routeCounterSnapshotTable=None,
   l3RouteConfig=None,
   routeCounterStatus=None )

gv = CliGlobal.CliGlobal( mountGlobals )

# Config command keywords.
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
   helpdesc='Show packet and byte count' )
routeKeyword = CliCommand.guardedKeyword( 'route', helpdesc='Route information',
   guard=featureSupportedGuard )
routeIpv6Keyword = CliCommand.guardedKeyword( 'ipv6',
   helpdesc='Show IPv6 route counters', guard=routeFeatureGuard )

def vrfNameFromId( mode, vrfId ):
   vrfEntry = gv.vrfIdStatus.vrfIdToName.get( vrfId )
   if vrfEntry is None:
      mode.addError( 'Unknown VRF ID %s' % vrfId )
      return None
   return vrfEntry.vrfName

def vrfIdFromName( mode, vrfName ):
   vrfId = gv.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )
   if vrfId is None:
      mode.addError( 'No such VRF %s' % vrfName )
      return None
   return vrfId

def populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry, counterInfo,
                              routeCounterFeature, vrfName, prefix ):

   routeCounterEntry.prefix = prefix.stringValue
   counterPackets = 0
   counterBytes = 0
   snapshotPackets = 0
   snapshotBytes = 0

   routeCounters = vrfRouteCounters.vrfs.get( vrfName )
   if routeCounters is None:
      routeCounters = RouteCountersModel.RouteCounters()

   if routeCounterFeature:
      counterKey = counterInfo.counterIndex
      counterTableEntry = gv.routeCounterTable.counter.get( counterKey )
      snapshotTableEntry = gv.routeCounterSnapshotTable.counter.get( counterKey )
      if counterTableEntry is not None:
         counterPackets = counterTableEntry.pkts
         counterBytes = counterTableEntry.octets
      if snapshotTableEntry is not None:
         snapshotPackets = snapshotTableEntry.pkts
         snapshotBytes = snapshotTableEntry.octets
   else:
      counterKey = counterInfo.key
      pick = PlatformIndependentCounterKey( counterInfo.pick )
      counterStatusEntry = gv.counterStatusTable.prefixCounterStatus.get( pick )

      # An entry is programmed in hardware if the counter status is flagged as
      # allocatedInuse
      routeCounterEntry.programmed = ( counterStatusEntry is not None and
         counterStatusEntry.status == PdckStatus.allocatedInUse )

      counterPrefixEntry = None
      counterTableVrfEntry = gv.prefixCounterTable.vrfs.get( vrfName )
      if counterTableVrfEntry is not None:
         counterPrefixEntry = counterTableVrfEntry.ipv4Entry.get( prefix.v4Prefix )

      if counterPrefixEntry is not None:
         counterTableEntry = counterPrefixEntry.state.counters
         counterPackets = counterTableEntry.packetsForwarded
         counterBytes = counterTableEntry.octetsForwarded
         snapshotPackets = counterTableEntry.snapshotPacketsForwarded
         snapshotBytes = counterTableEntry.snapshotOctetsForwarded
         backupPackets = counterTableEntry.packetsForwardedBackup
         backupBytes = counterTableEntry.octetsForwardedBackup
         snapBackupPackets = counterTableEntry.snapshotPacketsForwardedBackup
         snapBackupBytes = counterTableEntry.snapshotOctetsForwardedBackup
      else:
         backupPackets = 0
         backupBytes = 0
         snapBackupPackets = 0
         snapBackupBytes = 0

      # If any backup PICK in a VRF is programmed into the status we have a unique
      # backup. This is true whether or not an entry exists in the counter table.
      if pick.getBackupPrefixPick() in gv.counterStatusTable.prefixCounterStatus:
         routeCounters.uniqueBackup = True
         routeCounterEntry.backupPacketCount = backupPackets - snapBackupPackets
         routeCounterEntry.backupByteCount = backupBytes - snapBackupBytes

   routeCounterEntry.packetCount = counterPackets - snapshotPackets
   routeCounterEntry.byteCount = counterBytes - snapshotBytes
   routeCounters.counters.append( routeCounterEntry )
   vrfRouteCounters.vrfs[ vrfName ] = routeCounters

def showRouteCounters( mode, args ):
   if 'ipv4' in args:
      routeType = 'ipv4'
      prefix = args.get( 'PREFIX4' )
   else:
      routeType = 'ipv6'
      prefix = args.get( 'ADDR6' ) or args.get( 'PREFIX6' )

   vrfName = args.get( 'VRF' )

   if checkCounterFeatureSupported( "Route" ):
      routeCountersFunc = getRouteFeatureCounters
   elif checkCounterFeatureEnabled( 'RouteIpv4' ):
      routeCountersFunc = getRouteIpv4FeatureCounters
   else:
      # If the counter feature is disabled, print an error mentioning it.  In order
      # to avoid changing the existing functionality we don't do this check for the
      # "Route" feature.
      mode.addErrorAndStop( "'hardware counter feature route " + routeType +
         "' should be enabled first" )
   vrfRouteCounters = routeCountersFunc( mode, vrfName, prefix, routeType )

   if vrfRouteCounters is not None:
      return RouteCountersModel.VrfRouteCounters( vrfs=vrfRouteCounters.vrfs )

   # If either of the above routeCountersFuncs return None and expect to escape
   # they should add an error before returning as infra will expect it.
   return None

# Populates route counters for platforms advertising the RouteIpv4 feature.
def getRouteIpv4FeatureCounters( mode, vrfName, prefix,
                                 routeType ):
   vrfRouteCounters = RouteCountersModel.VrfRouteCounters()
   vrfIds = []
   counterKeyTable = gv.counterKeyTable
   if vrfName is None:
      vrfIds = list( gv.vrfNameStatus.nameToIdMap.vrfNameToId.values() )
   else:
      vrfId = vrfIdFromName( mode, vrfName )
      if vrfId is None:
         return None
      vrfIds.append( vrfId )

   if prefix is not None:
      prefixObj = Tac.newInstance( 'Arnet::IpGenPrefix', str( prefix ) )
      for vrfId in vrfIds:
         counterKey = Tac.Value(
            'Routing::Hardware::PrefixCounterKey', prefixObj, vrfId, True )
         counterInfo = counterKeyTable.counterKeyInfo.get( counterKey )
         if counterInfo is not None:
            vrfName = vrfNameFromId( mode, counterKey.vrfId )
            if vrfName is None:
               continue
            routeCounterEntry = RouteCountersModel.RouteCounterEntry()
            populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                      counterInfo, False, vrfName, prefixObj )
      if not vrfRouteCounters.vrfs:
         errorMsg = "No counter attached to route " + str( prefix )
         if vrfName:
            errorMsg += " in VRF " + vrfName
         mode.addError( errorMsg )
         return None
   else:
      for counterKey, counterInfo in counterKeyTable.counterKeyInfo.items():
         if counterKey.vrfId in vrfIds and counterKey.prefix.af == routeType:
            vrfName = vrfNameFromId( mode, counterKey.vrfId )
            if vrfName is None:
               continue
            routeCounterEntry = RouteCountersModel.RouteCounterEntry()
            populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                      counterInfo, False, vrfName,
                                      counterKey.prefix )

   return vrfRouteCounters

# Populates route counters for platforms advertising the Route flexcounter feature.
def getRouteFeatureCounters( mode, vrfName, prefix, routeType ):
   vrfRouteCounters = RouteCountersModel.VrfRouteCounters()
   if vrfName is None:
      vrfName = DEFAULT_VRF

   if not vrfExists( vrfName ):
      mode.addError( 'No such VRF %s' % vrfName )
      return None
   if prefix is not None:
      prefixObj = Tac.newInstance( 'Arnet::IpGenPrefix', str( prefix ) )
      vrfNameObj = Tac.newInstance( 'L3::VrfName', vrfName )
      vrfNameAndIpGenPrefix = Tac.Value(
         'Ale::FlexCounter::VrfNameAndIpGenPrefix', vrfNameObj, prefixObj )
      found = gv.routeCounterStatus.hasCounterInfo( vrfNameAndIpGenPrefix )
      if not found:
         mode.addError( 'No counter attached to route %s in vrf %s' %
            ( prefix, vrfName ) )
         return None
      else:
         counterInfo = gv.routeCounterStatus.getCounterInfo( vrfNameAndIpGenPrefix )
         routeCounterEntry = RouteCountersModel.RouteCounterEntry()
         populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                   counterInfo, True, vrfName, prefixObj )
   else:
      for vKey, vVal in gv.routeCounterStatus.vrfCounterInfo.items():
         for pKey, pVal in vVal.prefixCounterInfo.items():
            routeCounterEntry = RouteCountersModel.RouteCounterEntry()
            vrfNameAndIpGenPrefix = Tac.Value(
               'Ale::FlexCounter::VrfNameAndIpGenPrefix', vKey, pKey )
            if routeType == pKey.af:
               if vrfName is None or vKey == vrfName:
                  populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                            pVal, True, vrfName, pKey )
   return vrfRouteCounters

class ShowRouteIpCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show route ( ( ipv4 [ VRF ] [ PREFIX4 ] ) | ' \
                         '( ipv6 [ VRF ] [ ADDR6 | PREFIX6 ] ) ) ' \
                         'counters'
   data = {
      'route' : routeKeyword,
      'ipv4' : 'Show IPv4 route counters',
      'ipv6' : routeIpv6Keyword,
      'VRF' : VrfExprFactory( helpdesc='Show route counters with given VRF name',
                              guard=IraIpCli.vrfRoutingSupportedGuard ),
      'PREFIX4' : IpAddrMatcher.ipPrefixExpr(
                  'Match this IP address', 'Match this subnet mask',
                  'Match this IP prefix',
                  overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'ADDR6' : Ip6AddrMatcher.ip6AddrMatcher,
      'PREFIX6' : Ip6AddrMatcher.ip6PrefixMatcher,
      'counters' : matcherCounters,
   }
   cliModel = RouteCountersModel.VrfRouteCounters
   handler = showRouteCounters

BasicCli.addShowCommandClass( ShowRouteIpCounters )

def Plugin( em ):
   smir = SmashLazyMount.mountInfo( 'reader' )
   sharkMountInfo = SharkLazyMount.mountInfo( 'shadow' )

   # These mounts will be unused for the StrataCounters implementation.

   mountPath = "hardware/ale/ipv4PrefixCounter"
   gv.prefixCounterTable = SharkLazyMount.mount( em, mountPath,
      "Ale::FlexCounters::PrefixCountersTable", sharkMountInfo, True )
   gv.counterKeyTable = SmashLazyMount.mount( em,
      'hardware/ale/prefixToCounterKeyTable',
      'Routing::Hardware::PrefixToCounterKeyTable', smir )
   gv.counterStatusTable = SmashLazyMount.mount( em,
      'hardware/ale/prefixCounterStatus',
      'Ale::PrefixCounterStatusTable', smir )

   gv.vrfNameStatus = LazyMount.mount( em, Cell.path( 'vrf/vrfNameStatus' ),
      'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
   gv.vrfIdStatus = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
      "Vrf::VrfIdMap::Status", SmashLazyMount.mountInfo( 'reader' ),
      autoUnmount=True )

   # These mounts will be unused for the SandCounters implementation.

   mountPath = tableMountPath( em, featureIdEnum.Route, allFapsId, False )
   gv.routeCounterTable = SmashLazyMount.mount( em, mountPath,
      "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( em, featureIdEnum.Route, allFapsId, True )
   gv.routeCounterSnapshotTable = SmashLazyMount.mount( em, mountPath,
      "FlexCounters::CounterTable", smir )

   gv.routeCounterStatus = LazyMount.mount( em,
      'flexCounter/stratal3/l3RouteFlexCounterStatus',
      'Ale::FlexCounter::L3RouteFlexCounterStatus', 'r' )

   mountPath = 'hardware/counter/l3route/config'
   gv.l3RouteConfig = LazyMount.mount( em, mountPath,
      'Ale::FlexCounter::L3RouteConfig', 'r' )
