#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Hardware
import ShowCommand
import Tac
import ConfigMount
import LazyMount

# dummy import for 'hardware/ale/cpufabric/config'
# pkgdeps: rpm Ale

cpuFabricConfig = None
cpuFabricSliceStatus = None
thresholdShutdownSupported = False

def thresholdShutdownGuard( mode, token ):
   if thresholdShutdownSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

matcherCpu = CliMatcher.KeywordMatcher( 'cpu',
      helpdesc='CPU interface' )
matcherForwarding = CliMatcher.KeywordMatcher( 'forwarding',
      helpdesc='Forwarding enginee' )
matcherMonitor = CliMatcher.KeywordMatcher( 'monitor',
      helpdesc='Tx/Rx health monitoring' )
matcherShutdown = CliMatcher.KeywordMatcher( 'shutdown',
      helpdesc='Threshold of errors to cause linecard to be shutdown' )

#--------------------------------------------------------------------------------
# [ no | default ] hardware cpu forwarding monitor
#--------------------------------------------------------------------------------
class EnableHealthMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware cpu forwarding monitor'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliToken.Hardware.hardwareForConfigMatcher,
      'cpu' : matcherCpu,
      'forwarding' : matcherForwarding,
      'monitor' : matcherMonitor,
   }
   hidden = True

   handler = "AleCpuFabricHandler.handleEnableHealthMonitoring"

   defaultHandler = handler

   noHandler = "AleCpuFabricHandler.handleNoEnableHealthMonitoring"

BasicCli.GlobalConfigMode.addCommandClass( EnableHealthMonitoringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware cpu forwarding monitor threshold shutdown NUM_ERRORS
#--------------------------------------------------------------------------------
class ThresholdShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware cpu forwarding monitor threshold shutdown NUM_ERRORS'
   noOrDefaultSyntax = 'hardware cpu forwarding monitor threshold shutdown ...'
   data = {
      'hardware' : CliToken.Hardware.hardwareForConfigMatcher,
      'cpu' : matcherCpu,
      'forwarding' : matcherForwarding,
      'monitor' : matcherMonitor,
      'threshold' : 'Configurable threshold',
      'shutdown' : CliCommand.guardedKeyword( 'shutdown',
         helpdesc='threshold of errors to cause linecard to be shutdown',
         guard=thresholdShutdownGuard ),
      'NUM_ERRORS' : CliMatcher.IntegerMatcher( 1, 128,
         helpdesc='Number of DMA errors' ),
   }
   hidden = True

   handler = "AleCpuFabricHandler.handleThresholdShutdown"

   noOrDefaultHandler = "AleCpuFabricHandler.handleNoThresholdShutdown"

BasicCli.GlobalConfigMode.addCommandClass( ThresholdShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware cpu forwarding monitor action none
#--------------------------------------------------------------------------------
class ActionNoneCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware cpu forwarding monitor action none'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliToken.Hardware.hardwareForConfigMatcher,
      'cpu' : matcherCpu,
      'forwarding' : matcherForwarding,
      'monitor' : matcherMonitor,
      'action' : 'Tx/Rx health recovery action',
      'none' : 'No action',
   }
   hidden = True

   handler = "AleCpuFabricHandler.handleActionNone"

   noOrDefaultHandler = "AleCpuFabricHandler.handleNoActionNone"

BasicCli.GlobalConfigMode.addCommandClass( ActionNoneCmd )

#--------------------------------------------------------------------------------
# show hardware cpu forwarding monitor
#--------------------------------------------------------------------------------
# Allow platform-specific callbacks to registered
showMonitorStatusHook = []

def registerShowMonitorStatus( hook ):
   showMonitorStatusHook.append( hook )

class HardwareCpuForwardingMonitorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware cpu forwarding monitor'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'cpu' : matcherCpu,
      'forwarding' : matcherForwarding,
      'monitor' : matcherMonitor,
   }
   hidden = True

   handler = "AleCpuFabricHandler.handleHardwareCpuForwardingMonitor"

BasicCli.addShowCommandClass( HardwareCpuForwardingMonitorCmd )

# ------------------------------------------------------------------------------
# plugin definition
# ------------------------------------------------------------------------------

def Plugin( entityManager ):
   global cpuFabricConfig, cpuFabricSliceStatus
   cpuFabricConfig = ConfigMount.mount( entityManager,
                                        'hardware/ale/cpufabric/config',
                                        'Ale::CpuFabricConfig', 'w' )
   cpuFabricSliceStatus = LazyMount.mount( entityManager,
                                           'hardware/ale/cpufabric/sliceStatus',
                                           'Tac::Dir', 'ri' )
