#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import (
      Model,
      Str,
      Int,
      Bool,
      List,
      Dict,
      Enum,
      Float,
      Submodel,
      DeferredModel,
   )
from ArnetModel import (
   IpGenericAddress,
   IpGenericPrefix,
   MacAddress,
)
from IntfModels import Interface

largeInt = int

l2AdjTypeEnum = Tac.Type( 'Ale::NextHopEntryType' ).attributes
l2AdjStateEnum = ( 'uninitialized', 'installFailed', 'installed', 'installNotReady',
                   'installNotRequired', 'ecmpGroupsFull', 'pendingSync',
                   'garbageCollected', 'pendingDelete', 'installFailedPreserved' )
fecInfoUnavailable = Tac.Type( 'Ale::CliFecTypeStrings' ).fecInfoUnavailable
routeTypeEnum = tuple( set( Tac.Type( 'Ale::RouteType' ).attributes ).union(
                                                        { fecInfoUnavailable } ) )
forwardingActionEnum = Tac.Type( 'Ale::ForwardingAction' ).attributes
decimalPlaces = 5

# optimized - Routes in LEM ( for Sand )
# optimizedAndExpanded - Routes Expanded in Hardware
# optimizedAndCompressed - Routes Compressed in Hardware
routeOptimizedEnum = ( 'unoptimized', 'optimized',
                       'optimizedAndExpanded', 'optimizedAndCompressed' )

stateEnum = ( 'enabled', 'disabled' )
support = ( 'supported', 'notSupported' )
labelOperationEnum = ( 'push', 'pop', 'swap', 'forward' )

class AleStats( Model ):
   insertTime = Float( help='Insert timestamp' )
   lastUpdateTime = Float( help='Update timestamp' )
   numUpdates = Int( help='Number of updates' )

class AlePrevLevelFecs( Model ):
   fecId = Int( help='Previous level FEC ID' )
   tableId = Int( help='FEC table ID' )

class GreEncapInfo( Model ):
   destination = IpGenericPrefix(
             help='Location in the IP network where the tunnel terminates' )
   source = IpGenericPrefix(
             help='Location in the IP network where the tunnel starts' )
   tos = Int( optional=True, help='Type of service' )
   ttl = Int( optional=True, help='Time to live' )

class MplsEncapInfo( Model ):
   operation = Enum( values=labelOperationEnum, help='Label Operation' )
   labelStack = List( valueType=int, help='MPLS Label Stack' )
   entropyLabelIndices = List( valueType=int,
         help="Entropy label indices in the label stack of the tunnel",
         optional=True )
   entropyLabelIndicatorIndices = List( valueType=int,
         help="Entropy label indicator indices in the label stack of the tunnel",
         optional=True )

class VxlanEncapInfo( Model ):
   vtepIp = IpGenericAddress( help="VTEP IP address" )
   srcVti = Interface( help="Source VTI" )

class IpInIpEncapInfo( Model ):
   source = IpGenericPrefix( help='Source IP address' )
   destination = IpGenericPrefix( help='Destination IP address' )

class IpTunnelInfo( Model ):
   source = IpGenericAddress( help="Source IP address" )
   destination = IpGenericAddress( help="Destination IP address" )
   underlayL2Adj = Int( help="L2 Adjacency ID of the underlay FEC",
                        optional=True )
   interface = Interface( help="Tunnel interface", optional=True )

class FallbackInfo( Model ):
   fallbackVrf = Str( help="Fallback VRF name" )
   actionType = Enum( values=( 'none', 'ipip' ),
                      help="Decapsulation type" )

class DecapInfo( Model ):
   actionType = Enum( values=( 'ipip', ),
                      help="Decapsulation type" )

class VoqFabricEncapInfo( Model ):
   systemPortId = Int( help='System port index' )
   encapIndex = Int( help='Encapsulation index' )

class GueEncapInfo( Model ):
   destination = IpGenericPrefix(
      help='Address in the IP network where the tunnel terminates' )
   source = IpGenericPrefix(
      help='Address in the IP network where the tunnel starts' )
   tos = Int( optional=True, help='Type of service' )
   ttl = Int( optional=True, help='Time to live' )

class AleL2AdjVia( Model ):
   weight = Int( help='Metric representing the hop cost' )
   adjType = Enum( values=l2AdjTypeEnum, help='L2 Adjacency Type' )
   macAddr = MacAddress( help='Next-Hop Mac Address' )
   l3IntfId = Interface( help='L3 Interface Id' )
   l2IntfId = Interface( help='L2 Interface Id' )
   nextLevelL2AdjIndex = Int(
         optional=True,
         help='L2 Adjacency ID of the next FEC in a hierarchical FEC' )
   counterId = Int( optional=True, help='Next-Hop Counter Id' )
   forwardingAction = Enum( values=forwardingActionEnum, help='Forwarding action' )
   fallbackInfo = Submodel( valueType=FallbackInfo, help="Fallback information",
                            optional=True )
   decapInfo = Submodel( valueType=DecapInfo, help="Decap information",
                         optional=True )
   mplsEncapInfo = Submodel( valueType=MplsEncapInfo, help="MPLS Encap information",
                             optional=True )
   greEncapInfo = Submodel( valueType=GreEncapInfo, help="GRE Encap information",
                             optional=True )
   ipTunnelInfo = Submodel( valueType=IpTunnelInfo,
                            help="Tunnel resolution information", optional=True )
   vxlanEncapInfo = Submodel( valueType=VxlanEncapInfo,
                              help="VXLAN Encap information", optional=True )
   ipInIpEncapInfo = Submodel( valueType=IpInIpEncapInfo,
                               help="IP-in-IP Encap information", optional=True )
   voqFabricEncapInfo = Submodel( valueType=VoqFabricEncapInfo,
                                  help="VOQ Fabric Encap information",
                                  optional=True )
   gueEncapInfo = Submodel( valueType=GueEncapInfo, help="UDP Encap information",
                            optional=True )

class AleL3AdjViaTunnel( Model ):
   tunnelType = Str( help="Tunnel type" )
   tunnelAddressFamily = Enum( values=( 'IPv4', 'IPv6' ), help="Address family",
                               optional=True )
   tunnelIndex = Int( help="Tunnel table index" )

class AleL3AdjVia( Model ):
   nextHopIp = IpGenericPrefix( help='Ip Address of Next Hop' )
   weight = Int( help='Metric representing the hop cost' )
   l3IntfId = Interface( help='L3 Interface Id' )
   forwardingAction = Enum( values=forwardingActionEnum, help='Forwarding action' )
   tunnelDescriptor = Submodel( valueType=AleL3AdjViaTunnel,
                                help="Tunnel information", optional=True )
   fallbackInfo = Submodel( valueType=FallbackInfo, help="Fallback information",
                            optional=True )
   decapInfo = Submodel( valueType=DecapInfo, help="Decapsulation information",
                         optional=True )

class AleL2AdjInfo( Model ):
   adjCount = Int( optional=True, help='Number of L3 Adjacencies' )
   l2AdjState = Enum( values=l2AdjStateEnum, help='L2 Adjacency State' )
   l2AdjRequestState = Str( help='L2 Adjacency Request State' )
   l2AdjHardwareState = Str( help='L2 Adjacency Hardware State' )
   l2AdjVia = List( valueType=AleL2AdjVia, optional=True,
                    help='List storing the L2 Adjacency Vias' )
   backupL2AdjVia = List( valueType=AleL2AdjVia, optional=True,
                          help='List storing the backup L2 Adjacency Vias' )
   platformFecIndex = Int( optional=True,
                           help='Platform FEC table index used by the L2 Adjacency' )
   l2AdjStats = Submodel( valueType=AleStats, optional=True,
                          help='L2 Adjacency timestamps' )

class AlePrefix( Model ):
   prefix = Str( help='Prefix' )

class AleV4Prefixes( Model ):
   v4Prefixes = List( valueType=AlePrefix, optional=True,
      help="A mapping between VRF ID and V4 Prefixes" )

class AleV6Prefixes( Model ):
   v6Prefixes = List( valueType=AlePrefix, optional=True,
      help="A mapping between VRF ID and V6 Prefixes" )

class AleL3AdjInfo( Model ):
   routeType = Enum( values=routeTypeEnum, optional=True,
                     help='Specifies Route Type' )
   l2AdjIndex = Int( optional=True, help='Layer 2 Adjacency Index' )
   prefixCount = Int( optional=True, help='Number of Prefixes' )
   adjRefCount = Int( optional=True, help='Inter Adjacency references' )
   pbrRefCount = Int( optional=True, help='PBR references' )
   extRefCount = Int( optional=True, help='External FEC references' )
   proxyRefCount = Int( optional=True, help='proxy FEC references' )
   totalRefCount = Int( optional=True, help='Total references' )
   v4PrefixesByVrf = Dict( keyType=str, valueType=AleV4Prefixes,
                           optional=True,
                           help='Dict to list V4 Prefix references by VRF' )
   v6PrefixesByVrf = Dict( keyType=str, valueType=AleV6Prefixes,
                           optional=True,
                           help='Dict to list V6 Prefix references by VRF' )
   adjStats = Submodel( valueType=AleStats, optional=True,
                        help='SubModel storing Adjacency Timestamps' )
   l3AdjVia = List( valueType=AleL3AdjVia, optional=True,
                    help='List storing the L3 V4 Adjacency Vias' )
   backupL3AdjVia = List( valueType=AleL3AdjVia, optional=True,
                          help='List storing the backup L3 V4 Adjacency Vias' )
   l3AdjVia6 = List( valueType=AleL3AdjVia, optional=True,
                    help='List storing the L3 V6 Adjacency Vias' )
   backupL3AdjVia6 = List( valueType=AleL3AdjVia, optional=True,
                           help='List storing the backup L3 V6 Adjacency Vias' )
   transientAdj = Bool( optional=True,
                        help="ECMP Adj programmed as non ECMP in hardware" )
   orderedNexthops = Bool( optional=True,
                           help="The ordering of next-hops is honored" )
   ucmpRequested = Bool( optional=True, help="Auto UCMP requested" )
   sourceFec = Int( optional=True, help="Source FEC being used by this proxy" )
   fecRefs = List( valueType=AlePrevLevelFecs, optional=True,
                         help='List of previous level fec refs' )

class AleRouteInfo( Model ):
   cmdErr = Str( optional=True, help="Error Message" )
   routeProgrammed = Bool( optional=True,
                           help='Flag to indiacte Route is programmed or not' )
   routeOptimized = Enum( values=routeOptimizedEnum, optional=True,
                          help='Details of Route Optimization' )
   parent = Str( optional=True, help='Parent prefix' )
   l3AdjInfo = Dict( keyType=largeInt, valueType=AleL3AdjInfo,
                     optional=True, help='Dict to handle L3 Adjacency Info' )
   l2AdjInfo = Dict( keyType=largeInt, valueType=AleL2AdjInfo,
                     optional=True, help='Dict to handle L2 Adjacency Info' )
   routeStats = Submodel( valueType=AleStats, optional=True,
                      help='Stats of a route' )

class AleRoute( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   vrfName = Str( optional=True, help="VRF name" )
   numUnprog = Int( optional=True, help="Count of unprogrammed routes" )
   fibRoute = Dict( keyType=IpGenericPrefix, valueType=AleRouteInfo, optional=True,
                    help="Mapping prefix to its FibRoute Info" )

class AllAleRoutes( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   numUnprog = Int( optional=True, help="Count of unprogrammed routes in all VRFs" )
   vrfs = Dict( keyType=str,
                valueType=AleRoute,
                optional=True,
                help="Dictionary mapping VRF names to ale routes in that VRF" )

class AleL2Adj( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   l2Adj = Dict( keyType=largeInt, valueType=AleL2AdjInfo, optional=True,
                 help="Dictionary to map l2AdjIndex to l2Adj Info" )

class AleAdj( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   adj = Dict( keyType=largeInt, valueType=AleL3AdjInfo,
               optional=True,
               help="Dictionary to map fecId to L3Adj" )
   transientAdjsPresent = Bool( optional=True,
                                help='Output contains some transient adjs' )
