#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

from AleFlexCounterTypes import (
   counterFeatureMap,
   tokenHelpdesc,
)
import BasicCliModes
import CliCommand
from CliPlugin.AleCountersCli import (
   counterFeatureSupported,
   counterFeatureUnitsSupported,
   changeFeatureIdConfig,
)
from CliToken.Hardware import hardwareForConfigMatcher

__defaultTraceHandle__ = Tracing.Handle( 'AleHardwareCounterFeatureCli' )
t8 = __defaultTraceHandle__.trace8

#------------------------------------------------------------------
# The "[no | default] hardware counter feature <counterFeature>
#      [ <featureDirection> ] [ <featureOption> ] [ units packets ]"
# command, in "config" mode.
#------------------------------------------------------------------

def uniquify( token, featureId ):
   return token + '_' + featureId

def deuniquify( tokenName, token ):
   if not tokenName.startswith( token + '_' ):
      return None
   featureId = tokenName[ len( token + '_' ) : ]
   return featureId

def addToken( token, data, featureId=None, guard=None ):
   helpdesc = tokenHelpdesc( token )
   if guard is None:
      dataVal = data.get( token )
      if dataVal is None:
         data[ token ] = helpdesc
      else:
         assert dataVal == helpdesc

      return token
   else:
      assert featureId is not None
      uniqueToken = uniquify( token, featureId )

      # Here we need to make the token unique to the current feature, so that we can
      # add a guard that is specific to the feature. Note: this is not needed for
      # tokens that don't require a guard.
      assert uniqueToken not in data, 'uniqueToken is not unique'
      data[ uniqueToken ] = CliCommand.guardedKeyword( token, helpdesc, guard )

      return uniqueToken

# `hardwareCounterFeatureSyntaxGen` generates the BNF syntax for all auto-generated
# `hardware counter feature` commands, along with the data dictionary associated with
# it. It can then be used directly as the `syntax`, `noOrDefaultSyntax`. and `data`
# attributes of a CliCommandClass.
# The syntax looks as follow:
# > hardware counter feature (
# > ( acl_AclEgressIpv4 out ipv4 [ units_AclEgressIpv4 packets ] )
# > | ( acl_AclEgressIpv6 out ipv6 [ units_AclEgressIpv6 packets ] )
# > | ( acl_AclIngress in [ units_AclIngress packets ] )
# > | ( decap-group_DecapGroup [ units_DecapGroup packets ] )
# > [...]
#
# It also generates a more human readable version of the syntax:
# > hardware counter feature (
# >  ( acl in [ units packets ] )
# >  | ( acl out ipv4 [ units packets ] )
# >  | ( acl out ipv6 [ units packets ] )
# >  | ( decap-group [ units packets ] )
# > [...]
# The main difference is that some tokens are "uniquified" in the syntax to be passed
# to the CLI parser, since we need to have a unique name for each token where we want
# to add a guard.
def hardwareCounterFeatureSyntaxGen():
   baseSyntax = 'hardware counter feature '
   fullSyntax = baseSyntax
   fullHumanReadableSyntax = baseSyntax
   data = {
      'hardware' : hardwareForConfigMatcher,
      'counter' : 'Counter',
      'feature' : 'Counter feature',
   }

   fullSyntax += '( '
   fullHumanReadableSyntax += '(\n '
   fullSyntaxList = []
   fullHumanReadableSyntaxList = []
   for featureId, featureInfo in counterFeatureMap.items():
      if not featureInfo.autogenCliCfg:
         continue

      firstToken = featureInfo.cliConfigTokens[ 0 ]
      featureSyntax = addToken(
         firstToken, data, featureId=featureId,
         guard=counterFeatureSupported( featureId ) )
      humanReadableFeatureSyntax = firstToken

      for token in featureInfo.cliConfigTokens[ 1 : ]:
         assert token
         assert ' ' not in token

         featureSyntax += ' ' + addToken( token, data )
         humanReadableFeatureSyntax += ' ' + token

      # Generate the suffix feature attributes:
      # [ units packets ]
      featureSyntax += ' ['
      humanReadableFeatureSyntax += ' ['

      unitsToken = 'units'
      featureSyntax += ' ' + addToken(
         unitsToken, data, featureId=featureId,
         guard=counterFeatureUnitsSupported( featureId ) )
      humanReadableFeatureSyntax += ' ' + unitsToken

      packetsToken = 'packets'
      featureSyntax += ' ' + addToken( packetsToken, data )
      humanReadableFeatureSyntax += ' ' + packetsToken

      featureSyntax += ' ]'
      humanReadableFeatureSyntax += ' ]'

      fullSyntaxList.append( '( ' + featureSyntax + ' )' )
      fullHumanReadableSyntaxList.append( '( ' + humanReadableFeatureSyntax +
                                          ' )\n' )
   fullSyntax += ' | '.join( sorted( fullSyntaxList ) )
   fullHumanReadableSyntax += ' | '.join( sorted( fullHumanReadableSyntaxList ) )

   fullSyntax += ' )'
   fullHumanReadableSyntax += ' )'

   return fullHumanReadableSyntax, fullSyntax, data

hardwareCounterFeatureHumanReadableSyntax, hardwareCounterFeatureSyntax, \
   hardwareCounterFeatureSyntaxData = hardwareCounterFeatureSyntaxGen()

def hardwareCounterFeatureHandler( mode, args, state ):
   # Try and find which featureId we were called for.
   featureId = None
   for tokenName, token in args.items():
      featureId = deuniquify( tokenName, token )
      if featureId is None:
         # Not a uniquified token, so doesn't contain featureId, keep searching.
         continue
      assert featureId in counterFeatureMap
      break
   assert featureId is not None, "Couldn't deduce featureId from the tokens"

   # Handle extra feature attributes.
   units = None
   if uniquify( 'units', featureId ) in args:
      if 'packets' in args:
         units = 'packets'
   changeFeatureIdConfig( mode, featureId, state=state, units=units )

class HardwareCounterFeatureCmd( CliCommand.CliCommandClass ):
   syntax = hardwareCounterFeatureSyntax
   noOrDefaultSyntax = hardwareCounterFeatureSyntax
   data = hardwareCounterFeatureSyntaxData

   @staticmethod
   def handler( mode, args ):
      state = True
      hardwareCounterFeatureHandler( mode, args, state )

   @staticmethod
   def noHandler( mode, args ):
      state = False
      hardwareCounterFeatureHandler( mode, args, state )

   @staticmethod
   def defaultHandler( mode, args ):
      state = None
      hardwareCounterFeatureHandler( mode, args, state )

BasicCliModes.GlobalConfigMode.addCommandClass( HardwareCounterFeatureCmd )
