# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import BasicCli
import CliCommand
from CliMatcher import (
      KeywordMatcher
   )
from CliPlugin import AleCliLib
from CliPlugin import AleFibCli
from CliPlugin.AleL3CliModel import (
      AleL3HandledErrors,
   )
from CliPlugin import TechSupportCli
from CliPlugin import VrfCli
from CliDynamicSymbol import CliDynamicPlugin
import CliToken.Ip
import CliToken.Ipv6
import CommonGuards
import ShowCommand
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'AleL3Cli' )

t0 = Tracing.trace0

ipMatcherForShow = CliToken.Ip.ipMatcherForShow
ipv6MatcherForShow = CliToken.Ipv6.ipv6MatcherForShow
hardwareMatcherForShow = CliToken.Hardware.hardwareMatcherForShow
aleKwForShow = CliCommand.guardedKeyword( 'ale',
      helpdesc="Ale-specific information",
      guard=CommonGuards.standbyGuard )
vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='Specify the VRF' )

AleL3CliModel = CliDynamicPlugin( "AleL3CliModel" )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname] [ PREFIX ] acknowledgement [all]
# "show ipv6 hardware ale routes [vrf vrfname] [ PREFIX ] acknowledgement [all]
#------------------------------------------------------------------------------------

acknowledgementMatcher = KeywordMatcher(
      "acknowledgement", "Show the route acknowledgement" )

allMatcher = KeywordMatcher(
      "all", "Show all routes" )

class ShowIpHardwareAleAcknowledgementRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware ale routes [ VRF ] [ PREFIX ] ' \
      'acknowledgement [ all ]'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': AleCliLib.v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'PREFIX': AleCliLib.IpPrefixOrAddrMatcher,
            'acknowledgement': acknowledgementMatcher,
            'all': allMatcher,
          }
   cliModel = "AleL3CliModel.AcknowledgementRouteStatus"
   privileged = True
   handler = "AleL3CliHandler.doShowL3AleAcknowledgementRoutes"

BasicCli.addShowCommandClass( ShowIpHardwareAleAcknowledgementRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale adj [vrf vrfName] [fecId] acknowledgement [all]
#------------------------------------------------------------------------------------

class ShowIpHardwareAleAdjAcknowledgementCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj [ VRF ] [ FECID ] acknowledgement [ all ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': AleFibCli.adjMatcher,
            'VRF': vrfExprFactory,
            'FECID': AleFibCli.fecIdMatcher,
            'acknowledgement': 'Show the FEC acknowledgement',
            'all': 'Show all FECs'
          }
   cliModel = "AleL3CliModel.AcknowledgementFecStatus"
   privileged = True
   handler = "AleL3CliHandler.doShowL3AleAdjAcknowledgement"

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjAcknowledgementCmd )

# -----------------------------------------------------------------------------------
# (hidden) "show ip hardware ale errors"
# -----------------------------------------------------------------------------------
class ShowIpHardwareAleErrors( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale errors'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'errors': "Show AleL3Unicast's internally handled error details",
          }
   cliModel = AleL3HandledErrors
   privileged = True
   # This command will be very sentisitve to the internal implementation details
   # of AleL3Unicast. We might add or remove different errors as needed, or the
   # nature in which they are emitted, which will make this command unstable,
   # so it needs to be hidden. Realistically, it will only be used internally by
   # engineering and autotest for the sake of speed and convenience.
   hidden = True

   @staticmethod
   def handler( mode, args ):
      cmd = "AE#"
      return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, model=AleL3HandledErrors )

BasicCli.addShowCommandClass( ShowIpHardwareAleErrors )

# --------------------------------------------------------------------------
# register show tech-support to include Ale command outputs
# --------------------------------------------------------------------------
def _showTechGuard():
   return AleCliLib.vrfRootSmGuard( mode=None, token=None ) is None

TechSupportCli.registerShowTechSupportCmd(
   '2023-03-21 16:12:30',
   cmds=[ 'show ip hardware ale errors' ],
   cmdsGuard=_showTechGuard )
