#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import BasicCliModes
import CliToken
from CliToken.Trace import matcherTrace
from CliToken.Ip import ipMatcherForConfig
from CliToken.Ip import ipMatcherForClear
from CliToken.Hardware import hardwareForConfigMatcher
from CliToken.Clear import clearKwNode
import CliCommand
import CliMatcher
import ConfigMount
from CliPlugin import IpGenAddrMatcher
from CliPlugin.AleFibCli import fecIdMatcher
from CliPlugin.IraNexthopGroupCli import nexthopGroupNameMatcher
from CliPlugin.VrfCli import VrfExprFactory
from Toggles.AleL3UnicastToggleLib import toggleAleSelectiveTracingEnabled
import Tac
from TypeFuture import TacLazyType

startMatcher = CliMatcher.KeywordMatcher(
   'start',
   helpdesc='Start execution command' )
hardwareMatcher = CliToken.Hardware.hardwareForExecMatcher
aleMatcher = CliMatcher.KeywordMatcher(
   'ale',
   helpdesc='Ale-specific commands' )
reviewMatcher = CliMatcher.KeywordMatcher(
   'review',
   helpdesc='Review commands' )
adjMatcher = CliMatcher.KeywordMatcher(
   'adj',
   helpdesc='Adjacency commands' )
l3Matcher = CliMatcher.KeywordMatcher(
   'l3',
   helpdesc='Review L3 adjacencies' )
nexthopGroupKw = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'nexthop-group',
                                      helpdesc='Nexthop group commands' ) )
traceSelectMatcher = CliMatcher.KeywordMatcher(
   'select',
   helpdesc='Selective tracing configurations' )
traceSelectPrefixKwMatcher = CliMatcher.KeywordMatcher(
   'prefix',
   helpdesc='Selective tracing configurations for prefixes' )
traceSelectPrefixMatcher = IpGenAddrMatcher.IpGenPrefixMatcher(
   'IP prefix',
   helpdesc4='IPv4 prefix',
   helpdesc6='IPv6 prefix' )
traceSelectL2AdjKwMatcher = CliMatcher.KeywordMatcher(
   'l2-adj',
   helpdesc='Selective tracing configurations for L2 adjacencies' )
traceSelectAdjIndexMatcher = CliMatcher.IntegerMatcher(
   1,
   0xFFFFFFFF,
   helpdesc='Adjacency index' )
traceSelectAdjKwMatcher = CliMatcher.KeywordMatcher(
   'adj',
   helpdesc='Selective tracing configurations for adjacencies' )
traceSelectFecIdMatcher = CliMatcher.IntegerMatcher(
   1,
   0xFFFFFFFFFFFFFFFF,
   helpdesc='FEC identifier' )
traceSelectTraceSpecKwMatcher = CliMatcher.KeywordMatcher(
   'tracespec',
   helpdesc='Selective tracing configurations for tracespecs' )
traceSelectTraceSpecMatcher = CliMatcher.PatternMatcher(
   pattern='^([a-zA-Z0-9_]+)(,[a-zA-Z0-9_]+)*$',
   helpdesc='Trace specification (comma separated trace handles)',
   helpname='handleA,handleB,handleC,handleD' )
vrfExprFactory = VrfExprFactory( helpdesc='VRF name' )
routeMatcher = CliMatcher.KeywordMatcher(
   'route',
   helpdesc='Review routes'
)
matcherIp = CliMatcher.KeywordMatcher( 'ip', 'IPv4 related' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6', 'IPv6 related' )
singleRouteReviewPrefixMatcher = IpGenAddrMatcher.IpGenPrefixMatcher(
   'IP prefix to review',
   helpdesc4='IPv4 prefix to review',
   helpdesc6='IPv6 prefix to review' )

debugCliConfig = None

__defaultTraceHandle__ = Tracing.Handle( 'AleL3UnicastDebugCli' )
t0 = Tracing.trace0

SelectiveTraceDomain = TacLazyType( 'Ale::SelectiveTraceDomain' )

# --------------------------------------------------------------------------------
# "start hardware ale review adj l3"
# --------------------------------------------------------------------------------
class ForceL3AdjReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review adj l3'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'adj': adjMatcher,
      'l3': l3Matcher,
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceL3AdjReview"

BasicCliModes.EnableMode.addCommandClass( ForceL3AdjReviewCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review adj l3 FECID [ vrf vrfName ]"
# --------------------------------------------------------------------------------
class ForceSingleL3AdjReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review adj l3 FECID [ VRF ]'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'adj': adjMatcher,
      'l3': l3Matcher,
      'FECID': fecIdMatcher,
      'VRF': vrfExprFactory,
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceSingleL3AdjReview"

BasicCliModes.EnableMode.addCommandClass( ForceSingleL3AdjReviewCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review nexthop-group"
# --------------------------------------------------------------------------------
class ForceNhgReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review nexthop-group [ NHG ]'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'nexthop-group': nexthopGroupKw,
      'NHG': nexthopGroupNameMatcher,
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceNhgReview"

BasicCliModes.EnableMode.addCommandClass( ForceNhgReviewCmd )

# --------------------------------------------------------------------------------
# (hidden) "clear ip hardware ale errors"
# --------------------------------------------------------------------------------
class ForceClearInternalErrorsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip hardware ale errors'
   data = {
      'clear': clearKwNode,
      'ip': ipMatcherForClear,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'errors': "Clear AleL3Unicast's internal handled error details",
   }
   # This command is related to the "show ip hardware ale errors" command, which is
   # very sensitive to the implementation of AleL3Unicast. That command is hidden due
   # to this fact. Therefore, this command will also remain hidden, as it will only
   # be used internally by engineering and autotest for the sake of convenience.
   hidden = True
   handler = "AleL3UnicastDebugCliHandler.handleForceClearInternalErrors"

BasicCliModes.EnableMode.addCommandClass( ForceClearInternalErrorsCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select prefix PREFIX"
# --------------------------------------------------------------------------------
class SelectiveTracePrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select prefix PREFIX'
   noOrDefaultSyntax = 'ip hardware ale trace select prefix [ PREFIX ]'
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'prefix': traceSelectPrefixKwMatcher,
      'PREFIX': traceSelectPrefixMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTracePrefix"

   noOrDefaultHandler = \
      "AleL3UnicastDebugCliHandler.handleNoSelectiveTracePrefix"

BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTracePrefixCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select l2-adj ADJ_INDEX"
# --------------------------------------------------------------------------------
class SelectiveTraceL2AdjCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select l2-adj ADJ_INDEX'
   noOrDefaultSyntax = 'ip hardware ale trace select l2-adj [ ADJ_INDEX ]'
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'l2-adj': traceSelectL2AdjKwMatcher,
      'ADJ_INDEX': traceSelectAdjIndexMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTraceL2Adj"

   noOrDefaultHandler = \
      "AleL3UnicastDebugCliHandler.handleNoSelectiveTraceL2Adj"

BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTraceL2AdjCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select adj FECID"
# --------------------------------------------------------------------------------
class SelectiveTraceAdjCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select adj FECID'
   noOrDefaultSyntax = 'ip hardware ale trace select adj [ FECID ]'
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'adj': traceSelectAdjKwMatcher,
      'FECID': traceSelectFecIdMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTraceAdj"

   noOrDefaultHandler = "AleL3UnicastDebugCliHandler.handleNoSelectiveTraceAdj"

BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTraceAdjCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select prefix tracespec TRACESPEC"
# --------------------------------------------------------------------------------
class SelectiveTracePrefixTraceSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select prefix tracespec TRACESPEC'
   noOrDefaultSyntax = syntax.replace( 'TRACESPEC', '...' )
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'prefix': traceSelectPrefixKwMatcher,
      'tracespec': traceSelectTraceSpecKwMatcher,
      'TRACESPEC': traceSelectTraceSpecMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTracePrefixTraceSpec"

   noOrDefaultHandler = \
            "AleL3UnicastDebugCliHandler.handleNoSelectiveTracePrefixTraceSpec"

BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTracePrefixTraceSpecCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select l2-adj tracespec TRACESPEC"
# --------------------------------------------------------------------------------
class SelectiveTraceL2AdjTraceSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select l2-adj tracespec TRACESPEC'
   noOrDefaultSyntax = syntax.replace( 'TRACESPEC', '...' )
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'l2-adj': traceSelectL2AdjKwMatcher,
      'tracespec': traceSelectTraceSpecKwMatcher,
      'TRACESPEC': traceSelectTraceSpecMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTraceL2AdjTraceSpec"

   noOrDefaultHandler = \
            "AleL3UnicastDebugCliHandler.handleNoSelectiveTraceL2AdjTraceSpec"


BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTraceL2AdjTraceSpecCmd )

# --------------------------------------------------------------------------------
# "[ no | default ] ip hardware ale trace select adj tracespec TRACESPEC"
# --------------------------------------------------------------------------------
class SelectiveTraceAdjTraceSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware ale trace select adj tracespec TRACESPEC'
   noOrDefaultSyntax = syntax.replace( 'TRACESPEC', '...' )
   data = {
      'ip': ipMatcherForConfig,
      'hardware': hardwareForConfigMatcher,
      'ale': aleMatcher,
      'trace': matcherTrace,
      'select': traceSelectMatcher,
      'adj': traceSelectAdjKwMatcher,
      'tracespec': traceSelectTraceSpecKwMatcher,
      'TRACESPEC': traceSelectTraceSpecMatcher,
   }

   hidden = not toggleAleSelectiveTracingEnabled()

   handler = "AleL3UnicastDebugCliHandler.handleSelectiveTraceAdjTraceSpec"

   noOrDefaultHandler = \
            "AleL3UnicastDebugCliHandler.handleNoSelectiveTraceAdjTraceSpec"

BasicCliModes.GlobalConfigMode.addCommandClass( SelectiveTraceAdjTraceSpecCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review route ip"
# --------------------------------------------------------------------------------
class ForceIpRouteReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review route ip'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'route': routeMatcher,
      'ip': matcherIp
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceIpRouteReview"

BasicCliModes.EnableMode.addCommandClass( ForceIpRouteReviewCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review route ipv6"
# --------------------------------------------------------------------------------
class ForceIpv6RouteReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review route ipv6'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'route': routeMatcher,
      'ipv6': matcherIpv6
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceIpv6RouteReview"

BasicCliModes.EnableMode.addCommandClass( ForceIpv6RouteReviewCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review route ip PREFIX [ VRF ]"
# --------------------------------------------------------------------------------
class ForceSingleIpRouteReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review route ip PREFIX [ VRF ]'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'route': routeMatcher,
      'ip': matcherIp,
      'PREFIX': singleRouteReviewPrefixMatcher,
      'VRF': vrfExprFactory,
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceSingleIpRouteReview"

BasicCliModes.EnableMode.addCommandClass( ForceSingleIpRouteReviewCmd )

# --------------------------------------------------------------------------------
# "start hardware ale review route ipv6 PREFIX [ VRF ]"
# --------------------------------------------------------------------------------
class ForceSingleIpv6RouteReviewCmd( CliCommand.CliCommandClass ):
   syntax = 'start hardware ale review route ipv6 PREFIX [ VRF ]'
   data = {
      'start': startMatcher,
      'hardware': hardwareMatcher,
      'ale': aleMatcher,
      'review': reviewMatcher,
      'route': routeMatcher,
      'ipv6': matcherIpv6,
      'PREFIX': singleRouteReviewPrefixMatcher,
      'VRF': vrfExprFactory,
   }

   handler = "AleL3UnicastDebugCliHandler.handleForceSingleIpv6RouteReview"

BasicCliModes.EnableMode.addCommandClass( ForceSingleIpv6RouteReviewCmd )

# -------------------------------------------------------------------------------
# Mount the needed sysdb state
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global debugCliConfig
   debugCliConfig = ConfigMount.mount( entityManager,
                                       'hardware/ale/l3/debugcliconfig',
                                       'Ale::DebugCliConfig', 'w' )
