# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import BasicCli
import CliCommand
import CliGlobal
import CliParser
from CliPlugin.AleFibCli import (
   aleKwForShow,
   adjMatcher,
   fecIdMatcher,
   fecIdValid,
   hardwareMatcherForShow,
   ipMatcherForShow,
)
from CliPlugin.AleOcCliModel import AleOpenConfigExport
import ShowCommand
from typing import Any, Optional
import LazyMount

cliGlobal = CliGlobal.CliGlobal( dict(
   ocAftExportFlags=None,
) )

def aleOpenConfigGuard( mode, token ) -> Optional[ str ]:
   if cliGlobal.ocAftExportFlags.ocAftExportEnabled:
      return None
   return CliParser.guardNotThisPlatform

exportNode = CliCommand.guardedKeyword(
   'export',
   helpdesc="Display ale OpenConfig exported FECs",
   guard=aleOpenConfigGuard )

# -----------------------------------------------------------------------------------
# "show ip hardware ale adj [fecId] export
# -----------------------------------------------------------------------------------

def doShowL3AleAdjExport(
      mode: CliParser.Mode, args: dict[ str ] ) -> AleOpenConfigExport:
   fecId = args.get( 'FECID' )
   if fecId and not fecIdValid( fecId ):
      mode.addError( "fecId not valid" )
      return None
   return AgentCommandRequest.runCliPrintSocketCommandWithArgs(
      mode.entityManager, 'AleOpenConfig', 'showFecExport',
      args, mode=mode, model=AleOpenConfigExport )

class ShowIpHardwareAleAdjExportCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj [ FECID ] export'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': adjMatcher,
            'FECID': fecIdMatcher,
            'export': exportNode
          }
   cliModel = AleOpenConfigExport
   privileged = True
   handler = doShowL3AleAdjExport

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjExportCmd )

def Plugin( em: Any ):
   cliGlobal.ocAftExportFlags = LazyMount.mount(
      em, "openconfig/aft/status", "Ale::OcAftExportFlags", 'r' )
