# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pkgdeps: rpm L1Profile-cli

"""
Implementation of the CLI command(s):

   show interfaces [ <interface> ] interactions
"""
import re

import CliPlugin.IntfInteractionReferenceMatcher as RefMatcher
import CliPlugin.XcvrAllStatusDir
from CliPlugin import AlePhyIntfInteractionsModel as IxnModel
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
import Arnet
import BasicCli
import CliCommand
import CliParser
import IntfInteractionsReferenceLib as IxnRef
# pylint: disable-next=consider-using-from-import
import HwL1TopologyComponentLib.IntfSlotTypes as IntfSlotTypes
from HwL1TopologyComponentLib.Errors import HwL1ComponentError
import LazyMount
import Tracing
import Tac

traceHandle = Tracing.Handle( "AlePhyCli" )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

# Declare global names that will be used to refer to mounted paths
ethPhyDefaultCapsSliceDir = None
l1MappingDir = None
l1TopoDir = None
l1TopoRootSliceDir = None
resourceConsumerBaseDir = None
entmibStatus = None
xcvrAllStatusDir = None
portGroupDir = None
hwInvDir = None
l1CardProfileConfigDir = None

# Expose TAC-types with shorter names
EthIntfId = Tac.Type( "Arnet::EthIntfId" )
EthLinkModeSet = Tac.Type( "Interface::EthLinkModeSet" )
PhyScope = Tac.Type( "PhyEee::PhyScope" )
SerdesGroupMode = Tac.Type( "Hardware::Phy::SerdesGroupMode" )
CardConfigDir = Tac.Type( 'L1Profile::CardConfigDir' )
L1ProfileMountConstants = Tac.Type( 'L1Profile::MountConstants' )

#----------------------------------------------------------------------------
# show interfaces [ <interface> ] interactions
#----------------------------------------------------------------------------

fixedProductDenyList = [
   # Temporarily not support cedarville2 (BUG900731)
   re.compile( "DCS-7280SR3A[KM]?-48YC8" ),
   # Temporarily not support topanga (BUG862102)
   re.compile( "DCS-7280CR3A[KM]?-48D6" ),
   # Temporarily not support oliveville (BUG831857)
   re.compile( "DCS-7280CR3A[KM]?-24D12" ),
   # XXX michaelchin: products below here are not expected to support the
   # interactions CLI at this time. This primarily covers Mako systems and
   # J/J+-era Sand systems, which populate serdesResourceConsumers to facilitate
   # the command on Capitola and Nice.
   #
   # There are other products that don't need to support this command (e.g. old
   # Strata systems), but those don't populate serdesResourceConsumers and thus
   # will be guarded against by another check.
   re.compile( "DCS-7020.*" ), # QumranAX (Wheatland, Piedmont, etc.)
   re.compile( r"DCS-713\d.*" ), # Mako fixed-systems
   # The following blurb excludes ALL DCS-7280[CQS]R/R2 except Nice and Capitola
   # (i.e. all other (hopefully) J/J+/Q-MX fixed systems). Uses negative-lookahead
   # to not match against Nice's and Capitola's model numbers or 7280.*R3 SKUs
   # (anything J2 or newer).
   #
   # The negative look ahead on R3 SKUs means that the next generation of Sand
   # products (presumably R4) will be guarded by default. As a follow-up to this,
   # I need to make SandFap only publish serdesResourceConsumers for Nice/Capitola.
   # Then, all older products will be properly guarded by the more generic "any-
   # serdes-resource-consumers" check.
   re.compile( "DCS-(?!7280QRA-C36S|7280QRA?-C72)(?:7280[CQST]R(?!3)).*" ),
   # These are Caravan SKUs which do not support the command.
   # AWN-2600 is the prototype and AWE represents the production switches
   re.compile( "AWN-2600" ),
   re.compile( "AWE-5[35]10" ),
   re.compile( "AWE-72[235]0R" ),
   re.compile( "ZTX-7230F-4TX-4S" ), # CouncilBluffsRuby1
   # Redondo* SKUs are a unique customer-driven SKU and don't need to support
   # interactions. "SKN-" is a unique prefix that applies for specifically all
   # current and future Redondo products.
   re.compile( "SKN-" ),
]
linecardDenyList = [
   # XXX michaelchin: linecards below here don't use L1Topo and are not expected
   # to support the interactions CLI at this time.
   #
   # Same comment about negative lookahead on R3 (see above) applies here.
   re.compile( "7500R(?!3).*-.*-LC" ), # all Jericho/Jericho+ LC
]
switchcardDenyList = [
   re.compile( "7289R3AM?K?-SC" ), # BUG67778[13]: Anaconda switchcard
]

# Products in this list will force its interfaces into the model, 
# even if there are no serdes associated with the interfaces
serdesDomainExceptionList = [
   re.compile( "7388-16CD?" ),
]

def entmibModelName( cardSlot=None ):
   """
   Returns a model name from EntityMib::Status. Desired behavior:

   * For fixed-systems, returns the product model name (e.g. DCS-XXXX-...)
   * For modular systems, if cardSlot not provided, returns the chassis product
     name (e.g. DCS-7304, 7368, etc.)
   * For modular systems, if cardSlot IS provided and cardSlot is populated on
     chassis, return the card's (presumably linecard) product model name
     (e.g. 7368-4D)
   * Else, return the empty string

   Parameters
   ----------
   cardSlot: int, position of card in chassis
   """
   modelName = ""
   # A reactor in EntityMib::Status updates root to correctly point to
   # 'fixedSystem' or 'chassis', depending on the product.
   if entmibStatus.root:
      if cardSlot and entmibStatus.root == entmibStatus.chassis:
         # This is the code path to get the model name of a specific linecard.
         # If the slot isn't populated, we still return "".
         slot = entmibStatus.root.cardSlot.get( cardSlot )
         card = getattr( slot, "card", None )
         modelName = getattr( card, "modelName", "" )
      else:
         # Fall down here to return fixed-system or modular chassis name
         modelName = entmibStatus.root.modelName
   return modelName

def entmibCardSlot( cardSlot=None ):
   return ( entmibStatus.chassis.cardSlot if cardSlot is None 
            else entmibStatus.chassis.cardSlot[ cardSlot ] )

def anySerdesResouceConsumers():
   ''' Returns True if any serdesResourceConsumer collection nested under
   resourceConsumerBaseDir is not empty '''
   for sliceDir in resourceConsumerBaseDir.values():
      for subSliceDir in sliceDir.values():
         # serdesResourceConsumer evaluates to False if it is empty, otherwise True
         if subSliceDir.serdesResourceConsumer:
            return True
   return False

def productNotInDenyList( cardSlot=None ):
   """
   Command is supported under the following conditions:
   * For fixed-systems, product must not match any of the fixedProductDenyList
     regexes
   * For modular-systems, at least one installed linecard does not match any of
     the linecardDenyList regexes. Additionally, no card can match any of the
     switchCardDenyList regexes.

   If cardSlot is provided, for a modular chassis, the specific linecard indicated
   by the argument must not match any of the linecardDenyList entries.

   Note: products that don't populate any serdesResourceConsumers won't be handled
   by this helper function. Those products are excluded via a different check in the
   guard for the command.
   """
   supported = False
   if entmibStatus.fixedSystem:
      modelName = entmibModelName()
      supported = not any( r.search( modelName ) for r in fixedProductDenyList )
   elif entmibStatus.chassis:
      slots = list( entmibStatus.chassis.cardSlot.values() )
      if cardSlot and cardSlot in entmibStatus.chassis.cardSlot:
         slots = [ entmibStatus.chassis.cardSlot[ cardSlot ] ]
      anyLinecardSupported = False
      # Not all modular systems have switchcards. For those, assume command is
      # supported
      switchCardSupported = True
      for slot in slots:
         if not slot.card:
            # No card in this slot; ignore
            continue
         cardName = entmibModelName( slot.relPos )
         if slot.card.tag == "Linecard":
            # If any linecard is not in the deny list, claim support for the command.
            anyLinecardSupported = (
                  anyLinecardSupported or
                  not any( r.search( cardName ) for r in linecardDenyList ) )
         elif slot.card.tag == "Supervisor":
            # This might be a Linecard in Supervisor's clothing! If the card has
            # ports tagged as "Ethernet", treat it as a supported Linecard.
            anyLinecardSupported |= any( i.tag.startswith( "Ethernet" )
                                         for i in slot.card.port.values() )
         elif slot.card.tag == "Switchcard":
            switchCardSupported = (
                  switchCardSupported and
                  not any( r.search( cardName ) for r in switchcardDenyList ) )
      # If the switchcard is unsupported, the command is unsupported, regardless
      # of what linecards are supported.
      supported = anyLinecardSupported and switchCardSupported
   return supported

def intfInExceptionList( intfIds ):
   exceptionIntfs = []
   for intfId in intfIds:
      cardSlot = EthIntfId.module( intfId )
      if cardSlot not in entmibCardSlot():
         continue
      slot = entmibCardSlot( cardSlot )
      if not slot.card:
         # No card in this slot; ignore
         continue
      cardName = entmibModelName( slot.relPos )
      if ( slot.card.tag == "Linecard" and 
           any( r.search( cardName ) for r in serdesDomainExceptionList ) ):
         exceptionIntfs.append( intfId )
   return exceptionIntfs
      
def intfIxnGuard( mode, token ):

   # Interface Interactions CLI is not supported on systems with L1 profiles applied
   l1ProfileCardConfig = l1CardProfileConfigDir.cardConfig
   if any( applied for sliceCfg in l1ProfileCardConfig.values()
           if ( applied := sliceCfg.cardProfile ) and applied.visible ):
      return CliParser.guardNotThisPlatform

   # Infer that this product supports the cmd if an agent has populated any
   # serdesResourceConsumer and is not in the product deny list.
   if anySerdesResouceConsumers() and productNotInDenyList():
      return None
   return CliParser.guardNotThisPlatform

def getSerdesResourceConsumer( intfId ):
   # Although it is inefficient in most cases to search the entire world for each
   # interface, identifying the correct slicified path to search under isn't as
   # simple as identifying the linecard the interface exists on, particularly on
   # Glacier-like systems.
   #
   # In those cases, interface objects are populated under a SwitchcardX slice, not
   # a LinecardX path (e.g. object for Et9/3/1 would be found under the Switchard1
   # slice). While we can write logic to work around that, I think we can live with
   # the current solution until it becomes an actual performance issue.
   for sliceDir in resourceConsumerBaseDir.values():
      for subSliceDir in sliceDir.values():
         src = subSliceDir.serdesResourceConsumer.get( intfId )
         if src:
            return src
   t1( "SerdesResourceConsumer not found for", intfId )
   return None

def getSpeedGroupName( intfId ):
   for sliceDir in resourceConsumerBaseDir.values():
      for subSliceDir in sliceDir.values():
         if subSliceDir.speedGroupStatusDir:
            sgName = subSliceDir.speedGroupStatusDir.intfSpeedGroup.get( intfId )
            if sgName:
               return sgName 
   t1( "Speed-group not found for", intfId )
   return None

def getLogicalPortPoolDesc( intfId ):
   for sliceDir in resourceConsumerBaseDir.values():
      for subSliceDir in sliceDir.values():
         if subSliceDir.logicalPortPoolDescDir:
            poolId = subSliceDir.logicalPortPoolDescDir.poolIdByIntf.get( intfId )
            if poolId is not None:  # poolId can legitimately be 0
               lppDesc = subSliceDir.logicalPortPoolDescDir.pool.get( poolId )
               if lppDesc:
                  return lppDesc
   t1( "Logical port pool description not found for", intfId )
   return None

def getHardwarePortGroup( intfId ):
   if not portGroupSupported():
      return None
   # Generate a dict mapping an interface name to its port-group id
   intfPortGroup = {}
   for portGroup in portGroupDir.portGroup.values():
      portGroupId = portGroup.name
      for mode in portGroup.mode.values():
         for port in mode.ports:
            intfPortGroup[ port ] = portGroupId
   if intfId not in intfPortGroup:
      return None
   t1( "Hardware port-group", intfPortGroup[ intfId ], "found for", intfId )
   return intfPortGroup[ intfId ]

def getHardwarePortMode( portGroupId ):
   hardwarePortModes = []
   for mode in portGroupDir.portGroup[ portGroupId ].mode.values():
      hardwarePortModes.append( mode.name )
   t1( "Hardware port modes", hardwarePortModes,
       "found for port-group", portGroupId )
   return hardwarePortModes

def getBlackhawkAutoFecRestrictions( intfId ):
   for sliceDir in resourceConsumerBaseDir.values():
      for subSliceDir in sliceDir.values():
         if subSliceDir.blackhawkRestrictions:
            if subSliceDir.blackhawkRestrictions.get( intfId ):
               return subSliceDir.blackhawkRestrictions.get( intfId )
   t1( "Blackhawk restriction not found for", intfId )
   return None

def findXcvrSlotDir( slotNum ):
   xcvrSlotDir = None
   modularSystem = hwInvDir.get( "modularSystem" )
   if modularSystem:
      card = modularSystem.card.get( slotNum )
      if not card:
         return None
      lcInv = card.component.get( "norcal" )
      if not lcInv:
         return None
      elif isinstance( lcInv, Tac.Type( "Inventory::NorCalSupervisor" ) ):
         # xcvrSlotDir on supervisor linecards appear down a different tree. Start
         # from the last common point and work downwards.
         card = modularSystem.supeUplinkCard.get( slotNum )
         if not card:
            return None
         lcInv = card.component[ "norcal" ]
         xcvrSlotDir = lcInv.mainPowerDomain[ "xcvrSlotDir" ]
      else:
         xcvrSlotDir = lcInv.mainPowerDomain.get( "xcvrSlotDir" )
   # Presumably, "modularSystem" and "fixedSystem" are mutually exclusive under
   # hwInvDir, so this if-block can't clobber the other.
   fixedSystem = hwInvDir.get( "fixedSystem" )
   if fixedSystem:
      xcvrSlotDir = fixedSystem.component.get( "xcvrSlotDir" )
   return xcvrSlotDir

def portGroupSupported():
   """
   Indicate whether a system supports the "hardware port-group" CLI command
   """
   if entmibStatus.fixedSystem:
      for portGroup in portGroupDir.portGroup.values():
         if portGroup.defaultMode:
            return True
   return False

def fetchSingleSlotIntfMap( intfSet ):
   """
   Helper function. Given a set of intfIds, return a mapping in the form
   { "lane%d" : intfId } where "lane%d" is 1-indexed for multi-lane slots
   and 0-indexed for single-lane slots.

   Parameters
   ----------
   intfSet : set of IntfId string representations

   Returns
   -------
   dict of { "lane%d" : intfId } as described above.
   """
   # We can arbitrarily act on the first intf in intfSet, since we are
   # using the intf to find xcvrSlot information, and we've already
   # determined that all intfs in this intfSet are from the same xcvrSlot.
   intf = list( intfSet )[ 0 ]
   # Fetch xcvrSlotDir inventory. EthIntfId.module returns 0 for fixed-
   # systems, but note that findXcvrSlotDir doesn't use its argument if run
   # on a fixed-system.
   cardSlot = EthIntfId.module( intf )
   useFallbackMapping = False
   xcvrSlotDir = findXcvrSlotDir( cardSlot )
   if xcvrSlotDir:
      invXcvrSlot = xcvrSlotDir.xcvrSlot.get( EthIntfId.port( intf ) )
      xcvrLaneToIntfIdMap = { k: v.intfId
                              for k, v in invXcvrSlot.port.items() }

      # Find the number of host-lanes that would exist on a normal
      # version of this port type. Infer number of host-lanes based on
      # trace-length collection. Number of host-lanes is important for
      # the templates downstream to deal with port-renumbered intfs.
      numHostLanes = 0
      if invXcvrSlot.slotType == 'qsfp':
         numHostLanes = 4
      else:
         try:
            slotType = IntfSlotTypes.getIntfSlotType( invXcvrSlot.slotType )()
            numHostLanes = slotType.getAllInterfaceSlotLanes()
         except HwL1ComponentError:
            # For ports that didn't find a slot type, fall-back to the old
            # computation.
            t0( "Slot-type not found: ", invXcvrSlot.fullName )
            useFallbackMapping = True

      # In my infinite wisdom, previous-me made templates based off of
      # { "laneX": intfId } mappings, where "X" is 1-indexed lane IDs,
      # except for single-lane ports (like SFPs) where "X" is 0. The
      # following section rebuilds that for this approach of generating
      # lane-to-intf mappings.
      intfDict = {}
      if EthIntfId.lane( intf ) == 0:
         intfDict = { "lane0": intf }
      else:
         for i in range( numHostLanes ):
            # If not every host-lane is associated with an intfId (e.g.
            # depopulated ports or port-renumbered intfs), map it to None.
            # Downstream templates will handle the holes in intfIds.
            intfIdOrNone = xcvrLaneToIntfIdMap.get( i )
            intfDict[ "lane%d" % ( i + 1 ) ] = intfIdOrNone
   else:
      # Most btest testlets don't build out xcvrSlotDir and can use the
      # fallback mapping just fine.
      useFallbackMapping = True

   if useFallbackMapping:
      t0( "Failed to generate serdesDomain via XcvrSlotDir; fall back to "
          "name-based hard mapping" )
      # Multi-lane ports will have their members named by lane. Single-lane
      # ports will have its member named "lane0"
      intfDict = { "lane%d" % EthIntfId.lane( intf ): intf
                   for intf in intfSet }

   return intfDict

def generateSerdesIntfSets( intfs ):
   """
   Helper function to generate sets of interfaces that share serdes with each other

   Parameters
   ----------
   intfs : list of EthIntfCli.EthPhyIntf-based objects

   Returns
   -------
   list, where members are sets of IntfId string representations
   dict, mapping front-panel port number to superset of serdes used by the port

   Note
   ----
   The second return argument is used in generateSerdesDomains for multi-slot
   groupings to determine primary-secondary relationships. There's potential
   for that block to move into its own function, which may require a cache to
   avoid running that computation twice during the same command instance.
   """
   # First, create a mapping of front-panel port number to the superset of
   # serdes used by the port (typically switch-chip serdes). Also map the
   # port number to the list of member intfIds, which will be used later.
   portToSerdesSuperset = {}
   portToIntfIds = {}
   for intf in intfs:
      intfId = intf.name
      src = getSerdesResourceConsumer( intfId )
      if not src:
         continue
      port = EthIntfId.port( intfId )
      portToIntfIds.setdefault( port, [] ).append( intfId )
      serdesSuperset = portToSerdesSuperset.setdefault( port, set() )
      for sg in src.serdesGroup.values():
         serdesSuperset.update( sg.serdesId )

   # Next, invert portToSerdesSuperset to map individual serdes IDs to the
   # ports that use them.
   serdesIdToPorts = {}
   for port, serdesSet in portToSerdesSuperset.items():
      for serdes in serdesSet:
         serdesIdToPorts.setdefault( serdes, set() ).add( port )

   # Finally, reverse-sort the previous map by the length of the values. The
   # group of ports that make up a serdes domain will move to the front of the
   # iteration.
   portToSerdesDomain = {}
   intfSets = []
   for portSet in sorted( serdesIdToPorts.values(), key=len, reverse=True ):
      if all( portId in portToSerdesDomain for portId in portSet ):
         # Most of the ports will be processed in the beginning of the loop
         # because of how the entries are sorted. No need to process them again.
         #
         # BUG478057: there is an assumption here that each serdes-domain contains
         # a serdes that is used by all physical ports that are members of the
         # serdes-domain. If that assumption is broken, we need to revisit this.
         continue
      intfSet = set()
      # The following two loops must be separate because the intfSet is not
      # finalized until each port in the portSet has been processed. The
      # serdesDomain list must contain only finalized serdes domains.
      for portId in portSet:
         intfSet.update( portToIntfIds[ portId ] )
      for portId in portSet:
         portToSerdesDomain[ portId ] = intfSet
      # Happens outside the loop because we only need to register each intfSet once
      intfSets.append( intfSet )
   
   # We still add intfs to intfSets if they are on the exceptionList
   for port, serdesSet in portToSerdesSuperset.items():
      if not serdesSet:
         intfSet = set()
         intfSet.update( intfInExceptionList( portToIntfIds[ port ] ) )
         intfSets.append( intfSet )

   return intfSets, portToSerdesSuperset

def generateSerdesDomains( intfs ):
   """
   Generates a list of SerdesDomainBase-derived objects. A "serdes domain" is
   defined as a set of interfaces that share a set of serdes (typically switch-
   chip serdes).

   Returns
   -------
   serdesDomains: list of SerdesDomainBase-derived objects

   Note
   ----
   I found this to be a deceptively complex problem due to cases where a serdes
   domain can span multiple front-panel ports. I don't consider my approach to
   be efficient by any means, but I believe it to be correct. I'll leave the
   task of optimization to someone smarter than me...
   """
   # Group interfaces on the slice into groups of interfaces that share serdes
   intfSets, portToSerdesSuperset = generateSerdesIntfSets( intfs )

   # Construct a SerdesDomain object from each serdes domain. Each interface in
   # the domain must be "named" correctly to be understood correctly by the
   # IntfInteractionReference objects.
   serdesDomains = []
   for domain in intfSets:
      serdesDomain = None
      numPorts = len( { EthIntfId.port( intf ) for intf in domain } )
      if numPorts == 1:
         intfDict = fetchSingleSlotIntfMap( domain )
         serdesDomain = IxnRef.SerdesDomain( intfDict )
      elif numPorts == 2:
         # For now, dual port is the only multi-port domain we have. Need
         # to determine the primary/secondary port relationship. Total number
         # of serdes used should be a decent heuristic to deterimine a primary port.
         ports = sorted( { EthIntfId.port( i ) for i in domain } )
         if( len( portToSerdesSuperset[ ports[ 0 ] ] ) >=
             len( portToSerdesSuperset[ ports[ 1 ] ] ) ):
            # The usage of >= instead of strictly > is to handle the case where both
            # ports use the same number of serdes. I'm assuming that in such a case
            # we would define the primary port to be the one with a lower power
            # number. 'ports' is sorted, so the lower number should be guaranteed to
            # be first.
            primary, secondary = ports
         else:
            # Otherwise, the latter port uses more serdes and is likely the primary 
            secondary, primary = ports
         intfDict = { "primary%d" % EthIntfId.lane( intf ): intf
                      for intf in domain if EthIntfId.port( intf ) == primary }
         intfDict.update( { "secondary%d" % EthIntfId.lane( intf ): intf
                            for intf in domain
                            if EthIntfId.port( intf ) == secondary } )
         assert len( intfDict ) == len( domain )
         serdesDomain = IxnRef.SerdesDomain( intfDict )
      else:
         # Consciously do not add a serdes domain to avoid presenting bad input
         # to downstream code. We might still be able to generate useful
         # information if we keep going instead of asserting here.
         t1( "Unhandled number of ports for serdes domain: intfIds =", domain )
      if serdesDomain:
         # If this is defined, we were able to classify the serdesDomain
         serdesDomains.append( serdesDomain )
         t1( "Created SerdesDomain: memberIntfs =", Arnet.sortIntf( domain ) )
   t1( "Num serdes domains:", len( serdesDomains ) )
   return serdesDomains

def getL1TopoPhys( topoHelper, intfId ):
   module = EthIntfId.module( intfId )
   port = EthIntfId.port( intfId )
   lane = EthIntfId.lane( intfId )
   if lane > 0:
      # IntfId lanes start at 1, but topology lanes start at 0. EthIntfId.lane
      # returns the intfId lane or 0 if the intfId doesn't have a lane. This
      # means we want to subtract 1 from lane unless EthIntfId.lane returns 0.
      # Not sure if there is a way to do this that feels less hacky.
      lane -= 1
   sliceId = "FixedSystem"
   if module:
      sliceId = "Linecard%d" % module
   # Seems like the convention is to trace the TX path (that's what the boolean
   # argument represents).
   xcvrLaneDesc = Tac.Value( "Hardware::L1Topology::XcvrLaneDescriptor",
                             sliceId, port, lane, True )
   xcvrTopo = topoHelper.xcvrHelper.xcvrLaneTopology( xcvrLaneDesc )
   intfTopo = topoHelper.intfHelper.intfTopology( sliceId, intfId )

   # If L1 topology exists for this interface, take the union of phys used at
   # any configuration of the interface, in case they might be different.
   if xcvrTopo and intfTopo:
      return set( topoHelper.getPhyTypesForIntfId( intfId ).phyType )
   return set()

def identifyXcvrSlotType( intfId ):
   xcvrSlot = ""
   # BUG474967: need to define behavior for when xcvrAllStatusDir.xcvrStatus
   # is empty (e.g. early start-up). Will address this along with btests for
   # this implementation.
   module = EthIntfId.module( intfId )
   port = EthIntfId.port( intfId )
   # XXX michaelchin: XcvrLib.getXcvrSlotName does this, but in a way that I
   # disagree with, so writing my own.
   xcvrSlotName = ( "Ethernet%d/%d" % ( module, port )
                    if module else "Ethernet%d" % port )
   xStatus = xcvrAllStatusDir.xcvrStatus.get( xcvrSlotName )
   if xStatus:
      # We are looking for physical slot form-factor, so we can take xcvrType
      # as is (i.e. don't need to look at swizzled xcvrStatus)
      xcvrSlot = xStatus.xcvrType
   return xcvrSlot

def showInterfacesInteractions( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   # Get all intfs since we have to consider the relationships between all
   # interfaces.
   allIntfs = IntfCli.Intf.getAll( mode, None, mod, EthIntfCli.EthPhyIntf,
                                   exposeInactive=True )
   intfIdToEthPhyIntfObj = { intf.name: intf for intf in allIntfs }
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthPhyIntf,
                                exposeInactive=True )
   displayIntfNames = [ intf.name for intf in intfs ]
   t0( "Running `show interfaces interactions` for intfs:", intfs )
   model = IxnModel.InterfacesInteractions()

   # Create L1 Topo traversal helper to help identify criteria used to match
   # against interaction templates. Use globally-scoped traversal helper instead
   # of managing one helper per slice
   topoDir = LazyMount.force( l1TopoDir )
   mappingDir = LazyMount.force( l1MappingDir )
   topoHelper = Tac.newInstance( "Hardware::L1Topology::TraversalHelper",
                                 topoDir, mappingDir )
   # We will want to cache whether or not a particular linecard on a modular
   # system supports interactions. There's a helper method for it, but it
   # does O(N) regex matches, so it'd be nice to reduce the number of times
   # we have to call it.
   moduleSupportsIxns = {}
   # Identify serdes domains (i.e. interfaces that share a set of switch-chip
   # serdes). They will be used to match interfaces to the appropriate
   # interaction reference template. SerdesDomains must be computed per slice.
   # Once the serdes-domains have been created, they can be operated on
   # without regards to slices.
   intfsBySlice = {}
   for intf in allIntfs:
      intfId = intf.name
      module = EthIntfId.module( intfId )
      # 'module' is either 0 for fixed systems or a card slot for modular systems.
      # If a particular linecard does not support the interactions CLI, do not
      # attempt to generate interactions for it.
      if module not in moduleSupportsIxns:
         moduleSupportsIxns[ module ] = productNotInDenyList( module )
      if moduleSupportsIxns.get( module ):
         intfsBySlice.setdefault( module, [] ).append( intf )
   serdesDomains = []
   for intfs in intfsBySlice.values():
      serdesDomains.extend( generateSerdesDomains( intfs ) )

   # For each serdes domain, identify criteria that would be used to match against
   # a template.
   for serdesDomain in serdesDomains: 
      # First identify phys on interface via L1Topo. Use the first memberIntf as
      # a base-line.
      #
      # There is an assumption here that all intfIds in a serdes domain will pass
      # through the same phys as the others in the domain. If this assumption is
      # broken, we will have to loop over all intfIds in the domain and use the
      # union of all of their phys.
      domainIntf = next( iter( serdesDomain.notNoneIntfs() ) )
      groupPhys = getL1TopoPhys( topoHelper, domainIntf )

      # Next identify the xcvr slot type. Between this and groupPhys, we can
      # correctly match against a majority of the port types on our products.
      xcvrSlot = identifyXcvrSlotType( domainIntf )

      # Gather the remaining information that may be used to refine a reference match
      # productModel is expected to show the chassis model for a fixed system
      # and the linecard model for a modular system.
      productModel = entmibModelName( EthIntfId.module( domainIntf ) )
      portRange = set( map( EthIntfId.port, serdesDomain.notNoneIntfs() ) )
      refId = RefMatcher.InteractionReferenceId( groupPhys, xcvrSlot,
                                                 productModel, portRange )
      template = RefMatcher.findMatch( refId )
      if not template:
         continue
      # There is a risk of an internal error here if the serdesDomain is not
      # compatible with the reference template. This could happen if there is
      # a mistake in the matcher's registry. The pythonic thing to do would
      # be to wrap this in a try-except -- I'll leave that as a TODO when there
      # is a test that could exercise it (BUG479133).
      ixnRef = template( serdesDomain )

      # Populate optional parameters in ixnRef before generating the model
      # For capabilities, take the union of capabilities of all intfs in the
      # serdes domain
      domainCapabilities = EthLinkModeSet()
      for intfId in serdesDomain.notNoneIntfs():
         intfCaps = intfIdToEthPhyIntfObj[ intfId ].ethPhyDefaultCaps()
         domainCapabilities |= intfCaps.linkModeCapabilities
      ixnRef.groupLinkModes = domainCapabilities
      # For speed-groups, if the intf is associated with a speed-group name
      # according to SpeedGroupStatusDir, use that name.
      speedGroupName = getSpeedGroupName( domainIntf )
      if speedGroupName:
         ixnRef.speedGroupName = speedGroupName
      # For logical ports, if the intf is associated with a logical port pool
      # description, use the description to populate the reference.
      lppDesc = getLogicalPortPoolDesc( domainIntf )
      if lppDesc:
         # lppDesc.memberIntf is a set-void collection; reduce that to something
         # that easier for the rest of the command to work with.
         desc = IxnRef.LogicalPortInteractionDesc( lppDesc.poolId,
                                                   list( lppDesc.memberIntf ),
                                                   lppDesc.maxPorts )
         ixnRef.logicalPortPoolDesc = desc

      for intfId in serdesDomain.notNoneIntfs():
         blackhawkRestriction = getBlackhawkAutoFecRestrictions( intfId )
         if blackhawkRestriction:
            ixnRef.blackhawkAutoFecRequirement[ intfId ] = \
               blackhawkRestriction.primaryIntf

      # For hardware port-groups, if the intf is associated with a hardware
      # port-group name, use that name.
      hardwarePortGroup = getHardwarePortGroup( domainIntf )
      if hardwarePortGroup:
         ixnRef.hardwarePortGroup = hardwarePortGroup
         ixnRef.hardwarePortModes = getHardwarePortMode( hardwarePortGroup )

      # This is terribly inefficient, but its correct. Given this current
      # architecture, we would have to only loop over the serdes domains that
      # are relevant to the interfaces we want to display.
      model.interfaces.update( { k: v for k, v in ixnRef.model().interfaces.items()
                                 if k in displayIntfNames } )
   return model

# Register Cli command
interactionsKw = CliCommand.guardedKeyword(
   "interactions",
   "Show interactions with other Ethernet interfaces",
   intfIxnGuard )

class ShowIntfInteractions( IntfCli.ShowIntfCommand ):
   syntax = "show interfaces interactions"
   data = dict( interactions=interactionsKw )
   cliModel = IxnModel.InterfacesInteractions
   handler = showInterfacesInteractions
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfInteractions )

#----------------------------------------------------------------------------
# Plugin
#----------------------------------------------------------------------------
def Plugin( entityManager ):
   global ethPhyDefaultCapsSliceDir
   global l1MappingDir
   global l1TopoDir
   global l1TopoRootSliceDir
   global resourceConsumerBaseDir
   global xcvrAllStatusDir
   global portGroupDir
   global hwInvDir
   global l1CardProfileConfigDir

   ethPhyDefaultCapsSliceDir = LazyMount.mount( entityManager,
                        "interface/archer/status/eth/phy/capabilities/default/slice",
                        "Tac::Dir", "ri" )
   l1MappingDir = LazyMount.mount( entityManager,
                                   "hardware/l1/mapping", "Tac::Dir", "ri" )
   l1TopoDir = LazyMount.mount( entityManager,
                                "hardware/l1/topology", "Tac::Dir", "ri" )
   l1TopoRootSliceDir = LazyMount.mount( entityManager,
                                         "hardware/l1/topology/slice",
                                         "Tac::Dir", "ri" )
   resourceConsumerBaseDir = LazyMount.mount( entityManager,
                                              "interface/resources/consumers/slice",
                                              "Tac::Dir", "ri" )
   xcvrAllStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( entityManager )
   portGroupDir = LazyMount.mount( entityManager,
                                   'hardware/ale/portGroup',
                                   'Ale::PortGroupDir', 'ri' )
   hwInvDir = LazyMount.mount( entityManager, "hardware/inventory",
                               "Tac::Dir", "ri" )

   # BUG290864
   global entmibStatus
   entmibStatus = LazyMount.mount( entityManager, "hardware/entmib",
                                   "EntityMib::Status", "r" )

   l1CardProfileConfigDir = LazyMount.mount(
      entityManager,
      L1ProfileMountConstants.cardConfigPath(),
      CardConfigDir.tacType.fullTypeName,
      'r' )
