# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
"""
`show interfaces interactions` models
"""
import Arnet
import EthIntfLib
from EthIntfLib.Types import EthSpeed, EthSpeedApi

from CliModel import Bool, Dict, Enum, Int, List, Model, Str, Submodel
from IntfModels import Interface
from Intf.IntfRange import intfListToCanonical

# Declaring string used to indicate no interactions
NO_IXNS = "No interactions with other interfaces"
# Declaring 10G asterisk message. For simplicitly, we will display this line
# whenever we display interactions of any kind, regardless if any of the
# interfaces are <=10G capable.
ASTERISK_10G = "* = includes speeds slower than 10G that the interface is capable of"

#-------------------------------------------------------------------------------
# Helper functions
#-------------------------------------------------------------------------------
def _printIndent( indent, value ):
   """
   Helper function to print lines with an indent

   Parameters
   ----------
   indent : int, number of spaces to print before the message
   value : string, the message
   """
   print( " " * indent + value )
_print2 = lambda value : _printIndent( 2, value )
_print4 = lambda value : _printIndent( 4, value )
_print6 = lambda value : _printIndent( 6, value )

def _combineIntfNames( intfs, noHoleRange=False ):
   """
   Helper function that uses the CLI's interface range parser to combine intf
   names. Returns the long name of the interface range or None.

   Parameters
   ----------
   intf : list of interface names as strings
   """
   intfLists = intfListToCanonical( intfs,
                                    noHoleRange=noHoleRange,
                                    verifyIntfHoles=noHoleRange,
                                    shortName=False )

   if intfLists:
      assert len( intfLists ) == 1, "expected only one interface type"
      return intfLists[ 0 ]

   return None

#-------------------------------------------------------------------------------
# `show interfaces interactions` models
#-------------------------------------------------------------------------------

# Derives speed strings (e.g. 2.5G, 400G, 10M, etc.) from EthSpeed type. 'replace'
# statement handles speed2p5Gbps, since TACC enums don't contain '.'s
speedStrEnum = [ speedVal.strip( "spedb" ).replace( "p", "." )
                 for speedVal in EthSpeed.attributes
                 if EthSpeedApi.isIeeeSpeed( speedVal ) and
                    ( speedVal != EthSpeed.speedUnknown ) ]

class SpeedConfiguration( Model ):
   speed = Enum( values=speedStrEnum, help="Configuration speed" )
   lanes = Int( optional=True, help=( "Number of electrical lanes used on the "
                                      "host-side of the interface" ) )

   def populateFromStr( self, cfg ):
      speed = lanes = None
      if "-" in cfg:
         speed, lanes = cfg.split( "-" )
      else:
         speed = cfg
      self.speed = speed
      if lanes:
         self.lanes = int( lanes )

   def toStr( self ):
      strep = self.speed
      if self.lanes:
         strep += "-%d" % self.lanes # pylint: disable=consider-using-f-string
      return strep

class SpeedConfigurationList( Model ):
   configurations = List( valueType=SpeedConfiguration,
                          help="List of speed configurations" )

   def toStr( self ):
      cfgStrs = [ config.toStr() for config in self.configurations ]
      orderedSgkStrs = list( EthIntfLib.speedLanestoSpeedLanesStr.values() )
      return "/".join( sgk for sgk in orderedSgkStrs if sgk in cfgStrs )

class InactiveInterfacesInteraction( Model ):
   interfaces = List( valueType=Interface, help="Interfaces that become inactive" )

   def render( self ):
      ifRange = _combineIntfNames( self.interfaces )
      if len( self.interfaces ) == 1:
         fmt = "%s becomes inactive"
      else:
         fmt = "%s become inactive"
      _print4( fmt % ifRange )

class SpeedLimitedInterfacesInteraction( Model ):
   interfaces = Dict( keyType=Interface, valueType=str,
                      help=( "Mapping of interface name to what speed "
                             "configurations it becomes limited to. Speed "
                             "configurations are represented in the notation "
                             "<speed> or <speed>-<lane>. <speed> describes the "
                             "interface speed, and <lane> if specified describes "
                             "the number of electrical lanes used on the host-"
                             "side of the transceiver." ) )
   def render( self ):
      speedBuckets = {}
      for intf, speedStr in self.interfaces.items():
         if speedStr not in speedBuckets:
            speedBuckets[ speedStr ] = []
         speedBuckets[ speedStr ].append( intf )
      for speed, intfs in speedBuckets.items():
         ifRange = _combineIntfNames( intfs )
         if len( intfs ) == 1:
            fmt = "%s is limited to %s"
         else:
            fmt = "%s are limited to %s"
         _print4( fmt % ( ifRange, speed ) )

class PrimaryInterfaceConfigurationRequirement( Model ):
   interface = Interface( help="Name of primary interface" )
   requiredConfiguration = Enum(
         values=list( EthIntfLib.speedLanestoSpeedLanesStr.values() ),
         help="Speed configuration required on primary interface. Speed "
              "configuration is given in the notation <speed> or <speed>-<lane>. "
              "<speed> describes the interface speed, and <lane> if specified "
              "describes the number of electrical lanes used on the host-side of "
              "the transceiver." )

   def render( self ):
      fmt = "Primary interface %s must be operating at %s"
      _print4( fmt % ( self.interface.stringValue, self.requiredConfiguration ) )

class CompatibleParentInterfaceConfigurations( Model ):
   interfaces = Dict( keyType=Interface, valueType=SpeedConfigurationList,
                      help=( "Mapping of interface name to a slash-separated list "
                             "of its speed configurations that do not conflict "
                             "with the desired configuration on the desired "
                             "interface." ) )
   def render( self ):
      fmt = "%s must be operating at %s"
      for intf in Arnet.sortIntf( self.interfaces ):
         configs = self.interfaces[ intf ]
         _print4( fmt % ( intf, configs.toStr() ) )

class SpeedGroupRequirement( Model ):
   # Regarding typing: despite being a "number", speed-group identifiers are
   # implemented as Tac::Name since they can contain "/"s (hence using Str here)
   speedGroup = Str( help="Speed-group the referred interface is in" )
   serdes = Str( help="Speed-group SerDes configuration needed for the referred "
                      "interface speed to take effect" )

   def render( self ):
      fmt = "Hardware speed-group %s must include %s"
      _print4( fmt % ( self.speedGroup, self.serdes ) )

class HardwarePortGroupRequirement( Model ):
   # Regarding typing: despite being a "number", port-group identifiers are
   # implemented as Tac::String (hence using Str here)
   portGroup = Str( help="Port-group the referred interface is in" )
   portMode = Str( help="Port-group mode needed for the referred "
                        "interface speed to take effect" )

   def render( self ):
      fmt = "Hardware port-group %s must be %s"
      _print4( fmt % ( self.portGroup, self.portMode ) )

class AutonegFecRequirement( Model ):
   primaryIntf = Interface( help="Name of the primary interface" )

   def render( self ):
      fmt = ( "%s must match error-correction encoding configuration for speed "
              "auto 100G-4" )
      _print4( "For speed auto 100G-4" )
      _print6( fmt % self.primaryIntf.stringValue )

class InterfacesInteractionsData( Model ):
   inactiveInterfaces = Submodel(
      valueType=InactiveInterfacesInteraction, optional=True,
      help=( "List of interfaces that become inactive to facilitate desired "
             "speed configuration" ) )
   speedLimitedInterfaces = Submodel(
      valueType=SpeedLimitedInterfacesInteraction, optional=True,
      help=( "List of interfaces that become speed-limited and what speeds "
             "they are limited to to facilitate desired speed configuration" ) )
   primaryInterfaceConfigurationRequirement = Submodel(
      valueType=PrimaryInterfaceConfigurationRequirement, optional=True,
      help=( "Speed configuration required on primary interface to facilitate "
             "desired speed configuration" ) )
   compatibleParentInterfaceConfigurations = Submodel(
      valueType=CompatibleParentInterfaceConfigurations, optional=True,
      help=( "List of interfaces that can affect desired configuration and "
             "which of their configurations are compatible with desired "
             "configuration" ) )
   autonegRequirement = Submodel(
      valueType=AutonegFecRequirement, optional=True,
      help=( "Error-correct encoding restriction when auto-negotation is "
             "configured" ) )
   speedGroupRequirement = Submodel(
      valueType=SpeedGroupRequirement, optional=True,
      help=( "Speed-group configuration required to achieve stated speed "
             "configuration" ) )
   hardwarePortGroupRequirement = Submodel(
      valueType=HardwarePortGroupRequirement, optional=True,
      help=( "Hardware port-group configuration required to achieve stated speed "
             "configuration" ) )
   # New types of interactions should be added as optional attributes here

   def empty( self ):
      return not self.toDict()

   def render( self ):
      if self.empty():
         _print4( NO_IXNS )
         return
      if self.inactiveInterfaces:
         self.inactiveInterfaces.render()
      if self.speedLimitedInterfaces:
         self.speedLimitedInterfaces.render()
      if self.primaryInterfaceConfigurationRequirement:
         self.primaryInterfaceConfigurationRequirement.render()
      if self.compatibleParentInterfaceConfigurations:
         self.compatibleParentInterfaceConfigurations.render()
      if self.speedGroupRequirement:
         self.speedGroupRequirement.render()
      if self.autonegRequirement:
         self.autonegRequirement.render()
      if self.hardwarePortGroupRequirement:
         self.hardwarePortGroupRequirement.render()

class InterfaceHardwareResourcePool( Model ):
   poolId = Int( help="Name of the resource pool" )
   memberInterfaces = List( valueType=Interface,
                            help="Interfaces belonging to this resource pool" )
   capacity = Int( help=( "Maximum number of interface hardware resources available "
                          "in the pool" ) )

   def render( self ):
      ifRange = _combineIntfNames( self.memberInterfaces, noHoleRange=True )
      # pylint: disable-next=bad-string-format-type,consider-using-f-string
      _print2( "%s share %d logical ports in pool %d" % ( ifRange,
                                                          self.capacity,
                                                          self.poolId ) )

class InterfacesInteractionsDataBySpeed( Model ):
   # XXX michaelchin: keyType is a string because at the moment we are unable
   # to use an enum.
   speeds = Dict( keyType=str,
                  valueType=InterfacesInteractionsData,
                  help=( "A mapping of desired speed configurations to interactions "
                         "with other interfaces. Speed configurations are "
                         "represented in the notation <speed> or <speed>-<lane>. "
                         "<speed> describes the interface speed and, <lane> if "
                         "specified describes the number of electrical lanes used "
                         "on the host-side of the transceiver." ) )
   interfaceHardwareResourcePool= Submodel(
      valueType=InterfaceHardwareResourcePool, optional=True,
      help="Interface hardware resource pool that this interface belongs to" )
   inactiveReason = Str( default=None, optional=True,
                         help="Reason why the interface is inactive "
                              "due to having no available configurations" )
   # Following bool used to solely to modify render under certain conditions, thus
   # private.
   _includesSlowerSpeeds10g = Bool( help="Describes if the interface's 10G "
                                         "interactions also apply to its slower-"
                                         "than-10G speeds" )

   def empty( self ):
      speedsEmpty = self.speeds and all( v.empty() for v in
                                         self.speeds.values() )
      noHwResourcePool = not self.interfaceHardwareResourcePool
      return speedsEmpty and noHwResourcePool

   def includesSlowerSpeeds10gIs( self, value ):
      self._includesSlowerSpeeds10g = value

   def render( self ):
      if not self.speeds and self.inactiveReason:
         _print2( f"Inactive due to {self.inactiveReason}" )
         return
      if self.empty():
         _print2( NO_IXNS )
         return
      if self.interfaceHardwareResourcePool:
         self.interfaceHardwareResourcePool.render()
      for speed in EthIntfLib.speedLanestoSpeedLanesStr.values():
         value = self.speeds.get( speed )
         if value is not None:
            printedSpeed = speed
            if speed == "10G" and self._includesSlowerSpeeds10g:
               printedSpeed += "*"
            # pylint: disable-next=consider-using-f-string
            _print2( "For speed %s" % printedSpeed )
            value.render()

class InterfacesInteractions( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfacesInteractionsDataBySpeed,
                      help=( "A mapping of interface names to interactions with "
                             "other interfaces at a desired "
                             "speed configuration" ) )
   def render( self ):
      if not self.interfaces:
         # No interfaces at all. Print nothing.
         pass
      elif all( v.empty() for v in self.interfaces.values() ):
         # There are interfaces, but all are independent of each other.
         print( NO_IXNS )
      else:
         print( ASTERISK_10G )
         print()
         for intf in Arnet.sortIntf( self.interfaces ):
            value = self.interfaces[ intf ]
            print( intf + ":" )
            value.render()
      print()

