#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import AleCliLib
from CliPlugin.IraCommonCli import fibMatcher
import CliParser
import CliCommand
import BasicCli
import CliToken.Ip
import CliToken.Ipv6
from TypeFuture import TacLazyType
from CliMatcher import (
   DynamicIntegerMatcher,
   FloatMatcher,
   IntegerMatcher,
   KeywordMatcher,
)
import Tac # pylint: disable-msg=E0401
from Toggles.AleToggleLib import toggleAleCurrentBfdStateEnabled
from Toggles.AleToggleLib import toggleFecDampeningEnabled
from Toggles.AleToggleLib import toggleAleVxlanAirbEcmpSupportEnabled

# Global tokens or keyword matcher
ipForConfigMatcher = CliToken.Ip.ipMatcherForConfig
ipv6ForConfigMatcher = CliToken.Ipv6.ipv6MatcherForConfig

hardwareMatcher = CliToken.Hardware.hardwareForConfigMatcher

nextHopMatcher = KeywordMatcher(
                     'next-hop',
                     helpdesc='Next-hop configuration' )

programmedMatcher = KeywordMatcher( 'programmed', helpdesc='FIB programming' )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop add dampen"
# -------------------------------------------------------------------------------
def fecDampeningGuard( mode, token ):
   fecDampeningSupport = AleCliLib.routingHwStatus.fecDampeningSupport
   if fecDampeningSupport and fecDampeningSupport.fecDampeningSupported:
      return None
   return CliParser.guardNotThisPlatform

class AleFecDampeningCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop add dampen'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'add' : CliCommand.guardedKeyword( 'add',
                                         helpdesc='Add Fib routes',
                                         guard=fecDampeningGuard ),
      'dampen': 'Dampen next-hop creation'
   }
   handler = "AleRoutingHandler.setFecDampeningSupported"
   noOrDefaultHandler = "AleRoutingHandler.setFecDampeningSupported"

if toggleFecDampeningEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( AleFecDampeningCmd )

# -------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop sharing"
#-------------------------------------------------------------------------------

class AleAdjSharingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop sharing'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'sharing': "Configure next-hop sharing mode",
   }
   handler = "AleRoutingHandler.setAdjSharing"
   noOrDefaultHandler = "AleRoutingHandler.setAdjSharing"

BasicCli.GlobalConfigMode.addCommandClass( AleAdjSharingCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] ip hardware fib next-hop sharing resolved ip sr-te"
#-------------------------------------------------------------------------------
ProxyFecTuple = TacLazyType( 'Routing::Hardware::ProxyFecTuple' )
FibAdjType = TacLazyType( 'Smash::Fib::AdjType' )

class AdjProxyIpSrTeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop sharing resolved IP_KW sr-te'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'sharing': "Configure next-hop sharing mode",
      'resolved': "Adjacency type",
      'IP_KW': KeywordMatcher(
         'ip', helpdesc='Ipv4 and Ipv6 FIB RIB adjacency type' ),
      'sr-te': 'SR-TE Policy adjacency type',
   }

   handler = "AleRoutingHandler.handleAdjProxyIpSrTe"
   noOrDefaultHandler = "AleRoutingHandler.handleNoAdjProxyIpSrTe"

BasicCli.GlobalConfigMode.addCommandClass( AdjProxyIpSrTeCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop multi-path maximum <max>"
#-------------------------------------------------------------------------------
def maxAleEcmpRange( mode, context ):
   upperBound = max( AleCliLib.routingHwStatus.maxLogicalProtocolEcmp,
                     AleCliLib.routingHwStatus.maxResilientEcmp,
                     AleCliLib.routingHwStatus.maxNexthopGroupEcmp )
   return ( 1, upperBound or 0xFFFFFFFF )

class AleMultiPathMaxCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop multi-path maximum MAX_VAL'
   noOrDefaultSyntax = 'ip hardware fib next-hop multi-path maximum ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'multi-path': "Multi-path configuration",
      'maximum': "Configure max ale ecmp",
      'MAX_VAL': DynamicIntegerMatcher( maxAleEcmpRange,
                                        helpdesc="Max Ale Ecmp" )
   }

   handler = "AleRoutingHandler.handleAleMultiPathMax"
   noOrDefaultHandler = "AleRoutingHandler.handleNoMultiPathMax"

BasicCli.GlobalConfigMode.addCommandClass( AleMultiPathMaxCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib default route protected"
#-------------------------------------------------------------------------------

class AleDefaultRouteProtectCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib default route protected'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'default': "Configure default route",
      'route': "Configure route",
      'protected': "Configure default route protection"
   }
   handler = "AleRoutingHandler.setDefaultRouteProtection"
   noOrDefaultHandler = "AleRoutingHandler.setDefaultRouteProtection"

BasicCli.GlobalConfigMode.addCommandClass( AleDefaultRouteProtectCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib route delete delay <seconds>
#-------------------------------------------------------------------------------
class AleRouteDeleteDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib route delete delay SECONDS'
   noOrDefaultSyntax = 'ip hardware fib route delete delay ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'route': "Routes",
      'delete': "Delete",
      'delay': 'Delay',
      'SECONDS': IntegerMatcher( 0, 60, helpdesc="Number of seconds" )
   }

   handler = "AleRoutingHandler.handleRouteDeleteDelay"
   noOrDefaultHandler = "AleRoutingHandler.handleNoRouteDeleteDelay"


BasicCli.GlobalConfigMode.addCommandClass( AleRouteDeleteDelayCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib next-hop update event bfd
#-------------------------------------------------------------------------------

class AleUpdateEventBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop update event bfd'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'update': "Update Fib routes",
      'event': "Event type",
      'bfd': "BFD status"
   }

   handler = "AleRoutingHandler.setNhUpdateEvent"
   noOrDefaultHandler = "AleRoutingHandler.setNhUpdateEvent"

BasicCli.GlobalConfigMode.addCommandClass( AleUpdateEventBfdCmd )

# -------------------------------------------------------------------------------
# [no|default] ip hardware fib next-hop bfd mode current all
# -------------------------------------------------------------------------------

class AleUpdateBfdModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop bfd mode current all'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'bfd': "BFD status",
      'mode': 'BFD status operational mode',
      'current': 'Only consider current BFD session status',
      'all': 'Apply to all BFD session types'
   }

   noOrDefaultSyntax = syntax
   handler = "AleRoutingHandler.setBfdStatusMode"
   noOrDefaultHandler = "AleRoutingHandler.setBfdStatusMode"

if toggleAleCurrentBfdStateEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( AleUpdateBfdModeCmd )

# -------------------------------------------------------------------------------
# [no|default] ip hardware fib vxlan asymmetric-irb ecmp
# -------------------------------------------------------------------------------

def asymmetricIrbEcmpSupported( mode, token ):
   if AleCliLib.routingHwStatus.asymmetricIrbEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class AleAsymmetricIrbEcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib vxlan asymmetric-irb ecmp'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'vxlan': "VXLAN configuration",
      'asymmetric-irb': 'Asymmetric-irb configuration',
      'ecmp': CliCommand.Node(
         matcher=KeywordMatcher( 'ecmp',
                                 helpdesc='Enable ECMP' ),
         guard=asymmetricIrbEcmpSupported ),
   }

   noOrDefaultSyntax = syntax
   handler = "AleRoutingHandler.setAsymmetricIrbEcmp"
   noOrDefaultHandler = "AleRoutingHandler.setAsymmetricIrbEcmp"

if toggleAleVxlanAirbEcmpSupportEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( AleAsymmetricIrbEcmpCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib maximum routes <num-routes>"
# "[no|default] ipv6 hardware fib maximum routes <num-routes>"
#-------------------------------------------------------------------------------
class AleMaxRoutesCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) hardware fib maximum routes MAX_ROUTES'
   noOrDefaultSyntax = '( ip | ipv6 ) hardware fib maximum routes ...'
   data = {
      'ip': ipForConfigMatcher,
      'ipv6': ipv6ForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'maximum': "Softlimit of Max routes",
      'routes': "IP routes in this address family",
      'MAX_ROUTES': IntegerMatcher( 1, 0xFFFFFFFF,
                                    helpdesc="number of max Routes" )
   }

   @staticmethod
   def _getConfig( args ):
      return ( AleCliLib.routing6HwConfig if 'ipv6' in args
               else AleCliLib.routingHwConfig )

   handler = "AleRoutingHandler.handleMaxRoutes"
   noOrDefaultHandler = "AleRoutingHandler.handleNoMaxRoutes"

BasicCli.GlobalConfigMode.addCommandClass( AleMaxRoutesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop weight-deviation <percentage>"
#-------------------------------------------------------------------------------
class AleWeightDeviationCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop weight-deviation PERCENT'
   noOrDefaultSyntax = 'ip hardware fib next-hop weight-deviation ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'weight-deviation': "Configure weight deviation",
      'PERCENT': FloatMatcher( 0.01, 50.0,
                               helpdesc='Weight deviation percentage',
                               precisionString='%.2f' )
   }

   handler = "AleRoutingHandler.handleWeightDeviation"
   noOrDefaultHandler = "AleRoutingHandler.handleNoWeightDeviation"

BasicCli.GlobalConfigMode.addCommandClass( AleWeightDeviationCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop flattening weight-expansion
#              ucmp [ecmp]
#-------------------------------------------------------------------------------

hierarchicalMatcher = KeywordMatcher( 'hierarchical',
                                      helpdesc="Hardware resource configuration" )
hierarchcialNextHopMatcher = KeywordMatcher( 'next-hop',
                                             helpdesc='Next-hop configuration' )

class AleHierarchicalWeightExpansionCmd( CliCommand.CliCommandClass ):
   syntax = '''ip hardware fib hierarchical next-hop flattening weight-expansion
               ucmp [ ecmp ]
            '''
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'flattening': "Flattening of next-hop configuration",
      'weight-expansion': "Follow configured next-hop weights by replication",
      'ucmp': "Unequal cost multipath expansion",
      'ecmp': "Equal cost multipath expansion"
   }

   handler = "AleRoutingHandler.handleHierarchicalWeightExpansion"
   noOrDefaultHandler = "AleRoutingHandler.handleNoHierarchicalWeightExpansion"


BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalWeightExpansionCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib programmed monitor
# -------------------------------------------------------------------------------

class AleRoutingTelemetryCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib programmed monitor'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'programmed': programmedMatcher,
      'monitor': 'Track FIB programming',
   }
   handler = "AleRoutingHandler.setOrResetRoutingTelemetry"
   noOrDefaultHandler = "AleRoutingHandler.setOrResetRoutingTelemetry"

BasicCli.GlobalConfigMode.addCommandClass( AleRoutingTelemetryCmd )

# -------------------------------------------------------------------------------
# [no|default] ip hardware fib programmed error action preserved
# -------------------------------------------------------------------------------
def hwRetainStateOnFailureSupported( mode, token ):
   hwStateRetainSupport = AleCliLib.routingHwStatus.hwStateRetainSupport
   if hwStateRetainSupport and hwStateRetainSupport.retainStateOnFailureSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class AleHwRetainStateOnFailureCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib programmed error action preserved'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'programmed': programmedMatcher,
      'error': CliCommand.Node(
                      matcher=KeywordMatcher(
                         'error',
                         helpdesc='Configure programming error handling' ),
                      guard=hwRetainStateOnFailureSupported ),
      'action': 'Action taken upon programming error',
      'preserved': 'Preserve dependent routes and FECs after programming failures'
   }
   handler = "AleRoutingHandler.handleUseRetainedHwState"
   noOrDefaultHandler = "AleRoutingHandler.handleNoUseRetainedHwState"

BasicCli.GlobalConfigMode.addCommandClass( AleHwRetainStateOnFailureCmd )

# -------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop urpf
#-------------------------------------------------------------------------------

class AleHierarchicalNextHopPlusUrpfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop urpf'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'urpf': 'Enable urpf with hfec'
   }
   handler = "AleRoutingHandler.setHfecPlusUrpf"
   noOrDefaultHandler = "AleRoutingHandler.setHfecPlusUrpf"

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalNextHopPlusUrpfCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop disabled
#-------------------------------------------------------------------------------
def hierarchicalFecSupported( mode, token ):
   if AleCliLib.routingHwStatus.hierarchicalFecSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

class AleHierarchicalNextHopDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'disabled': "Disable hierarchical FECs"
   }
   handler = "AleRoutingHandler.setHfecDisabled"
   noOrDefaultHandler = "AleRoutingHandler.setHfecDisabled"

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalNextHopDisabledCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop max-level <level>
#-------------------------------------------------------------------------------
unconfigured = Tac.Type( "Routing::Hardware::HierarchicalFecsMaxLevel" ).invalid

def maxLevelRange( mode, context ):
   if AleCliLib.routingHwStatus.maxHierarchicalFecChainLength == 1:
      # Normally not possible thanks to the guard, but needed on startup where
      # we parse the config even if the feature agents have not populated
      # routingHwStatus yet. The configured value is bounded by the actual
      # hardware limit in L3FeatureConfigSm.
      return ( 1, unconfigured )
   return ( 1, AleCliLib.routingHwStatus.maxHierarchicalFecChainLength )

class AleHierarchicalNextHopMaxLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop max-level MAX_LEVEL'
   noOrDefaultSyntax = 'ip hardware fib hierarchical next-hop max-level ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'max-level': CliCommand.Node(
                      matcher=KeywordMatcher(
                         'max-level',
                         helpdesc='Max FEC hierarchy level configuration' ),
                      guard=hierarchicalFecSupported ),
      'MAX_LEVEL': DynamicIntegerMatcher( maxLevelRange,
                                          helpdesc='FEC hierarchy max level' )
   }

   handler = "AleRoutingHandler.handleHierarchicalNextHopMaxLevel"

   noOrDefaultHandler = "AleRoutingHandler.handleNoHierarchicalNextHopMaxLevel"

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalNextHopMaxLevelCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop resource optimization threshold
#               low <percentage> high <percentage>"
#-------------------------------------------------------------------------------
def nextHopResourceOptimizationSupported( mode, token ):
   if AleCliLib.routingHwStatus.adjResourceOptimizationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

resourceMatcher = KeywordMatcher( "resource",
                     helpdesc="Hardware resource configuration" )

nodeOptimization = CliCommand.Node(
      matcher=KeywordMatcher( "optimization",
                     helpdesc="Optimize hardware resource" ),
      guard=nextHopResourceOptimizationSupported )

matcherThresholdValue = IntegerMatcher( 0, 100,
                                 helpdesc="Resource utilization percentage "
                                          "for starting/stopping optimization." )

class AdjResourceOptimizationThresholdsCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib next-hop resource optimization thresholds '
              'low THRESHOLD_LOW high THRESHOLD_HIGH' )
   noOrDefaultSyntax = (
         'ip hardware fib next-hop resource optimization thresholds ...' )
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'resource': resourceMatcher,
      'optimization': nodeOptimization,
      'thresholds': "Utilization percentage for "
                    "starting or stopping optimization",
      'low': 'Stop optimization threshold',
      'THRESHOLD_LOW': matcherThresholdValue,
      'high': 'Start optimization threshold',
      'THRESHOLD_HIGH': matcherThresholdValue,
   }

   handler = "AleRoutingHandler.setResourceOptimizationThresholds"
   noOrDefaultHandler = "AleRoutingHandler.resetResourceOptimizationThresholds"

BasicCli.GlobalConfigMode.addCommandClass( AdjResourceOptimizationThresholdsCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop resource optimization disabled"
#-------------------------------------------------------------------------------

class AdjResourceOptimizationDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop resource optimization disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'resource': resourceMatcher,
      'optimization': nodeOptimization,
      'disabled': "Disable hardware resource optimization for adjacency programming",
   }

   handler = "AleRoutingHandler.setAdjResourceOptimization"

   noOrDefaultHandler = "AleRoutingHandler.setAdjResourceOptimization"

BasicCli.GlobalConfigMode.addCommandClass( AdjResourceOptimizationDisabledCmd )

#-------------------------------------------------------------------------------
# hidden
# "[ no | default ] ip hardware fib next-hop proxy disabled"
#-------------------------------------------------------------------------------
class AdjProxyDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop proxy disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      # Add CLI save plugin if you want to unhide this command
      'proxy': CliCommand.Node( matcher=KeywordMatcher(
             "proxy", helpdesc="Proxy configuration for adjacencies" ),
             hidden=True ),
      'disabled': "Disable proxy adjacency programming",
   }

   handler = "AleRoutingHandler.handleAdjProxyDisabled"

   noOrDefaultHandler = "AleRoutingHandler.handleNoAdjProxyDisabled"

BasicCli.GlobalConfigMode.addCommandClass( AdjProxyDisabledCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] hardware fec programmed all"
#-------------------------------------------------------------------------------
class HardwareFecProgrammedCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware fec programmed all'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcher,
      'fec': "Forward Equivalence Class configuration",
      'programmed': "Forward Equivalence Class programmed configuration",
      'all': "Program all FECs",
   }

   handler = "AleRoutingHandler.handleHardwareFecProgrammed"

   noOrDefaultHandler = "AleRoutingHandler.handleNoHardwareFecProgrammed"

BasicCli.GlobalConfigMode.addCommandClass( HardwareFecProgrammedCmd )

# -------------------------------------------------------------------------------
# "[ no | default ] hardware fec multi-path maximum <max>"
# -------------------------------------------------------------------------------
def maxConfiguredEcmpRange( mode, context ):
   return ( 1, AleCliLib.routingHwStatus.maxHwEcmp or 0xFFFFFFFF )

class HardwareFecMultiPathMaxCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware fec multi-path maximum MAX_VAL'
   noOrDefaultSyntax = 'hardware fec multi-path maximum ...'
   data = {
      'hardware': hardwareMatcher,
      'fec': "Forward Equivalence Class configuration",
      'multi-path': "Multi-path configuration",
      'maximum': "Configure maximum multi-path value for a single fec",
      'MAX_VAL': DynamicIntegerMatcher( maxConfiguredEcmpRange,
                                        helpdesc="Max multipath value" )
   }

   handler = "AleRoutingHandler.handleHardwareFecMultiPathMax"

   noOrDefaultHandler = "AleRoutingHandler.handleNoHardwareFecMultiPathMax"

BasicCli.GlobalConfigMode.addCommandClass( HardwareFecMultiPathMaxCmd )
