# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import BasicCliSession
import CliCommand
import CliMatcher
from CliMode.Alias import AliasMode
import CliParser
from CliPlugin import CliCliModel
import CliSession
import CliToken.Cli
import ConfigMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

Alias = TacLazyType( 'Cli::Config::Alias' )

cliConfig = None

def maybeAddAliasSessionOnCommitHandler( mode ):
   if mode.session.inConfigSession():
      callback = BasicCliSession.resetRegexAliases
      em = mode.entityManager
      CliSession.registerSessionOnCommitHandler( em, 'regexAliases', callback )

class AliasConfigMode( AliasMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'alias configuration'
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   # Maximum sequence number for command.
   MAX_SEQ = 0xFFFFFFFF # 32-bit integer
   # Maximum number of commands we allow for each alias.
   MAX_CMD = 32
   # The incremental step, default is 10.
   INC_STEP = 10

   #----------------------------------------------------------------------------
   # Constructs a new AliasConfigMode instance for an alias of multi-line
   # commands. This is called every time the user enters "alias-name" mode.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, cmdAlias=None, regex=None ):
      self.compiledRegex = regex
      if regex:
         self.commentKeyTemplate = 'alias-"%s"'
         self.aliases = cliConfig.regexAlias

         # Check if we are editing an existing alias.
         for idx, alias in self.aliases.items():
            if alias.cmdAlias == cmdAlias:
               self.key = idx
               break
         else:
            self.key = 0

         # Prompt with parens (likely in Regex) leads to issues with tests.
         prompt = 'regex'
      else:
         self.commentKeyTemplate = 'alias-%s'
         self.aliases = cliConfig.alias
         self.key = cmdAlias.lower()
         prompt = cmdAlias

      try:
         self.alias = Tac.nonConst( self.aliases[ self.key ] )
      except KeyError:
         self.alias = Alias( cmdAlias )

      AliasMode.__init__( self, prompt )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def _commitAlias( self ):
      maybeAddAliasSessionOnCommitHandler( self )
      if self.alias is not None:
         if not self.alias.originalCmd:
            # Remove associated comment if any.
            self.removeComment()
            self.addError( "Empty alias not allowed" )
         else:
            if not self.key:
               # Committing a brand new regex alias.
               self.key = BasicCliSession.addCompiledRegex( self.compiledRegex,
                                                            self.aliases )
            self.aliases[ self.key ] = self.alias

   def onExit( self ):
      self._commitAlias()
      BasicCli.ConfigModeBase.onExit( self )

   def commentKey( self ):
      # Comments are stored in Sysdb under the mode's key.
      # Regex alias config mode has a plain, 'regex', prompt,
      # because parens are likely in regex and that confuses tests,
      # but using 'regex' for this key puts all regex alias modes' comments
      # under the same key. So we override this.
      result = self.commentKeyTemplate % self.alias.cmdAlias
      return result

   def getAliasConfig( self, name ):
      return cliConfig.alias.get( name, None )

#------------------------------------------------------------------------------------
# alias ALIAS | REGEX
#------------------------------------------------------------------------------------
aliasKwMatcher = CliMatcher.KeywordMatcher( 'alias',
      helpdesc='Add a command alias' )
aliasRegexMatcher = CliMatcher.QuotedStringMatcher(
      helpdesc='Python-compatible RegEx command alias. Use ^V to escape \'?\'',
      helpname='QUOTED STRING',
      requireQuote=True,
      priority=CliParser.PRIO_HIGH )
cmdAliasMatcher = CliMatcher.PatternMatcher( r'.+',
      helpname='WORD',
      helpdesc='Command alias string' )
originalCmdMatcher = CliMatcher.StringMatcher(
      helpname='LINE',
      helpdesc='Original command string' )

class GotoAliasMode( CliCommand.CliCommandClass ):
   syntax = 'alias ALIAS'
   data = {
      'alias': aliasKwMatcher,
      'ALIAS': cmdAliasMatcher,
   }

   handler = 'AliasCli.gotoAliasMode'

BasicCli.GlobalConfigMode.addCommandClass( GotoAliasMode )

class GotoRegexAliasMode( CliCommand.CliCommandClass ):
   syntax = 'alias REGEX'
   data = {
      'alias': aliasKwMatcher,
      'REGEX': aliasRegexMatcher,
   }

   handler = 'AliasCli.gotoRegexAliasMode'

BasicCli.GlobalConfigMode.addCommandClass( GotoRegexAliasMode )

#------------------------------------------------------------------------------------
# alias commands path PATH
#------------------------------------------------------------------------------------
class CommandsPathCmd( CliCommand.CliCommandClass ):
   syntax = 'alias commands path PATH'
   noOrDefaultSyntax = 'alias commands path ...'
   data = {
      'alias': aliasKwMatcher,
      'commands': CliMatcher.KeywordMatcher( 'commands',
                      helpdesc='Create Cli aliases for all executable files '
                               'under a given path',
                      priority=CliParser.PRIO_HIGH ),
      'path': 'Load commands from path',
      'PATH': CliMatcher.PatternMatcher( '.+',
                                          helpdesc='Path to load commands',
                                          helpname='WORD' ),
   }

   handler = 'AliasCli.commandsPathCmd'
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( CommandsPathCmd )

#------------------------------------------------------------------------------------
# alias ALIAS COMMAND
#------------------------------------------------------------------------------------
def removeAlias( mode, cmdAlias, collection ):
   '''Remove an alias from a given collection and any associated comments.
   '''
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, AliasConfigMode ) and lastMode.alias
        and lastMode.alias.cmdAlias == cmdAlias ):
      # If we are deleting ourselves, make sure we don't write it back when
      # we exit.
      lastMode.alias = None
   # Remove associated comment if any. commentKey should be same as what user
   # entered initially.
   alias = collection.get( cmdAlias )
   if alias:
      BasicCliModes.removeCommentWithKey( 'alias-' + alias.cmdAlias )
      del collection[ cmdAlias ]

class ModifyAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'alias ALIAS COMMAND'
   noOrDefaultSyntax = 'alias ALIAS ...'
   data = {
      'alias': aliasKwMatcher,
      'ALIAS': cmdAliasMatcher,
      'COMMAND': originalCmdMatcher,
   }

   handler = 'AliasCli.modifyAliasCmd'
   noOrDefaultHandler = 'AliasCli.modifyAliasCmdNot'

BasicCli.GlobalConfigMode.addCommandClass( ModifyAliasCmd )

#------------------------------------------------------------------------------------
# alias REGEX TEMPLATE
#------------------------------------------------------------------------------------
class ModifyRegexAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'alias REGEX TEMPLATE'
   noOrDefaultSyntax = 'alias REGEX ...'
   data = {
      'alias': aliasKwMatcher,
      'REGEX': aliasRegexMatcher,
      'TEMPLATE': CliMatcher.StringMatcher(
                        helpname='LINE',
                        helpdesc='A string template for this alias' ),
   }

   handler = 'AliasCli.modifyRegexAliasCmd'
   noOrDefaultHandler = 'AliasCli.modifyRegexAliasCmdNot'

BasicCli.GlobalConfigMode.addCommandClass( ModifyRegexAliasCmd )

#------------------------------------------------------------------------------------
# command COMMAND
#------------------------------------------------------------------------------------
class AliasAddCmd( CliCommand.CliCommandClass ):
   syntax = 'command COMMAND'
   noOrDefaultSyntax = syntax
   data = {
      'command': 'Add a command to the end of the sequence',
      'COMMAND': originalCmdMatcher
   }

   handler = 'AliasCli.aliasAddCmd'
   noOrDefaultHandler = 'AliasCli.aliasRemoveCmd'

AliasConfigMode.addCommandClass( AliasAddCmd )

#------------------------------------------------------------------------------------
# [ no | default ] SEQ COMMAND
#------------------------------------------------------------------------------------
class AliasEditCmd( CliCommand.CliCommandClass ):
   syntax = 'SEQ COMMAND'
   noOrDefaultSyntax = 'SEQ ...'
   data = {
      'SEQ': CliMatcher.IntegerMatcher( 1, AliasConfigMode.MAX_SEQ,
         helpdesc='Unsigned sequence number' ),
      'COMMAND': originalCmdMatcher
   }

   handler = 'AliasCli.aliasEditCmd'
   noOrDefaultHandler = 'AliasCli.aliasEditCmdNot'

AliasConfigMode.addCommandClass( AliasEditCmd )

#------------------------------------------------------------------------------------
# abort
#------------------------------------------------------------------------------------
class AliasAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': CliToken.Cli.abortMatcher
   }

   handler = 'AliasCli.aliasAbortCmd'

AliasConfigMode.addCommandClass( AliasAbortCmd )

#------------------------------------------------------------------------------------
# show [ pending ]
#------------------------------------------------------------------------------------
class ShowPending( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = {
      'pending': 'Show pending list in this session'
   }

   handler = 'AliasCli.showPending'

AliasConfigMode.addShowCommandClass( ShowPending )

#------------------------------------------------------------------------------------
# show active
#------------------------------------------------------------------------------------
class ShowActive( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active'
   data = {
      'active': BasicCliModes.showActiveNode,
   }

   handler = 'AliasCli.showActive'

AliasConfigMode.addShowCommandClass( ShowActive )

#------------------------------------------------------------------------------------
# show diff
#------------------------------------------------------------------------------------
class ShowDiff( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show the difference between active and pending list'
   }

   handler = 'AliasCli.showDiff'

AliasConfigMode.addShowCommandClass( ShowDiff )

#------------------------------------------------------------------------------------
# show aliases
#-----------------------------------------------------------------------------------
class ShowAliases( ShowCommand.ShowCliCommandClass ):
   syntax = 'show aliases [ match STRING ]'
   data = {
      'aliases': 'List all configured aliases',
      'match': 'List all RegEx aliases matching the following string',
      'STRING': CliMatcher.QuotedStringMatcher( helpdesc='String to test against' ),
   }
   cliModel = CliCliModel.Aliases

   handler = 'AliasCli.showAliases'

BasicCli.addShowCommandClass( ShowAliases )

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )
