# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libIsisHostnameExport.so*

# CliPlugin module for BGP configuration commands

# pylint: disable-msg=W0611
# pylint: disable=consider-using-f-string

import AgentCommandRequest
import BasicCli
import Cell
import CliCommand
from CliGlobal import CliGlobal
from CliDynamicSymbol import CliDynamicPlugin
from CliDynamicSymbol import LazyCallback
import CliMatcher
from CliMatcher import KeywordMatcher
from CliMode.ArBgp import BgpSessionTrackerMode
import CliParser
from CliPlugin import RcfCliLib
from CliPlugin.RcfLibCliModels import (
   ShowAgentRcfModel,
   ShowAgentRcfDetailModel,
)
from CliPlugin import VrfCli
from CliPlugin import ArBgpCliCommon
from CliPlugin import BgpMaintenanceShowCli
from CliPlugin import ShowTaskSchedulerModel
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import TechSupportCli
from CliPlugin.BgpCliModels import (
   ShowNeighborsHistoryModel,
   ShowNeighborsHistorySocketModel,
   BgpOrrPositionByVrf,
)
from CliPlugin.ArBgpCliModels import (
   BgpRfdPolicies,
)
from CliPlugin.DynamicPrefixListCli import (
     routeMapShowCmdDict,
     DynamicPrefixListConfigMode,
)

from CliPlugin.EthIntfCli import EthIntf
from CliPlugin import IntfCli
from CliPlugin.LagIntfCli import EthLagIntf
from CliPlugin.LoopbackIntfCli import LoopbackIntf
from CliPlugin.RoutingBgpCli import (
   BgpCmdBaseClass,
   deleteRouterBgpMacVrfHook,
   PeerCliExpression,
   peergroupNameRegex,
   RouterBgpBaseMode,
   rfdPolicyNameMatcher,
   V4V6PeerKeyCliExpression,
)
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RouteMapCli import CommonTokens as RouteMapCommonTokens
from CliPlugin.PolicyEvalModels import PolicyEvalResults

from CliPlugin.RoutingBgpShowCli import (
        allVrfExprFactory,
        arBgpShowCmdDict,
        doShowBgpOutputNotSupported,
        routeSummaryVrfModel,
        summaryVrfModel,
)
import ConfigMount
import LazyMount
import ShowCommand
import Tac
from Toggles import (
   ArBgpToggleLib,
   RcfLibToggleLib,
   RouterGeneralCliToggleLib,
   RoutingLibToggleLib,
)
import Tracing
import Url

from AgentSnapshotCore import snapshotCoreCallBack
from ArBgpAgent import BgpAgentName
import BasicCliModes
from CliModel import cliPrinted
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel

import CliToken
from CliToken.Agent import agentKwForConfig
import CliToken.RoutingBgp as bgpTokens
import CliToken.RoutingArBgp as arBgpTokens
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from CliToken import RoutingBgpShowCliTokens
from CliToken.RoutingBgpShowCliTokens import (
   active,
   afiMatcher,
   aspath,
   aspathRegexMatcher,
   bgpAfterShow,
   commRegexp,
   detail,
   detailConfig,
   fieldSet,
   ipv4AfterShowBgp,
   ipv6AfterShowBgp,
   lldpAfterShow,
   regexp,
   scheduler,
   schedulerVerbose,
   showCommunity,
   unicastAfterShowBgp,
   unsupported,
   routeFlapDamping,
   policyInfo,
)

from CliSavePlugin.BgpMacVrfConfigCliSave import saveMacVrfConfig

traceHandle = Tracing.Handle( 'ArBgpCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

notSupported = "Not supported"
config = None
vrfConfigDir = None
mapConfig = None
asnConfig = None
securityConfig = None
vrfNameStatus = None
rtrGeneralConfigDir = None
peerInfoTable = None
sessionTrackerConfigDir = None
onBootLinkEnableOverrideConfig = None
startupConfigStatus = None

BgpPeerState = Tac.Type( 'Routing::Bgp::BgpPeerState' )
RoutingBgpShowCliHandler = CliDynamicPlugin( "RoutingBgpShowCliHandler" )

# BGP Session Tracker related configurations
# pylint: disable=unnecessary-comprehension
def getSessionTrackerNames( mode ):
   return [ trackername
            for trackername in sessionTrackerConfigDir.sessionTrackerConfig.keys() ]
# pylint: enable=unnecessary-comprehension

# BGP Session tracker is only supported for Ethernet and LAG interfaces at this time.
def bgpSessionTrackerSupportedIntfsGuard( mode, token ):
   allowedTypes = ( EthIntf, EthLagIntf )
   if ArBgpToggleLib.toggleSessionTrackerOnLoopbackEnabled():
      allowedTypes += ( LoopbackIntf, )
   intf = mode.intf
   if isinstance( intf, allowedTypes ) and intf.routingSupported():
      return None
   return CliParser.guardNotThisPlatform

sessionTrackerNameMatcher = CliMatcher.DynamicNameMatcher(
   getSessionTrackerNames, 'Name of session tracker', helpname='NAME' )
bgpKwWithGuard = CliCommand.guardedKeyword( 'bgp',
      helpdesc='Configure BGP commands',
      guard=bgpSessionTrackerSupportedIntfsGuard )
sessionTrackerFeatureName = "BgpSessionTracker"
sessionTrackerFeatureAgentName = "Bgp"

class SessionTrackerConfigMode( BgpSessionTrackerMode, BasicCli.ConfigModeBase ):
   name = 'Session tracker configuration'

   def __init__( self, parent, session, trackerName, mode ):
      BgpSessionTrackerMode.__init__( self, trackerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.name = trackerName
      self.session_ = session

      # Only create override config during startup config processing
      if ( getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent and
           startupConfigStatus.startupConfigStatus == 'inProgress' and
           sessionTrackerFeatureName not in onBootLinkEnableOverrideConfig.feature ):
         overrideConfig = Tac.newInstance(
                             'Interface::OnBootLinkEnableOverrideFeature',
                             sessionTrackerFeatureName )
         overrideConfig.time = Tac.now()
         overrideConfig.agents.add( sessionTrackerFeatureAgentName )
         onBootLinkEnableOverrideConfig.addFeature( overrideConfig )

      if self.name not in sessionTrackerConfigDir.sessionTrackerConfig:
         attrConfig = Tac.newInstance( 'Routing::Bgp::SessionTrackerAttrsCliConfig',
                                       self.name )
         sessionTrackerConfigDir.addSessionTrackerConfig( attrConfig )

   def updateRecoveryDelay( self, recoveryDelaySec, present=True ):
      attrConfig = Tac.nonConst(
                      sessionTrackerConfigDir.sessionTrackerConfig[ self.name ] )
      attrConfig.recoveryDelaySec = recoveryDelaySec
      attrConfig.recoveryDelaySecPresent = present
      sessionTrackerConfigDir.addSessionTrackerConfig( attrConfig )

   def updateNeighborAll( self, trackAllPeersUp ):
      attrConfig = Tac.nonConst(
                      sessionTrackerConfigDir.sessionTrackerConfig[ self.name ] )
      attrConfig.trackAllPeersUp = trackAllPeersUp
      sessionTrackerConfigDir.addSessionTrackerConfig( attrConfig )

#-------------------------------------------------------------------------------
# "[ no | default ] session tracker <name>" config command in "router bgp <>"
#-------------------------------------------------------------------------------
class EnterSessionTrackerConfigMode( CliCommand.CliCommandClass ):
   syntax = 'session tracker NAME'
   noOrDefaultSyntax = syntax
   data = {
      'session' : 'Configure BGP session state data',
      'tracker' : 'Enable session tracking for BGP peers',
      'NAME' : sessionTrackerNameMatcher,
   }

   handler = "ArBgpCliHandler.handlerEnterSessionTrackerConfigMode"
   noOrDefaultHandler = "ArBgpCliHandler." \
      "noOrDefaultHandlerEnterSessionTrackerConfigMode"

RouterBgpBaseMode.addCommandClass( EnterSessionTrackerConfigMode )

#-------------------------------------------------------------------------------
# "[ no | default ] recovery delay < 1-3600 > seconds" config command in
# "config-tracker-name" mode.
#-------------------------------------------------------------------------------
class SessionTrackerRecoveryDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'recovery delay DELAYSEC seconds'
   noOrDefaultSyntax = 'recovery delay ...'
   data = {
      'recovery' : 'Configure recovery action when BGP sessions come up',
      'delay' : 'Configure delay for session recovery',
      'DELAYSEC' : bgpTokens.secondsRangeMatcher,       # 1-3600 seconds
      'seconds' : bgpTokens.secondsKwMatcher,
   }

   handler = "ArBgpCliHandler.handlerSessionTrackerRecoveryDelayCmd"
   noOrDefaultHandler = "ArBgpCliHandler." \
      "noOrDefaultHandlerSessionTrackerRecoveryDelayCmd"

SessionTrackerConfigMode.addCommandClass( SessionTrackerRecoveryDelayCmd )

#-------------------------------------------------------------------------------
# "[ no | default ] neighbor all" config command in "config-tracker-name" mode.
#-------------------------------------------------------------------------------
class SessionTrackerNeighborAllCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor all'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : 'Configuration for neighbors tracked by this session tracker',
      'all' : 'Session tracker is up when all neighbors are in established state'
   }

   handler = "ArBgpCliHandler.handlerSessionTrackerNeighborAllCmd"
   noOrDefaultHandler = "ArBgpCliHandler." \
      "noOrDefaultHandlerSessionTrackerNeighborAllCmd"

if ArBgpToggleLib.toggleSessionTrackerUpAllModeEnabled():
   SessionTrackerConfigMode.addCommandClass( SessionTrackerNeighborAllCmd )

class SessionTrackerIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp session tracker NAME'
   noOrDefaultSyntax = 'bgp session tracker ...'
   data = {
      'bgp' : bgpKwWithGuard,
      'session' : 'Configure BGP session state data',
      'tracker' : 'Associate interface with BGP session tracker',
      'NAME' : sessionTrackerNameMatcher,
   }

   handler = "ArBgpCliHandler.handlerSessionTrackerIntfConfigCmd"
   noOrDefaultHandler = "ArBgpCliHandler." \
      "noOrDefaultHandlerSessionTrackerIntfConfigCmd"

IntfCli.IntfConfigMode.addCommandClass( SessionTrackerIntfConfigCmd )

# Clean-up hook for session tracker if bgpConfig is unconfigured.
def deleteSessionTrackerConfigHook():
   if sessionTrackerConfigDir:
      sessionTrackerConfigDir.sessionTrackerConfig.clear()

deleteRouterBgpMacVrfHook.addExtension( deleteSessionTrackerConfigHook )

# By default show commands execute in a forked process. This allows
# the BgpCore task scheduler to continue normal execution while the
# show command is in progress.
bgpShowCmdUseFork = True
# pylint: disable-msg=E0602

class ArBgpShowForkToggleCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp-show-fork ( enable | disable )'
   data = {
      'bgp-show-fork' :
         'Control whether ArBgp show commands execute in a forked child process',
      'enable' : 'Execute show command in a forked child',
      'disable' : 'Execute show command in the main ArBgp process',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      global bgpShowCmdUseFork
      bgpShowCmdUseFork = 'enable' in args

BasicCli.EnableMode.addCommandClass( ArBgpShowForkToggleCmd )

def peerGroupNames( mode ):
   return ( x.group for x in config.neighborConfig if x.type == 'peerGroup' )

class CommonTokens:
   '''
   This class is intended to keep common tokens that may be shared across several
   commands with the new CLI parser.
   '''

   showBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP information' )
   neighbors = RoutingBgpShowCliTokens.neighbors
   neighborAddr = IpAddrMatcher.IpAddrMatcher( 'Neighbor address' )
   neighborAddr6 = Ip6AddrMatcher.Ip6AddrMatcher( 'Neighbor address' )
   prefixExpr = IpAddrMatcher.ipPrefixExpr(
      'IPv4 address', 'Subnet mask', 'IPv4 prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   prefix6Expr = Ip6AddrMatcher.ip6PrefixExpr(
      'IPv6 address', 'Subnet mask', 'IPv6 prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   neighborPrefix = IpAddrMatcher.IpPrefixMatcher(
      'Neighbors covered by prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   neighborPrefix6 = Ip6AddrMatcher.Ip6PrefixMatcher(
      'Neighbors covered by prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   peerGroup = CliMatcher.KeywordMatcher( 'peer-group',
                                          helpdesc='Filter by peer group name' )
   peerGroupName = CliMatcher.DynamicNameMatcher( peerGroupNames,
                                                  'Name of the peer group',
                                                  helpname='WORD',
                                                  pattern=peergroupNameRegex )
   vrf = VrfCli.VrfExprFactory(
      helpdesc='Virtual routing and forwarding instance',
      inclDefaultVrf=True, inclAllVrf=True )
   rtMembership = CliMatcher.KeywordMatcher(
      'rt-membership', helpdesc='Route Target Membership address family' )

class ArBgpCliCommand:
   # Note: the following demilimiter fields must be kept in sync with
   #       BgpCommandParamsBase::commandDelimiter() and
   #       BgpCommandParamsBase::delimiterReplace().
   # Warning: do not use the Tac type to extract these fields as it will lead
   #       to the dlopen('libArBgp') which might take up to 9 seconds and also
   #       unnecessarily take more memory inside ConfigAgent.
   # Todo: BgpCommandParamsBase should be moved into its own lib (or with more
   #       types that are only relevant to the CLI rendering).
   #       It was very tempting to move it into libArBgpCommunity.so but that
   #       would end up generating more confusion.
   Delimiter = '&'
   DelimiterReplace = '\t'

   def __init__( self, mode, command, vrfName=None, lldp=None, nlriAfiSafi=None,
                 extraNlriAfiSafis=None, nlriType=None, transportAfi=None,
                 prefix=None, peerAddr=None, routerId=None, llIntf=None,
                 disableFork=False, revision=1 ):
      self.delimiter = ArBgpCliCommand.Delimiter
      self.replace = ArBgpCliCommand.DelimiterReplace
      self.entityManager = mode.entityManager
      self.command = command
      self.params = []
      self.addParam( 'fork', bgpShowCmdUseFork and not disableFork )
      # Process common parameters
      if vrfName:
         self.addParam( 'vrf', vrfName )
      if lldp:
         self.addParam( 'lldp', lldp )
      self.addNlriAfiSafi( nlriAfiSafi )
      if extraNlriAfiSafis:
         for afiSafi in extraNlriAfiSafis:
            self.addParam( 'extraNlriAfiSafis', afiSafi )
      if nlriType:
         self.addParam( 'nlriType', nlriType )
      if transportAfi:
         if transportAfi == 'ip':
            transportAfi = 'ipv4'
         self.addParam( 'transportAfi', transportAfi )
      if prefix:
         self.addParam( 'addr', prefix )
      if peerAddr:
         self.addParam( 'peerAddr', peerAddr )
      if routerId:
         self.addParam( 'routerId', routerId )
      if llIntf:
         self.addParam( 'llIntf', llIntf )

   def addNlriAfiSafi( self, nlriAfiSafi ):
      if not nlriAfiSafi:
         return
      if isinstance( nlriAfiSafi, Tac.Type( 'Routing::Bgp::AfiSafi' ) ):
         nlriAfiSafi = nlriAfiSafi.afiSafiU32()
      elif nlriAfiSafi in ( 'ip', 'ipv4' ):
         nlriAfiSafi = 'ipv4Unicast'
      elif nlriAfiSafi == 'ipv6':
         nlriAfiSafi = 'ipv6Unicast'
      self.addParam( 'nlriAfiSafi', nlriAfiSafi )

   # pylint: disable=consider-using-f-string
   def addParam( self, keyword, argument=None ):
      if argument is None:
         self.params.append( '%s' % ( keyword ) )
      else:
         self.params.append( f'{keyword}={argument}' )
   # pylint: enable=consider-using-f-string

   def updateParam( self, keyword, oldValue, newValue ):
      if oldValue == newValue:
         # Param is already set to intended value
         return
      param = f'{keyword}={oldValue}'
      self.params.remove( param )
      # All the core params (e.g. 'fork') should be in the beginning
      param = f'{keyword}={newValue}'
      self.params.insert( 0, param )

   def paramString( self ):
      ps = [ str( p ).replace( self.delimiter, self.replace ) \
             for p in self.params ]
      return str( self.delimiter ).join( ps )

   def run( self, **kwArgs ):
      t0( f"Running {self.command!r} {self.params!r}" )
      AgentCommandRequest.runSocketCommand( self.entityManager, BgpAgentName,
                                            self.command,
                                            self.paramString(), **kwArgs )

# Support commands that use TaskScheduler and yield after some time quantum or
# if the output buffer is full. This command class will disable 'fork' and
# enable 'keepalive' so the Bgp agent can keep the socket alive between task runs.
class ArBgpAsyncCliCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      # Pass common parameters on specifically.
      vrfName = kwargs.pop( 'vrfName', None )
      lldp = kwargs.pop( 'lldp', None )
      nlriAfiSafi = kwargs.pop( 'nlriAfiSafi', None )
      extraNlriAfiSafis = kwargs.pop( 'extraNlriAfiSafis', None )
      nlriType = kwargs.pop( 'nlriType', None )
      transportAfi = kwargs.pop( 'transportAfi', None )
      prefix = kwargs.pop( 'prefix', None )
      peerAddr = kwargs.pop( 'peerAddr', None )
      routerId = kwargs.pop( 'routerId', None )
      super().__init__(
         mode, command, vrfName=vrfName, lldp=lldp,
         nlriAfiSafi=nlriAfiSafi,
         extraNlriAfiSafis=extraNlriAfiSafis,
         nlriType=nlriType,
         transportAfi=transportAfi,
         prefix=prefix,
         peerAddr=peerAddr,
         routerId=routerId,
         disableFork=True )
      self.mode = mode

      # For whichever parameters are left, add to our list.
      for k, v in kwargs.items():
         self.addParam( k, v )

   def run( self, **kwArgs ):
      connErrMsg = 'BGP inactive'
      return AgentCommandRequest.runCliPrintSocketCommand(
         self.entityManager, BgpAgentName, self.command, self.paramString(),
         self.mode, keepalive=True, connErrMsg=connErrMsg, **kwArgs )

arBgpShowCmdDict[ 'doShowBgpInstance' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpInstance" )

arBgpShowCmdDict[ 'doShowBgpInstanceVpnSummary' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpInstanceVpnSummary" )

#-------------------------------------------------------------------------------
# "show bgp update-group [index | A.B.C.D | A:B:C:D:E:F:G:H] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpUpdateGroup' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpUpdateGroup" )

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpStatistics' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpStatistics" )

schedulerHistory = CliMatcher.KeywordMatcher( 'history',
                                              'Display scheduler event history' )
schedulerInternal = CliMatcher.KeywordMatcher( 'internal',
                                               'Display memory addresses of tasks' )

#--------------------------------------------------------------------------------
# Deprecate 'show bgp scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class ShowBgpSchedulerReset( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp scheduler reset'
   data = {
         "bgp" : bgpAfterShow,
         "scheduler" : scheduler,
         "reset" : CliCommand.singleNode(
            CliMatcher.KeywordMatcher( 'reset',
                                       helpdesc='Reset scheduler statistics' ),
            deprecatedByCmd='clear agent bgp task scheduler' ),
      }
   privileged = True
   handler = "ArBgpCliHandler.handlerShowBgpSchedulerReset"

BasicCli.addShowCommandClass( ShowBgpSchedulerReset )

class ShowBgpScheduler( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp scheduler [ { history | internal | verbose } ]'
   data = {
         "bgp" : bgpAfterShow,
         "scheduler" : scheduler,
         "history" : CliCommand.Node( matcher=schedulerHistory, maxMatches=1,
                                     hidden=True ),
         "internal" : CliCommand.Node( matcher=schedulerInternal, maxMatches=1,
                                     hidden=True ),
         "verbose" : CliCommand.Node( matcher=schedulerVerbose, maxMatches=1 ),
      }
   privileged = True

   cliModel = ShowTaskSchedulerModel.Overall
   handler = "ArBgpCliHandler.handlerShowBgpScheduler"

BasicCli.addShowCommandClass( ShowBgpScheduler )

class ShowBgpOrrPosition( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp optimal-route-reflection position'
   data = { 'bgp' : bgpAfterShow,
            'optimal-route-reflection' : bgpTokens.orr,
            'position' : bgpTokens.orrPosition }
   privileged = True

   cliModel = BgpOrrPositionByVrf
   handler = "ArBgpCliHandler.handlerDoShowBgpOrrPosition"

BasicCli.addShowCommandClass( ShowBgpOrrPosition )

#-------------------------------------------------------------------------------
# show bgp debug policy
#  { inbound neighbor { <address> | all } | outbound neighbor <address>
#    | redistribute }
#  <afi> <safi> [ vrf <vrf name> ]
#  [ { route-map <route-map> | rcf <function name>() ] <prefix>
#  [ path-id <path-id> ]
#-------------------------------------------------------------------------------
safiOptions = { 'unicast' : 'Unicast subsequent address family' }

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Specify the VRF',
      inclDefaultVrf=True )
vrfExcludeAllExprFactory = VrfCli.VrfExprFactory(
   helpdesc='Specify the VRF',
   inclAllVrf=False, inclDefaultVrf=True )

class ShowBgpDebugPolicyBase( ShowCommand.ShowCliCommandClass ):
   data = {
      'bgp' : CommonTokens.showBgp,
      'debug' : 'Debug BGP',
      'policy' : 'Debug BGP policy application',
      'inbound' : 'Debug inbound policy application',
      'outbound' : 'Debug outbound policy application',
      'redistribute' : 'Debug redistribute policy application',
      'neighbor' : 'Select BGP neighbor',
      'all' : 'Debug policy application to all neighbors',
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'ipv4' : 'IPv4 related',
      'ipv6' : 'IPv6 related',
      'SAFI' : CliMatcher.EnumMatcher( safiOptions ),
      'VRF' : vrfExprFactory,
      'route-map' : 'Route map to apply',
      'RMAP_NAME' : RouteMapCommonTokens.routeMapName,
      'rcf' : 'RCF function to apply',
      'RCF_FUNC_NAME' : RcfCliLib.rcfFunctionMatcher,
      'PREFIX' : CommonTokens.prefixExpr,
      'PREFIX6' : CommonTokens.prefix6Expr,
   }

   cliModel = PolicyEvalResults
   handler = "ArBgpCliHandler.policyDebugHandler"

class ShowBgpDebugPolicyInbound( ShowBgpDebugPolicyBase ):
   _routeMapOrRcfFunction = '( ( route-map RMAP_NAME ) | ( rcf RCF_FUNC_NAME ) )'
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) ' \
            '( ( ipv4 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX6 ) )' \
            .format(
               rmOrRcf=_routeMapOrRcfFunction )
   data = ShowBgpDebugPolicyBase.data.copy()
   if RcfLibToggleLib.toggleShowBgpDebugPolicyRcfAddPathEnabled():
      syntax += '[ path-id PATH_ID ]'
      data.update( {
         'path-id' : bgpTokens.pathId,
         'PATH_ID' : bgpTokens.pathIdRangeMatcher,
         } )

BasicCli.addShowCommandClass( ShowBgpDebugPolicyInbound )

class ShowBgpDebugPolicyOutbound( ShowBgpDebugPolicyBase ):
   _routeMapOrRcfFunction = '( ( route-map RMAP_NAME ) | ( rcf RCF_FUNC_NAME ) )'

   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) ' \
            '( ipv4 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX6 )'.format(
                  rmOrRcf=_routeMapOrRcfFunction )
   data = ShowBgpDebugPolicyBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyOutbound )

class ShowBgpDebugPolicyRedistribute( ShowBgpDebugPolicyBase ):
   _routeMapOrRcfFunction = '( ( route-map RMAP_NAME ) | ( rcf RCF_FUNC_NAME ) )'

   syntax = 'show bgp debug policy redistribute ' \
            '( ipv4 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ {rmOrRcf} ] PREFIX6 )'.format(
                  rmOrRcf=_routeMapOrRcfFunction )
   data = ShowBgpDebugPolicyBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyRedistribute )

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpConvergence' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpConvergence" )

#-------------------------------------------------------------------------------
# "show bgp traffic-policy field-set mappings [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpTrafficPolicyFieldSetMappings' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpTrafficPolicyFieldSetMappings" )

#-------------------------------------------------------------------------------
# "show bgp vrf selection policy field-set mappings
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpVrfSelectionPolicyFieldSetMappings' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpVrfSelectionPolicyFieldSetMappings" )

#-------------------------------------------------------------------------------
# "show ip bgp field-set <name> [ detail ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
def getIpv4FieldSetNames( mode ):
   return [ fsd.name for fsd in gv.fieldSetNameToIdMap.nameToId if not fsd.ipv6 ]

ipv4FieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv4FieldSetNames,
   helpdesc='IP prefix field-set name'
)

class ShowIpBgpFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = """show ip bgp field-set FSNAME [ detail ] [ VRF ]"""
   data = {
         "ip" : ipMatcherForShow,
         "bgp" : bgpAfterShow,
         "field-set" : fieldSet,
         "FSNAME" : ipv4FieldSetNameMatcher,
         "detail" : detail,
         "VRF" : allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel
   handler = "ArBgpCliHandler.handlerShowIpBgpFieldSet"

BasicCli.addShowCommandClass( ShowIpBgpFieldSet )

#-------------------------------------------------------------------------------
# "show ipv6 bgp field-set <name> [ detail ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
def getIpv6FieldSetNames( mode ):
   return [ fsd.name for fsd in gv.fieldSetNameToIdMap.nameToId if fsd.ipv6 ]

ipv6FieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv6FieldSetNames,
   helpdesc='IPv6 prefix field-set name'
)

class ShowIpv6BgpFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 bgp field-set FSNAME [ detail ] [ VRF ]"""
   data = {
         "ipv6" : ipv6MatcherForShow,
         "bgp" : bgpAfterShow,
         "field-set" : fieldSet,
         "FSNAME" : ipv6FieldSetNameMatcher,
         "detail" : detail,
         "VRF" : allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel
   handler = "ArBgpCliHandler.handlerShowIpv6BgpFieldSet"

BasicCli.addShowCommandClass( ShowIpv6BgpFieldSet )

#-------------------------------------------------------------------------------
# "show bgp ( ipv4 | ipv6 ) unicast field-set <name> [ detail ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
def getIpFieldSetNames( mode, context ):
   if context.sharedResult[ "AFI" ] == 'ipv4':
      return getIpv4FieldSetNames( mode )
   return getIpv6FieldSetNames( mode )

ipFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpFieldSetNames,
   helpdesc='IP prefix field-set name',
   passContext=True
)

class ShowBgpIpFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp AFI unicast field-set FSNAME [ detail ] [ VRF ]"""
   data = {
         "bgp" : bgpAfterShow,
         # not using RouteUnicastAddressFamilyRule because it's a CliExpression
         # and Node expects a CliMatcher
         "AFI" : CliCommand.Node( afiMatcher, storeSharedResult=True ),
         "unicast" : unicastAfterShowBgp,
         "field-set" : fieldSet,
         "FSNAME" : ipFieldSetNameMatcher,
         "detail" : detail,
         "VRF" : allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel
   handler = "ArBgpCliHandler.handlerShowBgpIpFieldSet"

BasicCli.addShowCommandClass( ShowBgpIpFieldSet )

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp [labeled-unicast] [<ip> | <prefix>] [longer-prefixes]
#  [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowAfBgpInternal' ] =  \
   LazyCallback( "ArBgpCliHandler.doShowAfBgpInternal" )

def mplsLabelSafiArg( kwargs ):
   labUni = kwargs.pop( "labUni", None )
   result = ""
   if labUni == "labeled-unicast":
      result = "mplslabel"
      kwargs[ 'safi' ] = result
   return result

#-------------------------------------------------------------------------------
# "show ip bgp community regex <comm-regex> as-path regex <asp-regex>
#     [vrf <vrfName>]"
# "show ip bgp [as-path] regex <asp-regex> community regex <comm-regex>
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------

class ShowIpBgpAndRegexp( ShowCommand.ShowCliCommandClass ):
   syntax = """show ip bgp
               ( ( community commregexp COMMREGEX aspath aspregexp ASPREGEX ) |
                 ( [ aspath ] aspregexp ASPREGEX community commregexp COMMREGEX ) )
               [ VRF ]"""
   data = {
         "ip" : ipMatcherForShow,
         "bgp" : bgpAfterShow,
         "community" : showCommunity,
         "commregexp" : commRegexp,
         "COMMREGEX" : RouteMapCommonTokens.commListRegexMatcher,
         "aspath" : aspath,
         "aspregexp" : regexp,
         "ASPREGEX" : aspathRegexMatcher,
         "VRF" : allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel
   handler = "ArBgpCliHandler.handlerShowIpBgpAndRegexp"

BasicCli.addShowCommandClass( ShowIpBgpAndRegexp )

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp summary [<prefix>] [vrf <vrfName>]
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowIpv6BgpSummary' ] = \
   LazyCallback( "ArBgpCliHandler.doShowIpv6BgpSummary" )
arBgpShowCmdDict[ 'doShowIpBgpSummary' ] = \
   LazyCallback( "ArBgpCliHandler.doShowIpBgpSummary" )

# -------------------------------------------------------------------------------
# "show ( ip | ipv6 ) bgp summary [lldp] [vrf <vrfName>]"
# -------------------------------------------------------------------------------
class ShowIpBgpSummaryLldp ( ShowCommand.ShowCliCommandClass ):
   syntax = """show ( ip | ipv6 ) bgp summary lldp [ VRF ]"""
   data = {
         "ip" : ipMatcherForShow,
         "ipv6" : ipv6MatcherForShow,
         "bgp" : bgpAfterShow,
         "summary" : RoutingBgpShowCliTokens.summary,
         "lldp" : lldpAfterShow,
         "VRF" : allVrfExprFactory,
      }
   cliModel = summaryVrfModel
   handler = "ArBgpCliHandler.handlerShowIpBgpSummaryLldp"

BasicCli.addShowCommandClass( ShowIpBgpSummaryLldp )

#-------------------------------------------------------------------------------
# "show bgp summary [vrf <vrfName>]"
#-------------------------------------------------------------------------------

class ShowBgpSummary( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp summary [ VRF ]"""
   data = {
         "bgp" : bgpAfterShow,
         "summary" : RoutingBgpShowCliTokens.summary,
         "VRF" : allVrfExprFactory,
      }
   cliModel = "ArBgpCliHandler.summaryAllVrfModel"
   handler = "ArBgpCliHandler.handlerShowBgpSummary"

BasicCli.addShowCommandClass( ShowBgpSummary )

#-------------------------------------------------------------------------------
# "show bgp (ipv4 | ipv6)
#       (unicast | multicast | labeled-unicast) summary [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowBgpNlriAfSummary' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpNlriAfSummary" )

# -------------------------------------------------------------------------------
# "show bgp (ipv4 | ipv6)
#       (unicast | multicast | labeled-unicast) summary lldp [vrf <vrfName>]"
# -------------------------------------------------------------------------------
class ShowBgpNlriAfSummaryLldp ( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp AFI_SAFI summary lldp [ VRF ]"""
   data = {
         "bgp" : bgpAfterShow,
         "AFI_SAFI" : RoutingBgpShowCliTokens.AddressFamilyAfterShowBgpRule,
         "summary" : RoutingBgpShowCliTokens.summary,
         "lldp" : lldpAfterShow,
         "VRF" : allVrfExprFactory,
      }
   cliModel = summaryVrfModel
   handler = "ArBgpCliHandler.handlerShowBgpNlriAfSummaryLldp"

BasicCli.addShowCommandClass( ShowBgpNlriAfSummaryLldp )

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowIpBgpUpdateErrors' ] = \
   LazyCallback( "ArBgpCliHandler.doShowIpBgpUpdateErrors" )

arBgpShowCmdDict[ 'doShowBgpNeighbors' ] = \
   LazyCallback( "ArBgpCliHandler.doShowBgpNeighborsImpl" )
arBgpShowCmdDict[ 'doShowIpBgpNeighbors' ] = \
   lambda mode, **kwargs: \
   LazyCallback( "ArBgpCliHandler.doShowBgpNeighborsImpl" )( mode, 'ip', **kwargs )
arBgpShowCmdDict[ 'doShowIpv6BgpNeighbors' ] = \
   lambda mode, **kwargs: \
   LazyCallback( "ArBgpCliHandler.doShowBgpNeighborsImpl" )( mode, 'ipv6', **kwargs )

arBgpShowCmdDict[ 'doShowIpBgpPeerGroup' ] = \
   LazyCallback( "ArBgpCliHandler.doShowIpBgpPeerGroup" )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
# "show ipv6 bgp peers <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowIpBgpNeighborsQueuedWithdrawals' ] = \
LazyCallback( "ArBgpCliHandler.doShowIpBgpNeighborsQueuedWithdrawals" )

arBgpShowCmdDict[ 'doShowIpv6BgpNeighborsQueuedWithdrawals' ] = \
LazyCallback( "ArBgpCliHandler.doShowIpv6BgpNeighborsQueuedWithdrawals" )

#-------------------------------------------------------------------------------
# "clear ip bgp [<ip> | *] (([vrf <vrfName>] [reason MESSAGE]) |
#                          ([soft [in | out]] [vrf <vrfName>]))"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doClearIpBgp' ] = LazyCallback( "ArBgpCliHandler.doClearIpBgp" )

#-------------------------------------------------------------------------------
# "clear (ip | ipv6) bgp neighbor [*] [vrf <vrfName>] [reason MESSAGE]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doClearIpBgpTransport' ] = \
   LazyCallback( "ArBgpCliHandler.doClearIpGenBgpTransport" )

# -------------------------------------------------------------------------------
# "clear bgp route-flap-damping [PEER] [UNICAST_AFI_SAFI [PREFIX]] [VRF]"
# -------------------------------------------------------------------------------
class ClearBgpRouteFlapDamping( CliCommand.CliCommandClass ):
   syntax = 'clear bgp route-flap-damping [PEER] '\
      '[ ( ipv4 unicast [IPPREFIX] ) | ( ipv6 unicast [IP6PREFIX] ) ] [VRF]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'bgp' : 'Bgp',
      'route-flap-damping' : 'Clear route flap damping information',
      'PEER' : PeerCliExpression,
      'ipv4' : ipv4AfterShowBgp,
      'unicast' : unicastAfterShowBgp,
      'ipv6' : ipv6AfterShowBgp,
      'IPPREFIX' : IpAddrMatcher.ipPrefixMatcher,
      'IP6PREFIX' : Ip6AddrMatcher.ip6PrefixMatcher,
      'VRF' : allVrfExprFactory,
      }
   handler = "ArBgpCliHandler.handlerdoClearBgpRouteFlapDamping"

BasicCli.EnableMode.addCommandClass( ClearBgpRouteFlapDamping )

# -------------------------------------------------------------------------------
# clear bgp route priority ( <AfiSafi/VRF> | all )
# -------------------------------------------------------------------------------

class BgpAfiSafiVrfMatcher( CliCommand.CliExpression ):
   expression = ( '( evpn | '
                  '( flow-spec ( FLOWSPECIPV4 | FLOWSPECIPV6 ) ) | '
                  '( ( IPV4 | IPV6 )'
                  '  ( unicast | multicast | labeled-unicast ) [ VRF ] ) | '
                  'link-state | '
                  'mvpn-ipv4 | '
                  'rt-membership | '
                  '( sr-te ( SRTEIPV4 | SRTEIPV6 ) ) | '
                  'vpls | '
                  'vpn-ipv4 | '
                  'vpn-ipv6 )' )
   data = {
      'evpn' : 'EVPN',
      'flow-spec' : 'Flow Specification',
      'FLOWSPECIPV4' : KeywordMatcher( 'ipv4', helpdesc='IPv4 Flow Specification' ),
      'FLOWSPECIPV6' : KeywordMatcher( 'ipv6', helpdesc='IPv6 Flow Specification' ),
      'IPV4' : KeywordMatcher( 'ipv4', helpdesc='IPv4' ),
      'IPV6' : KeywordMatcher( 'ipv6', helpdesc='IPv6' ),
      'unicast' : 'Unicast',
      'multicast' : 'Multicast',
      'labeled-unicast' : 'Labeled Unicast',
      'VRF' : allVrfExprFactory,
      'link-state' : 'Link State',
      'mvpn-ipv4' : 'Multicast VPN IPv4',
      'rt-membership' : 'Route Target Membership',
      'sr-te' : 'Segment Routing Traffic Engineering',
      'SRTEIPV4' : KeywordMatcher(
         'ipv4', helpdesc='IPv4 Segment Routing Traffic Engineering' ),
      'SRTEIPV6' : KeywordMatcher(
         'ipv6', helpdesc='IPv6 Segment Routing Traffic Engineering' ),
      'vpls' : 'Virtual Private LAN Service',
      'vpn-ipv4' : 'MPLS L3 VPN IPv4 unicast',
      'vpn-ipv6' : 'MPLS L3 VPN IPv6 unicast',
   }

   @staticmethod
   def adapter( mode, args, argList ):
      afiSafi, vrfAllowed = BgpAfiSafiVrfMatcher._getAfiSafi( args )
      vrfName = args.pop( 'VRF', None ) if vrfAllowed else 'default'
      if afiSafi:
         BgpAfiSafiVrfMatcher._applyCliVrfAndSetResult( mode, args, afiSafi,
                                                        vrfName=vrfName )

   # Method needed to allow VrfExecCmdDec to override vrfName when
   # unspecified.
   @staticmethod
   @VrfCli.VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
   def _applyCliVrfAndSetResult( mode, args, afiSafi, vrfName=None ):
      args[ 'afiSafiVrf' ] = ( afiSafi.afi, afiSafi.safi, vrfName )

   @staticmethod
   def _getAfiSafi( args ):
      AfiSafi = Tac.Type( 'Routing::Bgp::AfiSafi' )
      for name, afiSafi in ( ( 'evpn', AfiSafi.evpn() ),
                             ( 'link-state', AfiSafi.linkState() ),
                             ( 'mvpn-ipv4', AfiSafi.ipv4Mvpn() ),
                             ( 'rt-membership', AfiSafi.rtMembership() ),
                             ( 'vpls', AfiSafi.vpls() ),
                             ( 'vpn-ipv4', AfiSafi.vpnv4() ),
                             ( 'vpn-ipv6', AfiSafi.vpnv6() ) ):
         if args.pop( name, None ):
            return ( afiSafi, False )
      if args.pop( 'flow-spec', None ):
         if args.pop( 'FLOWSPECIPV4', None ):
            return ( AfiSafi.ipv4Flowspec(), False )
         args.pop( 'FLOWSPECIPV6' )
         return ( AfiSafi.ipv6Flowspec(), False )
      if args.pop( 'sr-te', None ):
         if args.pop( 'SRTEIPV4', None ):
            return ( AfiSafi.ipv4SrTe(), False )
         args.pop( 'SRTEIPV6' )
         return ( AfiSafi.ipv6SrTe(), False )
      if args.pop( 'IPV4', None ):
         afi = 'afiIpv4'
      elif args.pop( 'IPV6', None ):
         afi = 'afiIpv6'
      else:
         return ( None, False )
      if args.pop( 'unicast', None ):
         safi = 'safiUnicast'
      elif args.pop( 'multicast', None ):
         safi = 'safiMulticast'
      else:
         args.pop( 'labeled-unicast' )
         safi = 'safiMplsLabels'
      return ( AfiSafi( afi, safi ), True )

class ClearBgpRoutePriority( CliCommand.CliCommandClass ):
   syntax = 'clear bgp route priority ( AFISAFIVRF | all )'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'bgp' : 'Bgp',
      'route' : 'Clear BGP route information',
      'priority' : 'Reevaluate BGP route priorities for current policy config',
      'AFISAFIVRF' : BgpAfiSafiVrfMatcher,
      'all' : 'Reevaluate route priorities for all address families and VRFs',
   }
   handler = 'ArBgpCliHandler.clearBgpRoutePriority'

if RcfLibToggleLib.toggleRcfRoutePriorityEnabled():
   BasicCli.EnableMode.addCommandClass( ClearBgpRoutePriority )

#-------------------------------------------------------------------------------
# show tech-support bgp
# [ ( [ verbose ] [ ( file | flash:<file> ) [ compress ] ] [ line-numbers ] )
# | memory
# | import-error
# | evpn
# | graceful-restart
# | rcf
# | label-management
# | link-state
# | ( iar [ drain-undrain ] ) ]
#-------------------------------------------------------------------------------
showTechBgpKw = CliMatcher.KeywordMatcher(
   'bgp',
   'Show aggregated status and configuration details for the Bgp agent' )

class ShowTechBgpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tech-support bgp
               [ ( [ verbose ] [ DUMPFILE [ compress ] ] [ line-numbers ] )
               | memory
               | import-error
               | evpn
               | graceful-restart
               | rcf
               | label-management
               | link-state
               | ( iar [ drain-undrain ] ) ]'''
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'bgp' : showTechBgpKw,
      'verbose' : 'Verbose mode',
      'DUMPFILE' : Url.UrlMatcher( lambda fs : ( fs.realFileSystem() and
                                                fs.supportsWrite() ),
                                   'send output to a file',
                                   acceptSimpleFile=False ),
      'compress' : 'Compress the output',
      'memory' : 'Show memory related info',
      'import-error' : 'Show import error counts',
      'evpn' : 'Show EVPN related state',
      'graceful-restart' : 'Graceful restart information for Bgp agent',
      'line-numbers' : 'Include line numbers in output',
      'rcf' : 'Show router control-functions related state',
      'label-management' : 'Show MPLS label manager information',
      'link-state' : 'Show BGP Link State AFI/SAFI related information',
      'iar' : 'Show IAR related state',
      'drain-undrain' : 'Show drain-undrain IAR stats for all the peers'
      }

   privileged = True

   handler = "ArBgpCliHandler.showArBgpDump"

BasicCli.addShowCommandClass( ShowTechBgpCmd )

debugKw = CliMatcher.KeywordMatcher( 'debug',
                                     'BGP debugging commands' )
ipUribKw = CliMatcher.KeywordMatcher( 'ipUrib',
                                      'IP Unicast RIB' )

class ShowTechBgpDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support bgp debug [ ipUrib ]'
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'bgp' : showTechBgpKw,
      'debug' : CliCommand.Node( matcher=debugKw, hidden=True ),
      'ipUrib' : CliCommand.Node( matcher=ipUribKw, hidden=True )
   }
   privileged = True
   handler = "ArBgpCliHandler.handlerShowTechBgpDebugCmd"

BasicCli.addShowCommandClass( ShowTechBgpDebugCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ip all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
arBgpShowCmdDict[ 'doShowMaintenanceBgpIpAll' ] = \
   LazyCallback( "ArBgpCliHandler.doShowMaintenanceBgpIpAll" )

#-------------------------------------------------------------------------------
# "show maintenance bgp ipv6 all [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowMaintenanceBgpIp6All' ] = \
   LazyCallback( "ArBgpCliHandler.doShowMaintenanceBgpIp6All" )

#-------------------------------------------------------------------------------
# "show maintenance bgp [<addr>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowMaintenanceBgp' ] = \
   LazyCallback( "ArBgpCliHandler.doShowMaintenanceBgp" )

#-------------------------------------------------------------------------------
# CliHook for "show maintenance interface <intf> detail" command
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'bgpShowMaintenanceIntf' ] = \
   LazyCallback( "ArBgpCliHandler.bgpShowMaintenanceIntf" )

#-------------------------------------------------------------------------------
# show maintenance bgp receiver route-map
#-------------------------------------------------------------------------------

arBgpShowCmdDict[ 'doShowMaintenanceBgpReceiverRm' ] = \
   LazyCallback( "ArBgpCliHandler.doShowMaintenanceBgpReceiverRm" )

class ShowBgpConfiguration( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp configuration ( active | unsupported | detail )"""
   data = {
         "bgp" : bgpAfterShow,
         "configuration" : RoutingBgpShowCliTokens.bgpConfig,
         "active" : active,
         "unsupported" : unsupported,
         "detail" : detailConfig,
      }
   privileged = True
   handler = "ArBgpCliHandler.handlerShowBgpConfiguration"

BasicCli.addShowCommandClass( ShowBgpConfiguration )

class CommonHistoryTokens:
   '''
   Shared tokens and behavior between the history commands below.
   '''

   history = CliMatcher.KeywordMatcher( 'history', helpdesc='Neighbor history' )
   historyScope = CliCommand.setCliExpression(
      { 'connect-failures' : 'Connection failures' }, name='HISTORY_SCOPE' )

#------------------------------------------------------------------------------------
# show bgp neighbors [ <ip> | <prefix> | ( peer-group <peerGroupName> ) ]
#                    history [ connect-failures ] [ ( vrf < vrfName > ) ]
# -- See AID5484 for a description of this command
#-----------------------------------------------------------------------------------
class ShowBgpNeighborsHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp neighbors [ PEER | PREFIX | PREFIX6 | ' \
            '( peer-group PEERGROUP_NAME ) ] history [ HISTORY_SCOPE ]' \
            '[ VRF ]'

   data = {
      'bgp' : CommonTokens.showBgp,
      'neighbors' : CommonTokens.neighbors,
      'PEER' : V4V6PeerKeyCliExpression,
      'PREFIX' : CommonTokens.neighborPrefix,
      'PREFIX6' : CommonTokens.neighborPrefix6,
      'peer-group' : CommonTokens.peerGroup,
      'PEERGROUP_NAME' : CommonTokens.peerGroupName,
      'history' : CommonHistoryTokens.history,
      'HISTORY_SCOPE' : CommonHistoryTokens.historyScope,
      'VRF' : CommonTokens.vrf,
   }

   cliModel = ShowNeighborsHistoryModel
   handler = "ArBgpCliHandler.handlerShowBgpNeighborsHistoryCmd"

#-----------------------------------------------------------------------------
# show bgp neighbors history socket
#-----------------------------------------------------------------------------
class ShowBgpNeighborsHistorySocketCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp neighbors history socket'

   data = {
      'bgp' : CommonTokens.showBgp,
      'neighbors' : CommonTokens.neighbors,
      'history' : CommonHistoryTokens.history,
      'socket' : 'TCP socket statistics',
   }

   # need to return a cliModel for show command
   cliModel = ShowNeighborsHistorySocketModel
   handler = "ArBgpCliHandler.handlerShowBgpNeighborsHistorySocketCmd"

#-----------------------------------------------------------------------------
# clear bgp [ ip | ip6 | prefix | prefix6 | peer-group peerGroupName ] history
#           [ { connect-failures } ] [ vrf vrfname ]
# -- See AID5484 for a description of this command
#-----------------------------------------------------------------------------
class ClearBgpNeighborsHistoryCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bgp [ PEER | PREFIX | PREFIX6 | ' \
            '( peer-group PEERGROUP_NAME ) ] history [ HISTORY_SCOPE ] ' \
            '[ VRF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'bgp' : 'Bgp',
      'PEER' : V4V6PeerKeyCliExpression,
      'PREFIX' : CommonTokens.neighborPrefix,
      'PREFIX6' : CommonTokens.neighborPrefix6,
      'peer-group' : CommonTokens.peerGroup,
      'PEERGROUP_NAME' : CommonTokens.peerGroupName,
      'history' : CommonHistoryTokens.history,
      'HISTORY_SCOPE' : CommonHistoryTokens.historyScope,
      'VRF' : CommonTokens.vrf,
   }

   handler = "ArBgpCliHandler.handlerClearBgpNeighborsHistoryCmd"

BasicCli.addShowCommandClass( ShowBgpNeighborsHistoryCmd )
BasicCli.addShowCommandClass( ShowBgpNeighborsHistorySocketCmd )
BasicCli.EnableMode.addCommandClass( ClearBgpNeighborsHistoryCmd )

#---------------------------------------------------------------------------
# show dynamic prefix-list [ <dynamic prefix-list name> ] [ vrf <vrf-name> ]
#---------------------------------------------------------------------------
routeMapShowCmdDict[ 'doArBgpShowDynPfxList' ] = \
   LazyCallback( 'ArBgpCliHandler.doArBgpShowDynPfxList' )

# ----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] match rcf <rcf-function-name>()
# ----------------------------------------------------------------------------------

class DynamicPrefixListConfigModeMatchRcf( CliCommand.CliCommandClass ):
   syntax = 'match rcf FUNCTION'
   noOrDefaultSyntax = syntax
   data = {
      'match' : 'Configuration to determine the state of the dynamic prefix list',
      'rcf' : 'Routing control function configuration to evaluate state of routes',
      'FUNCTION' : RcfCliLib.rcfFunctionMatcher,
   }

   handler = "ArBgpCliHandler.handlerDynamicPrefixListConfigModeMatchRcf"
   noOrDefaultHandler = "ArBgpCliHandler." \
      "noOrDefaultHandlerDynamicPrefixListConfigModeMatchRcf"

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModeMatchRcf )

#-------------------------------------------------------------------------------
# agent Bgp snapshot core
#-------------------------------------------------------------------------------
def bgpAgentSnapshotCore( mode, ns ):
   entityManager = mode.entityManager
   command = 'snapshot core'
   AgentCommandRequest.runSocketCommand( entityManager, BgpAgentName,
                                         command,
                                         '' )

snapshotCoreCallBack[ 'Bgp' ] = bgpAgentSnapshotCore

#-------------------------------------------------------------------------------
# "show bgp med consistency [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowBgpMedConsistency( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp med consistency [ VRF ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'med' : CliCommand.Node( CliMatcher.KeywordMatcher( 'med',
                                                          helpdesc='BGP MED' ),
                               hidden=True ),
      'consistency' : CliCommand.Node(
         CliMatcher.KeywordMatcher( 'consistency', helpdesc='Check consistency' ),
         hidden=True ),
      "VRF" : allVrfExprFactory,
   }

   handler = "ArBgpCliHandler.handlerShowBgpMedConsistency"

BasicCli.addShowCommandClass( ShowBgpMedConsistency )

#-------------------------------------------------------------------------------
# show bgp rcf [ <func-name> ]
#-------------------------------------------------------------------------------

class ShowBgpRcf( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rcf [ FUNC_NAME ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'rcf' : RcfCliLib.rcfKw,
      'FUNC_NAME' : RcfCliLib.rcfFunctionMatcherParensOptional,
   }

   cliModel = ShowAgentRcfModel

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         return doShowBgpOutputNotSupported( mode, *args )
      cmd = ArBgpAsyncCliCommand( mode, 'show bgp rcf' )
      if 'FUNC_NAME' in args:
         funcNameParam = RcfCliLib.stripParens( args[ 'FUNC_NAME' ] )
         cmd.addParam( 'functionName', funcNameParam )
      ret = cmd.run( model=ShowBgpRcf.cliModel )
      if ret is None:
         return None
      return cliPrinted( ShowBgpRcf.cliModel )

BasicCli.addShowCommandClass( ShowBgpRcf )

# -------------------------------------------------------------------------------
# show bgp rcf detail [ <func-name> ]
# -------------------------------------------------------------------------------

class ShowBgpRcfDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rcf detail [ FUNC_NAME ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'rcf' : RcfCliLib.rcfKw,
      'detail' : detail,
      'FUNC_NAME' : RcfCliLib.rcfFunctionMatcherParensOptional,
   }
   cliModel = ShowAgentRcfDetailModel
   handler = "ArBgpCliHandler.handlerShowBgpRcfDetailCmd"

if RcfLibToggleLib.toggleRcfShowAgentRcfDetailEnabled():
   BasicCli.addShowCommandClass( ShowBgpRcfDetail )

# -------------------------------------------------------------------------------
# "[no|default] agent bgp threads ( single | ( bestpath BESTPATH_THREADS input
#  INPUT_THREADS ) )" command, in global configuration mode.
# -------------------------------------------------------------------------------
class SetThreadConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'agent bgp threads \
             ( single | ( bestpath BESTPATH_THREADS input INPUT_THREADS ) )'
   noOrDefaultSyntax = 'agent bgp threads ...'
   data = {
      'agent' : agentKwForConfig,
      'bgp' : bgpTokens.bgp,
      'threads' : arBgpTokens.threads,
      'single' : arBgpTokens.singleThreaded,
      'bestpath' : arBgpTokens.bestpathThreads,
      'BESTPATH_THREADS' : arBgpTokens.numBestpathThreadsRangeMatcher,
      'input' : arBgpTokens.inputThreads,
      'INPUT_THREADS' : arBgpTokens.numInputThreadsRangeMatcher,
   }
   hidden = True
   handler = "ArBgpCliHandler.handlerSetThreadConfigCmd"
   noOrDefaultHandler = "ArBgpCliHandler.noOrDefaultHandlerSetThreadConfigCmd"

BasicCliModes.GlobalConfigMode.addCommandClass( SetThreadConfigCmd )

# -------------------------------------------------------------------------------
# show bgp session tracker [ TRACKER_NAME ]
# -------------------------------------------------------------------------------

class ShowBgpSessionTracker( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp session tracker [ TRACKER_NAME ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'session' : 'BGP session state data',
      'tracker' : 'session tracking data for BGP peers',
      'TRACKER_NAME' : sessionTrackerNameMatcher,
   }
   cliModel = "ArBgpCliModels.BgpSessionTrackerAllModel"
   handler = "ArBgpCliHandler.handlerShowBgpSessionTracker"

BasicCli.addShowCommandClass( ShowBgpSessionTracker )

class ShowBgpRfdPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp route-flap-damping policy [ POLICY_NAME ]'
   data = {
      'bgp' : bgpAfterShow,
      'route-flap-damping' : routeFlapDamping,
      'policy' : policyInfo,
      'POLICY_NAME' : rfdPolicyNameMatcher,
   }
   cliModel = BgpRfdPolicies
   handler = "ArBgpCliHandler.doShowBgpRfdPolicy"

BasicCli.addShowCommandClass( ShowBgpRfdPolicy )

# --------------------------------------------------------------------------------
# "[ no | default ] next-hop resolution route igp-nexthop-cost protocol bgp"
# command, in "router-bgp" mode.
# --------------------------------------------------------------------------------
class SetNhResRouteIgpNhCostBgpCmd( BgpCmdBaseClass ):
   syntax = 'next-hop resolution route igp-nexthop-cost protocol bgp'
   noOrDefaultSyntax = 'next-hop resolution route ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'next-hop' : bgpTokens.neighborNextHop,
      'resolution' : 'Next hop resolution',
      'route' : 'Use resolving route',
      'igp-nexthop-cost' : 'IGP next hop cost',
      'protocol' : 'Protocol of resolving routes',
      'bgp' : 'Use BGP resolving routes',
   } )
   handler = "ArBgpCliHandler.handlerSetNhResRouteIgpNhCostBgpCmd"
   noOrDefaultHandler = \
      "ArBgpCliHandler.noOrDefaultHandlerSetNhResRouteIgpNhCostBgpCmd"

if RouterGeneralCliToggleLib.toggleNhResolutionRouteIgpNhCostEnabled():
   RouterBgpBaseMode.addCommandClass( SetNhResRouteIgpNhCostBgpCmd )

gv = CliGlobal( dict( fieldSetNameToIdMap=None ) )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, vrfConfigDir, mapConfig
   global asnConfig
   global securityConfig
   global vrfNameStatus
   global rtrGeneralConfigDir, peerInfoTable
   global sessionTrackerConfigDir
   global onBootLinkEnableOverrideConfig
   global startupConfigStatus

   config = LazyMount.mount( entityManager, 'routing/bgp/config',
                             'Routing::Bgp::Config', 'r' )
   vrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                   'Routing::Bgp::VrfConfigDir', 'r' )
   mapConfig = LazyMount.mount( entityManager, 'routing/routemap/config',
                                'Routing::RouteMap::Config', 'r' )
   asnConfig = LazyMount.mount( entityManager, 'routing/bgp/asn/config',
                                  'Routing::AsnConfig', 'r' )
   securityConfig = LazyMount.mount( entityManager, 'mgmt/security/config',
                                    'Mgmt::Security::Config', 'r' )

   # initialize the global ext comm parser state machinery, or get the existing one
   ArBgpCliCommon.getBgpExtCommParserContainerSm()

   # Mounts for dynamic prefix list, maintenance bgp show commands
   vrfNameStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   rtrGeneralConfigDir = LazyMount.mount( entityManager,
                                          'routing/general/config/dynPolicyRoutes',
                                          'Routing::DynPolicyRoutes::Config', 'r' )
   peerInfoTable = LazyMount.mount(
         entityManager,
         Cell.path( 'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' ),
         'Tac::Dir', 'ri' )
   sessionTrackerConfigDir = ConfigMount.mount(
      entityManager, 'routing/arbgp/sessionTracker/inst',
      'Routing::Bgp::SessionTrackerCliConfigDir', 'w' )
   onBootLinkEnableOverrideConfig = ConfigMount.mount(
      entityManager, 'interface/shutdown/onboot/config',
      'Interface::OnBootLinkEnableOverrideConfig', 'w' )
   startupConfigStatus = LazyMount.mount(
      entityManager, 'Sysdb/status', 'Sysdb::Status', 'r' )
   gv.fieldSetNameToIdMap = LazyMount.mount(
      entityManager,
      'routing/bgp/trafficpolicy/fieldSetNameToIdMap',
      'Routing::Bgp::FieldSetNameToIdMap',
      'r' )
