#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
from ArnetModel import Ip4Address
from CliModel import Enum
from CliModel import GeneratorDict
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliPlugin.BgpCliModels import generateAfiSafiModel, peerKeyFromData, stateHelp
from Toggles.BgpCommonToggleLib import toggleArBgpShowSummaryPfxAdvEnabled

##########
# Models for "show bgp summary"
##########
class BgpSummaryAllPeerAfiSafiEntry( Model ):
   afiSafiState = Enum( values=( 'advertised', 'configured', 'negotiated',
                                 'received' ),
                        help='current afiSafi state' )
   nlrisAccepted = Int( help='Number of nlris accepted from the peer' )
   nlrisReceived = Int( help='Number of nlris receieved from the peer' )
   if toggleArBgpShowSummaryPfxAdvEnabled():
      nlrisAdvertised = Int( help='Number of nlris advertised to the peer' )

BgpSummaryAllPeerEntryBase_ = generateAfiSafiModel( 'BgpSummaryAllPeerEntryBase_',
      Submodel, 'Peer information for %s',
      valueType=BgpSummaryAllPeerAfiSafiEntry, optional=True )

class BgpSummaryAllPeerEntry( BgpSummaryAllPeerEntryBase_ ):
   peerState = Enum( values=( 'Idle', 'Connect', 'Active', 'OpenSent',
                              'OpenConfirm', 'Established', 'NotNegotiated' ),
                     help=stateHelp )
   peerAsn = Str( help='Autonomous System Number' )

   def getKey( self, data ):
      return peerKeyFromData( data )

helpStr = 'Dictionary of BGP peer entries indexed by the peer address'

class BgpSummaryAll( Model ):
   asn = Str( help='Autonomous System Number' )
   routerId = Ip4Address( help='BGP Router Identity' )
   vrf = Str( help='VRF Name' )
   peers = GeneratorDict( valueType=BgpSummaryAllPeerEntry, help=helpStr )

class BgpRfdPolicy( Model ):
   attributeChangePenalty = Int( help="Penalty for path attribute change" )
   withdrawalPenalty = Int( help="Penalty for path withdrawal" )
   readvertisementAfterWithdrawalPenalty = \
         Int( help="Penalty for path readvertisement after it is withdrawn" )
   reusePenaltyThreshold = Int(
         help="Value of penalty below which suppressed path would be reused" )
   suppressionPenaltyThreshold = Int(
         help="Value of penalty above which path would be suppressed" )
   maximumPenaltyThreshold = Int( help="Maximum value of penalty imposed on a path" )
   penaltyDecayHalfLife = Int( help="Penalty decay half life in seconds" )
   suppressionIgnorePrefixListV4Unicast = \
        Str( help="Prefix list of IPv4 unicast prefixes ignored from suppression",
             optional=True )
   suppressionIgnorePrefixListV6Unicast = \
        Str( help="Prefix list of IPv6 unicast prefixes ignored from suppression",
             optional=True )
   maximumSuppressionTime = Int(
         help="Maximum time a path is suppressed after it is stable in seconds" )
   minimumSuppressionTime = Int(
         help="Minimum time a path is suppressed after it is stable in seconds" )

   def render( self ):
      print( "Attribute change penalty: {}".
              format( self.attributeChangePenalty ) )
      print( f"Withdrawal penalty: {self.withdrawalPenalty}" )
      print( "Readvertisement after withdrawal penalty: {}".
              format( self.readvertisementAfterWithdrawalPenalty ) )
      print( f"Reuse penalty threshold: {self.reusePenaltyThreshold}" )
      print( "Suppression penalty threshold: {}".
              format( self.suppressionPenaltyThreshold ) )
      print( f"Maximum penalty threshold: {self.maximumPenaltyThreshold}" )
      print( "Penalty decay half life: {} seconds".
              format( self.penaltyDecayHalfLife ) )
      if self.suppressionIgnorePrefixListV4Unicast:
         print( "Suppression ignore IPv4 prefix list: {}".
               format( self.suppressionIgnorePrefixListV4Unicast ) )
      if self.suppressionIgnorePrefixListV6Unicast:
         print( "Suppression ignore IPv6 prefix list: {}".
               format( self.suppressionIgnorePrefixListV6Unicast ) )
      print( "Maximum suppression time with no flaps: {} seconds".
              format( self.maximumSuppressionTime ) )
      print( "Minimum suppression time with no flaps: {} seconds\n".
             format( self.minimumSuppressionTime ) )

class BgpRfdPolicies( Model ):
   policies = GeneratorDict( keyType=str, valueType=BgpRfdPolicy,
              help="Dictionary of route flap damping policies keyed by policy name" )

   def render( self ):
      for policyName, rfdPolicy in self.policies:
         print( f"Route flap damping policy: {policyName}" )
         rfdPolicy.render()
